/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.AbstractScheduledService;
import com.google.common.util.concurrent.Service;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.ObjectClosedException;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.concurrent.Services;
import io.pravega.segmentstore.server.CachePolicy;
import io.pravega.segmentstore.server.CacheUtilizationProvider;
import io.pravega.segmentstore.server.SegmentStoreMetrics;
import io.pravega.segmentstore.storage.ThrottleSourceListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class CacheManager
extends AbstractScheduledService
implements AutoCloseable,
CacheUtilizationProvider {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(CacheManager.class);
    private static final String TRACE_OBJECT_ID = "CacheManager";
    @GuardedBy(value="clients")
    private final Collection<Client> clients;
    private final ScheduledExecutorService executorService;
    private final AtomicInteger currentGeneration;
    private final AtomicInteger oldestGeneration;
    private final AtomicLong cacheSize;
    private final CachePolicy policy;
    private final AtomicBoolean closed;
    private final SegmentStoreMetrics.CacheManager metrics;
    @GuardedBy(value="cleanupListeners")
    private final HashSet<ThrottleSourceListener> cleanupListeners;

    public CacheManager(CachePolicy policy, ScheduledExecutorService executorService) {
        Preconditions.checkNotNull((Object)policy, (Object)"policy");
        Preconditions.checkNotNull((Object)executorService, (Object)"executorService");
        this.policy = policy;
        this.clients = new HashSet<Client>();
        this.oldestGeneration = new AtomicInteger();
        this.currentGeneration = new AtomicInteger();
        this.cacheSize = new AtomicLong();
        this.executorService = executorService;
        this.closed = new AtomicBoolean();
        this.metrics = new SegmentStoreMetrics.CacheManager();
        this.cleanupListeners = new HashSet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        if (!this.closed.getAndSet(true)) {
            if (this.state() == Service.State.RUNNING) {
                Futures.await((CompletableFuture)Services.stopAsync((Service)this, (Executor)this.executorService));
            }
            Collection<Client> collection = this.clients;
            synchronized (collection) {
                this.clients.clear();
            }
            this.metrics.close();
            log.info("{} Closed.", (Object)TRACE_OBJECT_ID);
        }
    }

    protected ScheduledExecutorService executor() {
        return this.executorService;
    }

    protected void runOneIteration() {
        if (this.closed.get()) {
            return;
        }
        try {
            boolean anythingEvicted = this.applyCachePolicy();
            if (anythingEvicted) {
                this.notifyCleanupListeners();
            }
        }
        catch (Throwable ex) {
            if (Exceptions.mustRethrow((Throwable)ex)) {
                throw ex;
            }
            log.error("{}: Error.", (Object)TRACE_OBJECT_ID, (Object)ex);
        }
    }

    protected AbstractScheduledService.Scheduler scheduler() {
        long millis = this.policy.getGenerationDuration().toMillis();
        return AbstractScheduledService.Scheduler.newFixedDelaySchedule((long)millis, (long)millis, (TimeUnit)TimeUnit.MILLISECONDS);
    }

    @Override
    public double getCacheUtilization() {
        return (double)this.cacheSize.get() / (double)this.policy.getMaxSize();
    }

    @Override
    public double getCacheTargetUtilization() {
        return this.policy.getTargetUtilization();
    }

    @Override
    public double getCacheMaxUtilization() {
        return this.policy.getMaxUtilization();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void registerCleanupListener(@NonNull ThrottleSourceListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener is marked @NonNull but is null");
        }
        if (listener.isClosed()) {
            log.warn("{} Attempted to register a closed ThrottleSourceListener ({}).", (Object)TRACE_OBJECT_ID, (Object)listener);
            return;
        }
        HashSet<ThrottleSourceListener> hashSet = this.cleanupListeners;
        synchronized (hashSet) {
            this.cleanupListeners.add(listener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void register(Client client) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkNotNull((Object)client, (Object)"client");
        Collection<Client> collection = this.clients;
        synchronized (collection) {
            if (!this.clients.contains(client)) {
                this.clients.add(client);
                client.updateGenerations(this.currentGeneration.get(), this.oldestGeneration.get());
            }
        }
        log.info("{} Registered {}.", (Object)TRACE_OBJECT_ID, (Object)client);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void unregister(Client client) {
        if (this.closed.get()) {
            return;
        }
        Preconditions.checkNotNull((Object)client, (Object)"client");
        Collection<Client> collection = this.clients;
        synchronized (collection) {
            this.clients.remove(client);
        }
        log.info("{} Unregistered {}.", (Object)TRACE_OBJECT_ID, (Object)client);
    }

    protected boolean applyCachePolicy() {
        long sizeReduction;
        CacheStatus currentStatus = this.collectStatus();
        if (currentStatus == null || currentStatus.getSize() == 0L) {
            this.cacheSize.set(0L);
            return false;
        }
        boolean currentChanged = this.adjustCurrentGeneration(currentStatus);
        boolean oldestChanged = this.adjustOldestGeneration(currentStatus);
        if (!currentChanged && !oldestChanged) {
            return false;
        }
        boolean reducedOverall = false;
        do {
            if ((sizeReduction = this.updateClients()) <= 0L) continue;
            currentStatus = currentStatus.withUpdatedSize(-sizeReduction);
            this.logCurrentStatus(currentStatus);
            oldestChanged = this.adjustOldestGeneration(currentStatus);
            reducedOverall = true;
        } while (sizeReduction > 0L && oldestChanged);
        this.cacheSize.set(currentStatus.getSize());
        this.metrics.report(currentStatus.getSize(), currentStatus.getNewestGeneration() - currentStatus.getOldestGeneration());
        return reducedOverall;
    }

    private CacheStatus collectStatus() {
        int cg;
        int minGeneration = cg = this.currentGeneration.get();
        int maxGeneration = 0;
        long totalSize = 0L;
        Collection<Client> clients = this.getCurrentClients();
        for (Client c : clients) {
            CacheStatus clientStatus;
            try {
                clientStatus = c.getCacheStatus();
            }
            catch (ObjectClosedException ex) {
                log.warn("{} Detected closed client {}.", (Object)TRACE_OBJECT_ID, (Object)c);
                this.unregister(c);
                continue;
            }
            if (clientStatus.getSize() == 0L) continue;
            totalSize += clientStatus.getSize();
            if (clientStatus.oldestGeneration > cg || clientStatus.newestGeneration > cg) {
                log.warn("{} Client {} returned status that is out of bounds {}. CurrentGeneration = {}, OldestGeneration = {}.", new Object[]{TRACE_OBJECT_ID, c, clientStatus, this.currentGeneration, this.oldestGeneration});
            }
            minGeneration = Math.min(minGeneration, clientStatus.oldestGeneration);
            maxGeneration = Math.max(maxGeneration, clientStatus.newestGeneration);
        }
        if (minGeneration > maxGeneration) {
            return null;
        }
        return new CacheStatus(totalSize, minGeneration, maxGeneration);
    }

    private long updateClients() {
        long sizeReduction = 0L;
        int cg = this.currentGeneration.get();
        int og = this.oldestGeneration.get();
        for (Client c : this.getCurrentClients()) {
            try {
                sizeReduction += Math.max(0L, c.updateGenerations(cg, og));
            }
            catch (ObjectClosedException ex) {
                log.warn("{} Detected closed client {}.", (Object)TRACE_OBJECT_ID, (Object)c);
                this.unregister(c);
            }
            catch (Throwable ex) {
                if (Exceptions.mustRethrow((Throwable)ex)) {
                    throw ex;
                }
                log.warn("{} Unable to update client {}. {}", new Object[]{TRACE_OBJECT_ID, c, ex});
            }
        }
        return sizeReduction;
    }

    private boolean adjustCurrentGeneration(CacheStatus currentStatus) {
        boolean shouldIncrement;
        boolean bl = shouldIncrement = currentStatus.getNewestGeneration() >= this.currentGeneration.get();
        if (shouldIncrement) {
            this.currentGeneration.incrementAndGet();
        }
        return shouldIncrement;
    }

    private boolean adjustOldestGeneration(CacheStatus currentStatus) {
        boolean isAdjusted;
        int newOldestGeneration = this.oldestGeneration.get();
        if (this.exceedsPolicy(currentStatus)) {
            newOldestGeneration = Math.max(newOldestGeneration, currentStatus.oldestGeneration) + 1;
            newOldestGeneration = Math.max(newOldestGeneration, this.getOldestPermissibleGeneration());
            newOldestGeneration = Math.min(newOldestGeneration, this.currentGeneration.get());
        }
        boolean bl = isAdjusted = newOldestGeneration > this.oldestGeneration.get();
        if (isAdjusted) {
            this.oldestGeneration.set(newOldestGeneration);
        }
        return isAdjusted;
    }

    private boolean exceedsPolicy(CacheStatus currentStatus) {
        return currentStatus.getSize() > this.policy.getEvictionThreshold() || currentStatus.getOldestGeneration() < this.getOldestPermissibleGeneration();
    }

    private int getOldestPermissibleGeneration() {
        return this.currentGeneration.get() - this.policy.getMaxGenerations() + 1;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Collection<Client> getCurrentClients() {
        Collection<Client> collection = this.clients;
        synchronized (collection) {
            return new ArrayList<Client>(this.clients);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void logCurrentStatus(CacheStatus status) {
        int size;
        Collection<Client> collection = this.clients;
        synchronized (collection) {
            size = this.clients.size();
        }
        log.info("{} Current Generation = {}, Oldest Generation = {}, Clients = {},  CacheSize = {} MB", new Object[]{TRACE_OBJECT_ID, this.currentGeneration, this.oldestGeneration, size, status.getSize() / 0x100000L});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void notifyCleanupListeners() {
        ArrayList<ThrottleSourceListener> toNotify = new ArrayList<ThrottleSourceListener>();
        ArrayList<ThrottleSourceListener> toRemove = new ArrayList<ThrottleSourceListener>();
        HashSet<ThrottleSourceListener> hashSet = this.cleanupListeners;
        synchronized (hashSet) {
            for (ThrottleSourceListener l : this.cleanupListeners) {
                if (l.isClosed()) {
                    toRemove.add(l);
                    continue;
                }
                toNotify.add(l);
            }
            this.cleanupListeners.removeAll(toRemove);
        }
        for (ThrottleSourceListener l : toNotify) {
            try {
                l.notifyThrottleSourceChanged();
            }
            catch (Throwable ex) {
                if (Exceptions.mustRethrow((Throwable)ex)) {
                    throw ex;
                }
                log.error("{}: Error while notifying cleanup listener {}.", new Object[]{TRACE_OBJECT_ID, l, ex});
            }
        }
    }

    public static class CacheStatus {
        private final int oldestGeneration;
        private final int newestGeneration;
        private final long size;

        public CacheStatus(long size, int oldestGeneration, int newestGeneration) {
            Preconditions.checkArgument((size >= 0L ? 1 : 0) != 0, (Object)"size must be a non-negative number");
            Preconditions.checkArgument((oldestGeneration >= 0 ? 1 : 0) != 0, (Object)"oldestGeneration must be a non-negative number");
            Preconditions.checkArgument((newestGeneration >= oldestGeneration ? 1 : 0) != 0, (Object)"newestGeneration must be larger than or equal to oldestGeneration");
            this.size = size;
            this.oldestGeneration = oldestGeneration;
            this.newestGeneration = newestGeneration;
        }

        private CacheStatus withUpdatedSize(long sizeDelta) {
            long newSize = Math.max(0L, this.size + sizeDelta);
            return new CacheStatus(newSize, this.oldestGeneration, this.newestGeneration);
        }

        public String toString() {
            return String.format("Size = %d, OG-NG = %d-%d", this.size, this.oldestGeneration, this.newestGeneration);
        }

        @SuppressFBWarnings(justification="generated code")
        public int getOldestGeneration() {
            return this.oldestGeneration;
        }

        @SuppressFBWarnings(justification="generated code")
        public int getNewestGeneration() {
            return this.newestGeneration;
        }

        @SuppressFBWarnings(justification="generated code")
        public long getSize() {
            return this.size;
        }
    }

    public static interface Client {
        public CacheStatus getCacheStatus();

        public long updateGenerations(int var1, int var2);
    }
}

