/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.attributes;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.TimeoutTimer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.function.Callbacks;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.AsyncIterator;
import io.pravega.common.util.BitConverter;
import io.pravega.common.util.BufferView;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.IllegalDataFormatException;
import io.pravega.common.util.Retry;
import io.pravega.common.util.btree.BTreeIndex;
import io.pravega.common.util.btree.PageEntry;
import io.pravega.segmentstore.contracts.BadOffsetException;
import io.pravega.segmentstore.contracts.SegmentProperties;
import io.pravega.segmentstore.contracts.StreamSegmentExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentNotExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentSealedException;
import io.pravega.segmentstore.contracts.StreamSegmentTruncatedException;
import io.pravega.segmentstore.server.AttributeIndex;
import io.pravega.segmentstore.server.AttributeIterator;
import io.pravega.segmentstore.server.CacheManager;
import io.pravega.segmentstore.server.DataCorruptionException;
import io.pravega.segmentstore.server.SegmentMetadata;
import io.pravega.segmentstore.server.attributes.AttributeIndexConfig;
import io.pravega.segmentstore.server.attributes.CacheKey;
import io.pravega.segmentstore.storage.Cache;
import io.pravega.segmentstore.storage.SegmentHandle;
import io.pravega.segmentstore.storage.Storage;
import io.pravega.shared.segment.StreamSegmentNameUtils;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentAttributeBTreeIndex
implements AttributeIndex,
CacheManager.Client,
AutoCloseable {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(SegmentAttributeBTreeIndex.class);
    private static final Retry.RetryAndThrowBase<Exception> UPDATE_RETRY = Retry.withExpBackoff((long)10L, (int)2, (int)10, (long)1000L).retryingOn(BadOffsetException.class).throwingOn(Exception.class);
    private static final Retry.RetryAndThrowBase<Exception> READ_RETRY = Retry.withExpBackoff((long)10L, (int)2, (int)10, (long)1000L).retryingOn(StreamSegmentTruncatedException.class).throwingOn(Exception.class);
    private static final int KEY_LENGTH = 16;
    private static final int VALUE_LENGTH = 8;
    private final SegmentMetadata segmentMetadata;
    private final AtomicReference<SegmentHandle> handle;
    private final Storage storage;
    private final Cache cache;
    @GuardedBy(value="cacheEntries")
    private int currentCacheGeneration;
    @GuardedBy(value="cacheEntries")
    private final Map<Long, CacheEntry> cacheEntries;
    private final BTreeIndex index;
    private final AttributeIndexConfig config;
    private final ScheduledExecutorService executor;
    private final String traceObjectId;
    private final AtomicBoolean closed;

    SegmentAttributeBTreeIndex(@NonNull SegmentMetadata segmentMetadata, @NonNull Storage storage, @NonNull Cache cache, @NonNull AttributeIndexConfig config, @NonNull ScheduledExecutorService executor) {
        if (segmentMetadata == null) {
            throw new NullPointerException("segmentMetadata is marked @NonNull but is null");
        }
        if (storage == null) {
            throw new NullPointerException("storage is marked @NonNull but is null");
        }
        if (cache == null) {
            throw new NullPointerException("cache is marked @NonNull but is null");
        }
        if (config == null) {
            throw new NullPointerException("config is marked @NonNull but is null");
        }
        if (executor == null) {
            throw new NullPointerException("executor is marked @NonNull but is null");
        }
        this.segmentMetadata = segmentMetadata;
        this.storage = storage;
        this.cache = cache;
        this.config = config;
        this.executor = executor;
        this.handle = new AtomicReference();
        this.index = BTreeIndex.builder().keyLength(16).valueLength(8).maxPageSize(this.config.getMaxIndexPageSize()).executor((Executor)this.executor).getLength(this::getLength).readPage(this::readPage).writePages(this::writePages).build();
        this.cacheEntries = new HashMap<Long, CacheEntry>();
        this.traceObjectId = String.format("AttributeIndex[%d-%d]", this.segmentMetadata.getContainerId(), this.segmentMetadata.getId());
        this.closed = new AtomicBoolean();
    }

    CompletableFuture<Void> initialize(Duration timeout) {
        TimeoutTimer timer = new TimeoutTimer(timeout);
        Preconditions.checkState((!this.index.isInitialized() ? 1 : 0) != 0, (Object)"SegmentAttributeIndex is already initialized.");
        String attributeSegmentName = StreamSegmentNameUtils.getAttributeSegmentName((String)this.segmentMetadata.getName());
        return ((CompletableFuture)((CompletableFuture)Futures.exceptionallyExpecting((CompletableFuture)this.storage.openWrite(attributeSegmentName).thenAccept(this.handle::set), ex -> ex instanceof StreamSegmentNotExistsException, null).thenComposeAsync(v -> this.index.initialize(timer.getRemaining()), (Executor)this.executor)).thenRun(() -> log.debug("{}: Initialized.", (Object)this.traceObjectId))).exceptionally(this::handleIndexOperationException);
    }

    static CompletableFuture<Void> delete(String segmentName, Storage storage, Duration timeout) {
        TimeoutTimer timer = new TimeoutTimer(timeout);
        String attributeSegmentName = StreamSegmentNameUtils.getAttributeSegmentName((String)segmentName);
        return Futures.exceptionallyExpecting((CompletableFuture)storage.openWrite(attributeSegmentName).thenCompose(handle -> storage.delete(handle, timer.getRemaining())), ex -> ex instanceof StreamSegmentNotExistsException, null);
    }

    @Override
    public void close() {
        this.close(false);
    }

    void close(boolean cleanCache) {
        if (!this.closed.getAndSet(true)) {
            if (cleanCache) {
                this.executor.execute(() -> {
                    this.removeAllCacheEntries();
                    log.info("{}: Closed.", (Object)this.traceObjectId);
                });
            } else {
                log.info("{}: Closed (no cache cleanup).", (Object)this.traceObjectId);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void removeAllCacheEntries() {
        ArrayList<CacheEntry> entries;
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            entries = new ArrayList<CacheEntry>(this.cacheEntries.values());
            this.cacheEntries.clear();
        }
        this.removeFromCache(entries);
        if (entries.size() > 0) {
            log.debug("{}: Cleared all cache entries ({}).", (Object)this.traceObjectId, (Object)entries.size());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CacheManager.CacheStatus getCacheStatus() {
        int minGen = 0;
        int maxGen = 0;
        long size = 0L;
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            for (CacheEntry e : this.cacheEntries.values()) {
                if (e == null) continue;
                int g = e.getGeneration();
                minGen = Math.min(minGen, g);
                maxGen = Math.max(maxGen, g);
                size += (long)e.getSize();
            }
        }
        return new CacheManager.CacheStatus(size, minGen, maxGen);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long updateGenerations(int currentGeneration, int oldestGeneration) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long sizeRemoved = 0L;
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            this.currentCacheGeneration = currentGeneration;
            ArrayList<CacheEntry> toRemove = new ArrayList<CacheEntry>();
            for (CacheEntry entry : this.cacheEntries.values()) {
                if (entry.getGeneration() >= oldestGeneration) continue;
                sizeRemoved += (long)entry.getSize();
                toRemove.add(entry);
            }
            this.removeFromCache(toRemove);
        }
        return sizeRemoved;
    }

    @Override
    public CompletableFuture<Void> update(@NonNull Map<UUID, Long> values, @NonNull Duration timeout) {
        if (values == null) {
            throw new NullPointerException("values is marked @NonNull but is null");
        }
        if (timeout == null) {
            throw new NullPointerException("timeout is marked @NonNull but is null");
        }
        this.ensureInitialized();
        if (values.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        Collection entries = values.entrySet().stream().map(this::serialize).collect(Collectors.toList());
        return this.executeConditionally(tm -> this.index.update(entries, tm), timeout);
    }

    @Override
    public CompletableFuture<Map<UUID, Long>> get(@NonNull Collection<UUID> keys, @NonNull Duration timeout) {
        if (keys == null) {
            throw new NullPointerException("keys is marked @NonNull but is null");
        }
        if (timeout == null) {
            throw new NullPointerException("timeout is marked @NonNull but is null");
        }
        this.ensureInitialized();
        if (keys.isEmpty()) {
            return CompletableFuture.completedFuture(Collections.emptyMap());
        }
        ArrayList<UUID> keyList = new ArrayList<UUID>(keys.size());
        ArrayList<ByteArraySegment> serializedKeys = new ArrayList<ByteArraySegment>(keyList.size());
        for (UUID key : keys) {
            keyList.add(key);
            serializedKeys.add(this.serializeKey(key));
        }
        return ((CompletableFuture)READ_RETRY.runAsync(() -> this.index.get((List)serializedKeys, timeout), this.executor).thenApply(entries -> {
            assert (entries.size() == keys.size()) : "Unexpected number of entries returned by the index search.";
            HashMap result = new HashMap();
            for (int i = 0; i < keyList.size(); ++i) {
                ByteArraySegment v = (ByteArraySegment)entries.get(i);
                if (v == null) continue;
                result.put(keyList.get(i), this.deserializeValue(v));
            }
            return result;
        })).exceptionally(this::handleIndexOperationException);
    }

    @Override
    public CompletableFuture<Void> seal(@NonNull Duration timeout) {
        if (timeout == null) {
            throw new NullPointerException("timeout is marked @NonNull but is null");
        }
        this.ensureInitialized();
        SegmentHandle handle = this.handle.get();
        if (handle == null) {
            return CompletableFuture.completedFuture(null);
        }
        return Futures.exceptionallyExpecting((CompletableFuture)this.storage.seal(handle, timeout).thenRun(() -> log.info("{}: Sealed.", (Object)this.traceObjectId)), ex -> ex instanceof StreamSegmentSealedException, null);
    }

    @Override
    public AttributeIterator iterator(UUID fromId, UUID toId, Duration fetchTimeout) {
        this.ensureInitialized();
        return new AttributeIteratorImpl(fromId, (id, inclusive) -> this.index.iterator(this.serializeKey(id), inclusive, this.serializeKey(toId), true, fetchTimeout));
    }

    @VisibleForTesting
    SegmentHandle getAttributeSegmentHandle() {
        return this.handle.get();
    }

    public String toString() {
        return this.traceObjectId;
    }

    private <T> CompletableFuture<T> createAttributeSegmentIfNecessary(Supplier<CompletableFuture<T>> toRun, Duration timeout) {
        if (this.handle.get() == null) {
            String attributeSegmentName = StreamSegmentNameUtils.getAttributeSegmentName((String)this.segmentMetadata.getName());
            return Futures.exceptionallyComposeExpecting((CompletableFuture)this.storage.create(attributeSegmentName, this.config.getAttributeSegmentRollingPolicy(), timeout), ex -> ex instanceof StreamSegmentExistsException, () -> {
                log.info("{}: Attribute Segment did not exist in Storage when initialize() was called, but does now.", (Object)this.traceObjectId);
                return this.storage.openWrite(attributeSegmentName);
            }).thenComposeAsync(handle -> {
                this.handle.set((SegmentHandle)handle);
                return (CompletionStage)toRun.get();
            }, (Executor)this.executor);
        }
        return toRun.get();
    }

    private CompletableFuture<Void> executeConditionally(Function<Duration, CompletableFuture<Long>> indexOperation, Duration timeout) {
        TimeoutTimer timer = new TimeoutTimer(timeout);
        return ((CompletableFuture)UPDATE_RETRY.runAsync(() -> this.executeConditionallyOnce(indexOperation, timer), this.executor).exceptionally(this::handleIndexOperationException)).thenAccept(Callbacks::doNothing);
    }

    private CompletableFuture<Long> executeConditionallyOnce(Function<Duration, CompletableFuture<Long>> indexOperation, TimeoutTimer timer) {
        return Futures.exceptionallyCompose(indexOperation.apply(timer.getRemaining()), ex -> {
            BadOffsetException boe;
            if (Exceptions.unwrap((Throwable)ex) instanceof BadOffsetException && (boe = (BadOffsetException)Exceptions.unwrap((Throwable)ex)).getExpectedOffset() != this.index.getIndexLength()) {
                log.warn("{}: Conditional Index Update failed (expected {}, given {}). Reinitializing index.", new Object[]{this.traceObjectId, boe.getExpectedOffset(), boe.getGivenOffset()});
                return this.index.initialize(timer.getRemaining()).thenCompose(v -> Futures.failedFuture((Throwable)ex));
            }
            return Futures.failedFuture((Throwable)ex);
        });
    }

    private PageEntry serialize(Map.Entry<UUID, Long> entry) {
        return new PageEntry(this.serializeKey(entry.getKey()), this.serializeValue(entry.getValue()));
    }

    private ByteArraySegment serializeKey(UUID key) {
        byte[] result = new byte[16];
        BitConverter.writeUnsignedLong((byte[])result, (int)0, (long)key.getMostSignificantBits());
        BitConverter.writeUnsignedLong((byte[])result, (int)8, (long)key.getLeastSignificantBits());
        return new ByteArraySegment(result);
    }

    private UUID deserializeKey(ByteArraySegment key) {
        Preconditions.checkArgument((key.getLength() == 16 ? 1 : 0) != 0, (Object)"Unexpected key length.");
        long msb = BitConverter.readUnsignedLong((ArrayView)key, (int)0);
        long lsb = BitConverter.readUnsignedLong((ArrayView)key, (int)8);
        return new UUID(msb, lsb);
    }

    private ByteArraySegment serializeValue(Long value) {
        if (value == null || value == Long.MIN_VALUE) {
            return null;
        }
        byte[] result = new byte[8];
        BitConverter.writeLong((byte[])result, (int)0, (long)value);
        return new ByteArraySegment(result);
    }

    private long deserializeValue(ByteArraySegment value) {
        Preconditions.checkArgument((value.getLength() == 8 ? 1 : 0) != 0, (Object)"Unexpected value length.");
        return BitConverter.readLong((ArrayView)value, (int)0);
    }

    private CompletableFuture<Long> getLength(Duration timeout) {
        SegmentHandle handle = this.handle.get();
        if (handle == null) {
            return CompletableFuture.completedFuture(0L);
        }
        return this.storage.getStreamSegmentInfo(handle.getSegmentName(), timeout).thenApply(SegmentProperties::getLength);
    }

    private CompletableFuture<ByteArraySegment> readPage(long offset, int length, Duration timeout) {
        byte[] fromCache = this.getFromCache(offset, length);
        if (fromCache != null) {
            return CompletableFuture.completedFuture(new ByteArraySegment(fromCache));
        }
        SegmentHandle handle = this.handle.get();
        if (handle == null) {
            if (offset == 0L && length == 0) {
                return CompletableFuture.completedFuture(new ByteArraySegment(new byte[0]));
            }
            return Futures.failedFuture((Throwable)new ArrayIndexOutOfBoundsException(String.format("Attribute Index Segment has not been created yet. Cannot read %d byte(s) from offset (%d).", length, offset)));
        }
        byte[] buffer = new byte[length];
        return this.storage.read(handle, offset, buffer, 0, length, timeout).thenApplyAsync(bytesRead -> {
            Preconditions.checkArgument((length == bytesRead ? 1 : 0) != 0, (Object)"Unexpected number of bytes read.");
            this.storeInCache(offset, buffer);
            return new ByteArraySegment(buffer);
        }, (Executor)this.executor);
    }

    private CompletableFuture<Long> writePages(List<Map.Entry<Long, ByteArraySegment>> pages, Collection<Long> obsoleteOffsets, long truncateOffset, Duration timeout) {
        long writeOffset = pages.get(0).getKey();
        ArrayList<InputStream> streams = new ArrayList<InputStream>();
        AtomicInteger length = new AtomicInteger();
        for (Map.Entry<Long, ByteArraySegment> e : pages) {
            long pageOffset = e.getKey();
            Preconditions.checkArgument((pageOffset == writeOffset + (long)length.get() ? 1 : 0) != 0, (Object)"Unexpected page offset.");
            ByteArraySegment pageContents = e.getValue();
            streams.add(pageContents.getReader());
            length.addAndGet(pageContents.getLength());
        }
        TimeoutTimer timer = new TimeoutTimer(timeout);
        return ((CompletableFuture)this.createAttributeSegmentIfNecessary(() -> this.writeToSegment(streams, writeOffset, length.get(), timer), timer.getRemaining()).thenComposeAsync(v -> {
            if (this.storage.supportsTruncation() && truncateOffset >= 0L) {
                return this.storage.truncate(this.handle.get(), truncateOffset, timer.getRemaining());
            }
            log.debug("{}: Not truncating attribute segment. SupportsTruncation = {}, TruncateOffset = {}.", new Object[]{this.traceObjectId, this.storage.supportsTruncation(), truncateOffset});
            return CompletableFuture.completedFuture(null);
        }, (Executor)this.executor)).thenApplyAsync(v -> {
            this.storeInCache(pages, obsoleteOffsets);
            return writeOffset + (long)length.get();
        }, (Executor)this.executor);
    }

    private CompletableFuture<Void> writeToSegment(List<InputStream> streams, long writeOffset, int length, TimeoutTimer timer) {
        SequenceInputStream toWrite = new SequenceInputStream(Collections.enumeration(streams));
        return this.storage.write(this.handle.get(), writeOffset, (InputStream)toWrite, length, timer.getRemaining());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private byte[] getFromCache(long offset, int length) {
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            byte[] data;
            CacheEntry entry = this.cacheEntries.getOrDefault(offset, null);
            if (entry != null && (data = this.cache.get((Cache.Key)entry.getKey())) != null && data.length == length) {
                entry.setGeneration(this.currentCacheGeneration);
                return data;
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void storeInCache(long offset, byte[] data) {
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            CacheEntry entry = this.cacheEntries.getOrDefault(offset, null);
            if (entry == null || entry.getSize() != data.length) {
                entry = new CacheEntry(offset, data.length, this.currentCacheGeneration);
                this.cacheEntries.put(offset, entry);
            }
            this.cache.insert((Cache.Key)entry.getKey(), data);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void storeInCache(List<Map.Entry<Long, ByteArraySegment>> toAdd, Collection<Long> obsoleteOffsets) {
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            obsoleteOffsets.stream().map(this.cacheEntries::get).filter(Objects::nonNull).forEach(this::removeFromCache);
            for (Map.Entry<Long, ByteArraySegment> e : toAdd) {
                long offset = e.getKey();
                ByteArraySegment data = e.getValue();
                CacheEntry entry = this.cacheEntries.getOrDefault(offset, null);
                if (entry == null || entry.getSize() != data.getLength()) {
                    entry = new CacheEntry(offset, data.getLength(), this.currentCacheGeneration);
                    this.cacheEntries.put(offset, entry);
                }
                this.cache.insert((Cache.Key)entry.getKey(), (BufferView)data);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeFromCache(Collection<CacheEntry> entries) {
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            entries.forEach(this::removeFromCache);
        }
    }

    @GuardedBy(value="cacheEntries")
    private void removeFromCache(CacheEntry e) {
        this.cache.remove((Cache.Key)e.getKey());
        this.cacheEntries.remove(e.getOffset());
    }

    private void ensureInitialized() {
        Preconditions.checkState((boolean)this.index.isInitialized(), (Object)"SegmentAttributeIndex is not initialized.");
    }

    private <T> T handleIndexOperationException(Throwable ex) {
        ex = Exceptions.unwrap((Throwable)ex);
        if (ex instanceof IllegalDataFormatException) {
            throw new DataCorruptionException("BTreeIndex operation failed. Index corrupted.", ex, new Object[0]);
        }
        throw ex;
    }

    @FunctionalInterface
    private static interface CreatePageEntryIterator {
        public AsyncIterator<List<PageEntry>> apply(UUID var1, boolean var2);
    }

    private class AttributeIteratorImpl
    implements AttributeIterator {
        private final CreatePageEntryIterator getPageEntryIterator;
        private final AtomicReference<AsyncIterator<List<PageEntry>>> pageEntryIterator;
        private final AtomicReference<UUID> lastProcessedId;
        private final AtomicBoolean firstInvocation;

        AttributeIteratorImpl(UUID firstId, CreatePageEntryIterator getPageEntryIterator) {
            this.getPageEntryIterator = getPageEntryIterator;
            this.pageEntryIterator = new AtomicReference();
            this.lastProcessedId = new AtomicReference<UUID>(firstId);
            this.firstInvocation = new AtomicBoolean(true);
            this.reinitialize();
        }

        public CompletableFuture<List<Map.Entry<UUID, Long>>> getNext() {
            return ((CompletableFuture)READ_RETRY.runAsync(this::getNextPageEntries, SegmentAttributeBTreeIndex.this.executor).thenApply(pageEntries -> {
                if (pageEntries == null) {
                    return null;
                }
                List result = pageEntries.stream().map(e -> Maps.immutableEntry((Object)SegmentAttributeBTreeIndex.this.deserializeKey(e.getKey()), (Object)SegmentAttributeBTreeIndex.this.deserializeValue(e.getValue()))).collect(Collectors.toList());
                if (result.size() > 0) {
                    this.lastProcessedId.set((UUID)((Map.Entry)result.get(result.size() - 1)).getKey());
                    this.firstInvocation.set(false);
                }
                return result;
            })).exceptionally(x$0 -> (List)SegmentAttributeBTreeIndex.this.handleIndexOperationException(x$0));
        }

        private CompletableFuture<List<PageEntry>> getNextPageEntries() {
            return this.pageEntryIterator.get().getNext().exceptionally(ex -> {
                this.reinitialize();
                throw new CompletionException((Throwable)ex);
            });
        }

        private void reinitialize() {
            this.pageEntryIterator.set(this.getPageEntryIterator.apply(this.lastProcessedId.get(), this.firstInvocation.get()));
        }
    }

    private class CacheEntry {
        private final long offset;
        private final int size;
        @GuardedBy(value="this")
        private int generation;

        CacheEntry(long offset, int size, int currentGeneration) {
            this.offset = offset;
            this.size = size;
            this.generation = currentGeneration;
        }

        CacheKey getKey() {
            return new CacheKey(SegmentAttributeBTreeIndex.this.segmentMetadata.getId(), this.offset);
        }

        synchronized int getGeneration() {
            return this.generation;
        }

        synchronized void setGeneration(int value) {
            this.generation = value;
        }

        @SuppressFBWarnings(justification="generated code")
        public long getOffset() {
            return this.offset;
        }

        @SuppressFBWarnings(justification="generated code")
        public int getSize() {
            return this.size;
        }
    }
}

