/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.reading;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.ObjectClosedException;
import io.pravega.common.util.BufferView;
import io.pravega.segmentstore.contracts.ReadResult;
import io.pravega.segmentstore.contracts.StreamSegmentNotExistsException;
import io.pravega.segmentstore.server.CacheManager;
import io.pravega.segmentstore.server.ContainerMetadata;
import io.pravega.segmentstore.server.DataCorruptionException;
import io.pravega.segmentstore.server.ReadIndex;
import io.pravega.segmentstore.server.SegmentMetadata;
import io.pravega.segmentstore.server.reading.ReadIndexConfig;
import io.pravega.segmentstore.server.reading.StreamSegmentReadIndex;
import io.pravega.segmentstore.storage.Cache;
import io.pravega.segmentstore.storage.CacheFactory;
import io.pravega.segmentstore.storage.ReadOnlyStorage;
import io.pravega.segmentstore.storage.ThrottleSourceListener;
import java.io.InputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class ContainerReadIndex
implements ReadIndex {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(ContainerReadIndex.class);
    private final String traceObjectId;
    @GuardedBy(value="lock")
    private final HashMap<Long, StreamSegmentReadIndex> readIndices;
    private final Object lock = new Object();
    private final Cache cache;
    private final ReadOnlyStorage storage;
    private final ScheduledExecutorService executor;
    private final ReadIndexConfig config;
    private final CacheManager cacheManager;
    @GuardedBy(value="lock")
    private ContainerMetadata metadata;
    @GuardedBy(value="lock")
    private ContainerMetadata preRecoveryMetadata;
    private final AtomicBoolean closed;

    public ContainerReadIndex(ReadIndexConfig config, ContainerMetadata metadata, CacheFactory cacheFactory, ReadOnlyStorage storage, CacheManager cacheManager, ScheduledExecutorService executor) {
        Preconditions.checkNotNull((Object)config, (Object)"config");
        Preconditions.checkNotNull((Object)metadata, (Object)"metadata");
        Preconditions.checkNotNull((Object)cacheFactory, (Object)"cacheFactory");
        Preconditions.checkNotNull((Object)storage, (Object)"storage");
        Preconditions.checkNotNull((Object)cacheManager, (Object)"cacheManager");
        Preconditions.checkNotNull((Object)executor, (Object)"executor");
        Preconditions.checkArgument((!metadata.isRecoveryMode() ? 1 : 0) != 0, (Object)"Given ContainerMetadata is in Recovery Mode.");
        this.traceObjectId = String.format("ReadIndex[%s]", metadata.getContainerId());
        this.readIndices = new HashMap();
        this.config = config;
        this.cache = cacheFactory.getCache(String.format("Container_%d", metadata.getContainerId()));
        this.metadata = metadata;
        this.storage = storage;
        this.cacheManager = cacheManager;
        this.executor = executor;
        this.preRecoveryMetadata = null;
        this.closed = new AtomicBoolean();
    }

    @Override
    public void close() {
        if (!this.closed.getAndSet(true)) {
            this.closeAllIndices(false);
            this.cache.close();
            log.info("{}: Closed.", (Object)this.traceObjectId);
        }
    }

    @Override
    public void append(long streamSegmentId, long offset, BufferView data) throws StreamSegmentNotExistsException {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        log.debug("{}: append (StreamSegmentId = {}, Offset = {}, DataLength = {}).", new Object[]{this.traceObjectId, streamSegmentId, offset, data.getLength()});
        StreamSegmentReadIndex index = this.getOrCreateIndex(streamSegmentId);
        Exceptions.checkArgument((!index.isMerged() ? 1 : 0) != 0, (String)"streamSegmentId", (String)"StreamSegment is merged. Cannot append to it anymore.", (Object[])new Object[0]);
        index.append(offset, data);
    }

    @Override
    public void beginMerge(long targetStreamSegmentId, long offset, long sourceStreamSegmentId) throws StreamSegmentNotExistsException {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        log.debug("{}: beginMerge (TargetId = {}, Offset = {}, SourceId = {}).", new Object[]{this.traceObjectId, targetStreamSegmentId, offset, sourceStreamSegmentId});
        StreamSegmentReadIndex targetIndex = this.getOrCreateIndex(targetStreamSegmentId);
        StreamSegmentReadIndex sourceIndex = this.getOrCreateIndex(sourceStreamSegmentId);
        Exceptions.checkArgument((!targetIndex.isMerged() ? 1 : 0) != 0, (String)"targetStreamSegmentId", (String)"StreamSegment is merged. Cannot access it anymore.", (Object[])new Object[0]);
        Exceptions.checkArgument((!sourceIndex.isMerged() ? 1 : 0) != 0, (String)"sourceStreamSegmentId", (String)"StreamSegment is merged. Cannot access it anymore.", (Object[])new Object[0]);
        targetIndex.beginMerge(offset, sourceIndex);
        sourceIndex.markMerged();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void completeMerge(long targetStreamSegmentId, long sourceStreamSegmentId) throws StreamSegmentNotExistsException {
        SegmentMetadata sourceMetadata;
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        log.debug("{}: completeMerge (TargetId = {}, SourceId = {}.", new Object[]{this.traceObjectId, targetStreamSegmentId, sourceStreamSegmentId});
        Object object = this.lock;
        synchronized (object) {
            sourceMetadata = this.metadata.getStreamSegmentMetadata(sourceStreamSegmentId);
        }
        Preconditions.checkState((sourceMetadata != null ? 1 : 0) != 0, (String)"No Metadata found for Segment Id %s.", (long)sourceStreamSegmentId);
        StreamSegmentReadIndex targetIndex = this.getOrCreateIndex(targetStreamSegmentId);
        targetIndex.completeMerge(sourceMetadata);
        Object object2 = this.lock;
        synchronized (object2) {
            this.closeIndex(sourceStreamSegmentId, false);
        }
    }

    @Override
    public ReadResult read(long streamSegmentId, long offset, int maxLength, Duration timeout) throws StreamSegmentNotExistsException {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        log.debug("{}: read (StreamSegmentId = {}, Offset = {}, MaxLength = {}).", new Object[]{this.traceObjectId, streamSegmentId, offset, maxLength});
        StreamSegmentReadIndex index = this.getOrCreateIndex(streamSegmentId);
        Exceptions.checkArgument((!index.isMerged() ? 1 : 0) != 0, (String)"streamSegmentId", (String)"StreamSegment is merged. Cannot access it anymore.", (Object[])new Object[0]);
        return index.read(offset, maxLength, timeout);
    }

    @Override
    public InputStream readDirect(long streamSegmentId, long offset, int length) throws StreamSegmentNotExistsException {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        log.debug("{}: readDirect (StreamSegmentId = {}, Offset = {}, Length = {}).", new Object[]{this.traceObjectId, streamSegmentId, offset, length});
        StreamSegmentReadIndex index = this.getOrCreateIndex(streamSegmentId);
        return index.readDirect(offset, length);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void triggerFutureReads(Collection<Long> streamSegmentIds) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        log.debug("{}: triggerFutureReads (StreamSegmentIds = {}).", (Object)this.traceObjectId, streamSegmentIds);
        HashSet<Long> missingIds = new HashSet<Long>();
        for (long segmentId : streamSegmentIds) {
            StreamSegmentReadIndex index = this.getIndex(segmentId);
            if (index == null) {
                Object object = this.lock;
                synchronized (object) {
                    if (this.metadata.getStreamSegmentMetadata(segmentId) == null) {
                        missingIds.add(segmentId);
                    }
                    continue;
                }
            }
            try {
                index.triggerFutureReads();
            }
            catch (ObjectClosedException ex) {
                if (this.getIndex(segmentId) != null) {
                    throw ex;
                }
                log.debug("{}: triggerFutureReads: StreamSegmentId {} was skipped because it is no longer registered.", (Object)this.traceObjectId, (Object)segmentId);
            }
        }
        Exceptions.checkArgument((missingIds.size() == 0 ? 1 : 0) != 0, (String)"streamSegmentIds", (String)"At least one StreamSegmentId does not exist in the metadata: %s", (Object[])new Object[]{missingIds});
    }

    @Override
    public void clear() {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkState((boolean)this.isRecoveryMode(), (Object)"Read Index is not in recovery mode. Cannot clear ReadIndex.");
        this.closeAllIndices(true);
        log.info("{}: Cleared.", (Object)this.traceObjectId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void cleanup(Collection<Long> segmentIds) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        ArrayList<Long> removed = new ArrayList<Long>();
        ArrayList<Long> notRemoved = new ArrayList<Long>();
        Object object = this.lock;
        synchronized (object) {
            if (segmentIds == null) {
                segmentIds = new ArrayList<Long>(this.readIndices.keySet());
            }
            for (long streamSegmentId : segmentIds) {
                StreamSegmentReadIndex index = this.readIndices.get(streamSegmentId);
                if (index != null && !index.isActive()) {
                    this.closeIndex(streamSegmentId, true);
                    removed.add(streamSegmentId);
                    continue;
                }
                if (index == null) {
                    removed.add(streamSegmentId);
                    continue;
                }
                notRemoved.add(streamSegmentId);
            }
        }
        if (notRemoved.size() > 0) {
            log.debug("{}: Unable to clean up ReadIndex for Segments {} because no such index exists or the Segments are not deleted.", (Object)this.traceObjectId, notRemoved);
        }
        log.info("{}: Cleaned up ReadIndices for {} inactive or deleted Segments.", (Object)this.traceObjectId, removed);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void enterRecoveryMode(ContainerMetadata recoveryMetadataSource) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkState((!this.isRecoveryMode() ? 1 : 0) != 0, (Object)"Read Index is already in recovery mode.");
        Preconditions.checkNotNull((Object)recoveryMetadataSource, (Object)"recoveryMetadataSource");
        Preconditions.checkArgument((boolean)recoveryMetadataSource.isRecoveryMode(), (Object)"Given ContainerMetadata is not in recovery mode.");
        Object object = this.lock;
        synchronized (object) {
            Preconditions.checkArgument((this.metadata.getContainerId() == recoveryMetadataSource.getContainerId() ? 1 : 0) != 0, (Object)"Given ContainerMetadata refers to a different container than this ReadIndex.");
            assert (this.preRecoveryMetadata == null) : "preRecoveryMetadata is not null, which should not happen unless we already are in recovery mode";
            this.preRecoveryMetadata = this.metadata;
            this.metadata = recoveryMetadataSource;
        }
        log.info("{} Enter RecoveryMode.", (Object)this.traceObjectId);
        this.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void exitRecoveryMode(boolean successfulRecovery) throws DataCorruptionException {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkState((boolean)this.isRecoveryMode(), (Object)"Read Index is not in recovery mode.");
        Object object = this.lock;
        synchronized (object) {
            assert (this.preRecoveryMetadata != null) : "preRecoveryMetadata is null, which should only be the case when we are not in recovery mode";
            Preconditions.checkState((!this.preRecoveryMetadata.isRecoveryMode() ? 1 : 0) != 0, (Object)"Cannot take ReadIndex out of recovery: ContainerMetadata is still in recovery mode.");
            if (successfulRecovery) {
                for (Map.Entry<Long, StreamSegmentReadIndex> e : this.readIndices.entrySet()) {
                    SegmentMetadata metadata = this.preRecoveryMetadata.getStreamSegmentMetadata(e.getKey());
                    if (metadata == null) {
                        throw new DataCorruptionException(String.format("ContainerMetadata has no knowledge of StreamSegment Id %s.", e.getKey()), new Object[0]);
                    }
                    e.getValue().exitRecoveryMode(metadata);
                }
            } else {
                this.clear();
            }
            this.metadata = this.preRecoveryMetadata;
            this.preRecoveryMetadata = null;
        }
        log.info("{} Exit RecoveryMode.", (Object)this.traceObjectId);
    }

    @Override
    public double getCacheUtilization() {
        return this.cacheManager.getCacheUtilization();
    }

    @Override
    public double getCacheTargetUtilization() {
        return this.cacheManager.getCacheTargetUtilization();
    }

    @Override
    public double getCacheMaxUtilization() {
        return this.cacheManager.getCacheMaxUtilization();
    }

    @Override
    public void registerCleanupListener(ThrottleSourceListener listener) {
        this.cacheManager.registerCleanupListener(listener);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean isRecoveryMode() {
        Object object = this.lock;
        synchronized (object) {
            return this.preRecoveryMetadata != null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    StreamSegmentReadIndex getIndex(long streamSegmentId) {
        Object object = this.lock;
        synchronized (object) {
            return this.readIndices.getOrDefault(streamSegmentId, null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private StreamSegmentReadIndex getOrCreateIndex(long streamSegmentId) throws StreamSegmentNotExistsException {
        StreamSegmentReadIndex index;
        Object object = this.lock;
        synchronized (object) {
            index = this.getIndex(streamSegmentId);
            if (index != null && !index.isActive()) {
                this.closeIndex(streamSegmentId, true);
                index = null;
            }
            if (index == null) {
                SegmentMetadata segmentMetadata = this.metadata.getStreamSegmentMetadata(streamSegmentId);
                if (segmentMetadata == null) {
                    throw new IllegalArgumentException(String.format("Segment Id %d does not exist in the metadata.", streamSegmentId));
                }
                if (!segmentMetadata.isActive()) {
                    throw new IllegalArgumentException(String.format("Segment Id %d does exist in the metadata but is inactive.", streamSegmentId));
                }
                if (segmentMetadata.isDeleted()) {
                    throw new StreamSegmentNotExistsException(segmentMetadata.getName());
                }
                index = new StreamSegmentReadIndex(this.config, segmentMetadata, this.cache, this.storage, this.executor, this.isRecoveryMode());
                this.cacheManager.register(index);
                this.readIndices.put(streamSegmentId, index);
            }
        }
        return index;
    }

    @GuardedBy(value="lock")
    private boolean closeIndex(long streamSegmentId, boolean cleanCache) {
        StreamSegmentReadIndex index = this.readIndices.remove(streamSegmentId);
        if (index != null) {
            this.cacheManager.unregister(index);
            index.close(cleanCache);
        }
        return index != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeAllIndices(boolean cleanCache) {
        Object object = this.lock;
        synchronized (object) {
            ArrayList<Long> segmentIds = new ArrayList<Long>(this.readIndices.keySet());
            segmentIds.forEach(segmentId -> this.closeIndex((long)segmentId, cleanCache));
            this.readIndices.clear();
        }
    }
}

