/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.tables;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Maps;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.ObjectClosedException;
import io.pravega.common.TimeoutTimer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.concurrent.MultiKeySequentialProcessor;
import io.pravega.segmentstore.contracts.ReadResult;
import io.pravega.segmentstore.contracts.SegmentProperties;
import io.pravega.segmentstore.contracts.tables.BadKeyVersionException;
import io.pravega.segmentstore.contracts.tables.KeyNotExistsException;
import io.pravega.segmentstore.contracts.tables.TableKey;
import io.pravega.segmentstore.contracts.tables.TableSegmentNotEmptyException;
import io.pravega.segmentstore.server.CacheManager;
import io.pravega.segmentstore.server.DirectSegmentAccess;
import io.pravega.segmentstore.server.reading.AsyncReadResultProcessor;
import io.pravega.segmentstore.server.tables.AsyncTableEntryReader;
import io.pravega.segmentstore.server.tables.CacheBucketOffset;
import io.pravega.segmentstore.server.tables.ContainerKeyCache;
import io.pravega.segmentstore.server.tables.EntrySerializer;
import io.pravega.segmentstore.server.tables.IndexReader;
import io.pravega.segmentstore.server.tables.KeyHasher;
import io.pravega.segmentstore.server.tables.TableBucket;
import io.pravega.segmentstore.server.tables.TableBucketReader;
import io.pravega.segmentstore.server.tables.TableKeyBatch;
import io.pravega.segmentstore.storage.CacheFactory;
import java.beans.ConstructorProperties;
import java.io.InputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
class ContainerKeyIndex
implements AutoCloseable {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(ContainerKeyIndex.class);
    @VisibleForTesting
    static final Duration RECOVERY_TIMEOUT = Duration.ofSeconds(60L);
    private static final int MAX_TAIL_CACHE_PRE_INDEX_LENGTH = 0x4000000;
    private final IndexReader indexReader;
    private final ScheduledExecutorService executor;
    private final ContainerKeyCache cache;
    private final CacheManager cacheManager;
    private final MultiKeySequentialProcessor<Map.Entry<Long, UUID>> conditionalUpdateProcessor;
    private final RecoveryTracker recoveryTracker;
    private final AtomicBoolean closed;
    private final KeyHasher keyHasher;
    private final String traceObjectId;

    ContainerKeyIndex(int containerId, @NonNull CacheFactory cacheFactory, @NonNull CacheManager cacheManager, @NonNull KeyHasher keyHasher, @NonNull ScheduledExecutorService executor) {
        if (cacheFactory == null) {
            throw new NullPointerException("cacheFactory is marked @NonNull but is null");
        }
        if (cacheManager == null) {
            throw new NullPointerException("cacheManager is marked @NonNull but is null");
        }
        if (keyHasher == null) {
            throw new NullPointerException("keyHasher is marked @NonNull but is null");
        }
        if (executor == null) {
            throw new NullPointerException("executor is marked @NonNull but is null");
        }
        this.cache = new ContainerKeyCache(containerId, cacheFactory);
        this.cacheManager = cacheManager;
        this.cacheManager.register(this.cache);
        this.executor = executor;
        this.indexReader = new IndexReader(executor);
        this.conditionalUpdateProcessor = new MultiKeySequentialProcessor((Executor)this.executor);
        this.recoveryTracker = new RecoveryTracker();
        this.keyHasher = keyHasher;
        this.closed = new AtomicBoolean();
        this.traceObjectId = String.format("KeyIndex[%d]", containerId);
    }

    @Override
    public void close() {
        if (!this.closed.getAndSet(true)) {
            this.conditionalUpdateProcessor.close();
            this.cacheManager.unregister(this.cache);
            this.cache.close();
            this.recoveryTracker.close();
            log.info("{}: Closed.", (Object)this.traceObjectId);
        }
    }

    <T> CompletableFuture<T> executeIfEmpty(DirectSegmentAccess segment, Supplier<CompletableFuture<T>> action, TimeoutTimer timer) {
        return this.recoveryTracker.waitIfNeeded(segment, ignored -> this.conditionalUpdateProcessor.addWithFilter(conditionKey -> ((Long)conditionKey.getKey()).longValue() == segment.getSegmentId(), () -> this.lambda$null$2(segment, timer, (Supplier)action)));
    }

    private CompletableFuture<Boolean> isTableSegmentEmpty(DirectSegmentAccess segment, TimeoutTimer timer) {
        Map<UUID, CacheBucketOffset> tailHashes = this.cache.getTailHashes(segment.getSegmentId());
        List<UUID> tailRemovals = tailHashes.entrySet().stream().filter(e -> ((CacheBucketOffset)e.getValue()).isRemoval()).map(Map.Entry::getKey).collect(Collectors.toList());
        if (tailHashes.size() > tailRemovals.size()) {
            return CompletableFuture.completedFuture(false);
        }
        SegmentProperties sp = segment.getInfo();
        long indexedBucketCount = this.indexReader.getBucketCount(sp);
        if (tailRemovals.isEmpty()) {
            return CompletableFuture.completedFuture(indexedBucketCount <= 0L);
        }
        return this.indexReader.locateBuckets(segment, tailRemovals, timer).thenApply(buckets -> {
            long removedCount = buckets.values().stream().filter(TableBucket::exists).count();
            return indexedBucketCount <= removedCount;
        });
    }

    CompletableFuture<Map<UUID, Long>> getBucketOffsets(DirectSegmentAccess segment, Collection<UUID> hashes, TimeoutTimer timer) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        if (hashes.isEmpty()) {
            return CompletableFuture.completedFuture(Collections.emptyMap());
        }
        HashMap<UUID, Long> result = new HashMap<UUID, Long>();
        ArrayList<UUID> toLookup = new ArrayList<UUID>();
        for (UUID hash : hashes) {
            if (result.containsKey(hash)) continue;
            CacheBucketOffset existingValue = this.cache.get(segment.getSegmentId(), hash);
            if (existingValue == null) {
                result.put(hash, -1L);
                toLookup.add(hash);
                continue;
            }
            if (!existingValue.isRemoval()) {
                result.put(hash, existingValue.getSegmentOffset());
                continue;
            }
            long backpointerOffset = this.cache.getBackpointer(segment.getSegmentId(), existingValue.getSegmentOffset());
            if (backpointerOffset < 0L) {
                result.put(hash, -1L);
                toLookup.add(hash);
                continue;
            }
            result.put(hash, existingValue.getSegmentOffset());
        }
        if (toLookup.isEmpty()) {
            return CompletableFuture.completedFuture(result);
        }
        return this.recoveryTracker.waitIfNeeded(segment, cacheUpdated -> this.getBucketOffsetFromSegment(segment, (Map<UUID, Long>)result, (Collection<UUID>)toLookup, (boolean)cacheUpdated, timer));
    }

    CompletableFuture<Long> getBucketOffsetDirect(DirectSegmentAccess segment, UUID keyHash, TimeoutTimer timer) {
        return this.recoveryTracker.waitIfNeeded(segment, cacheUpdated -> this.getBucketOffsetFromSegment(segment, Collections.synchronizedMap(new HashMap()), (Collection<UUID>)Collections.singleton(keyHash), (boolean)cacheUpdated, timer).thenApply(result -> (Long)result.get(keyHash)));
    }

    private CompletableFuture<Map<UUID, Long>> getBucketOffsetFromSegment(DirectSegmentAccess segment, Map<UUID, Long> result, Collection<UUID> toLookup, boolean tryCache, TimeoutTimer timer) {
        return this.indexReader.locateBuckets(segment, toLookup, timer).thenApplyAsync(bucketsByHash -> {
            for (Map.Entry e : bucketsByHash.entrySet()) {
                UUID keyHash = (UUID)e.getKey();
                TableBucket bucket = (TableBucket)e.getValue();
                if (bucket.exists()) {
                    long highestOffset = this.cache.includeExistingKey(segment.getSegmentId(), keyHash, bucket.getSegmentOffset());
                    result.put(keyHash, highestOffset);
                    continue;
                }
                if (tryCache) {
                    CacheBucketOffset existingValue = this.cache.get(segment.getSegmentId(), keyHash);
                    result.put(keyHash, existingValue == null || existingValue.isRemoval() ? -1L : existingValue.getSegmentOffset());
                    continue;
                }
                result.put(keyHash, -1L);
            }
            return result;
        }, (Executor)this.executor);
    }

    CompletableFuture<Long> getBackpointerOffset(DirectSegmentAccess segment, long offset, Duration timeout) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        long cachedBackpointer = this.cache.getBackpointer(segment.getSegmentId(), offset);
        if (cachedBackpointer >= 0L) {
            return CompletableFuture.completedFuture(cachedBackpointer);
        }
        if (offset <= this.cache.getSegmentIndexOffset(segment.getSegmentId())) {
            return this.indexReader.getBackpointerOffset(segment, offset, timeout);
        }
        return this.recoveryTracker.waitIfNeeded(segment, ignored -> this.indexReader.getBackpointerOffset(segment, offset, timeout));
    }

    CompletableFuture<List<Long>> update(DirectSegmentAccess segment, TableKeyBatch batch, Supplier<CompletableFuture<Long>> persist, TimeoutTimer timer) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        if (batch.isConditional()) {
            List keys = batch.getVersionedItems().stream().map(item -> Maps.immutableEntry((Object)segment.getSegmentId(), (Object)item.getHash())).collect(Collectors.toList());
            return this.conditionalUpdateProcessor.add(keys, () -> ((CompletableFuture)this.validateConditionalUpdate(segment, batch, timer).thenComposeAsync(arg_0 -> ContainerKeyIndex.lambda$null$12((Supplier)persist, arg_0), (Executor)this.executor)).thenApplyAsync(batchOffset -> this.updateCache(segment, batch, (long)batchOffset), (Executor)this.executor));
        }
        return persist.get().thenApplyAsync(batchOffset -> this.updateCache(segment, batch, (long)batchOffset), (Executor)this.executor);
    }

    private List<Long> updateCache(DirectSegmentAccess segment, TableKeyBatch batch, long batchOffset) {
        this.cache.updateSegmentIndexOffsetIfMissing(segment.getSegmentId(), () -> this.indexReader.getLastIndexedOffset(segment.getInfo()));
        return this.cache.includeUpdateBatch(segment.getSegmentId(), batch, batchOffset);
    }

    private CompletableFuture<Void> validateConditionalUpdate(DirectSegmentAccess segment, TableKeyBatch batch, TimeoutTimer timer) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        List<UUID> hashes = batch.getVersionedItems().stream().map(TableKeyBatch.Item::getHash).collect(Collectors.toList());
        CompletionStage result = this.getBucketOffsets(segment, hashes, timer).thenAccept(offsets -> this.validateConditionalUpdate(batch.getVersionedItems(), (Map<UUID, Long>)offsets, segment.getInfo().getName()));
        return Futures.exceptionallyCompose((CompletableFuture)result, ex -> {
            if ((ex = Exceptions.unwrap((Throwable)ex)) instanceof BadKeyVersionException) {
                return this.validateConditionalUpdateFailures(segment, ((BadKeyVersionException)ex).getExpectedVersions(), timer);
            }
            return Futures.failedFuture((Throwable)ex);
        });
    }

    private void validateConditionalUpdate(List<TableKeyBatch.Item> items, Map<UUID, Long> bucketOffsets, String segmentName) {
        HashMap<TableKey, Long> badKeyVersions = new HashMap<TableKey, Long>();
        for (TableKeyBatch.Item item : items) {
            TableKey key = item.getKey();
            Long bucketOffset = bucketOffsets.get(item.getHash());
            assert (key.hasVersion()) : "validateConditionalUpdate for TableKey with no compare version";
            if (bucketOffset == -1L) {
                if (key.getVersion() == -1L) continue;
                throw new KeyNotExistsException(segmentName, key.getKey());
            }
            if (bucketOffset.longValue() == key.getVersion()) continue;
            badKeyVersions.put(key, bucketOffset);
        }
        if (!badKeyVersions.isEmpty()) {
            throw new BadKeyVersionException(segmentName, badKeyVersions);
        }
    }

    private CompletableFuture<Void> validateConditionalUpdateFailures(DirectSegmentAccess segment, Map<TableKey, Long> expectedVersions, TimeoutTimer timer) {
        assert (!expectedVersions.isEmpty());
        TableBucketReader<TableKey> bucketReader = TableBucketReader.key(segment, this::getBackpointerOffset, this.executor);
        Map<TableKey, CompletableFuture> searches = expectedVersions.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> bucketReader.find(((TableKey)e.getKey()).getKey(), (Long)e.getValue(), timer)));
        return Futures.allOf(searches.values()).thenRun(() -> {
            HashMap failed = new HashMap();
            for (Map.Entry e : searches.entrySet()) {
                TableKey actual = (TableKey)((CompletableFuture)e.getValue()).join();
                boolean isValid = actual == null ? ((TableKey)e.getKey()).getVersion() == -1L : ((TableKey)e.getKey()).getVersion() == actual.getVersion();
                if (isValid) continue;
                failed.put(e.getKey(), actual == null ? -1L : actual.getVersion());
            }
            if (!failed.isEmpty()) {
                throw new CompletionException((Throwable)new BadKeyVersionException(segment.getInfo().getName(), failed));
            }
        });
    }

    void notifyIndexOffsetChanged(long segmentId, long indexOffset) {
        this.cache.updateSegmentIndexOffset(segmentId, indexOffset);
        this.recoveryTracker.updateSegmentIndexOffset(segmentId, indexOffset);
    }

    CompletableFuture<Map<UUID, CacheBucketOffset>> getUnindexedKeyHashes(DirectSegmentAccess segment) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        return this.recoveryTracker.waitIfNeeded(segment, ignored -> CompletableFuture.completedFuture(this.cache.getTailHashes(segment.getSegmentId())));
    }

    private void triggerCacheTailIndex(DirectSegmentAccess segment, long lastIndexedOffset, long segmentLength) {
        long tailIndexLength = segmentLength - lastIndexedOffset;
        if (lastIndexedOffset >= segmentLength) {
            log.debug("{}: Table Segment {} fully indexed.", (Object)this.traceObjectId, (Object)segment.getSegmentId());
            return;
        }
        if (tailIndexLength > this.getMaxTailCachePreIndexLength()) {
            log.debug("{}: Table Segment {} cannot perform tail-caching because tail index too long ({}).", new Object[]{this.traceObjectId, segment.getSegmentId(), tailIndexLength});
            return;
        }
        log.debug("{}: Tail-caching started for Table Segment {}. LastIndexedOffset={}, SegmentLength={}.", new Object[]{this.traceObjectId, segment.getSegmentId(), lastIndexedOffset, segmentLength});
        ReadResult rr = segment.read(lastIndexedOffset, (int)tailIndexLength, this.getRecoveryTimeout());
        ((CompletableFuture)AsyncReadResultProcessor.processAll(rr, this.executor, this.getRecoveryTimeout()).thenAcceptAsync(inputStream -> {
            Map<UUID, CacheBucketOffset> updates = this.collectLatestOffsets((InputStream)inputStream, lastIndexedOffset, (int)tailIndexLength);
            this.cache.includeTailCache(segment.getSegmentId(), updates);
            log.debug("{}: Tail-caching complete for Table Segment {}. Update Count={}.", new Object[]{this.traceObjectId, segment.getSegmentId(), updates.size()});
            this.recoveryTracker.updateSegmentIndexOffset(segment.getSegmentId(), segmentLength, updates.size() > 0);
        }, (Executor)this.executor)).exceptionally(ex -> {
            log.warn("{}: Tail-caching failed for Table Segment {}.", new Object[]{this.traceObjectId, segment.getSegmentId(), Exceptions.unwrap((Throwable)ex)});
            return null;
        });
    }

    private Map<UUID, CacheBucketOffset> collectLatestOffsets(InputStream input, long startOffset, int maxLength) {
        AsyncTableEntryReader.DeserializedEntry e;
        EntrySerializer serializer = new EntrySerializer();
        HashMap<UUID, CacheBucketOffset> entries = new HashMap<UUID, CacheBucketOffset>();
        long maxOffset = startOffset + (long)maxLength;
        for (long nextOffset = startOffset; nextOffset < maxOffset; nextOffset += (long)e.getHeader().getTotalLength()) {
            e = AsyncTableEntryReader.readEntryComponents(input, nextOffset, serializer);
            UUID hash = this.keyHasher.hash(e.getKey());
            entries.put(hash, new CacheBucketOffset(nextOffset, e.getHeader().isDeletion()));
        }
        return entries;
    }

    @VisibleForTesting
    protected long getMaxTailCachePreIndexLength() {
        return 0x4000000L;
    }

    @VisibleForTesting
    protected Duration getRecoveryTimeout() {
        return RECOVERY_TIMEOUT;
    }

    @SuppressFBWarnings(justification="generated code")
    public IndexReader getIndexReader() {
        return this.indexReader;
    }

    private static /* synthetic */ CompletionStage lambda$null$12(Supplier persist, Void v) {
        return (CompletableFuture)persist.get();
    }

    private /* synthetic */ CompletableFuture lambda$null$2(DirectSegmentAccess segment, TimeoutTimer timer, Supplier action) {
        return this.isTableSegmentEmpty(segment, timer).thenCompose(arg_0 -> ContainerKeyIndex.lambda$null$1((Supplier)action, segment, arg_0));
    }

    private static /* synthetic */ CompletionStage lambda$null$1(Supplier action, DirectSegmentAccess segment, Boolean isEmpty) {
        if (isEmpty.booleanValue()) {
            return (CompletionStage)action.get();
        }
        return Futures.failedFuture((Throwable)new TableSegmentNotEmptyException(segment.getInfo().getName()));
    }

    @ThreadSafe
    private class RecoveryTracker
    implements AutoCloseable {
        @GuardedBy(value="this")
        private final HashSet<Long> recoveredSegments = new HashSet();
        @GuardedBy(value="this")
        private final HashMap<Long, RecoveryTask> recoveryTasks = new HashMap();

        private RecoveryTracker() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() {
            ArrayList<RecoveryTask> toCancel;
            RecoveryTracker recoveryTracker = this;
            synchronized (recoveryTracker) {
                toCancel = new ArrayList<RecoveryTask>(this.recoveryTasks.values());
                this.recoveryTasks.clear();
            }
            ObjectClosedException ex = new ObjectClosedException((Object)ContainerKeyIndex.this);
            toCancel.forEach(task -> {
                task.task.completeExceptionally(ex);
                log.info("{}: Cancelled one or more tasks that were waiting on Table Segment {} recovery.", (Object)ContainerKeyIndex.this.traceObjectId, (Object)task.segmentId);
            });
        }

        void updateSegmentIndexOffset(long segmentId, long indexOffset) {
            this.updateSegmentIndexOffset(segmentId, indexOffset, false);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void updateSegmentIndexOffset(long segmentId, long indexOffset, boolean cacheUpdated) {
            RecoveryTask task;
            boolean removed = indexOffset < 0L;
            RecoveryTracker recoveryTracker = this;
            synchronized (recoveryTracker) {
                task = this.recoveryTasks.get(segmentId);
                if (removed) {
                    this.recoveredSegments.remove(segmentId);
                }
                if (task != null && !removed) {
                    if (indexOffset < task.triggerIndexOffset) {
                        log.debug("{}: For TableSegment {}, IndexOffset={}, TriggerOffset={}.", new Object[]{ContainerKeyIndex.this.traceObjectId, segmentId, indexOffset, task.triggerIndexOffset});
                        task = null;
                    } else {
                        this.recoveredSegments.add(segmentId);
                    }
                }
            }
            if (task != null) {
                if (removed) {
                    log.debug("{}: TableSegment {} evicted; cancelling dependent tasks.", (Object)ContainerKeyIndex.this.traceObjectId, (Object)segmentId);
                    task.task.cancel(true);
                } else {
                    log.debug("{}: TableSegment {} fully recovered; triggering dependent tasks.", (Object)ContainerKeyIndex.this.traceObjectId, (Object)segmentId);
                    task.task.complete(cacheUpdated);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        <T> CompletableFuture<T> waitIfNeeded(DirectSegmentAccess segment, Function<Boolean, CompletableFuture<T>> toExecute) {
            RecoveryTask task = null;
            long segmentLength = -1L;
            long lastIndexedOffset = -1L;
            boolean firstTask = false;
            RecoveryTracker recoveryTracker = this;
            synchronized (recoveryTracker) {
                if (!this.recoveredSegments.contains(segment.getSegmentId()) && (task = this.recoveryTasks.get(segment.getSegmentId())) == null) {
                    SegmentProperties sp = segment.getInfo();
                    segmentLength = sp.getLength();
                    lastIndexedOffset = ContainerKeyIndex.this.indexReader.getLastIndexedOffset(sp);
                    if (lastIndexedOffset >= segmentLength) {
                        this.recoveredSegments.add(segment.getSegmentId());
                    } else {
                        task = new RecoveryTask(segment.getSegmentId(), segmentLength);
                        this.recoveryTasks.put(segment.getSegmentId(), task);
                        firstTask = true;
                    }
                }
            }
            if (task == null) {
                return toExecute.apply(false);
            }
            log.debug("{}: TableSegment {} is not fully recovered. Queuing 1 task.", (Object)ContainerKeyIndex.this.traceObjectId, (Object)segment.getSegmentId());
            if (firstTask) {
                this.setupRecoveryTask(task);
                assert (lastIndexedOffset >= 0L);
                ContainerKeyIndex.this.triggerCacheTailIndex(segment, lastIndexedOffset, segmentLength);
            }
            return task.task.thenComposeAsync(toExecute, (Executor)ContainerKeyIndex.this.executor);
        }

        private void setupRecoveryTask(RecoveryTask task) {
            ScheduledFuture<Boolean> sf = ContainerKeyIndex.this.executor.schedule(() -> task.task.completeExceptionally(new TimeoutException(String.format("Table Segment %d recovery timed out.", task.segmentId))), ContainerKeyIndex.this.getRecoveryTimeout().toMillis(), TimeUnit.MILLISECONDS);
            task.task.whenComplete((r, ex) -> {
                sf.cancel(true);
                RecoveryTracker recoveryTracker = this;
                synchronized (recoveryTracker) {
                    RecoveryTask removed = this.recoveryTasks.remove(task.segmentId);
                    if (removed != task) {
                        this.recoveryTasks.put(task.segmentId, removed);
                    }
                }
            });
        }

        private class RecoveryTask {
            final long segmentId;
            final long triggerIndexOffset;
            final CompletableFuture<Boolean> task = new CompletableFuture();

            @ConstructorProperties(value={"segmentId", "triggerIndexOffset"})
            @SuppressFBWarnings(justification="generated code")
            public RecoveryTask(long segmentId, long triggerIndexOffset) {
                this.segmentId = segmentId;
                this.triggerIndexOffset = triggerIndexOffset;
            }
        }
    }
}

