/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.tables;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.BitConverter;
import io.pravega.common.util.ByteArraySegment;
import java.beans.ConstructorProperties;
import java.util.UUID;
import java.util.function.Function;
import lombok.NonNull;

abstract class KeyHasher {
    static final int HASH_SIZE_BYTES = 16;
    static final UUID MIN_HASH = new UUID(-9223372036854775807L, Long.MIN_VALUE);
    static final UUID MAX_HASH = new UUID(0x7FFFFFFFFFFFFFFEL, Long.MAX_VALUE);

    KeyHasher() {
    }

    public UUID hash(@NonNull byte[] key) {
        if (key == null) {
            throw new NullPointerException("key is marked @NonNull but is null");
        }
        return this.hash((ArrayView)new ByteArraySegment(key));
    }

    public abstract UUID hash(@NonNull ArrayView var1);

    protected UUID toUUID(byte[] rawHash) {
        assert (rawHash.length == 16);
        long msb = BitConverter.readLong((byte[])rawHash, (int)0);
        long lsb = BitConverter.readLong((byte[])rawHash, (int)8);
        if (msb == Long.MIN_VALUE) {
            ++msb;
        } else if (msb == Long.MAX_VALUE) {
            --msb;
        }
        return new UUID(msb, lsb);
    }

    static UUID getNextHash(UUID hash) {
        if (hash == null) {
            hash = MIN_HASH;
        } else if (hash.compareTo(MAX_HASH) >= 0) {
            return null;
        }
        long msb = hash.getMostSignificantBits();
        long lsb = hash.getLeastSignificantBits();
        if (lsb == Long.MAX_VALUE) {
            ++msb;
            lsb = Long.MIN_VALUE;
        } else {
            ++lsb;
        }
        return new UUID(msb, lsb);
    }

    static boolean isValid(UUID keyHash) {
        return MIN_HASH.compareTo(keyHash) <= 0 && MAX_HASH.compareTo(keyHash) >= 0;
    }

    static KeyHasher sha256() {
        return new Sha256Hasher();
    }

    @VisibleForTesting
    static KeyHasher custom(Function<ArrayView, byte[]> hashFunction) {
        return new CustomHasher(hashFunction);
    }

    private static class CustomHasher
    extends KeyHasher {
        @NonNull
        private final Function<ArrayView, byte[]> hashFunction;

        @Override
        public UUID hash(@NonNull ArrayView key) {
            if (key == null) {
                throw new NullPointerException("key is marked @NonNull but is null");
            }
            byte[] rawHash = this.hashFunction.apply(key);
            Preconditions.checkState((rawHash.length == 16 ? 1 : 0) != 0, (Object)"Resulting KeyHash has incorrect length.");
            return this.toUUID(rawHash);
        }

        @ConstructorProperties(value={"hashFunction"})
        @SuppressFBWarnings(justification="generated code")
        public CustomHasher(@NonNull Function<ArrayView, byte[]> hashFunction) {
            if (hashFunction == null) {
                throw new NullPointerException("hashFunction is marked @NonNull but is null");
            }
            this.hashFunction = hashFunction;
        }
    }

    private static class Sha256Hasher
    extends KeyHasher {
        private static final HashFunction HASH = Hashing.sha256();

        private Sha256Hasher() {
        }

        @Override
        public UUID hash(@NonNull ArrayView key) {
            if (key == null) {
                throw new NullPointerException("key is marked @NonNull but is null");
            }
            byte[] rawHash = new byte[16];
            int c = HASH.hashBytes(key.array(), key.arrayOffset(), key.getLength()).writeBytesTo(rawHash, 0, rawHash.length);
            assert (c == rawHash.length);
            return this.toUUID(rawHash);
        }
    }
}

