/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.tables;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.MathHelpers;
import io.pravega.common.TimeoutTimer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.BufferView;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.HashedArray;
import io.pravega.segmentstore.contracts.AttributeUpdate;
import io.pravega.segmentstore.contracts.AttributeUpdateType;
import io.pravega.segmentstore.contracts.ReadResult;
import io.pravega.segmentstore.contracts.SegmentProperties;
import io.pravega.segmentstore.contracts.tables.TableAttributes;
import io.pravega.segmentstore.contracts.tables.TableEntry;
import io.pravega.segmentstore.contracts.tables.TableKey;
import io.pravega.segmentstore.server.DataCorruptionException;
import io.pravega.segmentstore.server.DirectSegmentAccess;
import io.pravega.segmentstore.server.reading.AsyncReadResultProcessor;
import io.pravega.segmentstore.server.tables.AsyncTableEntryReader;
import io.pravega.segmentstore.server.tables.IndexReader;
import io.pravega.segmentstore.server.tables.TableBucket;
import io.pravega.segmentstore.server.tables.TableBucketReader;
import io.pravega.segmentstore.server.tables.TableWriterConnector;
import java.beans.ConstructorProperties;
import java.io.EOFException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.BiConsumer;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TableCompactor {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(TableCompactor.class);
    @NonNull
    private final TableWriterConnector connector;
    @NonNull
    private final IndexReader indexReader;
    @NonNull
    private final Executor executor;

    boolean isCompactionRequired(SegmentProperties info) {
        long startOffset = this.getCompactionStartOffset(info);
        long lastIndexOffset = this.indexReader.getLastIndexedOffset(info);
        if (startOffset + (long)this.connector.getMaxCompactionSize() >= lastIndexOffset) {
            return false;
        }
        long totalEntryCount = this.indexReader.getTotalEntryCount(info);
        long entryCount = this.indexReader.getEntryCount(info);
        long utilization = totalEntryCount == 0L ? 100L : MathHelpers.minMax((long)Math.round(100.0 * (double)entryCount / (double)totalEntryCount), (long)0L, (long)100L);
        long utilizationThreshold = (int)MathHelpers.minMax((long)this.indexReader.getCompactionUtilizationThreshold(info), (long)0L, (long)100L);
        return utilization < utilizationThreshold;
    }

    long calculateTruncationOffset(SegmentProperties info, long highestCopiedOffset) {
        long truncateOffset = -1L;
        if (highestCopiedOffset > 0L) {
            truncateOffset = highestCopiedOffset;
        } else if (this.indexReader.getLastIndexedOffset(info) >= info.getLength()) {
            truncateOffset = this.indexReader.getCompactionOffset(info);
        }
        if (truncateOffset <= info.getStartOffset()) {
            truncateOffset = -1L;
        }
        return truncateOffset;
    }

    CompletableFuture<Void> compact(@NonNull DirectSegmentAccess segment, TimeoutTimer timer) {
        if (segment == null) {
            throw new NullPointerException("segment is marked @NonNull but is null");
        }
        SegmentProperties info = segment.getInfo();
        long startOffset = this.getCompactionStartOffset(info);
        int maxLength = (int)Math.min((long)this.connector.getMaxCompactionSize(), this.indexReader.getLastIndexedOffset(info) - startOffset);
        if (startOffset < 0L || maxLength < 0) {
            return Futures.failedFuture((Throwable)((Object)new DataCorruptionException(String.format("Segment[%s] (%s) has CompactionStartOffset=%s and CompactionLength=%s.", segment.getSegmentId(), info.getName(), startOffset, maxLength), new Object[0])));
        }
        if (maxLength == 0) {
            log.debug("TableCompactor[{}]: Up to date.", (Object)segment.getSegmentId());
            return CompletableFuture.completedFuture(null);
        }
        return this.readCandidates(segment, startOffset, maxLength, timer).thenComposeAsync(candidates -> ((CompletableFuture)this.indexReader.locateBuckets(segment, candidates.candidates.keySet(), timer).thenComposeAsync(buckets -> this.excludeObsolete(segment, (CompactionArgs)candidates, (Map<UUID, TableBucket>)buckets, timer), this.executor)).thenComposeAsync(v -> this.copyCandidates(segment, (CompactionArgs)candidates, timer), this.executor), this.executor);
    }

    private CompletableFuture<CompactionArgs> readCandidates(DirectSegmentAccess segment, long startOffset, int maxLength, TimeoutTimer timer) {
        ReadResult rr = segment.read(startOffset, maxLength, timer.getRemaining());
        return AsyncReadResultProcessor.processAll(rr, this.executor, timer.getRemaining()).thenApply(inputStream -> this.parseEntries((InputStream)inputStream, startOffset, maxLength));
    }

    private CompactionArgs parseEntries(InputStream input, long startOffset, int maxLength) {
        long nextOffset;
        HashMap<UUID, CandidateSet> entries = new HashMap<UUID, CandidateSet>();
        int count = 0;
        long maxOffset = startOffset + (long)maxLength;
        try {
            AsyncTableEntryReader.DeserializedEntry e;
            for (nextOffset = startOffset; nextOffset < maxOffset; nextOffset += (long)e.getHeader().getTotalLength()) {
                e = AsyncTableEntryReader.readEntryComponents(input, nextOffset, this.connector.getSerializer());
                if (!e.getHeader().isDeletion()) {
                    UUID hash = this.connector.getKeyHasher().hash(e.getKey());
                    CandidateSet candidates = entries.computeIfAbsent(hash, h -> new CandidateSet());
                    candidates.add(new Candidate(nextOffset, TableEntry.versioned((ArrayView)new ByteArraySegment(e.getKey()), (ArrayView)new ByteArraySegment(e.getValue()), (long)e.getVersion())));
                }
                ++count;
            }
        }
        catch (EOFException ex) {
            input.close();
        }
        return new CompactionArgs(startOffset, nextOffset, count, entries);
    }

    private CompletableFuture<Void> excludeObsolete(DirectSegmentAccess segment, CompactionArgs args, Map<UUID, TableBucket> buckets, TimeoutTimer timer) {
        args.candidates.keySet().removeIf(hash -> {
            TableBucket bucket = (TableBucket)buckets.get(hash);
            return bucket == null || !bucket.exists();
        });
        TableBucketReader<TableKey> br = TableBucketReader.key(segment, this.indexReader::getBackpointerOffset, this.executor);
        Iterator<Map.Entry<UUID, CandidateSet>> candidates = args.candidates.entrySet().iterator();
        return Futures.loop(candidates::hasNext, () -> {
            Map.Entry e = (Map.Entry)candidates.next();
            long bucketOffset = ((TableBucket)buckets.get(e.getKey())).getSegmentOffset();
            BiConsumer<TableKey, Long> handler = (key, offset) -> ((CandidateSet)e.getValue()).handleExistingKey((TableKey)key, (long)offset);
            return br.findAll(bucketOffset, handler, timer);
        }, (Executor)this.executor);
    }

    private CompletableFuture<Void> copyCandidates(DirectSegmentAccess segment, CompactionArgs args, TimeoutTimer timer) {
        CompletableFuture<Object> result;
        ArrayList<TableEntry> toWrite = new ArrayList<TableEntry>();
        int totalLength = 0;
        for (CandidateSet list : args.candidates.values()) {
            for (Candidate c2 : list.getAll()) {
                toWrite.add(c2.entry);
                totalLength += this.connector.getSerializer().getUpdateLength(c2.entry);
            }
        }
        Collection<AttributeUpdate> attributes = this.generateAttributeUpdates(args);
        if (totalLength == 0) {
            assert (toWrite.size() == 0);
            result = segment.updateAttributes(attributes, timer.getRemaining());
        } else {
            toWrite.sort(Comparator.comparingLong(c -> c.getKey().getVersion()));
            byte[] appendData = new byte[totalLength];
            this.connector.getSerializer().serializeUpdateWithExplicitVersion(toWrite, appendData);
            result = segment.append((BufferView)new ByteArraySegment(appendData), attributes, timer.getRemaining());
            log.debug("TableCompactor[{}]: Compacting {}, CopyCount={}, CopyLength={}.", new Object[]{segment.getSegmentId(), args, toWrite, totalLength});
        }
        return Futures.toVoid(result);
    }

    private Collection<AttributeUpdate> generateAttributeUpdates(CompactionArgs candidates) {
        return Arrays.asList(new AttributeUpdate(TableAttributes.COMPACTION_OFFSET, AttributeUpdateType.ReplaceIfEquals, candidates.endOffset, candidates.startOffset), new AttributeUpdate(TableAttributes.TOTAL_ENTRY_COUNT, AttributeUpdateType.Accumulate, (long)(-candidates.count)));
    }

    private long getCompactionStartOffset(SegmentProperties info) {
        return Math.max(this.indexReader.getCompactionOffset(info), info.getStartOffset());
    }

    @ConstructorProperties(value={"connector", "indexReader", "executor"})
    @SuppressFBWarnings(justification="generated code")
    public TableCompactor(@NonNull TableWriterConnector connector, @NonNull IndexReader indexReader, @NonNull Executor executor) {
        if (connector == null) {
            throw new NullPointerException("connector is marked @NonNull but is null");
        }
        if (indexReader == null) {
            throw new NullPointerException("indexReader is marked @NonNull but is null");
        }
        if (executor == null) {
            throw new NullPointerException("executor is marked @NonNull but is null");
        }
        this.connector = connector;
        this.indexReader = indexReader;
        this.executor = executor;
    }

    private static class Candidate {
        final long segmentOffset;
        final TableEntry entry;

        public String toString() {
            return String.format("Offset = %d, Entry = {%s}", this.segmentOffset, this.entry);
        }

        @ConstructorProperties(value={"segmentOffset", "entry"})
        @SuppressFBWarnings(justification="generated code")
        public Candidate(long segmentOffset, TableEntry entry) {
            this.segmentOffset = segmentOffset;
            this.entry = entry;
        }
    }

    private static class CandidateSet {
        final Map<HashedArray, Candidate> byKey = new HashMap<HashedArray, Candidate>();

        private CandidateSet() {
        }

        void add(Candidate c) {
            HashedArray key = new HashedArray(c.entry.getKey().getKey());
            Candidate existing = this.byKey.get(key);
            if (existing == null || existing.entry.getKey().getVersion() < c.entry.getKey().getVersion()) {
                this.byKey.put(key, c);
            }
        }

        void handleExistingKey(TableKey existingKey, long existingKeyOffset) {
            HashedArray key = new HashedArray(existingKey.getKey());
            Candidate c = this.byKey.get(key);
            if (c != null && c.segmentOffset < existingKeyOffset) {
                this.byKey.remove(key);
            }
        }

        Collection<Candidate> getAll() {
            return this.byKey.values();
        }
    }

    private static class CompactionArgs {
        final long startOffset;
        final long endOffset;
        final int count;
        final Map<UUID, CandidateSet> candidates;

        public String toString() {
            return String.format("StartOffset=%s, EndOffset=%s, ProcessedCount=%s, CandidateCount=%s", this.startOffset, this.endOffset, this.count, this.candidates.size());
        }

        @ConstructorProperties(value={"startOffset", "endOffset", "count", "candidates"})
        @SuppressFBWarnings(justification="generated code")
        public CompactionArgs(long startOffset, long endOffset, int count, Map<UUID, CandidateSet> candidates) {
            this.startOffset = startOffset;
            this.endOffset = endOffset;
            this.count = count;
            this.candidates = candidates;
        }
    }
}

