/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.tables;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterators;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.TimeoutTimer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.HashedArray;
import io.pravega.segmentstore.contracts.BadAttributeUpdateException;
import io.pravega.segmentstore.contracts.ReadResult;
import io.pravega.segmentstore.contracts.SegmentProperties;
import io.pravega.segmentstore.contracts.tables.TableAttributes;
import io.pravega.segmentstore.server.DataCorruptionException;
import io.pravega.segmentstore.server.DirectSegmentAccess;
import io.pravega.segmentstore.server.SegmentOperation;
import io.pravega.segmentstore.server.WriterFlushResult;
import io.pravega.segmentstore.server.WriterSegmentProcessor;
import io.pravega.segmentstore.server.logs.operations.CachedStreamSegmentAppendOperation;
import io.pravega.segmentstore.server.tables.AsyncTableEntryReader;
import io.pravega.segmentstore.server.tables.BucketUpdate;
import io.pravega.segmentstore.server.tables.IndexWriter;
import io.pravega.segmentstore.server.tables.KeyUpdateCollection;
import io.pravega.segmentstore.server.tables.TableBucketReader;
import io.pravega.segmentstore.server.tables.TableCompactor;
import io.pravega.segmentstore.server.tables.TableWriterConnector;
import java.beans.ConstructorProperties;
import java.io.IOException;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WriterTableProcessor
implements WriterSegmentProcessor {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(WriterTableProcessor.class);
    private final TableWriterConnector connector;
    private final IndexWriter indexWriter;
    private final ScheduledExecutorService executor;
    private final OperationAggregator aggregator;
    private final AtomicLong lastAddedOffset;
    private final AtomicBoolean closed;
    private final String traceObjectId;
    private final TableCompactor compactor;

    WriterTableProcessor(@NonNull TableWriterConnector connector, @NonNull ScheduledExecutorService executor) {
        if (connector == null) {
            throw new NullPointerException("connector is marked @NonNull but is null");
        }
        if (executor == null) {
            throw new NullPointerException("executor is marked @NonNull but is null");
        }
        this.connector = connector;
        this.executor = executor;
        this.indexWriter = new IndexWriter(connector.getKeyHasher(), executor);
        this.aggregator = new OperationAggregator(this.indexWriter.getLastIndexedOffset(this.connector.getMetadata()));
        this.lastAddedOffset = new AtomicLong(-1L);
        this.closed = new AtomicBoolean();
        this.traceObjectId = String.format("TableProcessor[%d-%d]", this.connector.getMetadata().getContainerId(), this.connector.getMetadata().getId());
        this.compactor = new TableCompactor(connector, this.indexWriter, this.executor);
    }

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true)) {
            this.connector.close();
            log.info("{}: Closed.", (Object)this.traceObjectId);
        }
    }

    @Override
    public void add(SegmentOperation operation) throws DataCorruptionException {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkArgument((operation.getStreamSegmentId() == this.connector.getMetadata().getId() ? 1 : 0) != 0, (String)"Operation '%s' refers to a different Segment than this one (%s).", (Object)operation, (long)this.connector.getMetadata().getId());
        Preconditions.checkArgument((operation.getSequenceNumber() != Long.MIN_VALUE ? 1 : 0) != 0, (String)"Operation '%s' does not have a Sequence Number assigned.", (Object)operation);
        if (this.connector.getMetadata().isDeleted() || !(operation instanceof CachedStreamSegmentAppendOperation)) {
            return;
        }
        CachedStreamSegmentAppendOperation append = (CachedStreamSegmentAppendOperation)operation;
        if (this.lastAddedOffset.get() >= 0L) {
            if (this.lastAddedOffset.get() != append.getStreamSegmentOffset()) {
                throw new DataCorruptionException(String.format("Wrong offset for Operation '%s'. Expected: %s, actual: %d.", operation, this.lastAddedOffset, append.getStreamSegmentOffset()), new Object[0]);
            }
        } else if (this.aggregator.getLastIndexedOffset() < append.getStreamSegmentOffset()) {
            throw new DataCorruptionException(String.format("Operation '%s' begins after TABLE_INDEXED_OFFSET. Expected: %s, actual: %d.", operation, this.aggregator.getLastIndexedOffset(), append.getStreamSegmentOffset()), new Object[0]);
        }
        if (append.getStreamSegmentOffset() >= this.aggregator.getLastIndexedOffset()) {
            this.aggregator.add(append);
            this.lastAddedOffset.set(append.getLastStreamSegmentOffset());
            log.debug("{}: Add {} (State={}).", new Object[]{this.traceObjectId, operation, this.aggregator});
        } else {
            log.debug("{}: Skipped {} (State={}).", new Object[]{this.traceObjectId, operation, this.aggregator});
        }
    }

    @Override
    public boolean isClosed() {
        return this.closed.get();
    }

    @Override
    public long getLowestUncommittedSequenceNumber() {
        return this.aggregator.getFirstSequenceNumber();
    }

    @Override
    public boolean mustFlush() {
        if (this.connector.getMetadata().isDeleted()) {
            return false;
        }
        return !this.aggregator.isEmpty();
    }

    @Override
    public CompletableFuture<WriterFlushResult> flush(Duration timeout) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        TimeoutTimer timer = new TimeoutTimer(timeout);
        return this.connector.getSegment(timer.getRemaining()).thenComposeAsync(segment -> this.flushWithSingleRetry((DirectSegmentAccess)segment, timer).thenComposeAsync(flushResult -> {
            this.flushComplete(flushResult.lastIndexedOffset);
            return this.compactIfNeeded((DirectSegmentAccess)segment, flushResult.highestCopiedOffset, timer).thenApply(v -> flushResult);
        }, (Executor)this.executor), (Executor)this.executor);
    }

    public String toString() {
        return String.format("[%d: %s] Count = %d, LastOffset = %s, LUSN = %d", this.connector.getMetadata().getId(), this.connector.getMetadata().getName(), this.aggregator.size(), this.lastAddedOffset, this.getLowestUncommittedSequenceNumber());
    }

    private CompletableFuture<Void> compactIfNeeded(DirectSegmentAccess segment, long highestCopiedOffset, TimeoutTimer timer) {
        CompletableFuture<Void> result;
        SegmentProperties info = segment.getInfo();
        if (this.compactor.isCompactionRequired(info)) {
            result = this.compactor.compact(segment, timer);
        } else {
            log.debug("{}: No compaction required at this time.", (Object)this.traceObjectId);
            result = CompletableFuture.completedFuture(null);
        }
        return ((CompletableFuture)result.thenComposeAsync(v -> {
            long truncateOffset = this.compactor.calculateTruncationOffset(segment.getInfo(), highestCopiedOffset);
            if (truncateOffset > 0L) {
                log.debug("{}: Truncating segment at offset {}.", (Object)this.traceObjectId, (Object)truncateOffset);
                return segment.truncate(truncateOffset, timer.getRemaining());
            }
            log.debug("{}: No segment truncation possible now.", (Object)this.traceObjectId);
            return CompletableFuture.completedFuture(null);
        }, (Executor)this.executor)).exceptionally(ex -> {
            log.error("{}: Compaction failed.", (Object)this.traceObjectId, ex);
            return null;
        });
    }

    private CompletableFuture<TableWriterFlushResult> flushWithSingleRetry(DirectSegmentAccess segment, TimeoutTimer timer) {
        return Futures.exceptionallyComposeExpecting(this.flushOnce(segment, timer), this::canRetryFlushException, () -> {
            this.reconcileTableIndexOffset();
            return this.flushOnce(segment, timer);
        });
    }

    private void flushComplete(long lastIndexedOffset) {
        this.aggregator.reset();
        this.aggregator.setLastIndexedOffset(lastIndexedOffset);
        this.connector.notifyIndexOffsetChanged(this.aggregator.getLastIndexedOffset());
        log.debug("{}: FlushComplete (State={}).", (Object)this.traceObjectId, (Object)this.aggregator);
    }

    private CompletableFuture<TableWriterFlushResult> flushOnce(DirectSegmentAccess segment, TimeoutTimer timer) {
        KeyUpdateCollection keyUpdates = this.readKeysFromSegment(segment, this.aggregator.getFirstOffset(), this.aggregator.getLastOffset(), timer);
        log.debug("{}: Flush.ReadFromSegment {} UpdateKeys(s).", (Object)this.traceObjectId, (Object)keyUpdates.getUpdates().size());
        return ((CompletableFuture)this.indexWriter.groupByBucket(segment, keyUpdates.getUpdates(), timer).thenComposeAsync(builders -> this.fetchExistingKeys((Collection<BucketUpdate.Builder>)builders, segment, timer).thenComposeAsync(v -> {
            List<BucketUpdate> bucketUpdates = builders.stream().map(BucketUpdate.Builder::build).collect(Collectors.toList());
            this.logBucketUpdates(bucketUpdates);
            return this.indexWriter.updateBuckets(segment, bucketUpdates, this.aggregator.getLastIndexedOffset(), keyUpdates.getLastIndexedOffset(), keyUpdates.getTotalUpdateCount(), timer.getRemaining());
        }, (Executor)this.executor), (Executor)this.executor)).thenApply(ignored -> new TableWriterFlushResult(keyUpdates.getLastIndexedOffset(), keyUpdates.getHighestCopiedOffset()));
    }

    private void reconcileTableIndexOffset() {
        long tableIndexOffset = this.indexWriter.getLastIndexedOffset(this.connector.getMetadata());
        if (tableIndexOffset < this.aggregator.getLastIndexedOffset()) {
            throw new DataCorruptionException(String.format("Cannot reconcile INDEX_OFFSET attribute (%s) for Segment '%s'. It is lower than our known value (%s).", tableIndexOffset, this.connector.getMetadata().getId(), this.aggregator.getLastIndexedOffset()), new Object[0]);
        }
        if (!this.aggregator.setLastIndexedOffset(tableIndexOffset)) {
            throw new DataCorruptionException(String.format("Cannot reconcile INDEX_OFFSET attribute (%s) for Segment '%s'. Most likely it does not conform to an append boundary.  Existing value: %s.", tableIndexOffset, this.connector.getMetadata().getId(), this.aggregator.getLastIndexedOffset()), new Object[0]);
        }
        log.info("{}: ReconcileTableIndexOffset (State={}).", (Object)this.traceObjectId, (Object)this.aggregator);
    }

    private boolean canRetryFlushException(Throwable ex) {
        if (ex instanceof BadAttributeUpdateException) {
            BadAttributeUpdateException bau = (BadAttributeUpdateException)ex;
            return bau.getAttributeId() != null && bau.getAttributeId().equals(TableAttributes.INDEX_OFFSET);
        }
        return false;
    }

    private KeyUpdateCollection readKeysFromSegment(DirectSegmentAccess segment, long firstOffset, long lastOffset, TimeoutTimer timer) {
        KeyUpdateCollection keyUpdates = new KeyUpdateCollection();
        try (InputStream input = this.readFromInMemorySegment(segment, firstOffset, lastOffset, timer);){
            for (long segmentOffset = firstOffset; segmentOffset < lastOffset; segmentOffset += (long)this.indexSingleKey(input, segmentOffset, keyUpdates)) {
            }
        }
        return keyUpdates;
    }

    private int indexSingleKey(InputStream input, long entryOffset, KeyUpdateCollection keyUpdateCollection) throws IOException {
        AsyncTableEntryReader.DeserializedEntry e = AsyncTableEntryReader.readEntryComponents(input, entryOffset, this.connector.getSerializer());
        HashedArray key = new HashedArray(e.getKey());
        BucketUpdate.KeyUpdate update = new BucketUpdate.KeyUpdate(key, entryOffset, e.getVersion(), e.getHeader().isDeletion());
        keyUpdateCollection.add(update, e.getHeader().getTotalLength(), e.getHeader().getEntryVersion());
        return e.getHeader().getTotalLength();
    }

    private InputStream readFromInMemorySegment(DirectSegmentAccess segment, long startOffset, long endOffset, TimeoutTimer timer) {
        long readOffset = startOffset;
        long remainingLength = endOffset - startOffset;
        ArrayList inputs = new ArrayList();
        while (remainingLength > 0L) {
            int readLength = (int)Math.min(remainingLength, Integer.MAX_VALUE);
            ReadResult readResult = segment.read(readOffset, readLength, timer.getRemaining());
            Throwable throwable = null;
            try {
                inputs.addAll(readResult.readRemaining(readLength, timer.getRemaining()));
                assert (readResult.getConsumedLength() == readLength) : "Expecting a full read (from memory).";
                remainingLength -= (long)readResult.getConsumedLength();
                readOffset += (long)readResult.getConsumedLength();
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (readResult == null) continue;
                if (throwable != null) {
                    try {
                        readResult.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                readResult.close();
            }
        }
        return new SequenceInputStream(Iterators.asEnumeration(inputs.iterator()));
    }

    private CompletableFuture<Void> fetchExistingKeys(Collection<BucketUpdate.Builder> builders, DirectSegmentAccess segment, TimeoutTimer timer) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        return Futures.loop(builders, bucketUpdate -> this.fetchExistingKeys((BucketUpdate.Builder)bucketUpdate, segment, timer).thenApply(v -> true), (Executor)this.executor);
    }

    private CompletableFuture<Void> fetchExistingKeys(BucketUpdate.Builder builder, DirectSegmentAccess segment, TimeoutTimer timer) {
        return TableBucketReader.key(segment, this.indexWriter::getBackpointerOffset, this.executor).findAll(builder.getBucket().getSegmentOffset(), (key, offset) -> builder.withExistingKey(new BucketUpdate.KeyInfo(new HashedArray(key.getKey()), (long)offset, key.getVersion())), timer);
    }

    private void logBucketUpdates(Collection<BucketUpdate> bucketUpdates) {
        if (!log.isTraceEnabled()) {
            return;
        }
        log.trace("{}: Updating {} TableBucket(s).", (Object)this.traceObjectId, (Object)bucketUpdates.size());
        bucketUpdates.forEach(bu -> log.trace("{}: TableBucket [Offset={}, {}]: ExistingKeys=[{}], Updates=[{}].", new Object[]{this.traceObjectId, bu.getBucketOffset(), bu.getBucket(), bu.getExistingKeys().stream().map(Object::toString).collect(Collectors.joining("; ")), bu.getKeyUpdates().stream().map(Object::toString).collect(Collectors.joining("; "))}));
    }

    private static class TableWriterFlushResult
    extends WriterFlushResult {
        final long lastIndexedOffset;
        final long highestCopiedOffset;

        @ConstructorProperties(value={"lastIndexedOffset", "highestCopiedOffset"})
        @SuppressFBWarnings(justification="generated code")
        public TableWriterFlushResult(long lastIndexedOffset, long highestCopiedOffset) {
            this.lastIndexedOffset = lastIndexedOffset;
            this.highestCopiedOffset = highestCopiedOffset;
        }
    }

    @ThreadSafe
    private static class OperationAggregator {
        @GuardedBy(value="this")
        private long firstSeqNo;
        @GuardedBy(value="this")
        private long lastOffset;
        @GuardedBy(value="this")
        private long lastIndexedOffset;
        @GuardedBy(value="this")
        private final ArrayList<Long> appendOffsets = new ArrayList();

        OperationAggregator(long lastIndexedOffset) {
            this.reset();
            this.lastIndexedOffset = lastIndexedOffset;
        }

        synchronized void reset() {
            this.firstSeqNo = Long.MIN_VALUE;
            this.lastOffset = -1L;
            this.appendOffsets.clear();
        }

        synchronized void add(CachedStreamSegmentAppendOperation op) {
            if (this.appendOffsets.size() == 0) {
                this.firstSeqNo = op.getSequenceNumber();
            }
            this.lastOffset = op.getLastStreamSegmentOffset();
            this.appendOffsets.add(op.getStreamSegmentOffset());
        }

        synchronized boolean isEmpty() {
            return this.appendOffsets.isEmpty();
        }

        synchronized long getFirstSequenceNumber() {
            return this.firstSeqNo;
        }

        synchronized long getFirstOffset() {
            return this.appendOffsets.size() == 0 ? -1L : this.appendOffsets.get(0);
        }

        synchronized long getLastOffset() {
            return this.lastOffset;
        }

        synchronized long getLastIndexedOffset() {
            return this.lastIndexedOffset;
        }

        synchronized boolean setLastIndexedOffset(long value) {
            if (this.appendOffsets.size() > 0) {
                if (value >= this.getLastOffset()) {
                    this.reset();
                } else {
                    int index = this.appendOffsets.indexOf(value);
                    if (index < 0) {
                        return false;
                    }
                    this.appendOffsets.subList(0, index).clear();
                }
            }
            if (value >= 0L) {
                this.lastIndexedOffset = value;
            }
            return true;
        }

        synchronized int size() {
            return this.appendOffsets.size();
        }

        public synchronized String toString() {
            return String.format("Count = %d, FirstSN = %d, FirstOffset = %d, LastOffset = %d, LIdx = %s", this.appendOffsets.size(), this.firstSeqNo, this.getFirstOffset(), this.getLastOffset(), this.getLastIndexedOffset());
        }
    }
}

