/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.MoreObjects;
import com.google.common.collect.MapMaker;
import com.google.common.collect.Sets;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.buffer.Unpooled;
import io.netty.util.concurrent.FastThreadLocal;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BufferedChannel;
import org.apache.bookkeeper.bookie.BufferedChannelBase;
import org.apache.bookkeeper.bookie.BufferedReadChannel;
import org.apache.bookkeeper.bookie.EntryLogMetadata;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.util.IOUtils;
import org.apache.bookkeeper.util.collections.ConcurrentLongLongHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntryLogger {
    private static final Logger LOG = LoggerFactory.getLogger(EntryLogger.class);
    @VisibleForTesting
    static final int UNINITIALIZED_LOG_ID = -57005;
    volatile File currentDir;
    private final LedgerDirsManager ledgerDirsManager;
    private final boolean entryLogPerLedgerEnabled;
    private final AtomicBoolean shouldCreateNewEntryLog = new AtomicBoolean(false);
    private volatile long leastUnflushedLogId;
    private final Object compactionLogLock = new Object();
    final long logSizeLimit;
    List<BufferedLogChannel> logChannelsToFlush;
    volatile BufferedLogChannel logChannel;
    private volatile BufferedLogChannel compactionLogChannel;
    private final EntryLoggerAllocator entryLoggerAllocator;
    private final boolean entryLogPreAllocationEnabled;
    private final CopyOnWriteArrayList<EntryLogListener> listeners = new CopyOnWriteArrayList();
    private static final int HEADER_V0 = 0;
    private static final int HEADER_V1 = 1;
    private static final int HEADER_CURRENT_VERSION = 1;
    static final int LOGFILE_HEADER_SIZE = 1024;
    final ByteBuf logfileHeader = Unpooled.buffer((int)1024);
    static final int HEADER_VERSION_POSITION = 4;
    static final int LEDGERS_MAP_OFFSET_POSITION = 8;
    static final int LEDGERS_MAP_HEADER_SIZE = 24;
    static final int LEDGERS_MAP_ENTRY_SIZE = 16;
    static final int LEDGERS_MAP_MAX_BATCH_SIZE = 10000;
    static final long INVALID_LID = -1L;
    static final long LEDGERS_MAP_ENTRY_ID = -2L;
    static final int MIN_SANE_ENTRY_SIZE = 16;
    static final long MB = 0x100000L;
    private final int maxSaneEntrySize;
    final ServerConfiguration conf;
    private final ThreadLocal<Map<Long, BufferedReadChannel>> logid2Channel = new ThreadLocal<Map<Long, BufferedReadChannel>>(){

        @Override
        public Map<Long, BufferedReadChannel> initialValue() {
            return new MapMaker().concurrencyLevel(1).weakValues().makeMap();
        }
    };
    private final ConcurrentMap<Long, FileChannel> logid2FileChannel = new ConcurrentHashMap<Long, FileChannel>();
    private final FastThreadLocal<ByteBuf> sizeBuffer = new FastThreadLocal<ByteBuf>(){

        protected ByteBuf initialValue() throws Exception {
            return Unpooled.buffer((int)4);
        }
    };

    public EntryLogger(ServerConfiguration conf, LedgerDirsManager ledgerDirsManager) throws IOException {
        this(conf, ledgerDirsManager, null);
    }

    public EntryLogger(ServerConfiguration conf, LedgerDirsManager ledgerDirsManager, EntryLogListener listener) throws IOException {
        this.maxSaneEntrySize = conf.getNettyMaxFrameSizeBytes() - 500;
        this.ledgerDirsManager = ledgerDirsManager;
        if (listener != null) {
            this.addListener(listener);
        }
        this.logSizeLimit = Math.min(conf.getEntryLogSizeLimit(), 0x40000000L);
        this.entryLogPreAllocationEnabled = conf.isEntryLogFilePreAllocationEnabled();
        this.logfileHeader.writeBytes("BKLO".getBytes(Charsets.UTF_8));
        this.logfileHeader.writeInt(1);
        this.logfileHeader.writerIndex(1024);
        long logId = -1L;
        for (File dir : ledgerDirsManager.getAllLedgerDirs()) {
            if (!dir.exists()) {
                throw new FileNotFoundException("Entry log directory '" + dir + "' does not exist");
            }
            long lastLogId = this.getLastLogId(dir);
            if (lastLogId <= logId) continue;
            logId = lastLogId;
        }
        this.leastUnflushedLogId = logId + 1L;
        this.entryLoggerAllocator = new EntryLoggerAllocator(logId);
        this.conf = conf;
        this.entryLogPerLedgerEnabled = conf.isEntryLogPerLedgerEnabled();
        this.initialize();
    }

    void addListener(EntryLogListener listener) {
        if (null != listener) {
            this.listeners.add(listener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int readFromLogChannel(long entryLogId, BufferedReadChannel channel, ByteBuf buff, long pos) throws IOException {
        BufferedLogChannel bc = this.logChannel;
        if (null != bc && entryLogId == bc.getLogId()) {
            BufferedLogChannel bufferedLogChannel = bc;
            synchronized (bufferedLogChannel) {
                if (pos + (long)buff.writableBytes() >= bc.getFileChannelPosition()) {
                    return bc.read(buff, pos);
                }
            }
        }
        return channel.read(buff, pos);
    }

    public BufferedReadChannel putInReadChannels(long logId, BufferedReadChannel bc) {
        Map<Long, BufferedReadChannel> threadMap = this.logid2Channel.get();
        return threadMap.put(logId, bc);
    }

    public void removeFromChannelsAndClose(long logId) {
        FileChannel fileChannel = (FileChannel)this.logid2FileChannel.remove(logId);
        if (null != fileChannel) {
            try {
                fileChannel.close();
            }
            catch (IOException e) {
                LOG.warn("Exception while closing channel for log file:" + logId);
            }
        }
    }

    public BufferedReadChannel getFromChannels(long logId) {
        return this.logid2Channel.get().get(logId);
    }

    synchronized long getLeastUnflushedLogId() {
        return this.leastUnflushedLogId;
    }

    synchronized long getCurrentLogId() {
        BufferedLogChannel channel = this.logChannel;
        if (null == channel) {
            return -57005L;
        }
        return channel.getLogId();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    File getCurCompactionLogFile() {
        Object object = this.compactionLogLock;
        synchronized (object) {
            if (this.compactionLogChannel == null) {
                return null;
            }
            return this.compactionLogChannel.getLogFile();
        }
    }

    protected void initialize() throws IOException {
        this.ledgerDirsManager.addLedgerDirsListener(this.getLedgerDirsListener());
        if (this.ledgerDirsManager.hasWritableLedgerDirs()) {
            this.createNewLog();
        }
    }

    private LedgerDirsManager.LedgerDirsListener getLedgerDirsListener() {
        return new LedgerDirsManager.LedgerDirsListener(){

            @Override
            public void diskFull(File disk) {
                if (EntryLogger.this.currentDir != null && EntryLogger.this.currentDir.equals(disk)) {
                    EntryLogger.this.shouldCreateNewEntryLog.set(true);
                }
            }

            @Override
            public void diskAlmostFull(File disk) {
                if (EntryLogger.this.currentDir != null && EntryLogger.this.currentDir.equals(disk)) {
                    EntryLogger.this.shouldCreateNewEntryLog.set(true);
                }
            }
        };
    }

    synchronized void rollLog() throws IOException {
        this.createNewLog();
    }

    void createNewLog() throws IOException {
        if (null != this.logChannel) {
            if (null == this.logChannelsToFlush) {
                this.logChannelsToFlush = new LinkedList<BufferedLogChannel>();
            }
            this.logChannel.flush();
            this.appendLedgersMap(this.logChannel);
            BufferedLogChannel newLogChannel = this.entryLoggerAllocator.createNewLog();
            this.logChannelsToFlush.add(this.logChannel);
            LOG.info("Flushing entry logger {} back to filesystem, pending for syncing entry loggers : {}.", (Object)this.logChannel.getLogId(), this.logChannelsToFlush);
            for (EntryLogListener listener : this.listeners) {
                listener.onRotateEntryLog();
            }
            this.logChannel = newLogChannel;
        } else {
            this.logChannel = this.entryLoggerAllocator.createNewLog();
        }
        this.currentDir = this.logChannel.getLogFile().getParentFile();
    }

    EntryLoggerAllocator getEntryLoggerAllocator() {
        return this.entryLoggerAllocator;
    }

    private void appendLedgersMap(final BufferedLogChannel entryLogChannel) throws IOException {
        long ledgerMapOffset = entryLogChannel.position();
        ConcurrentLongLongHashMap ledgersMap = entryLogChannel.getLedgersMap();
        final int numberOfLedgers = (int)ledgersMap.size();
        int maxMapSize = 160024;
        final ByteBuf serializedMap = ByteBufAllocator.DEFAULT.buffer(160024);
        try {
            ledgersMap.forEach(new ConcurrentLongLongHashMap.BiConsumerLong(){
                int remainingLedgers;
                boolean startNewBatch;
                int remainingInBatch;
                {
                    this.remainingLedgers = numberOfLedgers;
                    this.startNewBatch = true;
                    this.remainingInBatch = 0;
                }

                @Override
                public void accept(long ledgerId, long size) {
                    if (this.startNewBatch) {
                        int batchSize = Math.min(this.remainingLedgers, 10000);
                        int ledgerMapSize = 24 + 16 * batchSize;
                        serializedMap.clear();
                        serializedMap.writeInt(ledgerMapSize - 4);
                        serializedMap.writeLong(-1L);
                        serializedMap.writeLong(-2L);
                        serializedMap.writeInt(batchSize);
                        this.startNewBatch = false;
                        this.remainingInBatch = batchSize;
                    }
                    serializedMap.writeLong(ledgerId);
                    serializedMap.writeLong(size);
                    --this.remainingLedgers;
                    if (--this.remainingInBatch == 0) {
                        try {
                            entryLogChannel.write(serializedMap);
                        }
                        catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                        this.startNewBatch = true;
                    }
                }
            });
        }
        catch (RuntimeException e) {
            if (e.getCause() instanceof IOException) {
                throw (IOException)e.getCause();
            }
            throw e;
        }
        finally {
            serializedMap.release();
        }
        entryLogChannel.flush();
        ByteBuffer mapInfo = ByteBuffer.allocate(12);
        mapInfo.putLong(ledgerMapOffset);
        mapInfo.putInt(numberOfLedgers);
        mapInfo.flip();
        entryLogChannel.fileChannel.write(mapInfo, 8L);
    }

    protected boolean removeEntryLog(long entryLogId) {
        File entryLogFile;
        this.removeFromChannelsAndClose(entryLogId);
        try {
            entryLogFile = this.findFile(entryLogId);
        }
        catch (FileNotFoundException e) {
            LOG.error("Trying to delete an entryLog file that could not be found: " + entryLogId + ".log");
            return false;
        }
        if (!entryLogFile.delete()) {
            LOG.warn("Could not delete entry log file {}", (Object)entryLogFile);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setLastLogId(File dir, long logId) throws IOException {
        FileOutputStream fos = new FileOutputStream(new File(dir, "lastId"));
        BufferedWriter bw = new BufferedWriter(new OutputStreamWriter((OutputStream)fos, Charsets.UTF_8));
        try {
            bw.write(Long.toHexString(logId) + "\n");
            bw.flush();
        }
        catch (IOException e) {
            LOG.warn("Failed write lastId file");
        }
        finally {
            try {
                bw.close();
            }
            catch (IOException e) {
                LOG.error("Could not close lastId file in {}", (Object)dir.getPath());
            }
        }
    }

    private long getLastLogId(File dir) {
        long id = this.readLastLogId(dir);
        if (id > 0L) {
            return id;
        }
        File[] logFiles = dir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File file) {
                return file.getName().endsWith(".log");
            }
        });
        ArrayList<Long> logs = new ArrayList<Long>();
        if (logFiles != null) {
            for (File lf : logFiles) {
                long logId = EntryLogger.fileName2LogId(lf.getName());
                logs.add(logId);
            }
        }
        if (0 == logs.size()) {
            return -1L;
        }
        Collections.sort(logs);
        return (Long)logs.get(logs.size() - 1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long readLastLogId(File f) {
        FileInputStream fis;
        try {
            fis = new FileInputStream(new File(f, "lastId"));
        }
        catch (FileNotFoundException e) {
            return -1L;
        }
        BufferedReader br = new BufferedReader(new InputStreamReader((InputStream)fis, Charsets.UTF_8));
        try {
            String lastIdString = br.readLine();
            long l = Long.parseLong(lastIdString, 16);
            return l;
        }
        catch (IOException e) {
            long l = -1L;
            return l;
        }
        catch (NumberFormatException e) {
            long l = -1L;
            return l;
        }
        finally {
            try {
                br.close();
            }
            catch (IOException iOException) {}
        }
    }

    void checkpoint() throws IOException {
        this.flushRotatedLogs();
        if (this.entryLogPerLedgerEnabled) {
            this.flushCurrentLog();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void flushRotatedLogs() throws IOException {
        List<BufferedLogChannel> channels = null;
        long flushedLogId = -1L;
        EntryLogger entryLogger = this;
        synchronized (entryLogger) {
            channels = this.logChannelsToFlush;
            this.logChannelsToFlush = null;
        }
        if (null == channels) {
            return;
        }
        Iterator<BufferedLogChannel> chIter = channels.iterator();
        while (chIter.hasNext()) {
            BufferedLogChannel channel = chIter.next();
            try {
                channel.flushAndForceWrite(false);
            }
            catch (IOException ioe) {
                EntryLogger entryLogger2 = this;
                synchronized (entryLogger2) {
                    if (null == this.logChannelsToFlush) {
                        this.logChannelsToFlush = channels;
                    } else {
                        this.logChannelsToFlush.addAll(0, channels);
                    }
                }
                throw ioe;
            }
            chIter.remove();
            EntryLogger.closeFileChannel(channel);
            if (channel.getLogId() > flushedLogId) {
                flushedLogId = channel.getLogId();
            }
            LOG.info("Synced entry logger {} to disk.", (Object)channel.getLogId());
        }
        this.leastUnflushedLogId = flushedLogId + 1L;
    }

    public void flush() throws IOException {
        this.flushRotatedLogs();
        this.flushCurrentLog();
    }

    synchronized void flushCurrentLog() throws IOException {
        if (this.logChannel != null) {
            this.logChannel.flushAndForceWrite(false);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Flush and sync current entry logger {}.", (Object)this.logChannel.getLogId());
            }
        }
    }

    long addEntry(long ledger, ByteBuffer entry) throws IOException {
        return this.addEntry(ledger, Unpooled.wrappedBuffer((ByteBuffer)entry), true);
    }

    long addEntry(long ledger, ByteBuf entry) throws IOException {
        return this.addEntry(ledger, entry, true);
    }

    public synchronized long addEntry(long ledger, ByteBuf entry, boolean rollLog) throws IOException {
        if (null == this.logChannel) {
            this.createNewLog();
        }
        int entrySize = entry.readableBytes() + 4;
        boolean reachEntryLogLimit = rollLog ? this.reachEntryLogLimit(entrySize) : this.readEntryLogHardLimit(entrySize);
        boolean createNewLog = this.shouldCreateNewEntryLog.get();
        if (createNewLog || reachEntryLogLimit) {
            this.createNewLog();
            if (createNewLog) {
                this.shouldCreateNewEntryLog.set(false);
            }
        }
        ByteBuf sizeBuffer = (ByteBuf)this.sizeBuffer.get();
        sizeBuffer.clear();
        sizeBuffer.writeInt(entry.readableBytes());
        this.logChannel.write(sizeBuffer);
        long pos = this.logChannel.position();
        this.logChannel.write(entry);
        this.logChannel.registerWrittenEntry(ledger, entrySize);
        return this.logChannel.getLogId() << 32 | pos;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long addEntryForCompaction(long ledgerId, ByteBuf entry) throws IOException {
        Object object = this.compactionLogLock;
        synchronized (object) {
            int entrySize = entry.readableBytes() + 4;
            if (this.compactionLogChannel == null) {
                this.createNewCompactionLog();
            }
            ByteBuf sizeBuffer = (ByteBuf)this.sizeBuffer.get();
            sizeBuffer.clear();
            sizeBuffer.writeInt(entry.readableBytes());
            this.compactionLogChannel.write(sizeBuffer);
            long pos = this.compactionLogChannel.position();
            this.compactionLogChannel.write(entry);
            this.compactionLogChannel.registerWrittenEntry(ledgerId, entrySize);
            return this.compactionLogChannel.getLogId() << 32 | pos;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void flushCompactionLog() throws IOException {
        Object object = this.compactionLogLock;
        synchronized (object) {
            if (this.compactionLogChannel == null) {
                throw new IOException("Failed to flush compaction log which has already been removed.");
            }
            this.appendLedgersMap(this.compactionLogChannel);
            this.compactionLogChannel.flushAndForceWrite(false);
            LOG.info("Flushed compaction log file {} with logId.", (Object)this.compactionLogChannel.getLogFile(), (Object)this.compactionLogChannel.getLogId());
            EntryLogger.closeFileChannel(this.compactionLogChannel);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void createNewCompactionLog() throws IOException {
        Object object = this.compactionLogLock;
        synchronized (object) {
            if (this.compactionLogChannel == null) {
                this.compactionLogChannel = this.entryLoggerAllocator.createNewLogForCompaction();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeCurCompactionLog() {
        Object object = this.compactionLogLock;
        synchronized (object) {
            if (this.compactionLogChannel != null) {
                if (!this.compactionLogChannel.getLogFile().delete()) {
                    LOG.warn("Could not delete compaction log file {}", (Object)this.compactionLogChannel.getLogFile());
                }
                try {
                    EntryLogger.closeFileChannel(this.compactionLogChannel);
                }
                catch (IOException e) {
                    LOG.error("Failed to close file channel for compaction log {}", (Object)this.compactionLogChannel.getLogId());
                }
                this.compactionLogChannel = null;
            }
        }
    }

    static long logIdForOffset(long offset) {
        return offset >> 32;
    }

    synchronized boolean reachEntryLogLimit(long size) {
        return this.logChannel.position() + size > this.logSizeLimit;
    }

    synchronized boolean readEntryLogHardLimit(long size) {
        return this.logChannel.position() + size > Integer.MAX_VALUE;
    }

    public ByteBuf internalReadEntry(long ledgerId, long entryId, long location) throws IOException, Bookie.NoEntryException {
        BufferedReadChannel fc;
        long entryLogId = EntryLogger.logIdForOffset(location);
        long pos = location & 0xFFFFFFFFL;
        ByteBuf sizeBuff = (ByteBuf)this.sizeBuffer.get();
        sizeBuff.clear();
        pos -= 4L;
        try {
            fc = this.getChannelForLogId(entryLogId);
        }
        catch (FileNotFoundException e) {
            FileNotFoundException newe = new FileNotFoundException(e.getMessage() + " for " + ledgerId + " with location " + location);
            newe.setStackTrace(e.getStackTrace());
            throw newe;
        }
        if (this.readFromLogChannel(entryLogId, fc, sizeBuff, pos) != sizeBuff.capacity()) {
            throw new Bookie.NoEntryException("Short read from entrylog " + entryLogId, ledgerId, entryId);
        }
        pos += 4L;
        int entrySize = sizeBuff.readInt();
        if (entrySize > this.maxSaneEntrySize) {
            LOG.warn("Sanity check failed for entry size of " + entrySize + " at location " + pos + " in " + entryLogId);
        }
        if (entrySize < 16) {
            LOG.error("Read invalid entry length {}", (Object)entrySize);
            throw new IOException("Invalid entry length " + entrySize);
        }
        ByteBuf data = PooledByteBufAllocator.DEFAULT.directBuffer(entrySize, entrySize);
        int rc = this.readFromLogChannel(entryLogId, fc, data, pos);
        if (rc != entrySize) {
            data.release();
            throw new Bookie.NoEntryException("Short read for " + ledgerId + "@" + entryId + " in " + entryLogId + "@" + pos + "(" + rc + "!=" + entrySize + ")", ledgerId, entryId);
        }
        data.writerIndex(entrySize);
        return data;
    }

    public ByteBuf readEntry(long ledgerId, long entryId, long location) throws IOException, Bookie.NoEntryException {
        long entryLogId = EntryLogger.logIdForOffset(location);
        long pos = location & 0xFFFFFFFFL;
        ByteBuf data = this.internalReadEntry(ledgerId, entryId, location);
        long thisLedgerId = data.getLong(0);
        if (thisLedgerId != ledgerId) {
            data.release();
            throw new IOException("problem found in " + entryLogId + "@" + entryId + " at position + " + pos + " entry belongs to " + thisLedgerId + " not " + ledgerId);
        }
        long thisEntryId = data.getLong(8);
        if (thisEntryId != entryId) {
            data.release();
            throw new IOException("problem found in " + entryLogId + "@" + entryId + " at position + " + pos + " entry is " + thisEntryId + " not " + entryId);
        }
        return data;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Header getHeaderForLogId(long entryLogId) throws IOException {
        BufferedReadChannel bc = this.getChannelForLogId(entryLogId);
        ByteBuf headers = PooledByteBufAllocator.DEFAULT.directBuffer(1024);
        try {
            bc.read(headers, 0L);
            headers.readInt();
            int headerVersion = headers.readInt();
            if (headerVersion < 0 || headerVersion > 1) {
                LOG.info("Unknown entry log header version for log {}: {}", (Object)entryLogId, (Object)headerVersion);
            }
            long ledgersMapOffset = headers.readLong();
            int ledgersCount = headers.readInt();
            Header header = new Header(headerVersion, ledgersMapOffset, ledgersCount);
            return header;
        }
        finally {
            headers.release();
        }
    }

    private BufferedReadChannel getChannelForLogId(long entryLogId) throws IOException {
        BufferedReadChannel fc = this.getFromChannels(entryLogId);
        if (fc != null) {
            return fc;
        }
        File file = this.findFile(entryLogId);
        FileChannel newFc = new RandomAccessFile(file, "r").getChannel();
        FileChannel oldFc = this.logid2FileChannel.putIfAbsent(entryLogId, newFc);
        if (null != oldFc) {
            newFc.close();
            newFc = oldFc;
        }
        fc = new BufferedReadChannel(newFc, this.conf.getReadBufferBytes());
        this.putInReadChannels(entryLogId, fc);
        return fc;
    }

    boolean logExists(long logId) {
        for (File d : this.ledgerDirsManager.getAllLedgerDirs()) {
            File f = new File(d, Long.toHexString(logId) + ".log");
            if (!f.exists()) continue;
            return true;
        }
        return false;
    }

    public Set<Long> getEntryLogsSet() throws IOException {
        TreeSet entryLogs = Sets.newTreeSet();
        FilenameFilter logFileFilter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".log");
            }
        };
        for (File d : this.ledgerDirsManager.getAllLedgerDirs()) {
            File[] files = d.listFiles(logFileFilter);
            if (files == null) {
                throw new IOException("Failed to get list of files in directory " + d);
            }
            for (File f : files) {
                Long entryLogId = Long.parseLong(f.getName().split(".log")[0], 16);
                entryLogs.add(entryLogId);
            }
        }
        return entryLogs;
    }

    private File findFile(long logId) throws FileNotFoundException {
        for (File d : this.ledgerDirsManager.getAllLedgerDirs()) {
            File f = new File(d, Long.toHexString(logId) + ".log");
            if (!f.exists()) continue;
            return f;
        }
        throw new FileNotFoundException("No file for log " + Long.toHexString(logId));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void scanEntryLog(long entryLogId, EntryLogScanner scanner) throws IOException {
        BufferedReadChannel bc;
        ByteBuf headerBuffer = Unpooled.buffer((int)12);
        try {
            bc = this.getChannelForLogId(entryLogId);
        }
        catch (IOException e) {
            LOG.warn("Failed to get channel to scan entry log: " + entryLogId + ".log");
            throw e;
        }
        long pos = 1024L;
        ByteBuf data = PooledByteBufAllocator.DEFAULT.directBuffer(0x100000);
        try {
            while (pos < bc.size()) {
                if (this.readFromLogChannel(entryLogId, bc, headerBuffer, pos) != headerBuffer.capacity()) {
                    LOG.warn("Short read for entry size from entrylog {}", (Object)entryLogId);
                    return;
                }
                long offset = pos;
                pos += 4L;
                int entrySize = headerBuffer.readInt();
                long ledgerId = headerBuffer.readLong();
                headerBuffer.clear();
                if (ledgerId == -1L || !scanner.accept(ledgerId)) {
                    pos += (long)entrySize;
                    continue;
                }
                data.clear();
                data.capacity(entrySize);
                int rc = this.readFromLogChannel(entryLogId, bc, data, pos);
                if (rc != entrySize) {
                    LOG.warn("Short read for ledger entry from entryLog {}@{} ({} != {})", new Object[]{entryLogId, pos, rc, entrySize});
                    return;
                }
                scanner.process(ledgerId, offset, data);
                pos += (long)entrySize;
            }
        }
        finally {
            data.release();
        }
    }

    public EntryLogMetadata getEntryLogMetadata(long entryLogId) throws IOException {
        try {
            return this.extractEntryLogMetadataFromIndex(entryLogId);
        }
        catch (Exception e) {
            LOG.info("Failed to get ledgers map index from: {}.log : {}", (Object)entryLogId, (Object)e.getMessage());
            return this.extractEntryLogMetadataByScanning(entryLogId);
        }
    }

    EntryLogMetadata extractEntryLogMetadataFromIndex(long entryLogId) throws IOException {
        Header header = this.getHeaderForLogId(entryLogId);
        if (header.version < 1) {
            throw new IOException("Old log file header without ledgers map on entryLogId " + entryLogId);
        }
        if (header.ledgersMapOffset == 0L) {
            throw new IOException("No ledgers map index found on entryLogId" + entryLogId);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Recovering ledgers maps for log {} at offset: {}", (Object)entryLogId, (Object)header.ledgersMapOffset);
        }
        BufferedReadChannel bc = this.getChannelForLogId(entryLogId);
        EntryLogMetadata meta = new EntryLogMetadata(entryLogId);
        int maxMapSize = 160024;
        ByteBuf ledgersMap = ByteBufAllocator.DEFAULT.directBuffer(160024);
        try {
            int ledgersMapSize;
            for (long offset = header.ledgersMapOffset; offset < bc.size(); offset += (long)(ledgersMapSize + 4)) {
                ((ByteBuf)this.sizeBuffer.get()).clear();
                bc.read((ByteBuf)this.sizeBuffer.get(), offset);
                ledgersMapSize = ((ByteBuf)this.sizeBuffer.get()).readInt();
                ledgersMap.clear();
                bc.read(ledgersMap, offset + 4L, ledgersMapSize);
                long lid = ledgersMap.readLong();
                if (lid != -1L) {
                    throw new IOException("Cannot deserialize ledgers map from ledger " + lid);
                }
                long entryId = ledgersMap.readLong();
                if (entryId != -2L) {
                    throw new IOException("Cannot deserialize ledgers map from entryId " + entryId);
                }
                int ledgersCount = ledgersMap.readInt();
                for (int i = 0; i < ledgersCount; ++i) {
                    long ledgerId = ledgersMap.readLong();
                    long size = ledgersMap.readLong();
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Recovering ledgers maps for log {} -- Found ledger: {} with size: {}", new Object[]{entryLogId, ledgerId, size});
                    }
                    meta.addLedgerSize(ledgerId, size);
                }
                if (!ledgersMap.isReadable()) continue;
                throw new IOException("Invalid entry size when reading ledgers map");
            }
        }
        catch (IndexOutOfBoundsException e) {
            throw new IOException(e);
        }
        finally {
            ledgersMap.release();
        }
        if (meta.getLedgersMap().size() != (long)header.ledgersCount) {
            throw new IOException("Not all ledgers were found in ledgers map index. expected: " + header.ledgersCount + " -- found: " + meta.getLedgersMap().size() + " -- entryLogId: " + entryLogId);
        }
        return meta;
    }

    private EntryLogMetadata extractEntryLogMetadataByScanning(long entryLogId) throws IOException {
        final EntryLogMetadata meta = new EntryLogMetadata(entryLogId);
        this.scanEntryLog(entryLogId, new EntryLogScanner(){

            @Override
            public void process(long ledgerId, long offset, ByteBuf entry) throws IOException {
                meta.addLedgerSize(ledgerId, entry.readableBytes() + 4);
            }

            @Override
            public boolean accept(long ledgerId) {
                return ledgerId >= 0L;
            }
        });
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrieved entry log meta data entryLogId: {}, meta: {}", (Object)entryLogId, (Object)meta);
        }
        return meta;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        LOG.info("Stopping EntryLogger");
        try {
            this.flush();
            for (FileChannel fc : this.logid2FileChannel.values()) {
                fc.close();
            }
            this.logid2FileChannel.clear();
            EntryLogger.closeFileChannel(this.logChannel);
            Object object = this.compactionLogLock;
            synchronized (object) {
                EntryLogger.closeFileChannel(this.compactionLogChannel);
                this.compactionLogChannel = null;
            }
        }
        catch (IOException ie) {
            LOG.error("Error flush entry log during shutting down, which may cause entry log corrupted.", (Throwable)ie);
        }
        finally {
            for (FileChannel fc : this.logid2FileChannel.values()) {
                IOUtils.close(LOG, (Closeable)fc);
            }
            EntryLogger.forceCloseFileChannel(this.logChannel);
            Object object = this.compactionLogLock;
            synchronized (object) {
                EntryLogger.forceCloseFileChannel(this.compactionLogChannel);
            }
        }
        this.entryLoggerAllocator.stop();
    }

    private static void closeFileChannel(BufferedChannelBase channel) throws IOException {
        if (null == channel) {
            return;
        }
        FileChannel fileChannel = channel.getFileChannel();
        if (null != fileChannel) {
            fileChannel.close();
        }
    }

    private static void forceCloseFileChannel(BufferedChannelBase channel) {
        if (null == channel) {
            return;
        }
        FileChannel fileChannel = channel.getFileChannel();
        if (null != fileChannel) {
            IOUtils.close(LOG, (Closeable)fileChannel);
        }
    }

    protected LedgerDirsManager getLedgerDirsManager() {
        return this.ledgerDirsManager;
    }

    static long fileName2LogId(String fileName) {
        if (fileName != null && fileName.contains(".")) {
            fileName = fileName.split("\\.")[0];
        }
        try {
            return Long.parseLong(fileName, 16);
        }
        catch (Exception nfe) {
            LOG.error("Invalid log file name {} found when trying to convert to logId.", (Object)fileName, (Object)nfe);
            return -1L;
        }
    }

    static String logId2HexString(long logId) {
        return Long.toHexString(logId);
    }

    class EntryLoggerAllocator {
        private long preallocatedLogId;
        private Future<BufferedLogChannel> preallocation = null;
        private ExecutorService allocatorExecutor;
        private final Object createEntryLogLock = new Object();
        private final Object createCompactionLogLock = new Object();

        EntryLoggerAllocator(long logId) {
            this.preallocatedLogId = logId;
            this.allocatorExecutor = Executors.newSingleThreadExecutor();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        BufferedLogChannel createNewLog() throws IOException {
            Object object = this.createEntryLogLock;
            synchronized (object) {
                BufferedLogChannel bc;
                if (!EntryLogger.this.entryLogPreAllocationEnabled) {
                    BufferedLogChannel bc2 = this.allocateNewLog();
                    return bc2;
                }
                if (null == this.preallocation) {
                    bc = this.allocateNewLog();
                } else {
                    try {
                        bc = this.preallocation.get();
                    }
                    catch (ExecutionException ee) {
                        if (ee.getCause() instanceof IOException) {
                            throw (IOException)ee.getCause();
                        }
                        throw new IOException("Error to execute entry log allocation.", ee);
                    }
                    catch (CancellationException ce) {
                        throw new IOException("Task to allocate a new entry log is cancelled.", ce);
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        throw new IOException("Intrrupted when waiting a new entry log to be allocated.", ie);
                    }
                }
                this.preallocation = this.allocatorExecutor.submit(() -> this.allocateNewLog());
                return bc;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        BufferedLogChannel createNewLogForCompaction() throws IOException {
            Object object = this.createCompactionLogLock;
            synchronized (object) {
                return this.allocateNewLog(".log.compacting");
            }
        }

        private BufferedLogChannel allocateNewLog() throws IOException {
            return this.allocateNewLog(".log");
        }

        private BufferedLogChannel allocateNewLog(String suffix) throws IOException {
            List<File> list = EntryLogger.this.ledgerDirsManager.getWritableLedgerDirsForNewLog();
            Collections.shuffle(list);
            File newLogFile = null;
            block0: do {
                this.preallocatedLogId = this.preallocatedLogId >= Integer.MAX_VALUE ? 0L : ++this.preallocatedLogId;
                String logFileName = Long.toHexString(this.preallocatedLogId) + suffix;
                for (File dir : list) {
                    newLogFile = new File(dir, logFileName);
                    if (!newLogFile.exists()) continue;
                    LOG.warn("Found existed entry log " + newLogFile + " when trying to create it as a new log.");
                    newLogFile = null;
                    continue block0;
                }
            } while (newLogFile == null);
            FileChannel channel = new RandomAccessFile(newLogFile, "rw").getChannel();
            BufferedLogChannel logChannel = new BufferedLogChannel(channel, EntryLogger.this.conf.getWriteBufferBytes(), EntryLogger.this.conf.getReadBufferBytes(), this.preallocatedLogId, newLogFile, EntryLogger.this.conf.getFlushIntervalInBytes());
            EntryLogger.this.logfileHeader.readerIndex(0);
            logChannel.write(EntryLogger.this.logfileHeader);
            for (File f : list) {
                EntryLogger.this.setLastLogId(f, this.preallocatedLogId);
            }
            LOG.info("Created new entry log file {} for logId {}.", (Object)newLogFile, (Object)this.preallocatedLogId);
            return logChannel;
        }

        void stop() {
            this.allocatorExecutor.shutdown();
            LOG.info("Stopped entry logger preallocator.");
        }

        Future<BufferedLogChannel> getPreallocationFuture() {
            return this.preallocation;
        }
    }

    static interface EntryLogListener {
        public void onRotateEntryLog();
    }

    public static interface EntryLogScanner {
        public boolean accept(long var1);

        public void process(long var1, long var3, ByteBuf var5) throws IOException;
    }

    private static class Header {
        final int version;
        final long ledgersMapOffset;
        final int ledgersCount;

        Header(int version, long ledgersMapOffset, int ledgersCount) {
            this.version = version;
            this.ledgersMapOffset = ledgersMapOffset;
            this.ledgersCount = ledgersCount;
        }
    }

    static class BufferedLogChannel
    extends BufferedChannel {
        private final long logId;
        private final EntryLogMetadata entryLogMetadata;
        private final File logFile;

        public BufferedLogChannel(FileChannel fc, int writeCapacity, int readCapacity, long logId, File logFile, long unpersistedBytesBound) throws IOException {
            super(fc, writeCapacity, readCapacity, unpersistedBytesBound);
            this.logId = logId;
            this.entryLogMetadata = new EntryLogMetadata(logId);
            this.logFile = logFile;
        }

        public long getLogId() {
            return this.logId;
        }

        public File getLogFile() {
            return this.logFile;
        }

        public void registerWrittenEntry(long ledgerId, long entrySize) {
            this.entryLogMetadata.addLedgerSize(ledgerId, entrySize);
        }

        public ConcurrentLongLongHashMap getLedgersMap() {
            return this.entryLogMetadata.getLedgersMap();
        }

        public String toString() {
            return MoreObjects.toStringHelper(BufferedChannel.class).add("logId", this.logId).add("logFile", (Object)this.logFile).toString();
        }
    }
}

