/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.logs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.io.BoundedInputStream;
import io.pravega.common.io.SerializationException;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.BitConverter;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.CloseableIterator;
import io.pravega.segmentstore.storage.LogAddress;
import java.beans.ConstructorProperties;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
public class DataFrame {
    static final int MIN_ENTRY_LENGTH_NEEDED = 6;
    private static final byte CURRENT_VERSION = 0;
    private final ByteArraySegment data;
    private WriteFrameHeader header;
    private ByteArraySegment contents;
    private LogAddress address;
    private int writeEntryStartIndex;
    private WriteEntryHeader writeEntryHeader;
    private int writePosition;
    private boolean sealed;

    DataFrame(ByteArraySegment source) {
        Preconditions.checkArgument((!source.isReadOnly() ? 1 : 0) != 0, (Object)"Cannot create a WriteFrame for a readonly source.");
        this.data = source;
        this.writeEntryStartIndex = -1;
        this.sealed = source.isReadOnly();
        this.writePosition = this.sealed ? -1 : 0;
        int sourceLength = this.data.getLength();
        Exceptions.checkArgument((sourceLength > 6 ? 1 : 0) != 0, (String)"data", (String)"Insufficient array length. Byte array must have a length of at least %d.", (Object[])new Object[]{7});
        this.header = new WriteFrameHeader(0, this.data.subSegment(0, 6));
        this.contents = this.data.subSegment(6, sourceLength - 6);
    }

    @VisibleForTesting
    static DataFrame ofSize(int maxSize) {
        return new DataFrame(new ByteArraySegment(new byte[maxSize]));
    }

    public int getLength() {
        return this.header.getSerializationLength() + this.header.getContentLength();
    }

    ArrayView getData() {
        if (this.data.isReadOnly()) {
            return this.data;
        }
        return this.data.subSegment(0, this.getLength());
    }

    boolean isEmpty() {
        return this.header.getContentLength() == 0;
    }

    boolean isSealed() {
        return this.sealed || this.contents.isReadOnly();
    }

    boolean startNewEntry(boolean firstRecordEntry) {
        Preconditions.checkState((!this.sealed ? 1 : 0) != 0, (Object)"DataFrame is sealed and cannot accept any more entries.");
        this.endEntry(true);
        if (this.getAvailableLength() < 6) {
            return false;
        }
        this.writeEntryStartIndex = this.writePosition;
        this.writeEntryHeader = new WriteEntryHeader(this.contents.subSegment(this.writePosition, 5));
        this.writeEntryHeader.setFirstRecordEntry(firstRecordEntry);
        this.writePosition += 5;
        return true;
    }

    void discardEntry() {
        if (this.writeEntryStartIndex < 0) {
            return;
        }
        this.writePosition = this.writeEntryStartIndex;
        this.writeEntryStartIndex = -1;
        this.writeEntryHeader = null;
    }

    boolean endEntry(boolean endOfRecord) {
        if (this.writeEntryStartIndex >= 0) {
            int entryLength = this.writePosition - this.writeEntryStartIndex - 5;
            assert (entryLength >= 0) : "entryLength is negative.";
            this.writeEntryHeader.setEntryLength(entryLength);
            this.writeEntryHeader.setLastRecordEntry(endOfRecord);
            this.writeEntryHeader.serialize();
            this.writeEntryHeader = null;
            this.writeEntryStartIndex = -1;
        }
        return this.getAvailableLength() >= 6;
    }

    int append(byte b) {
        this.ensureAppendConditions();
        if (this.getAvailableLength() >= 1) {
            this.contents.set(this.writePosition, b);
            ++this.writePosition;
            return 1;
        }
        return 0;
    }

    int append(ByteArraySegment data) {
        this.ensureAppendConditions();
        int actualLength = Math.min(data.getLength(), this.getAvailableLength());
        if (actualLength > 0) {
            this.contents.copyFrom(data, this.writePosition, actualLength);
            this.writePosition += actualLength;
        }
        return actualLength;
    }

    void seal() {
        if (!this.sealed && !this.contents.isReadOnly()) {
            Preconditions.checkState((this.writeEntryStartIndex < 0 ? 1 : 0) != 0, (Object)"An open entry exists. Any open entries must be closed prior to sealing.");
            this.header.setContentLength(this.writePosition);
            this.header.commit();
            this.sealed = true;
        }
    }

    private int getAvailableLength() {
        return this.contents.getLength() - this.writePosition;
    }

    private void ensureAppendConditions() {
        Preconditions.checkState((!this.sealed ? 1 : 0) != 0, (Object)"DataFrame is sealed.");
        Preconditions.checkState((this.writeEntryStartIndex >= 0 ? 1 : 0) != 0, (Object)"No entry started.");
    }

    public static DataFrameEntryIterator read(InputStream source, int length, LogAddress address) throws IOException {
        ReadFrameHeader header = new ReadFrameHeader(source);
        if (length < 6 + header.getContentLength()) {
            throw new SerializationException(String.format("Given buffer has insufficient number of bytes for this DataFrame. Expected %d, actual %d.", 6 + header.getContentLength(), length));
        }
        BoundedInputStream contents = new BoundedInputStream(source, header.getContentLength());
        return new DataFrameEntryIterator(contents, address, 6);
    }

    @SuppressFBWarnings(justification="generated code")
    public LogAddress getAddress() {
        return this.address;
    }

    @SuppressFBWarnings(justification="generated code")
    public void setAddress(LogAddress address) {
        this.address = address;
    }

    static class DataFrameEntryIterator
    implements CloseableIterator<DataFrameEntry, IOException> {
        private final BoundedInputStream contents;
        private final LogAddress frameAddress;
        private final int bufferOffset;
        private BoundedInputStream lastEntryContents;

        public void close() {
            this.closeLast();
            this.contents.close();
        }

        private void closeLast() throws IOException {
            BoundedInputStream last = this.lastEntryContents;
            if (last != null) {
                last.close();
                this.lastEntryContents = null;
            }
        }

        @VisibleForTesting
        int getLength() {
            return this.contents.getBound();
        }

        public DataFrameEntry getNext() throws IOException {
            BoundedInputStream resultContents;
            this.closeLast();
            if (this.reachedEnd()) {
                return null;
            }
            if (this.contents.getRemaining() < 5) {
                throw new SerializationException(String.format("Data Frame is corrupt. InputStream has insufficient bytes for a new Entry Header (%d).", this.contents.getRemaining()));
            }
            ReadEntryHeader header = new ReadEntryHeader((InputStream)this.contents);
            if (this.contents.getRemaining() < header.getEntryLength()) {
                throw new SerializationException(String.format("Data Frame is corrupt. Found Entry Length %d which cannot fit in the Frame's remaining length of %d.", header.getEntryLength(), this.contents.getRemaining()));
            }
            int frameOffset = this.bufferOffset + this.contents.getBound() - this.contents.getRemaining();
            this.lastEntryContents = resultContents = this.contents.subStream(header.getEntryLength());
            return new DataFrameEntry(header, resultContents, this.frameAddress, this.reachedEnd(), frameOffset);
        }

        private boolean reachedEnd() {
            return this.contents.getRemaining() <= 0;
        }

        @ConstructorProperties(value={"contents", "frameAddress", "bufferOffset"})
        @SuppressFBWarnings(justification="generated code")
        public DataFrameEntryIterator(BoundedInputStream contents, LogAddress frameAddress, int bufferOffset) {
            this.contents = contents;
            this.frameAddress = frameAddress;
            this.bufferOffset = bufferOffset;
        }

        @SuppressFBWarnings(justification="generated code")
        public LogAddress getFrameAddress() {
            return this.frameAddress;
        }
    }

    public static class DataFrameEntry {
        private final boolean firstRecordEntry;
        private final boolean lastRecordEntry;
        private final boolean lastEntryInDataFrame;
        private final LogAddress frameAddress;
        private final InputStream data;
        private final int length;
        private final int frameOffset;

        private DataFrameEntry(EntryHeader header, BoundedInputStream data, LogAddress frameAddress, boolean lastEntryInDataFrame, int frameOffset) {
            this.firstRecordEntry = header.isFirstRecordEntry();
            this.lastRecordEntry = header.isLastRecordEntry();
            this.lastEntryInDataFrame = lastEntryInDataFrame;
            this.frameAddress = frameAddress;
            this.data = data;
            this.length = data.getBound();
            this.frameOffset = frameOffset;
        }

        public String toString() {
            return String.format("Address = %s, Size = %d, First = %s, Last = %s, LastInDataFrame = %s", this.frameAddress, this.getLength(), this.isFirstRecordEntry(), this.isLastRecordEntry(), this.isLastEntryInDataFrame());
        }

        @SuppressFBWarnings(justification="generated code")
        public boolean isFirstRecordEntry() {
            return this.firstRecordEntry;
        }

        @SuppressFBWarnings(justification="generated code")
        public boolean isLastRecordEntry() {
            return this.lastRecordEntry;
        }

        @SuppressFBWarnings(justification="generated code")
        public boolean isLastEntryInDataFrame() {
            return this.lastEntryInDataFrame;
        }

        @SuppressFBWarnings(justification="generated code")
        public LogAddress getFrameAddress() {
            return this.frameAddress;
        }

        @SuppressFBWarnings(justification="generated code")
        public InputStream getData() {
            return this.data;
        }

        @SuppressFBWarnings(justification="generated code")
        public int getLength() {
            return this.length;
        }

        @SuppressFBWarnings(justification="generated code")
        public int getFrameOffset() {
            return this.frameOffset;
        }
    }

    private static final class ReadFrameHeader
    extends FrameHeader {
        ReadFrameHeader(InputStream source) throws IOException {
            Preconditions.checkNotNull((Object)source, (Object)"source");
            byte version = (byte)source.read();
            this.setVersion(version);
            if (version < 0) {
                throw new EOFException();
            }
            this.setContentLength(BitConverter.readInt((InputStream)source));
            byte flags = (byte)source.read();
            if (flags < 0) {
                throw new EOFException();
            }
            this.decodeFlags(flags, version);
        }
    }

    private static class WriteFrameHeader
    extends FrameHeader {
        private final int serializationLength;
        private ByteArraySegment buffer;

        WriteFrameHeader(byte version, ByteArraySegment target) {
            Exceptions.checkArgument((target.getLength() == 6 ? 1 : 0) != 0, (String)"target", (String)"Unexpected length for target buffer. Expected %d, given %d.", (Object[])new Object[]{6, target.getLength()});
            this.setVersion(version);
            this.serializationLength = 6;
            this.buffer = target;
        }

        void commit() {
            Preconditions.checkState((this.buffer != null && !this.buffer.isReadOnly() ? 1 : 0) != 0, (Object)"Cannot commit a read-only FrameHeader");
            assert (this.buffer.getLength() == 6);
            int bufferOffset = 0;
            this.buffer.set(bufferOffset, this.getVersion());
            ++bufferOffset;
            bufferOffset += BitConverter.writeInt((ArrayView)this.buffer, (int)bufferOffset, (int)this.getContentLength());
            this.buffer.set(bufferOffset, this.encodeFlags());
        }

        @SuppressFBWarnings(justification="generated code")
        public int getSerializationLength() {
            return this.serializationLength;
        }
    }

    private static abstract class FrameHeader {
        static final int SERIALIZATION_LENGTH = 6;
        private byte version;
        private int contentLength;

        private FrameHeader() {
        }

        byte encodeFlags() {
            return 0;
        }

        void decodeFlags(byte flags, byte version) {
        }

        public String toString() {
            return String.format("Version = %d, ContentLength = %d", this.getVersion(), this.getContentLength());
        }

        @SuppressFBWarnings(justification="generated code")
        public byte getVersion() {
            return this.version;
        }

        @SuppressFBWarnings(justification="generated code")
        public void setVersion(byte version) {
            this.version = version;
        }

        @SuppressFBWarnings(justification="generated code")
        public int getContentLength() {
            return this.contentLength;
        }

        @SuppressFBWarnings(justification="generated code")
        public void setContentLength(int contentLength) {
            this.contentLength = contentLength;
        }
    }

    private static class ReadEntryHeader
    extends EntryHeader {
        ReadEntryHeader(InputStream inputStream) throws IOException {
            this.setEntryLength(BitConverter.readInt((InputStream)inputStream));
            byte flags = (byte)inputStream.read();
            if (flags < 0) {
                throw new EOFException();
            }
            this.setFirstRecordEntry((flags & 1) == 1);
            this.setLastRecordEntry((flags & 2) == 2);
        }
    }

    private static class WriteEntryHeader
    extends EntryHeader {
        private ByteArraySegment data;

        WriteEntryHeader(ByteArraySegment headerContents) {
            Exceptions.checkArgument((headerContents.getLength() == 5 ? 1 : 0) != 0, (String)"headerContents", (String)"Invalid headerContents size. Expected %d, given %d.", (Object[])new Object[]{5, headerContents.getLength()});
            this.data = headerContents;
        }

        void serialize() {
            Preconditions.checkState((this.data != null && !this.data.isReadOnly() ? 1 : 0) != 0, (Object)"Cannot serialize a read-only EntryHeader.");
            BitConverter.writeInt((ArrayView)this.data, (int)0, (int)this.getEntryLength());
            byte flags = this.isFirstRecordEntry() ? (byte)1 : 0;
            flags = (byte)(flags | (this.isLastRecordEntry() ? 2 : 0));
            this.data.set(4, flags);
        }
    }

    private static abstract class EntryHeader {
        static final int HEADER_SIZE = 5;
        static final int FLAGS_OFFSET = 4;
        static final byte FIRST_ENTRY_MASK = 1;
        static final byte LAST_ENTRY_MASK = 2;
        private int entryLength;
        private boolean firstRecordEntry;
        private boolean lastRecordEntry;

        private EntryHeader() {
        }

        public String toString() {
            return String.format("Length = %d, FirstEntry = %s, LastEntry = %s", this.getEntryLength(), this.isFirstRecordEntry(), this.isLastRecordEntry());
        }

        @SuppressFBWarnings(justification="generated code")
        public int getEntryLength() {
            return this.entryLength;
        }

        @SuppressFBWarnings(justification="generated code")
        public void setEntryLength(int entryLength) {
            this.entryLength = entryLength;
        }

        @SuppressFBWarnings(justification="generated code")
        public boolean isFirstRecordEntry() {
            return this.firstRecordEntry;
        }

        @SuppressFBWarnings(justification="generated code")
        public void setFirstRecordEntry(boolean firstRecordEntry) {
            this.firstRecordEntry = firstRecordEntry;
        }

        @SuppressFBWarnings(justification="generated code")
        public boolean isLastRecordEntry() {
            return this.lastRecordEntry;
        }

        @SuppressFBWarnings(justification="generated code")
        public void setLastRecordEntry(boolean lastRecordEntry) {
            this.lastRecordEntry = lastRecordEntry;
        }
    }
}

