/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.logs;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.SimpleMovingAverage;
import io.pravega.common.io.SerializationException;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.segmentstore.server.logs.DataFrame;
import java.beans.ConstructorProperties;
import java.io.IOException;
import java.io.OutputStream;
import java.util.function.Consumer;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
class DataFrameOutputStream
extends OutputStream {
    private final Consumer<DataFrame> dataFrameCompleteCallback;
    private DataFrame currentFrame;
    private boolean hasDataInCurrentFrame;
    private boolean closed;
    private final BufferFactory bufferFactory;

    DataFrameOutputStream(int maxDataFrameSize, Consumer<DataFrame> dataFrameCompleteCallback) {
        Exceptions.checkArgument((maxDataFrameSize > 6 ? 1 : 0) != 0, (String)"maxDataFrameSize", (String)"Must be a at least %s.", (Object[])new Object[]{6});
        this.bufferFactory = new BufferFactory(maxDataFrameSize);
        this.dataFrameCompleteCallback = (Consumer)Preconditions.checkNotNull(dataFrameCompleteCallback, (Object)"dataFrameCompleteCallback");
    }

    @Override
    public void write(int b) throws IOException {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((this.currentFrame != null ? 1 : 0) != 0, (Object)"No current frame exists. Most likely no record is started.");
        int totalBytesWritten = 0;
        for (int attemptCount = 0; totalBytesWritten == 0 && attemptCount < 2; ++attemptCount) {
            if ((totalBytesWritten += this.currentFrame.append((byte)b)) != 0) continue;
            this.currentFrame.endEntry(false);
            this.flush();
            this.createNewFrame();
            this.startNewRecordInCurrentFrame(false);
        }
        if (totalBytesWritten == 0) {
            throw new SerializationException("Unable to make progress in serializing to DataFrame.");
        }
    }

    @Override
    public void write(byte[] data, int offset, int length) throws IOException {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((this.currentFrame != null ? 1 : 0) != 0, (Object)"No current frame exists. Most likely no record is started.");
        int totalBytesWritten = 0;
        int attemptsWithNoProgress = 0;
        while (totalBytesWritten < length) {
            int bytesWritten = this.currentFrame.append(new ByteArraySegment(data, offset + totalBytesWritten, length - totalBytesWritten));
            int n = attemptsWithNoProgress = bytesWritten == 0 ? attemptsWithNoProgress + 1 : 0;
            if (attemptsWithNoProgress > 1) {
                throw new IOException("Unable to make progress in serializing to DataFrame.");
            }
            if ((totalBytesWritten += bytesWritten) >= length) continue;
            this.currentFrame.endEntry(false);
            this.flush();
            this.createNewFrame();
            this.startNewRecordInCurrentFrame(false);
        }
    }

    @Override
    public void flush() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        if (!this.hasDataInCurrentFrame) {
            return;
        }
        this.currentFrame.seal();
        if (!this.currentFrame.isEmpty()) {
            this.bufferFactory.markUsed(this.currentFrame.getLength());
            this.dataFrameCompleteCallback.accept(this.currentFrame);
        }
        this.reset();
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.closed = true;
            this.currentFrame = null;
        }
    }

    void startNewRecord() throws IOException {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        if (this.currentFrame == null) {
            this.createNewFrame();
            this.startNewRecordInCurrentFrame(true);
        } else if (!this.currentFrame.startNewEntry(true)) {
            this.flush();
            this.createNewFrame();
            this.startNewRecordInCurrentFrame(true);
        }
    }

    void endRecord() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        if (this.currentFrame != null) {
            this.currentFrame.endEntry(true);
        }
    }

    void discardRecord() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        if (this.currentFrame != null) {
            this.currentFrame.discardEntry();
        }
    }

    void reset() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        this.currentFrame = null;
        this.hasDataInCurrentFrame = false;
    }

    void releaseBuffer() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        this.bufferFactory.reset();
    }

    private void createNewFrame() {
        Preconditions.checkState((this.currentFrame == null || this.currentFrame.isSealed() ? 1 : 0) != 0, (Object)"Cannot create a new frame if we currently have a non-sealed frame.");
        this.currentFrame = new DataFrame(this.bufferFactory.next());
        this.hasDataInCurrentFrame = false;
    }

    private void startNewRecordInCurrentFrame(boolean firstRecordEntry) throws SerializationException {
        if (!this.currentFrame.startNewEntry(firstRecordEntry)) {
            throw new SerializationException("Unable to start a new record.");
        }
        this.hasDataInCurrentFrame = true;
    }

    @SuppressFBWarnings(justification="generated code")
    public boolean isClosed() {
        return this.closed;
    }

    @NotThreadSafe
    private static class BufferFactory {
        private static final int MIN_LENGTH = 1024;
        private final SimpleMovingAverage lastBuffers = new SimpleMovingAverage(10);
        private final int maxLength;
        private byte[] current;
        private int currentUsed;

        ByteArraySegment next() {
            if (this.current == null) {
                this.current = new byte[this.maxLength];
                this.currentUsed = 0;
            }
            return new ByteArraySegment(this.current, this.currentUsed, this.current.length - this.currentUsed);
        }

        void markUsed(int length) {
            this.currentUsed += length;
            this.lastBuffers.add(length);
            int minLength = (int)Math.max(1024.0, this.lastBuffers.getAverage(0.0));
            if (this.current != null && this.current.length - this.currentUsed < minLength) {
                this.current = null;
            }
        }

        void reset() {
            this.current = null;
            this.lastBuffers.reset();
        }

        @ConstructorProperties(value={"maxLength"})
        @SuppressFBWarnings(justification="generated code")
        public BufferFactory(int maxLength) {
            this.maxLength = maxLength;
        }
    }
}

