/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.storage.impl.bookkeeper;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.ObjectClosedException;
import io.pravega.common.Timer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.CloseableIterator;
import io.pravega.common.util.RetriesExhaustedException;
import io.pravega.common.util.Retry;
import io.pravega.segmentstore.storage.DataLogDisabledException;
import io.pravega.segmentstore.storage.DataLogInitializationException;
import io.pravega.segmentstore.storage.DataLogNotAvailableException;
import io.pravega.segmentstore.storage.DataLogWriterNotPrimaryException;
import io.pravega.segmentstore.storage.DurableDataLog;
import io.pravega.segmentstore.storage.DurableDataLogException;
import io.pravega.segmentstore.storage.LogAddress;
import io.pravega.segmentstore.storage.QueueStats;
import io.pravega.segmentstore.storage.WriteFailureException;
import io.pravega.segmentstore.storage.WriteTooLongException;
import io.pravega.segmentstore.storage.impl.bookkeeper.BookKeeperConfig;
import io.pravega.segmentstore.storage.impl.bookkeeper.BookKeeperMetrics;
import io.pravega.segmentstore.storage.impl.bookkeeper.HierarchyUtils;
import io.pravega.segmentstore.storage.impl.bookkeeper.LedgerAddress;
import io.pravega.segmentstore.storage.impl.bookkeeper.LedgerMetadata;
import io.pravega.segmentstore.storage.impl.bookkeeper.Ledgers;
import io.pravega.segmentstore.storage.impl.bookkeeper.LogMetadata;
import io.pravega.segmentstore.storage.impl.bookkeeper.LogReader;
import io.pravega.segmentstore.storage.impl.bookkeeper.SequentialAsyncProcessor;
import io.pravega.segmentstore.storage.impl.bookkeeper.Write;
import io.pravega.segmentstore.storage.impl.bookkeeper.WriteLedger;
import io.pravega.segmentstore.storage.impl.bookkeeper.WriteQueue;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.LedgerHandle;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.api.ACLBackgroundPathAndBytesable;
import org.apache.curator.framework.api.BackgroundPathAndBytesable;
import org.apache.curator.framework.api.WatchPathable;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
class BookKeeperLog
implements DurableDataLog {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(BookKeeperLog.class);
    private static final long REPORT_INTERVAL = 1000L;
    private final String logNodePath;
    private final CuratorFramework zkClient;
    private final BookKeeper bookKeeper;
    private final BookKeeperConfig config;
    private final ScheduledExecutorService executorService;
    private final AtomicBoolean closed;
    private final Object lock = new Object();
    private final String traceObjectId;
    @GuardedBy(value="lock")
    private WriteLedger writeLedger;
    @GuardedBy(value="lock")
    private LogMetadata logMetadata;
    private final WriteQueue writes;
    private final SequentialAsyncProcessor writeProcessor;
    private final SequentialAsyncProcessor rolloverProcessor;
    private final BookKeeperMetrics.BookKeeperLog metrics;
    private final ScheduledFuture<?> metricReporter;

    BookKeeperLog(int containerId, CuratorFramework zkClient, BookKeeper bookKeeper, BookKeeperConfig config, ScheduledExecutorService executorService) {
        Preconditions.checkArgument((containerId >= 0 ? 1 : 0) != 0, (Object)"containerId must be a non-negative integer.");
        this.zkClient = (CuratorFramework)Preconditions.checkNotNull((Object)zkClient, (Object)"zkClient");
        this.bookKeeper = (BookKeeper)Preconditions.checkNotNull((Object)bookKeeper, (Object)"bookKeeper");
        this.config = (BookKeeperConfig)Preconditions.checkNotNull((Object)config, (Object)"config");
        this.executorService = (ScheduledExecutorService)Preconditions.checkNotNull((Object)executorService, (Object)"executorService");
        this.closed = new AtomicBoolean();
        this.logNodePath = HierarchyUtils.getPath(containerId, this.config.getZkHierarchyDepth());
        this.traceObjectId = String.format("Log[%d]", containerId);
        this.writes = new WriteQueue();
        Retry.RetryAndThrowBase<? extends Exception> retry = this.createRetryPolicy(this.config.getMaxWriteAttempts(), this.config.getBkWriteTimeoutMillis());
        this.writeProcessor = new SequentialAsyncProcessor(this::processWritesSync, retry, this::handleWriteProcessorFailures, this.executorService);
        this.rolloverProcessor = new SequentialAsyncProcessor(this::rollover, retry, this::handleRolloverFailure, this.executorService);
        this.metrics = new BookKeeperMetrics.BookKeeperLog(containerId);
        this.metricReporter = this.executorService.scheduleWithFixedDelay(this::reportMetrics, 1000L, 1000L, TimeUnit.MILLISECONDS);
    }

    private Retry.RetryAndThrowBase<? extends Exception> createRetryPolicy(int maxWriteAttempts, int writeTimeout) {
        int initialDelay = writeTimeout / maxWriteAttempts;
        int maxDelay = writeTimeout * maxWriteAttempts;
        return Retry.withExpBackoff((long)initialDelay, (int)2, (int)maxWriteAttempts, (long)maxDelay).retryWhen(ex -> true);
    }

    private void handleWriteProcessorFailures(Throwable exception) {
        log.warn("{}: Too many write processor failures; closing.", (Object)this.traceObjectId, (Object)exception);
        this.close();
    }

    private void handleRolloverFailure(Throwable exception) {
        log.warn("{}: Too many rollover failures; closing.", (Object)this.traceObjectId, (Object)exception);
        this.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() {
        if (!this.closed.getAndSet(true)) {
            WriteLedger writeLedger;
            this.metricReporter.cancel(true);
            this.metrics.close();
            this.rolloverProcessor.close();
            this.writeProcessor.close();
            Object object = this.lock;
            synchronized (object) {
                writeLedger = this.writeLedger;
                this.writeLedger = null;
                this.logMetadata = null;
            }
            this.writes.close().forEach(w -> w.fail(new CancellationException("BookKeeperLog has been closed."), true));
            if (writeLedger != null) {
                try {
                    Ledgers.close(writeLedger.ledger);
                }
                catch (DurableDataLogException bkEx) {
                    log.error("{}: Unable to close LedgerHandle for Ledger {}.", new Object[]{this.traceObjectId, writeLedger.ledger.getId(), bkEx});
                }
            }
            log.info("{}: Closed.", (Object)this.traceObjectId);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void initialize(Duration timeout) throws DurableDataLogException {
        List<Long> ledgersToDelete;
        LogMetadata newMetadata;
        Object object = this.lock;
        synchronized (object) {
            Preconditions.checkState((this.writeLedger == null ? 1 : 0) != 0, (Object)"BookKeeperLog is already initialized.");
            assert (this.logMetadata == null) : "writeLedger == null but logMetadata != null";
            LogMetadata oldMetadata = this.loadMetadata();
            if (oldMetadata != null) {
                if (!oldMetadata.isEnabled()) {
                    throw new DataLogDisabledException("BookKeeperLog is disabled. Cannot initialize.");
                }
                Map<Long, Long> emptyLedgerIds = Ledgers.fenceOut(oldMetadata.getLedgers(), this.bookKeeper, this.config, this.traceObjectId);
                oldMetadata = oldMetadata.updateLedgerStatus(emptyLedgerIds);
            }
            LedgerHandle newLedger = Ledgers.create(this.bookKeeper, this.config);
            log.info("{}: Created Ledger {}.", (Object)this.traceObjectId, (Object)newLedger.getId());
            newMetadata = this.updateMetadata(oldMetadata, newLedger, true);
            LedgerMetadata ledgerMetadata = newMetadata.getLedger(newLedger.getId());
            assert (ledgerMetadata != null) : "cannot find newly added ledger metadata";
            this.writeLedger = new WriteLedger(newLedger, ledgerMetadata);
            this.logMetadata = newMetadata;
            ledgersToDelete = this.getLedgerIdsToDelete(oldMetadata, newMetadata);
        }
        ledgersToDelete.forEach(id -> {
            try {
                Ledgers.delete(id, this.bookKeeper);
                log.info("{}: Deleted orphan empty ledger {}.", (Object)this.traceObjectId, id);
            }
            catch (DurableDataLogException ex) {
                log.warn("{}: Unable to delete orphan empty ledger {}.", new Object[]{this.traceObjectId, id, ex});
            }
        });
        log.info("{}: Initialized (Epoch = {}, UpdateVersion = {}).", new Object[]{this.traceObjectId, newMetadata.getEpoch(), newMetadata.getUpdateVersion()});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void enable() throws DurableDataLogException {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Object object = this.lock;
        synchronized (object) {
            Preconditions.checkState((this.writeLedger == null ? 1 : 0) != 0, (Object)"BookKeeperLog is already initialized; cannot re-enable.");
            assert (this.logMetadata == null) : "writeLedger == null but logMetadata != null";
            LogMetadata metadata = this.loadMetadata();
            Preconditions.checkState((metadata != null && !metadata.isEnabled() ? 1 : 0) != 0, (Object)"BookKeeperLog is already enabled.");
            metadata = metadata.asEnabled();
            this.persistMetadata(metadata, false);
            log.info("{}: Enabled (Epoch = {}, UpdateVersion = {}).", new Object[]{this.traceObjectId, metadata.getEpoch(), metadata.getUpdateVersion()});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void disable() throws DurableDataLogException {
        Object object = this.lock;
        synchronized (object) {
            this.ensurePreconditions();
            LogMetadata metadata = this.getLogMetadata();
            Preconditions.checkState((boolean)metadata.isEnabled(), (Object)"BookKeeperLog is already disabled.");
            metadata = this.logMetadata.asDisabled();
            this.persistMetadata(metadata, false);
            this.logMetadata = metadata;
            log.info("{}: Disabled (Epoch = {}, UpdateVersion = {}).", new Object[]{this.traceObjectId, metadata.getEpoch(), metadata.getUpdateVersion()});
        }
        this.close();
    }

    public CompletableFuture<LogAddress> append(ArrayView data, Duration timeout) {
        this.ensurePreconditions();
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"append", (Object[])new Object[]{data.getLength()});
        if (data.getLength() > this.getMaxAppendLength()) {
            return Futures.failedFuture((Throwable)new WriteTooLongException(data.getLength(), this.getMaxAppendLength()));
        }
        Timer timer = new Timer();
        CompletableFuture<LogAddress> result = new CompletableFuture<LogAddress>();
        this.writes.add(new Write(data, this.getWriteLedger(), result));
        this.writeProcessor.runAsync();
        result.whenCompleteAsync((address, ex) -> {
            if (ex != null) {
                this.handleWriteException((Throwable)ex);
            } else {
                this.metrics.writeCompleted(timer.getElapsed());
                LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"append", (long)traceId, (Object[])new Object[]{data.getLength(), address});
            }
        }, (Executor)this.executorService);
        return result;
    }

    public CompletableFuture<Void> truncate(LogAddress upToAddress, Duration timeout) {
        this.ensurePreconditions();
        Preconditions.checkArgument((boolean)(upToAddress instanceof LedgerAddress), (Object)"upToAddress must be of type LedgerAddress.");
        return CompletableFuture.runAsync(() -> this.tryTruncate((LedgerAddress)upToAddress), this.executorService);
    }

    public CloseableIterator<DurableDataLog.ReadItem, DurableDataLogException> getReader() throws DurableDataLogException {
        this.ensurePreconditions();
        return new LogReader(this.getLogMetadata(), this.bookKeeper, this.config);
    }

    public int getMaxAppendLength() {
        return 1047552;
    }

    public long getEpoch() {
        this.ensurePreconditions();
        return this.getLogMetadata().getEpoch();
    }

    public QueueStats getQueueStatistics() {
        return this.writes.getStatistics();
    }

    private void processWritesSync() {
        if (this.closed.get()) {
            return;
        }
        if (this.getWriteLedger().ledger.isClosed()) {
            this.rolloverProcessor.runAsync();
        } else if (!this.processPendingWrites() && !this.closed.get()) {
            this.writeProcessor.runAsync();
        }
    }

    private boolean processPendingWrites() {
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"processPendingWrites", (Object[])new Object[0]);
        WriteQueue.CleanupStatus cs = this.writes.removeFinishedWrites();
        if (cs == WriteQueue.CleanupStatus.WriteFailed) {
            this.close();
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"processPendingWrites", (long)traceId, (Object[])new Object[]{cs});
            return false;
        }
        if (cs == WriteQueue.CleanupStatus.QueueEmpty) {
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"processPendingWrites", (long)traceId, (Object[])new Object[]{cs});
            return true;
        }
        List<Write> toExecute = this.getWritesToExecute();
        boolean success = true;
        if (!toExecute.isEmpty() && (success = this.executeWrites(toExecute))) {
            this.rolloverProcessor.runAsync();
        }
        LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"processPendingWrites", (long)traceId, (Object[])new Object[]{toExecute.size(), success});
        return success;
    }

    private List<Write> getWritesToExecute() {
        long maxTotalSize = (long)this.config.getBkLedgerMaxSize() - this.getWriteLedger().ledger.getLength();
        List<Write> toExecute = this.writes.getWritesToExecute(maxTotalSize);
        if (this.handleClosedLedgers(toExecute)) {
            toExecute = this.writes.getWritesToExecute(maxTotalSize);
        }
        return toExecute;
    }

    private boolean executeWrites(List<Write> toExecute) {
        log.debug("{}: Executing {} writes.", (Object)this.traceObjectId, (Object)toExecute.size());
        for (int i = 0; i < toExecute.size(); ++i) {
            Write w = toExecute.get(i);
            try {
                int attemptCount = w.beginAttempt();
                if (attemptCount > this.config.getMaxWriteAttempts()) {
                    throw new RetriesExhaustedException(w.getFailureCause());
                }
                w.getWriteLedger().ledger.asyncAddEntry(w.data.array(), w.data.arrayOffset(), w.data.getLength(), this::addCallback, (Object)w);
                continue;
            }
            catch (Throwable ex) {
                boolean isFinal = !BookKeeperLog.isRetryable(ex);
                w.fail(ex, isFinal);
                for (int j = i + 1; j < toExecute.size(); ++j) {
                    toExecute.get(j).fail(new DurableDataLogException("Previous write failed.", ex), isFinal);
                }
                return false;
            }
        }
        return true;
    }

    private boolean handleClosedLedgers(List<Write> writes) {
        if (writes.size() == 0 || !writes.get((int)0).getWriteLedger().ledger.isClosed()) {
            return false;
        }
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"handleClosedLedgers", (Object[])new Object[]{writes.size()});
        WriteLedger currentLedger = this.getWriteLedger();
        HashMap<Long, Long> lastAddsConfirmed = new HashMap<Long, Long>();
        boolean anythingChanged = false;
        for (Write w : writes) {
            if (w.isDone() || !w.getWriteLedger().ledger.isClosed()) continue;
            long lac = this.fetchLastAddConfirmed(w.getWriteLedger(), lastAddsConfirmed);
            if (w.getEntryId() >= 0L && w.getEntryId() <= lac) {
                this.completeWrite(w);
                anythingChanged = true;
                continue;
            }
            if (currentLedger.ledger.getId() == w.getWriteLedger().ledger.getId()) continue;
            w.setWriteLedger(currentLedger);
            anythingChanged = true;
        }
        LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"handleClosedLedgers", (long)traceId, (Object[])new Object[]{writes.size(), anythingChanged});
        return anythingChanged;
    }

    private long fetchLastAddConfirmed(WriteLedger writeLedger, Map<Long, Long> lastAddsConfirmed) {
        long ledgerId = writeLedger.ledger.getId();
        long lac = lastAddsConfirmed.getOrDefault(ledgerId, -1L);
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"fetchLastAddConfirmed", (Object[])new Object[]{ledgerId, lac});
        if (lac < 0L) {
            lac = writeLedger.isRolledOver() ? writeLedger.ledger.getLastAddConfirmed() : Ledgers.readLastAddConfirmed(ledgerId, this.bookKeeper, this.config);
            lastAddsConfirmed.put(ledgerId, lac);
            log.info("{}: Fetched actual LastAddConfirmed ({}) for LedgerId {}.", new Object[]{this.traceObjectId, lac, ledgerId});
        }
        LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"fetchLastAddConfirmed", (long)traceId, (Object[])new Object[]{ledgerId, lac});
        return lac;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addCallback(int rc, LedgerHandle handle, long entryId, Object ctx) {
        Write write = (Write)ctx;
        try {
            assert (handle.getId() == write.getWriteLedger().ledger.getId()) : "Handle.Id mismatch: " + write.getWriteLedger().ledger.getId() + " vs " + handle.getId();
            write.setEntryId(entryId);
            if (rc == 0) {
                this.completeWrite(write);
                return;
            }
            this.handleWriteException(rc, write);
        }
        catch (Throwable ex) {
            write.fail(ex, !BookKeeperLog.isRetryable(ex));
        }
        finally {
            try {
                this.writeProcessor.runAsync();
            }
            catch (ObjectClosedException ex) {
                log.warn("{}: Not running WriteProcessor as part of callback due to BookKeeperLog being closed.", (Object)this.traceObjectId, (Object)ex);
            }
        }
    }

    private void completeWrite(Write write) {
        Timer t = write.complete();
        if (t != null) {
            this.metrics.bookKeeperWriteCompleted(write.data.getLength(), t.getElapsed());
        }
    }

    private void handleWriteException(Throwable ex) {
        if (ex instanceof ObjectClosedException && !this.closed.get()) {
            log.warn("{}: Caught ObjectClosedException but not closed; closing now.", (Object)this.traceObjectId, (Object)ex);
            this.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleWriteException(int responseCode, Write write) {
        assert (responseCode != 0) : "cannot handle an exception when responseCode == 0";
        Object ex = BKException.create((int)responseCode);
        try {
            ex = ex instanceof BKException.BKLedgerFencedException ? new DataLogWriterNotPrimaryException("BookKeeperLog is not primary anymore.", (Throwable)ex) : (ex instanceof BKException.BKNotEnoughBookiesException ? new DataLogNotAvailableException("BookKeeperLog is not available.", (Throwable)ex) : (ex instanceof BKException.BKLedgerClosedException ? new WriteFailureException("Active Ledger is closed.", (Throwable)ex) : (ex instanceof BKException.BKWriteException ? new WriteFailureException("Unable to write to active Ledger.", (Throwable)ex) : (ex instanceof BKException.BKClientClosedException ? new ObjectClosedException((Object)this, (Throwable)ex) : new DurableDataLogException("General exception while accessing BookKeeper.", (Throwable)ex)))));
        }
        catch (Throwable throwable) {
            write.fail((Throwable)ex, !BookKeeperLog.isRetryable((Throwable)ex));
            throw throwable;
        }
        write.fail((Throwable)ex, !BookKeeperLog.isRetryable((Throwable)ex));
    }

    private static boolean isRetryable(Throwable ex) {
        return (ex = Exceptions.unwrap((Throwable)ex)) instanceof WriteFailureException || ex instanceof DataLogNotAvailableException;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void tryTruncate(LedgerAddress upToAddress) {
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"tryTruncate", (Object[])new Object[]{upToAddress});
        LogMetadata oldMetadata = this.getLogMetadata();
        LogMetadata newMetadata = oldMetadata.truncate(upToAddress);
        this.persistMetadata(newMetadata, false);
        Object object = this.lock;
        synchronized (object) {
            this.logMetadata = newMetadata;
        }
        Set ledgerIdsToKeep = newMetadata.getLedgers().stream().map(LedgerMetadata::getLedgerId).collect(Collectors.toSet());
        Iterator ledgersToDelete = oldMetadata.getLedgers().stream().filter(lm -> !ledgerIdsToKeep.contains(lm.getLedgerId())).iterator();
        while (ledgersToDelete.hasNext()) {
            LedgerMetadata lm2 = (LedgerMetadata)ledgersToDelete.next();
            try {
                Ledgers.delete(lm2.getLedgerId(), this.bookKeeper);
            }
            catch (DurableDataLogException ex) {
                log.error("{}: Unable to delete truncated ledger {}.", new Object[]{this.traceObjectId, lm2.getLedgerId(), ex});
            }
        }
        log.info("{}: Truncated up to {}.", (Object)this.traceObjectId, (Object)upToAddress);
        LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"tryTruncate", (long)traceId, (Object[])new Object[]{upToAddress});
    }

    @VisibleForTesting
    LogMetadata loadMetadata() throws DataLogInitializationException {
        try {
            Stat storingStatIn = new Stat();
            byte[] serializedMetadata = (byte[])((WatchPathable)this.zkClient.getData().storingStatIn(storingStatIn)).forPath(this.logNodePath);
            LogMetadata result = (LogMetadata)LogMetadata.SERIALIZER.deserialize(serializedMetadata);
            result.withUpdateVersion(storingStatIn.getVersion());
            return result;
        }
        catch (KeeperException.NoNodeException nne) {
            log.warn("{}: No ZNode found for path '{}{}'. This is OK if this is the first time accessing this log.", new Object[]{this.traceObjectId, this.zkClient.getNamespace(), this.logNodePath});
            return null;
        }
        catch (Exception ex) {
            throw new DataLogInitializationException(String.format("Unable to load ZNode contents for path '%s%s'.", this.zkClient.getNamespace(), this.logNodePath), (Throwable)ex);
        }
    }

    private LogMetadata updateMetadata(LogMetadata currentMetadata, LedgerHandle newLedger, boolean clearEmptyLedgers) throws DurableDataLogException {
        boolean create;
        boolean bl = create = currentMetadata == null;
        if (create) {
            currentMetadata = new LogMetadata(newLedger.getId());
        } else {
            currentMetadata = currentMetadata.addLedger(newLedger.getId());
            if (clearEmptyLedgers) {
                currentMetadata = currentMetadata.removeEmptyLedgers(2);
            }
        }
        try {
            this.persistMetadata(currentMetadata, create);
        }
        catch (DurableDataLogException ex) {
            try {
                Ledgers.delete(newLedger.getId(), this.bookKeeper);
            }
            catch (Exception deleteEx) {
                log.warn("{}: Unable to delete newly created ledger {}.", new Object[]{this.traceObjectId, newLedger.getId(), deleteEx});
                ex.addSuppressed((Throwable)deleteEx);
            }
            throw ex;
        }
        log.info("{} Metadata updated ({}).", (Object)this.traceObjectId, (Object)currentMetadata);
        return currentMetadata;
    }

    private void persistMetadata(LogMetadata metadata, boolean create) throws DurableDataLogException {
        try {
            Stat result;
            byte[] serializedMetadata = LogMetadata.SERIALIZER.serialize((Object)metadata).getCopy();
            if (create) {
                result = new Stat();
                ((ACLBackgroundPathAndBytesable)this.zkClient.create().creatingParentsIfNeeded().storingStatIn(result)).forPath(this.logNodePath, serializedMetadata);
            } else {
                result = (Stat)((BackgroundPathAndBytesable)this.zkClient.setData().withVersion(metadata.getUpdateVersion())).forPath(this.logNodePath, serializedMetadata);
            }
            metadata.withUpdateVersion(result.getVersion());
        }
        catch (KeeperException.BadVersionException | KeeperException.NodeExistsException keeperEx) {
            throw new DataLogWriterNotPrimaryException(String.format("Unable to acquire exclusive write lock for log (path = '%s%s').", this.zkClient.getNamespace(), this.logNodePath), keeperEx);
        }
        catch (Exception generalEx) {
            throw new DataLogInitializationException(String.format("Unable to update ZNode for path '%s%s'.", this.zkClient.getNamespace(), this.logNodePath), (Throwable)generalEx);
        }
        log.info("{} Metadata persisted ({}).", (Object)this.traceObjectId, (Object)metadata);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void rollover() {
        if (this.closed.get()) {
            return;
        }
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"rollover", (Object[])new Object[0]);
        LedgerHandle l = this.getWriteLedger().ledger;
        if (!l.isClosed() && l.getLength() < (long)this.config.getBkLedgerMaxSize()) {
            this.writeProcessor.runAsync();
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"rollover", (long)traceId, (Object[])new Object[]{false});
            return;
        }
        try {
            LedgerHandle oldLedger;
            LedgerHandle newLedger = Ledgers.create(this.bookKeeper, this.config);
            log.debug("{}: Rollover: created new ledger {}.", (Object)this.traceObjectId, (Object)newLedger.getId());
            LogMetadata metadata = this.getLogMetadata();
            metadata = this.updateMetadata(metadata, newLedger, false);
            LedgerMetadata ledgerMetadata = metadata.getLedger(newLedger.getId());
            assert (ledgerMetadata != null) : "cannot find newly added ledger metadata";
            log.debug("{}: Rollover: updated metadata '{}.", (Object)this.traceObjectId, (Object)metadata);
            Object object = this.lock;
            synchronized (object) {
                oldLedger = this.writeLedger.ledger;
                if (!oldLedger.isClosed()) {
                    this.writeLedger.setRolledOver(true);
                }
                this.writeLedger = new WriteLedger(newLedger, ledgerMetadata);
                this.logMetadata = metadata;
            }
            Ledgers.close(oldLedger);
            log.info("{}: Rollover: swapped ledger and metadata pointers (Old = {}, New = {}) and closed old ledger.", new Object[]{this.traceObjectId, oldLedger.getId(), newLedger.getId()});
            this.writeProcessor.runAsync();
        }
        catch (Throwable throwable) {
            this.writeProcessor.runAsync();
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"rollover", (long)traceId, (Object[])new Object[]{true});
            throw throwable;
        }
        LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"rollover", (long)traceId, (Object[])new Object[]{true});
    }

    @GuardedBy(value="lock")
    private List<Long> getLedgerIdsToDelete(LogMetadata oldMetadata, LogMetadata currentMetadata) {
        if (oldMetadata == null) {
            return Collections.emptyList();
        }
        Set existingIds = currentMetadata.getLedgers().stream().map(LedgerMetadata::getLedgerId).collect(Collectors.toSet());
        return oldMetadata.getLedgers().stream().map(LedgerMetadata::getLedgerId).filter(id -> !existingIds.contains(id)).collect(Collectors.toList());
    }

    private void reportMetrics() {
        this.metrics.ledgerCount(this.getLogMetadata().getLedgers().size());
        this.metrics.queueStats(this.writes.getStatistics());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private LogMetadata getLogMetadata() {
        Object object = this.lock;
        synchronized (object) {
            return this.logMetadata;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private WriteLedger getWriteLedger() {
        Object object = this.lock;
        synchronized (object) {
            return this.writeLedger;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void ensurePreconditions() {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Object object = this.lock;
        synchronized (object) {
            Preconditions.checkState((this.writeLedger != null ? 1 : 0) != 0, (Object)"BookKeeperLog is not initialized.");
            assert (this.logMetadata != null) : "writeLedger != null but logMetadata == null";
        }
    }
}

