/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.shared.security.token;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.JwtBuilder;
import io.jsonwebtoken.JwtException;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.MalformedJwtException;
import io.jsonwebtoken.SignatureAlgorithm;
import io.jsonwebtoken.SignatureException;
import io.pravega.auth.InvalidTokenException;
import io.pravega.auth.TokenException;
import io.pravega.auth.TokenExpiredException;
import java.time.Instant;
import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JsonWebToken {
    @SuppressFBWarnings(justification="generated code")
    private static final Logger log = LoggerFactory.getLogger(JsonWebToken.class);
    private final String subject;
    private final String audience;
    private final byte[] signingKey;
    private final Date expirationTime;
    private final Instant currentInstant;
    private final Map<String, Object> permissionsByResource;
    private final SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.HS512;

    public JsonWebToken(String subject, String audience, byte[] signingKey) {
        this(subject, audience, signingKey, null, null);
    }

    public JsonWebToken(@NonNull String subject, @NonNull String audience, @NonNull byte[] signingKey, Integer timeToLiveInSeconds, Map<String, Object> resourcePermissionClaims) {
        if (subject == null) {
            throw new NullPointerException("subject is marked @NonNull but is null");
        }
        if (audience == null) {
            throw new NullPointerException("audience is marked @NonNull but is null");
        }
        if (signingKey == null) {
            throw new NullPointerException("signingKey is marked @NonNull but is null");
        }
        if (timeToLiveInSeconds != null) {
            Preconditions.checkArgument((timeToLiveInSeconds >= -1 ? 1 : 0) != 0);
        }
        this.subject = subject;
        this.audience = audience;
        this.signingKey = (byte[])signingKey.clone();
        this.currentInstant = Instant.now();
        this.expirationTime = timeToLiveInSeconds != null && timeToLiveInSeconds != -1 ? Date.from(this.currentInstant.plusSeconds(timeToLiveInSeconds.intValue())) : null;
        this.permissionsByResource = resourcePermissionClaims;
    }

    public String toCompactString() {
        JwtBuilder builder = Jwts.builder().setSubject(this.subject).setAudience(this.audience).setIssuedAt(Date.from(this.currentInstant));
        if (this.permissionsByResource != null) {
            builder.addClaims(this.permissionsByResource);
        }
        if (this.expirationTime != null) {
            builder.setExpiration(this.expirationTime);
        }
        builder.signWith(this.signatureAlgorithm, this.signingKey);
        return builder.compact();
    }

    @VisibleForTesting
    static Claims parseClaims(String token, byte[] signingKey) throws TokenExpiredException, InvalidTokenException, TokenException {
        if (Strings.isNullOrEmpty((String)token)) {
            throw new InvalidTokenException("Token is null or empty");
        }
        try {
            Jws claimsJws = Jwts.parser().setSigningKey(signingKey).parseClaimsJws(token);
            log.debug("Successfully parsed JWT token.");
            return (Claims)claimsJws.getBody();
        }
        catch (ExpiredJwtException e) {
            throw new TokenExpiredException((Exception)((Object)e));
        }
        catch (MalformedJwtException | SignatureException e) {
            throw new InvalidTokenException((Exception)e);
        }
        catch (JwtException e) {
            throw new TokenException((Exception)((Object)e));
        }
    }

    public static Set<Map.Entry<String, Object>> fetchClaims(String token, byte[] signingKey) throws TokenException {
        return JsonWebToken.parseClaims(token, signingKey).entrySet();
    }

    @SuppressFBWarnings(justification="generated code")
    public String toString() {
        return "JsonWebToken(subject=" + this.subject + ", audience=" + this.audience + ", signingKey=" + Arrays.toString(this.signingKey) + ", expirationTime=" + this.expirationTime + ", currentInstant=" + this.currentInstant + ", permissionsByResource=" + this.permissionsByResource + ", signatureAlgorithm=" + this.signatureAlgorithm + ")";
    }
}

