/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.host;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.cluster.Host;
import io.pravega.common.cluster.HostContainerMap;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.CollectionHelpers;
import io.pravega.segmentstore.server.ContainerHandle;
import io.pravega.segmentstore.server.SegmentContainerRegistry;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import lombok.Generated;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.NodeCache;
import org.apache.curator.utils.ZKPaths;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKSegmentContainerMonitor
implements AutoCloseable {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(ZKSegmentContainerMonitor.class);
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private final Object $lock = new Object[0];
    private static final Duration INIT_TIMEOUT_PER_CONTAINER = Duration.ofSeconds(30L);
    private static final Duration CLOSE_TIMEOUT_PER_CONTAINER = Duration.ofSeconds(30L);
    private static final Duration MONITOR_INTERVAL = Duration.ofSeconds(10L);
    private static final long REPORT_INTERVAL_MILLIS = Duration.ofMinutes(10L).toMillis();
    private static final Supplier<Long> CURRENT_TIME_MILLIS = System::currentTimeMillis;
    private final Host host;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final NodeCache hostContainerMapNode;
    private final SegmentContainerRegistry registry;
    private final Map<Integer, ContainerHandle> handles;
    private final Set<Integer> pendingTasks;
    private final ScheduledExecutorService executor;
    private AtomicReference<ScheduledFuture<?>> assignmentTask;
    private final AtomicLong lastReportTime;
    private final Semaphore parallelContainerStartsSemaphore;

    ZKSegmentContainerMonitor(SegmentContainerRegistry containerRegistry, CuratorFramework zkClient, Host pravegaServiceEndpoint, int parallelContainerStarts, ScheduledExecutorService executor) {
        Preconditions.checkNotNull((Object)zkClient, (Object)"zkClient");
        Preconditions.checkArgument((parallelContainerStarts > 0 ? 1 : 0) != 0, (Object)"parallelContainerStarts");
        this.registry = (SegmentContainerRegistry)Preconditions.checkNotNull((Object)containerRegistry, (Object)"containerRegistry");
        this.host = (Host)Preconditions.checkNotNull((Object)pravegaServiceEndpoint, (Object)"pravegaServiceEndpoint");
        this.executor = (ScheduledExecutorService)Preconditions.checkNotNull((Object)executor, (Object)"executor");
        this.handles = new ConcurrentHashMap<Integer, ContainerHandle>();
        this.pendingTasks = new ConcurrentSkipListSet<Integer>();
        String clusterPath = ZKPaths.makePath((String)"cluster", (String)"segmentContainerHostMapping");
        this.hostContainerMapNode = new NodeCache(zkClient, clusterPath);
        this.assignmentTask = new AtomicReference();
        this.lastReportTime = new AtomicLong(CURRENT_TIME_MILLIS.get());
        this.parallelContainerStartsSemaphore = new Semaphore(parallelContainerStarts);
    }

    public void initialize() {
        this.initialize(MONITOR_INTERVAL);
    }

    @VisibleForTesting
    public void initialize(Duration monitorInterval) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        this.hostContainerMapNode.start();
        this.assignmentTask.set(this.executor.scheduleWithFixedDelay(this::checkAssignment, 0L, monitorInterval.getSeconds(), TimeUnit.SECONDS));
        this.hostContainerMapNode.getListenable().addListener(this::checkAssignment, (Executor)this.executor);
    }

    @Override
    public void close() {
        Preconditions.checkState((boolean)this.closed.compareAndSet(false, true));
        try {
            this.hostContainerMapNode.close();
        }
        catch (IOException e) {
            log.warn("Failed to close hostContainerMapNode", (Throwable)e);
        }
        ScheduledFuture task = this.assignmentTask.getAndSet(null);
        if (task != null) {
            task.cancel(true);
        }
        ArrayList<ContainerHandle> toClose = new ArrayList<ContainerHandle>(this.handles.values());
        ArrayList<CompletionStage> results = new ArrayList<CompletionStage>();
        for (ContainerHandle handle : toClose) {
            results.add(this.registry.stopContainer(handle, CLOSE_TIMEOUT_PER_CONTAINER).thenAccept(v -> this.unregisterHandle(handle.getContainerId())));
        }
        Futures.await((CompletableFuture)Futures.allOf(results), (long)CLOSE_TIMEOUT_PER_CONTAINER.toMillis());
    }

    @VisibleForTesting
    Collection<Integer> getRegisteredContainers() {
        return this.handles.keySet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkAssignment() {
        Object object = this.$lock;
        synchronized (object) {
            long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"checkAssignment", (Object[])new Object[0]);
            try {
                Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
                Set<Integer> desiredList = this.getDesiredContainerList();
                if (desiredList != null) {
                    boolean logReport;
                    HashSet<Integer> runningContainers = new HashSet<Integer>(this.handles.keySet());
                    HashSet<Integer> containersPendingTasks = new HashSet<Integer>(this.pendingTasks);
                    Collection containersToBeStarted = CollectionHelpers.filterOut(desiredList, runningContainers);
                    containersToBeStarted = CollectionHelpers.filterOut((Collection)containersToBeStarted, containersPendingTasks);
                    Collection containersToBeStopped = CollectionHelpers.filterOut(runningContainers, desiredList);
                    containersToBeStopped = CollectionHelpers.filterOut((Collection)containersToBeStopped, containersPendingTasks);
                    boolean bl = logReport = !containersPendingTasks.isEmpty() || !containersToBeStarted.isEmpty() || !containersToBeStopped.isEmpty();
                    if (logReport || CURRENT_TIME_MILLIS.get() - this.lastReportTime.get() >= REPORT_INTERVAL_MILLIS) {
                        log.info("Container Changes: Desired = {}, Current = {}, PendingTasks = {}, ToStart = {}, ToStop = {}.", new Object[]{desiredList, runningContainers, containersPendingTasks, containersToBeStarted, containersToBeStopped});
                        this.lastReportTime.set(CURRENT_TIME_MILLIS.get());
                    }
                    containersToBeStarted.forEach(this::startContainer);
                    containersToBeStopped.forEach(this::stopContainer);
                } else {
                    log.warn("No segment container assignments found");
                }
            }
            catch (Throwable e) {
                log.warn("Failed to monitor the segmentcontainer assignment: ", e);
            }
            finally {
                LoggerHelpers.traceLeave((Logger)log, (String)"checkAssignment", (long)traceId, (Object[])new Object[0]);
            }
        }
    }

    private CompletableFuture<Void> stopContainer(int containerId) {
        log.info("Stopping Container {}.", (Object)containerId);
        ContainerHandle handle = this.handles.get(containerId);
        if (handle == null) {
            log.warn("Container {} handle is null, container is pending start or already unregistered.", (Object)containerId);
            return null;
        }
        this.pendingTasks.add(containerId);
        try {
            return this.registry.stopContainer(handle, CLOSE_TIMEOUT_PER_CONTAINER).whenComplete((aVoid, throwable) -> {
                if (throwable != null) {
                    log.warn("Stopping container {} failed: {}", (Object)containerId, throwable);
                }
                try {
                    this.unregisterHandle(containerId);
                }
                finally {
                    this.pendingTasks.remove(containerId);
                }
            });
        }
        catch (Throwable e) {
            this.pendingTasks.remove(containerId);
            throw e;
        }
    }

    private CompletableFuture<ContainerHandle> startContainer(int containerId) {
        log.info("Starting Container {}.", (Object)containerId);
        this.pendingTasks.add(containerId);
        try {
            return ((CompletableFuture)CompletableFuture.runAsync(() -> Exceptions.handleInterrupted(this.parallelContainerStartsSemaphore::acquire)).thenCompose(v -> this.registry.startContainer(containerId, INIT_TIMEOUT_PER_CONTAINER))).whenComplete((handle, ex) -> {
                try {
                    if (ex == null) {
                        if (this.handles.putIfAbsent(handle.getContainerId(), (ContainerHandle)handle) != null) {
                            log.warn("Starting container {} succeeded but handle is already registered.", (Object)handle.getContainerId());
                        } else {
                            handle.setContainerStoppedListener(this::unregisterHandle);
                            log.info("Container {} has been registered.", (Object)handle.getContainerId());
                        }
                    } else {
                        log.warn("Starting container {} failed: {}", (Object)containerId, ex);
                    }
                }
                finally {
                    this.pendingTasks.remove(containerId);
                    this.parallelContainerStartsSemaphore.release();
                }
            });
        }
        catch (Throwable e) {
            this.pendingTasks.remove(containerId);
            throw e;
        }
    }

    private void unregisterHandle(int containerId) {
        if (this.handles.remove(containerId) == null) {
            log.warn("Attempted to unregister non-registered container {}.", (Object)containerId);
        } else {
            log.info("Container {} has been unregistered.", (Object)containerId);
        }
    }

    private Set<Integer> getDesiredContainerList() {
        byte[] containerToHostMapSer;
        log.debug("Fetching the latest container assignment from ZooKeeper.");
        if (this.hostContainerMapNode.getCurrentData() != null && (containerToHostMapSer = this.hostContainerMapNode.getCurrentData().getData()) != null) {
            HostContainerMap controlMapping = HostContainerMap.fromBytes((byte[])containerToHostMapSer);
            return controlMapping.getHostContainerMap().entrySet().stream().filter(ep -> ((Host)ep.getKey()).equals((Object)this.host)).map(Map.Entry::getValue).findFirst().orElse(Collections.emptySet());
        }
        return null;
    }
}

