/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.host.handler;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Iterators;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.pravega.auth.AuthHandler;
import io.pravega.auth.TokenException;
import io.pravega.auth.TokenExpiredException;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.Timer;
import io.pravega.common.function.Callbacks;
import io.pravega.common.tracing.TagLogger;
import io.pravega.common.util.BufferView;
import io.pravega.segmentstore.contracts.AttributeId;
import io.pravega.segmentstore.contracts.AttributeUpdate;
import io.pravega.segmentstore.contracts.AttributeUpdateCollection;
import io.pravega.segmentstore.contracts.AttributeUpdateType;
import io.pravega.segmentstore.contracts.Attributes;
import io.pravega.segmentstore.contracts.BadAttributeUpdateException;
import io.pravega.segmentstore.contracts.BadOffsetException;
import io.pravega.segmentstore.contracts.ContainerNotFoundException;
import io.pravega.segmentstore.contracts.MergeStreamSegmentResult;
import io.pravega.segmentstore.contracts.ReadResult;
import io.pravega.segmentstore.contracts.ReadResultEntry;
import io.pravega.segmentstore.contracts.ReadResultEntryType;
import io.pravega.segmentstore.contracts.SegmentType;
import io.pravega.segmentstore.contracts.StreamSegmentExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentMergedException;
import io.pravega.segmentstore.contracts.StreamSegmentNotExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentSealedException;
import io.pravega.segmentstore.contracts.StreamSegmentStore;
import io.pravega.segmentstore.contracts.StreamSegmentTruncatedException;
import io.pravega.segmentstore.contracts.tables.BadKeyVersionException;
import io.pravega.segmentstore.contracts.tables.IteratorArgs;
import io.pravega.segmentstore.contracts.tables.KeyNotExistsException;
import io.pravega.segmentstore.contracts.tables.TableEntry;
import io.pravega.segmentstore.contracts.tables.TableKey;
import io.pravega.segmentstore.contracts.tables.TableSegmentConfig;
import io.pravega.segmentstore.contracts.tables.TableSegmentNotEmptyException;
import io.pravega.segmentstore.contracts.tables.TableStore;
import io.pravega.segmentstore.server.IllegalContainerStateException;
import io.pravega.segmentstore.server.host.delegationtoken.DelegationTokenVerifier;
import io.pravega.segmentstore.server.host.delegationtoken.PassingTokenVerifier;
import io.pravega.segmentstore.server.host.handler.ConnectionTracker;
import io.pravega.segmentstore.server.host.handler.ServerConnection;
import io.pravega.segmentstore.server.host.handler.TrackedConnection;
import io.pravega.segmentstore.server.host.stat.SegmentStatsRecorder;
import io.pravega.segmentstore.server.host.stat.TableSegmentStatsRecorder;
import io.pravega.segmentstore.server.tables.DeltaIteratorState;
import io.pravega.shared.protocol.netty.ByteBufWrapper;
import io.pravega.shared.protocol.netty.FailingRequestProcessor;
import io.pravega.shared.protocol.netty.RequestProcessor;
import io.pravega.shared.protocol.netty.WireCommand;
import io.pravega.shared.protocol.netty.WireCommands;
import java.time.Duration;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import lombok.Generated;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PravegaRequestProcessor
extends FailingRequestProcessor
implements RequestProcessor {
    static final Duration TIMEOUT = Duration.ofMinutes(1L);
    private static final TagLogger log = new TagLogger(LoggerFactory.getLogger(PravegaRequestProcessor.class));
    private static final int MAX_READ_SIZE = 0x200000;
    private static final String EMPTY_STACK_TRACE = "";
    private final StreamSegmentStore segmentStore;
    private final TableStore tableStore;
    private final SegmentStatsRecorder statsRecorder;
    private final TableSegmentStatsRecorder tableStatsRecorder;
    private final DelegationTokenVerifier tokenVerifier;
    private final boolean replyWithStackTraceOnError;
    private final TrackedConnection connection;

    @VisibleForTesting
    public PravegaRequestProcessor(StreamSegmentStore segmentStore, TableStore tableStore, ServerConnection connection) {
        this(segmentStore, tableStore, new TrackedConnection(connection, new ConnectionTracker()), SegmentStatsRecorder.noOp(), TableSegmentStatsRecorder.noOp(), new PassingTokenVerifier(), false);
    }

    PravegaRequestProcessor(@NonNull StreamSegmentStore segmentStore, @NonNull TableStore tableStore, @NonNull TrackedConnection connection, @NonNull SegmentStatsRecorder statsRecorder, @NonNull TableSegmentStatsRecorder tableStatsRecorder, @NonNull DelegationTokenVerifier tokenVerifier, boolean replyWithStackTraceOnError) {
        if (segmentStore == null) {
            throw new NullPointerException("segmentStore is marked non-null but is null");
        }
        if (tableStore == null) {
            throw new NullPointerException("tableStore is marked non-null but is null");
        }
        if (connection == null) {
            throw new NullPointerException("connection is marked non-null but is null");
        }
        if (statsRecorder == null) {
            throw new NullPointerException("statsRecorder is marked non-null but is null");
        }
        if (tableStatsRecorder == null) {
            throw new NullPointerException("tableStatsRecorder is marked non-null but is null");
        }
        if (tokenVerifier == null) {
            throw new NullPointerException("tokenVerifier is marked non-null but is null");
        }
        this.segmentStore = segmentStore;
        this.tableStore = tableStore;
        this.connection = connection;
        this.tokenVerifier = tokenVerifier;
        this.statsRecorder = statsRecorder;
        this.tableStatsRecorder = tableStatsRecorder;
        this.replyWithStackTraceOnError = replyWithStackTraceOnError;
    }

    public void readSegment(WireCommands.ReadSegment readSegment) {
        Timer timer = new Timer();
        String segment = readSegment.getSegment();
        String operation = "readSegment";
        if (!this.verifyToken(segment, readSegment.getOffset(), readSegment.getDelegationToken(), "readSegment")) {
            return;
        }
        int readSize = Math.min(0x200000, Math.max(8, readSegment.getSuggestedLength()));
        long trace = LoggerHelpers.traceEnter((Logger)log, (String)"readSegment", (Object[])new Object[]{readSegment});
        ((CompletableFuture)this.segmentStore.read(segment, readSegment.getOffset(), readSize, TIMEOUT).thenAccept(readResult -> {
            LoggerHelpers.traceLeave((Logger)log, (String)"readSegment", (long)trace, (Object[])new Object[]{readResult});
            this.handleReadResult(readSegment, (ReadResult)readResult);
            this.statsRecorder.readComplete(timer.getElapsed());
        })).exceptionally(ex -> this.handleException(readSegment.getRequestId(), segment, readSegment.getOffset(), "readSegment", this.wrapCancellationException((Throwable)ex)));
    }

    protected boolean verifyToken(String segment, long requestId, String delegationToken, String operation) {
        boolean isTokenValid = false;
        try {
            this.tokenVerifier.verifyToken(segment, delegationToken, AuthHandler.Permissions.READ);
            isTokenValid = true;
        }
        catch (TokenException e) {
            this.handleException(requestId, segment, operation, e);
        }
        return isTokenValid;
    }

    private void handleReadResult(WireCommands.ReadSegment request, ReadResult result) {
        boolean atTail;
        String segment = request.getSegment();
        ArrayList<BufferView> cachedEntries = new ArrayList<BufferView>();
        ReadResultEntry nonCachedEntry = this.collectCachedEntries(request.getOffset(), result, cachedEntries);
        String operation = "readSegment";
        boolean truncated = nonCachedEntry != null && nonCachedEntry.getType() == ReadResultEntryType.Truncated;
        boolean endOfSegment = nonCachedEntry != null && nonCachedEntry.getType() == ReadResultEntryType.EndOfStreamSegment;
        boolean bl = atTail = nonCachedEntry != null && nonCachedEntry.getType() == ReadResultEntryType.Future;
        if (!cachedEntries.isEmpty() || endOfSegment) {
            ByteBuf data = this.toByteBuf(cachedEntries);
            WireCommands.SegmentRead reply = new WireCommands.SegmentRead(segment, request.getOffset(), atTail, endOfSegment, data, request.getRequestId());
            this.connection.send((WireCommand)reply);
            this.statsRecorder.read(segment, reply.getData().readableBytes());
        } else if (truncated) {
            ((CompletableFuture)this.segmentStore.getStreamSegmentInfo(segment, TIMEOUT).thenAccept(info -> this.connection.send((WireCommand)new WireCommands.SegmentIsTruncated(request.getRequestId(), segment, info.getStartOffset(), EMPTY_STACK_TRACE, nonCachedEntry.getStreamSegmentOffset())))).exceptionally(e -> this.handleException(request.getRequestId(), segment, nonCachedEntry.getStreamSegmentOffset(), "readSegment", this.wrapCancellationException((Throwable)e)));
        } else {
            Preconditions.checkState((nonCachedEntry != null ? 1 : 0) != 0, (Object)"No ReadResultEntries returned from read!?");
            nonCachedEntry.requestContent(TIMEOUT);
            ((CompletableFuture)((CompletableFuture)nonCachedEntry.getContent().thenAccept(contents -> {
                ByteBuf data = this.toByteBuf(Collections.singletonList(contents));
                WireCommands.SegmentRead reply = new WireCommands.SegmentRead(segment, nonCachedEntry.getStreamSegmentOffset(), atTail, endOfSegment, data, request.getRequestId());
                this.connection.send((WireCommand)reply);
                this.statsRecorder.read(segment, reply.getData().readableBytes());
            })).exceptionally(exception -> {
                Throwable e = Exceptions.unwrap((Throwable)exception);
                if (e instanceof StreamSegmentTruncatedException) {
                    String clientReplyStackTrace = this.replyWithStackTraceOnError ? e.getMessage() : EMPTY_STACK_TRACE;
                    this.connection.send((WireCommand)new WireCommands.SegmentIsTruncated(request.getRequestId(), segment, ((StreamSegmentTruncatedException)e).getStartOffset(), clientReplyStackTrace, nonCachedEntry.getStreamSegmentOffset()));
                } else {
                    this.handleException(request.getRequestId(), segment, nonCachedEntry.getStreamSegmentOffset(), "readSegment", this.wrapCancellationException(e));
                }
                return null;
            })).exceptionally(e -> this.handleException(request.getRequestId(), segment, nonCachedEntry.getStreamSegmentOffset(), "readSegment", this.wrapCancellationException((Throwable)e)));
        }
    }

    private Throwable wrapCancellationException(Throwable u) {
        Throwable wrappedException = null;
        if (u != null && (wrappedException = Exceptions.unwrap((Throwable)u)) instanceof CancellationException) {
            wrappedException = new ReadCancellationException(wrappedException);
        }
        return wrappedException;
    }

    private ReadResultEntry collectCachedEntries(long initialOffset, ReadResult readResult, ArrayList<BufferView> cachedEntries) {
        long expectedOffset = initialOffset;
        while (readResult.hasNext()) {
            ReadResultEntry entry = (ReadResultEntry)readResult.next();
            if (entry.getType() == ReadResultEntryType.Cache) {
                Preconditions.checkState((entry.getStreamSegmentOffset() == expectedOffset ? 1 : 0) != 0, (Object)"Data returned from read was not contiguous.");
                BufferView content = entry.getContent().getNow(null);
                expectedOffset += (long)content.getLength();
                cachedEntries.add(content);
                continue;
            }
            return entry;
        }
        return null;
    }

    private ByteBuf toByteBuf(List<BufferView> contents) {
        Iterator iterators = Iterators.concat((Iterator)Iterators.transform(contents.iterator(), BufferView::iterateBuffers));
        Iterator b = Iterators.transform((Iterator)iterators, Unpooled::wrappedBuffer);
        return Unpooled.wrappedUnmodifiableBuffer((ByteBuf[])((ByteBuf[])Iterators.toArray((Iterator)b, ByteBuf.class)));
    }

    private ByteBuf toByteBuf(BufferView bufferView) {
        Iterator iterators = Iterators.transform((Iterator)bufferView.iterateBuffers(), Unpooled::wrappedBuffer);
        return Unpooled.wrappedUnmodifiableBuffer((ByteBuf[])((ByteBuf[])Iterators.toArray((Iterator)iterators, ByteBuf.class)));
    }

    public void updateSegmentAttribute(WireCommands.UpdateSegmentAttribute updateSegmentAttribute) {
        long requestId = updateSegmentAttribute.getRequestId();
        String segmentName = updateSegmentAttribute.getSegmentName();
        AttributeId attributeId = updateSegmentAttribute.getAttributeId() == null ? null : AttributeId.fromUUID((UUID)updateSegmentAttribute.getAttributeId());
        long newValue = updateSegmentAttribute.getNewValue();
        long expectedValue = updateSegmentAttribute.getExpectedValue();
        String operation = "updateSegmentAttribute";
        if (!this.verifyToken(segmentName, updateSegmentAttribute.getRequestId(), updateSegmentAttribute.getDelegationToken(), "updateSegmentAttribute")) {
            return;
        }
        long trace = LoggerHelpers.traceEnter((Logger)log, (String)"updateSegmentAttribute", (Object[])new Object[]{updateSegmentAttribute});
        AttributeUpdate update = new AttributeUpdate(attributeId, AttributeUpdateType.ReplaceIfEquals, newValue, expectedValue);
        this.segmentStore.updateAttributes(segmentName, AttributeUpdateCollection.from((AttributeUpdate[])new AttributeUpdate[]{update}), TIMEOUT).whenComplete((v, e) -> {
            LoggerHelpers.traceLeave((Logger)log, (String)"updateSegmentAttribute", (long)trace, (Object[])new Object[]{e});
            Consumer<Throwable> failureHandler = t -> {
                log.error(requestId, "Error (Segment = '{}', Operation = '{}')", new Object[]{segmentName, "handling result of updateSegmentAttribute", t});
                this.connection.close();
            };
            if (e == null) {
                Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentAttributeUpdated(requestId, true), failureHandler);
            } else if (Exceptions.unwrap((Throwable)e) instanceof BadAttributeUpdateException) {
                log.debug("Updating segment attribute {} failed due to: {}", (Object)update, (Object)e.getMessage());
                Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentAttributeUpdated(requestId, false), failureHandler);
            } else {
                this.handleException(requestId, segmentName, "updateSegmentAttribute", (Throwable)e);
            }
        });
    }

    public void getSegmentAttribute(WireCommands.GetSegmentAttribute getSegmentAttribute) {
        long requestId = getSegmentAttribute.getRequestId();
        String segmentName = getSegmentAttribute.getSegmentName();
        AttributeId attributeId = getSegmentAttribute.getAttributeId() == null ? null : AttributeId.fromUUID((UUID)getSegmentAttribute.getAttributeId());
        String operation = "getSegmentAttribute";
        if (!this.verifyToken(segmentName, getSegmentAttribute.getRequestId(), getSegmentAttribute.getDelegationToken(), "getSegmentAttribute")) {
            return;
        }
        long trace = LoggerHelpers.traceEnter((Logger)log, (String)"getSegmentAttribute", (Object[])new Object[]{getSegmentAttribute});
        ((CompletableFuture)this.segmentStore.getStreamSegmentInfo(segmentName, TIMEOUT).thenAccept(properties -> {
            LoggerHelpers.traceLeave((Logger)log, (String)"getSegmentAttribute", (long)trace, (Object[])new Object[]{properties});
            if (properties == null) {
                this.connection.send((WireCommand)new WireCommands.NoSuchSegment(requestId, segmentName, EMPTY_STACK_TRACE, -1L));
            } else {
                Long value = (Long)properties.getAttributes().get(attributeId);
                if (value == null) {
                    value = Long.MIN_VALUE;
                }
                this.connection.send((WireCommand)new WireCommands.SegmentAttribute(requestId, value.longValue()));
            }
        })).exceptionally(e -> this.handleException(requestId, segmentName, "getSegmentAttribute", (Throwable)e));
    }

    public void getStreamSegmentInfo(WireCommands.GetStreamSegmentInfo getStreamSegmentInfo) {
        String segmentName = getStreamSegmentInfo.getSegmentName();
        String operation = "getStreamSegmentInfo";
        if (!this.verifyToken(segmentName, getStreamSegmentInfo.getRequestId(), getStreamSegmentInfo.getDelegationToken(), "getStreamSegmentInfo")) {
            return;
        }
        ((CompletableFuture)this.segmentStore.getStreamSegmentInfo(segmentName, TIMEOUT).thenAccept(properties -> {
            if (properties != null) {
                WireCommands.StreamSegmentInfo result = new WireCommands.StreamSegmentInfo(getStreamSegmentInfo.getRequestId(), properties.getName(), true, properties.isSealed(), properties.isDeleted(), properties.getLastModified().getTime(), properties.getLength(), properties.getStartOffset());
                log.trace("Read stream segment info: {}", (Object)result);
                this.connection.send((WireCommand)result);
            } else {
                log.trace("getStreamSegmentInfo could not find segment {}", (Object)segmentName);
                this.connection.send((WireCommand)new WireCommands.StreamSegmentInfo(getStreamSegmentInfo.getRequestId(), segmentName, false, true, true, 0L, 0L, 0L));
            }
        })).exceptionally(e -> this.handleException(getStreamSegmentInfo.getRequestId(), segmentName, "getStreamSegmentInfo", (Throwable)e));
    }

    public void createSegment(WireCommands.CreateSegment createStreamSegment) {
        Timer timer = new Timer();
        String operation = "createSegment";
        if (createStreamSegment.getRolloverSizeBytes() < 0L) {
            log.warn("Segment rollover size bytes cannot be less than 0, actual is {}, fall back to default value", (Object)createStreamSegment.getRolloverSizeBytes());
        }
        long rolloverSizeBytes = createStreamSegment.getRolloverSizeBytes() < 0L ? 0L : createStreamSegment.getRolloverSizeBytes();
        List<AttributeUpdate> attributes = Arrays.asList(new AttributeUpdate(Attributes.SCALE_POLICY_TYPE, AttributeUpdateType.Replace, Byte.valueOf(createStreamSegment.getScaleType()).longValue()), new AttributeUpdate(Attributes.SCALE_POLICY_RATE, AttributeUpdateType.Replace, Integer.valueOf(createStreamSegment.getTargetRate()).longValue()), new AttributeUpdate(Attributes.ROLLOVER_SIZE, AttributeUpdateType.Replace, rolloverSizeBytes), new AttributeUpdate(Attributes.CREATION_TIME, AttributeUpdateType.None, System.currentTimeMillis()));
        if (!this.verifyToken(createStreamSegment.getSegment(), createStreamSegment.getRequestId(), createStreamSegment.getDelegationToken(), "createSegment")) {
            return;
        }
        log.info(createStreamSegment.getRequestId(), "Creating stream segment {}.", new Object[]{createStreamSegment});
        ((CompletableFuture)this.segmentStore.createStreamSegment(createStreamSegment.getSegment(), SegmentType.STREAM_SEGMENT, attributes, TIMEOUT).thenAccept(v -> this.connection.send((WireCommand)new WireCommands.SegmentCreated(createStreamSegment.getRequestId(), createStreamSegment.getSegment())))).whenComplete((res, e) -> {
            if (e == null) {
                this.statsRecorder.createSegment(createStreamSegment.getSegment(), createStreamSegment.getScaleType(), createStreamSegment.getTargetRate(), timer.getElapsed());
            } else {
                this.handleException(createStreamSegment.getRequestId(), createStreamSegment.getSegment(), "createSegment", (Throwable)e);
            }
        });
    }

    public void mergeSegments(WireCommands.MergeSegments mergeSegments) {
        String operation = "mergeSegments";
        if (!this.verifyToken(mergeSegments.getSource(), mergeSegments.getRequestId(), mergeSegments.getDelegationToken(), "mergeSegments")) {
            return;
        }
        log.info(mergeSegments.getRequestId(), "Merging Segments {} ", new Object[]{mergeSegments});
        AttributeUpdateCollection attributeUpdates = new AttributeUpdateCollection();
        if (mergeSegments.getAttributeUpdates() != null) {
            for (WireCommands.ConditionalAttributeUpdate update : mergeSegments.getAttributeUpdates()) {
                attributeUpdates.add(new AttributeUpdate(AttributeId.fromUUID((UUID)update.getAttributeId()), AttributeUpdateType.get((byte)update.getAttributeUpdateType()), update.getNewValue(), update.getOldValue()));
            }
        }
        ((CompletableFuture)this.segmentStore.mergeStreamSegment(mergeSegments.getTarget(), mergeSegments.getSource(), attributeUpdates, TIMEOUT).thenAccept(mergeResult -> {
            this.recordStatForTransaction((MergeStreamSegmentResult)mergeResult, mergeSegments.getTarget());
            this.connection.send((WireCommand)new WireCommands.SegmentsMerged(mergeSegments.getRequestId(), mergeSegments.getTarget(), mergeSegments.getSource(), mergeResult.getTargetSegmentLength()));
        })).exceptionally(e -> {
            if (Exceptions.unwrap((Throwable)e) instanceof StreamSegmentMergedException) {
                log.info(mergeSegments.getRequestId(), "Stream segment is already merged '{}'.", new Object[]{mergeSegments.getSource()});
                this.segmentStore.getStreamSegmentInfo(mergeSegments.getTarget(), TIMEOUT).thenAccept(properties -> this.connection.send((WireCommand)new WireCommands.SegmentsMerged(mergeSegments.getRequestId(), mergeSegments.getTarget(), mergeSegments.getSource(), properties.getLength())));
                return null;
            }
            if (Exceptions.unwrap((Throwable)e) instanceof BadAttributeUpdateException) {
                log.debug(mergeSegments.getRequestId(), "Conditional merge failed (Source segment={}, Target segment={}): {}", new Object[]{mergeSegments.getSource(), mergeSegments.getTarget(), e.toString()});
                this.connection.send((WireCommand)new WireCommands.SegmentAttributeUpdated(mergeSegments.getRequestId(), false));
                return null;
            }
            return this.handleException(mergeSegments.getRequestId(), mergeSegments.getSource(), "mergeSegments", (Throwable)e);
        });
    }

    public void sealSegment(WireCommands.SealSegment sealSegment) {
        String segment = sealSegment.getSegment();
        String operation = "sealSegment";
        if (!this.verifyToken(segment, sealSegment.getRequestId(), sealSegment.getDelegationToken(), "sealSegment")) {
            return;
        }
        log.info(sealSegment.getRequestId(), "Sealing segment {} ", new Object[]{sealSegment});
        ((CompletableFuture)this.segmentStore.sealStreamSegment(segment, TIMEOUT).thenAccept(size -> this.connection.send((WireCommand)new WireCommands.SegmentSealed(sealSegment.getRequestId(), segment)))).whenComplete((r, e) -> {
            if (e != null) {
                this.handleException(sealSegment.getRequestId(), segment, "sealSegment", (Throwable)e);
            } else {
                this.statsRecorder.sealSegment(sealSegment.getSegment());
            }
        });
    }

    public void truncateSegment(WireCommands.TruncateSegment truncateSegment) {
        String segment = truncateSegment.getSegment();
        String operation = "truncateSegment";
        if (!this.verifyToken(segment, truncateSegment.getRequestId(), truncateSegment.getDelegationToken(), "truncateSegment")) {
            return;
        }
        long offset = truncateSegment.getTruncationOffset();
        log.info(truncateSegment.getRequestId(), "Truncating segment {} at offset {}.", new Object[]{segment, offset});
        ((CompletableFuture)this.segmentStore.truncateStreamSegment(segment, offset, TIMEOUT).thenAccept(v -> this.connection.send((WireCommand)new WireCommands.SegmentTruncated(truncateSegment.getRequestId(), segment)))).exceptionally(e -> this.handleException(truncateSegment.getRequestId(), segment, offset, "truncateSegment", (Throwable)e));
    }

    public void deleteSegment(WireCommands.DeleteSegment deleteSegment) {
        String segment = deleteSegment.getSegment();
        String operation = "deleteSegment";
        if (!this.verifyToken(segment, deleteSegment.getRequestId(), deleteSegment.getDelegationToken(), "deleteSegment")) {
            return;
        }
        log.info(deleteSegment.getRequestId(), "Deleting segment {} ", new Object[]{deleteSegment});
        ((CompletableFuture)this.segmentStore.deleteStreamSegment(segment, TIMEOUT).thenRun(() -> {
            this.connection.send((WireCommand)new WireCommands.SegmentDeleted(deleteSegment.getRequestId(), segment));
            this.statsRecorder.deleteSegment(segment);
        })).exceptionally(e -> this.handleException(deleteSegment.getRequestId(), segment, "deleteSegment", (Throwable)e));
    }

    public void updateSegmentPolicy(WireCommands.UpdateSegmentPolicy updateSegmentPolicy) {
        String operation = "updateSegmentPolicy";
        if (!this.verifyToken(updateSegmentPolicy.getSegment(), updateSegmentPolicy.getRequestId(), updateSegmentPolicy.getDelegationToken(), "updateSegmentPolicy")) {
            return;
        }
        AttributeUpdateCollection attributes = AttributeUpdateCollection.from((AttributeUpdate[])new AttributeUpdate[]{new AttributeUpdate(Attributes.SCALE_POLICY_TYPE, AttributeUpdateType.Replace, (long)updateSegmentPolicy.getScaleType()), new AttributeUpdate(Attributes.SCALE_POLICY_RATE, AttributeUpdateType.Replace, (long)updateSegmentPolicy.getTargetRate())});
        log.info(updateSegmentPolicy.getRequestId(), "Updating segment policy {} ", new Object[]{updateSegmentPolicy});
        ((CompletableFuture)this.segmentStore.updateAttributes(updateSegmentPolicy.getSegment(), attributes, TIMEOUT).thenRun(() -> this.connection.send((WireCommand)new WireCommands.SegmentPolicyUpdated(updateSegmentPolicy.getRequestId(), updateSegmentPolicy.getSegment())))).whenComplete((r, e) -> {
            if (e != null) {
                this.handleException(updateSegmentPolicy.getRequestId(), updateSegmentPolicy.getSegment(), "updateSegmentPolicy", (Throwable)e);
            } else {
                this.statsRecorder.policyUpdate(updateSegmentPolicy.getSegment(), updateSegmentPolicy.getScaleType(), updateSegmentPolicy.getTargetRate());
            }
        });
    }

    public void getTableSegmentInfo(WireCommands.GetTableSegmentInfo getInfo) {
        String operation = "getTableSegmentInfo";
        if (!this.verifyToken(getInfo.getSegmentName(), getInfo.getRequestId(), getInfo.getDelegationToken(), "getTableSegmentInfo")) {
            return;
        }
        Timer timer = new Timer();
        log.debug(getInfo.getRequestId(), "Get Table Segment Info {}.", new Object[]{getInfo.getSegmentName()});
        ((CompletableFuture)this.tableStore.getInfo(getInfo.getSegmentName(), TIMEOUT).thenAccept(info -> {
            this.connection.send((WireCommand)new WireCommands.TableSegmentInfo(getInfo.getRequestId(), getInfo.getSegmentName(), info.getStartOffset(), info.getLength(), info.getEntryCount(), info.getKeyLength()));
            this.tableStatsRecorder.getInfo(getInfo.getSegmentName(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(getInfo.getRequestId(), getInfo.getSegmentName(), "getTableSegmentInfo", (Throwable)e));
    }

    public void createTableSegment(WireCommands.CreateTableSegment createTableSegment) {
        String operation = "createTableSegment";
        if (!this.verifyToken(createTableSegment.getSegment(), createTableSegment.getRequestId(), createTableSegment.getDelegationToken(), "createTableSegment")) {
            return;
        }
        log.info(createTableSegment.getRequestId(), "Creating table segment {}.", new Object[]{createTableSegment});
        Timer timer = new Timer();
        SegmentType.Builder typeBuilder = SegmentType.builder().tableSegment();
        TableSegmentConfig.TableSegmentConfigBuilder configBuilder = TableSegmentConfig.builder();
        if (createTableSegment.getKeyLength() > 0) {
            typeBuilder.fixedKeyLengthTableSegment();
            configBuilder.keyLength(createTableSegment.getKeyLength());
        }
        if (createTableSegment.getRolloverSizeBytes() < 0L) {
            log.warn("Table segment rollover size bytes cannot be less than 0, actual is {}, fall back to default value", (Object)createTableSegment.getRolloverSizeBytes());
        }
        long rolloverSizeByes = createTableSegment.getRolloverSizeBytes() < 0L ? 0L : createTableSegment.getRolloverSizeBytes();
        configBuilder.rolloverSizeBytes(rolloverSizeByes);
        ((CompletableFuture)this.tableStore.createSegment(createTableSegment.getSegment(), typeBuilder.build(), configBuilder.build(), TIMEOUT).thenAccept(v -> {
            this.connection.send((WireCommand)new WireCommands.SegmentCreated(createTableSegment.getRequestId(), createTableSegment.getSegment()));
            this.tableStatsRecorder.createTableSegment(createTableSegment.getSegment(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(createTableSegment.getRequestId(), createTableSegment.getSegment(), "createTableSegment", (Throwable)e));
    }

    public void deleteTableSegment(WireCommands.DeleteTableSegment deleteTableSegment) {
        String segment = deleteTableSegment.getSegment();
        String operation = "deleteTableSegment";
        if (!this.verifyToken(segment, deleteTableSegment.getRequestId(), deleteTableSegment.getDelegationToken(), "deleteTableSegment")) {
            return;
        }
        log.info(deleteTableSegment.getRequestId(), "Deleting table segment {}.", new Object[]{deleteTableSegment});
        Timer timer = new Timer();
        ((CompletableFuture)this.tableStore.deleteSegment(segment, deleteTableSegment.isMustBeEmpty(), TIMEOUT).thenRun(() -> {
            this.connection.send((WireCommand)new WireCommands.SegmentDeleted(deleteTableSegment.getRequestId(), segment));
            this.tableStatsRecorder.deleteTableSegment(segment, timer.getElapsed());
        })).exceptionally(e -> this.handleException(deleteTableSegment.getRequestId(), segment, "deleteTableSegment", (Throwable)e));
    }

    public void updateTableEntries(WireCommands.UpdateTableEntries updateTableEntries) {
        String segment = updateTableEntries.getSegment();
        String operation = "updateTableEntries";
        if (!this.verifyToken(segment, updateTableEntries.getRequestId(), updateTableEntries.getDelegationToken(), "updateTableEntries")) {
            updateTableEntries.release();
            return;
        }
        log.debug(updateTableEntries.getRequestId(), "Update Table Segment Entries: Segment={}, Offset={}, Count={}.", new Object[]{updateTableEntries.getSegment(), updateTableEntries.getTableSegmentOffset(), updateTableEntries.getTableEntries().getEntries().size()});
        ArrayList<TableEntry> entries = new ArrayList<TableEntry>(updateTableEntries.getTableEntries().getEntries().size());
        AtomicBoolean conditional = new AtomicBoolean(false);
        AtomicInteger size = new AtomicInteger(0);
        for (Map.Entry e2 : updateTableEntries.getTableEntries().getEntries()) {
            TableEntry v = TableEntry.versioned((BufferView)new ByteBufWrapper(((WireCommands.TableKey)e2.getKey()).getData()), (BufferView)new ByteBufWrapper(((WireCommands.TableValue)e2.getValue()).getData()), (long)((WireCommands.TableKey)e2.getKey()).getKeyVersion());
            entries.add(v);
            size.addAndGet(v.getKey().getKey().getLength() + v.getValue().getLength());
            if (!v.getKey().hasVersion()) continue;
            conditional.set(true);
        }
        Timer timer = new Timer();
        this.connection.adjustOutstandingBytes(size.get());
        ((CompletableFuture)((CompletableFuture)this.tableStore.put(segment, entries, updateTableEntries.getTableSegmentOffset(), TIMEOUT).thenAccept(versions -> {
            this.connection.send((WireCommand)new WireCommands.TableEntriesUpdated(updateTableEntries.getRequestId(), versions));
            this.tableStatsRecorder.updateEntries(updateTableEntries.getSegment(), entries.size(), conditional.get(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(updateTableEntries.getRequestId(), segment, updateTableEntries.getTableSegmentOffset(), "updateTableEntries", (Throwable)e))).whenComplete((r, ex) -> {
            this.connection.adjustOutstandingBytes(-size.get());
            updateTableEntries.release();
        });
    }

    public void removeTableKeys(WireCommands.RemoveTableKeys removeTableKeys) {
        String segment = removeTableKeys.getSegment();
        String operation = "removeTableKeys";
        if (!this.verifyToken(segment, removeTableKeys.getRequestId(), removeTableKeys.getDelegationToken(), "removeTableKeys")) {
            removeTableKeys.release();
            return;
        }
        log.debug(removeTableKeys.getRequestId(), "Remove Table Segment Keys: Segment={}, Offset={}, Count={}.", new Object[]{removeTableKeys.getSegment(), removeTableKeys.getTableSegmentOffset(), removeTableKeys.getKeys().size()});
        ArrayList<TableKey> keys = new ArrayList<TableKey>(removeTableKeys.getKeys().size());
        AtomicBoolean conditional = new AtomicBoolean(false);
        AtomicInteger size = new AtomicInteger(0);
        for (WireCommands.TableKey k : removeTableKeys.getKeys()) {
            TableKey v = TableKey.versioned((BufferView)new ByteBufWrapper(k.getData()), (long)k.getKeyVersion());
            keys.add(v);
            size.addAndGet(v.getKey().getLength());
            if (!v.hasVersion()) continue;
            conditional.set(true);
        }
        Timer timer = new Timer();
        this.connection.adjustOutstandingBytes(size.get());
        ((CompletableFuture)((CompletableFuture)this.tableStore.remove(segment, keys, removeTableKeys.getTableSegmentOffset(), TIMEOUT).thenRun(() -> {
            this.connection.send((WireCommand)new WireCommands.TableKeysRemoved(removeTableKeys.getRequestId(), segment));
            this.tableStatsRecorder.removeKeys(removeTableKeys.getSegment(), keys.size(), conditional.get(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(removeTableKeys.getRequestId(), segment, removeTableKeys.getTableSegmentOffset(), "removeTableKeys", (Throwable)e))).whenComplete((r, ex) -> {
            this.connection.adjustOutstandingBytes(-size.get());
            removeTableKeys.release();
        });
    }

    public void readTable(WireCommands.ReadTable readTable) {
        String segment = readTable.getSegment();
        String operation = "readTable";
        if (!this.verifyToken(segment, readTable.getRequestId(), readTable.getDelegationToken(), "readTable")) {
            readTable.release();
            return;
        }
        log.debug(readTable.getRequestId(), "Get Table Segment Keys: Segment={}, Count={}.", new Object[]{readTable.getSegment(), readTable.getKeys()});
        List keys = readTable.getKeys().stream().map(k -> new ByteBufWrapper(k.getData())).collect(Collectors.toList());
        Timer timer = new Timer();
        ((CompletableFuture)((CompletableFuture)this.tableStore.get(segment, keys, TIMEOUT).thenAccept(values -> {
            this.connection.send((WireCommand)new WireCommands.TableRead(readTable.getRequestId(), segment, this.getTableEntriesCommand(keys, (List<TableEntry>)values)));
            this.tableStatsRecorder.getKeys(readTable.getSegment(), keys.size(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(readTable.getRequestId(), segment, "readTable", (Throwable)e))).whenComplete((r, ex) -> readTable.release());
    }

    public void readTableKeys(WireCommands.ReadTableKeys readTableKeys) {
        String segment = readTableKeys.getSegment();
        String operation = "readTableKeys";
        if (!this.verifyToken(segment, readTableKeys.getRequestId(), readTableKeys.getDelegationToken(), "readTableKeys")) {
            return;
        }
        log.debug(readTableKeys.getRequestId(), "Iterate Table Segment Keys: Segment={}, Count={}.", new Object[]{readTableKeys.getSegment(), readTableKeys.getSuggestedKeyCount()});
        int suggestedKeyCount = readTableKeys.getSuggestedKeyCount();
        IteratorArgs args = this.getIteratorArgs(readTableKeys.getArgs());
        IteratorResult result = new IteratorResult(segment.getBytes().length + 8);
        Timer timer = new Timer();
        ((CompletableFuture)((CompletableFuture)this.tableStore.keyIterator(segment, args).thenCompose(itr -> itr.collectRemaining(e -> {
            IteratorResult iteratorResult = result;
            synchronized (iteratorResult) {
                if (result.getItemCount() >= suggestedKeyCount || result.getSizeBytes() >= 0x200000) {
                    return false;
                }
                for (TableKey key : e.getEntries()) {
                    WireCommands.TableKey k = new WireCommands.TableKey(this.toByteBuf(key.getKey()), key.getVersion());
                    result.add(k, k.size());
                }
                result.setContinuationToken(e.getState());
                return true;
            }
        }))).thenAccept(v -> {
            log.debug(readTableKeys.getRequestId(), "Iterate Table Segment Keys complete ({}).", new Object[]{result.getItemCount()});
            this.connection.send((WireCommand)new WireCommands.TableKeysRead(readTableKeys.getRequestId(), segment, result.getItems(), this.toByteBuf(result.getContinuationToken())));
            this.tableStatsRecorder.iterateKeys(readTableKeys.getSegment(), result.getItemCount(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(readTableKeys.getRequestId(), segment, "readTableKeys", (Throwable)e));
    }

    public void readTableEntries(WireCommands.ReadTableEntries readTableEntries) {
        String segment = readTableEntries.getSegment();
        String operation = "readTableEntries";
        if (!this.verifyToken(segment, readTableEntries.getRequestId(), readTableEntries.getDelegationToken(), "readTableEntries")) {
            return;
        }
        log.debug(readTableEntries.getRequestId(), "Iterate Table Segment Entries: Segment={}, Count={}.", new Object[]{readTableEntries.getSegment(), readTableEntries.getSuggestedEntryCount()});
        int suggestedEntryCount = readTableEntries.getSuggestedEntryCount();
        IteratorArgs args = this.getIteratorArgs(readTableEntries.getArgs());
        IteratorResult result = new IteratorResult(segment.getBytes().length + 8);
        Timer timer = new Timer();
        ((CompletableFuture)((CompletableFuture)this.tableStore.entryIterator(segment, args).thenCompose(itr -> itr.collectRemaining(e -> {
            if (result.getItemCount() >= suggestedEntryCount || result.getSizeBytes() >= 0x200000) {
                return false;
            }
            for (TableEntry entry : e.getEntries()) {
                WireCommands.TableKey k = new WireCommands.TableKey(this.toByteBuf(entry.getKey().getKey()), entry.getKey().getVersion());
                WireCommands.TableValue v = new WireCommands.TableValue(this.toByteBuf(entry.getValue()));
                result.add(new AbstractMap.SimpleImmutableEntry<WireCommands.TableKey, WireCommands.TableValue>(k, v), k.size() + v.size());
            }
            result.setContinuationToken(e.getState());
            return true;
        }))).thenAccept(v -> {
            log.debug(readTableEntries.getRequestId(), "Iterate Table Segment Entries complete ({}).", new Object[]{result.getItemCount()});
            this.connection.send((WireCommand)new WireCommands.TableEntriesRead(readTableEntries.getRequestId(), segment, new WireCommands.TableEntries(result.getItems()), this.toByteBuf(result.getContinuationToken())));
            this.tableStatsRecorder.iterateEntries(readTableEntries.getSegment(), result.getItemCount(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(readTableEntries.getRequestId(), segment, "readTableEntries", (Throwable)e));
    }

    private IteratorArgs getIteratorArgs(WireCommands.TableIteratorArgs rawArgs) {
        return IteratorArgs.builder().fetchTimeout(TIMEOUT).continuationToken((BufferView)this.wrap(rawArgs.getContinuationToken())).from((BufferView)this.wrap(rawArgs.getFromKey())).to((BufferView)this.wrap(rawArgs.getToKey())).build();
    }

    private ByteBufWrapper wrap(ByteBuf buf) {
        return buf == null || buf.equals((Object)Unpooled.EMPTY_BUFFER) ? null : new ByteBufWrapper(buf);
    }

    public void readTableEntriesDelta(WireCommands.ReadTableEntriesDelta readTableEntriesDelta) {
        String segment = readTableEntriesDelta.getSegment();
        String operation = "readTableEntriesDelta";
        if (!this.verifyToken(segment, readTableEntriesDelta.getRequestId(), readTableEntriesDelta.getDelegationToken(), "readTableEntriesDelta")) {
            return;
        }
        int suggestedEntryCount = readTableEntriesDelta.getSuggestedEntryCount();
        long fromPosition = readTableEntriesDelta.getFromPosition();
        log.info(readTableEntriesDelta.getRequestId(), "Iterate Table Entries Delta: Segment={} Count={} FromPositon={}.", new Object[]{readTableEntriesDelta.getSegment(), readTableEntriesDelta.getSuggestedEntryCount(), readTableEntriesDelta.getFromPosition()});
        Timer timer = new Timer();
        DeltaIteratorResult result = new DeltaIteratorResult(segment.getBytes().length + 8);
        ((CompletableFuture)((CompletableFuture)this.tableStore.entryDeltaIterator(segment, fromPosition, TIMEOUT).thenCompose(itr -> itr.collectRemaining(e -> {
            if (result.getItemCount() >= suggestedEntryCount || result.getSizeBytes() >= 0x200000) {
                return false;
            }
            TableEntry entry = (TableEntry)e.getEntries().iterator().next();
            DeltaIteratorState state = DeltaIteratorState.deserialize((BufferView)e.getState());
            WireCommands.TableKey k = new WireCommands.TableKey(this.toByteBuf(entry.getKey().getKey()), entry.getKey().getVersion());
            WireCommands.TableValue v = new WireCommands.TableValue(this.toByteBuf(entry.getValue()));
            if (state.isDeletionRecord()) {
                result.remove(entry.getKey().getKey(), k.size() + v.size());
            } else {
                Map.Entry old = (Map.Entry)result.getItem(entry.getKey().getKey());
                if (old != null && ((WireCommands.TableKey)old.getKey()).getKeyVersion() < entry.getKey().getVersion()) {
                    int sizeBytes = k.size() + v.size() - (((WireCommands.TableKey)old.getKey()).size() + ((WireCommands.TableValue)old.getValue()).size());
                    result.add(entry.getKey().getKey(), new AbstractMap.SimpleImmutableEntry<WireCommands.TableKey, WireCommands.TableValue>(k, v), sizeBytes);
                } else {
                    result.add(entry.getKey().getKey(), new AbstractMap.SimpleImmutableEntry<WireCommands.TableKey, WireCommands.TableValue>(k, v), k.size() + v.size());
                }
            }
            result.setState(state);
            return true;
        }))).thenAccept(v -> {
            log.debug(readTableEntriesDelta.getRequestId(), "Iterate Table Segment Entries Delta complete ({}).", new Object[]{result.getItemCount()});
            this.connection.send((WireCommand)new WireCommands.TableEntriesDeltaRead(readTableEntriesDelta.getRequestId(), segment, new WireCommands.TableEntries(result.getItems()), result.getState().isShouldClear(), result.getState().isReachedEnd(), result.getState().getFromPosition()));
            this.tableStatsRecorder.iterateEntries(readTableEntriesDelta.getSegment(), result.getItemCount(), timer.getElapsed());
        })).exceptionally(e -> this.handleException(readTableEntriesDelta.getRequestId(), segment, "readTableEntriesDelta", (Throwable)e));
    }

    private WireCommands.TableEntries getTableEntriesCommand(List<BufferView> inputKeys, List<TableEntry> resultEntries) {
        Preconditions.checkArgument((resultEntries.size() == inputKeys.size() ? 1 : 0) != 0, (Object)"Number of input keys should match result entry count.");
        List entries = IntStream.range(0, resultEntries.size()).mapToObj(i -> {
            TableEntry resultTableEntry = (TableEntry)resultEntries.get(i);
            if (resultTableEntry == null) {
                BufferView k = (BufferView)inputKeys.get(i);
                WireCommands.TableKey keyWireCommand = new WireCommands.TableKey(this.toByteBuf(k), -1L);
                return new AbstractMap.SimpleImmutableEntry<WireCommands.TableKey, WireCommands.TableValue>(keyWireCommand, WireCommands.TableValue.EMPTY);
            }
            TableEntry te = (TableEntry)resultEntries.get(i);
            TableKey k = te.getKey();
            WireCommands.TableKey keyWireCommand = new WireCommands.TableKey(this.toByteBuf(k.getKey()), k.getVersion());
            WireCommands.TableValue valueWireCommand = new WireCommands.TableValue(this.toByteBuf(te.getValue()));
            return new AbstractMap.SimpleImmutableEntry<WireCommands.TableKey, WireCommands.TableValue>(keyWireCommand, valueWireCommand);
        }).collect(Collectors.toList());
        return new WireCommands.TableEntries(entries);
    }

    Void handleException(long requestId, String segment, String operation, Throwable u) {
        return this.handleException(requestId, segment, -1L, operation, u);
    }

    private Void handleException(long requestId, String segment, long offset, String operation, Throwable u) {
        if (u == null) {
            IllegalStateException exception = new IllegalStateException("No exception to handle.");
            log.error(requestId, "Error (Segment = '{}', Operation = '{}')", new Object[]{segment, operation, exception});
            throw exception;
        }
        u = Exceptions.unwrap((Throwable)u);
        String clientReplyStackTrace = this.replyWithStackTraceOnError ? Throwables.getStackTraceAsString((Throwable)u) : EMPTY_STACK_TRACE;
        Consumer<Throwable> failureHandler = t -> {
            log.error(requestId, "Error (Segment = '{}', Operation = '{}')", new Object[]{segment, "handling result of " + operation, t});
            this.connection.close();
        };
        if (u instanceof StreamSegmentExistsException) {
            log.info(requestId, "Segment '{}' already exists and cannot perform operation '{}'.", new Object[]{segment, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentAlreadyExists(requestId, segment, clientReplyStackTrace), failureHandler);
        } else if (u instanceof StreamSegmentNotExistsException) {
            log.warn(requestId, "Segment '{}' does not exist and cannot perform operation '{}'.", new Object[]{segment, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.NoSuchSegment(requestId, segment, clientReplyStackTrace, offset), failureHandler);
        } else if (u instanceof StreamSegmentSealedException) {
            log.info(requestId, "Segment '{}' is sealed and cannot perform operation '{}'.", new Object[]{segment, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentIsSealed(requestId, segment, clientReplyStackTrace, offset), failureHandler);
        } else if (u instanceof ContainerNotFoundException) {
            int containerId = ((ContainerNotFoundException)u).getContainerId();
            log.warn(requestId, "Wrong host. Segment = '{}' (Container {}) is not owned. Operation = '{}').", new Object[]{segment, containerId, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.WrongHost(requestId, segment, EMPTY_STACK_TRACE, clientReplyStackTrace), failureHandler);
        } else if (u instanceof ReadCancellationException) {
            log.info(requestId, "Sending empty response on connection {} while reading segment {} due to CancellationException.", new Object[]{this.connection, segment});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentRead(segment, offset, true, false, Unpooled.EMPTY_BUFFER, requestId), failureHandler);
        } else if (u instanceof CancellationException) {
            log.info(requestId, "Closing connection {} while performing {} due to {}.", new Object[]{this.connection, operation, u.toString()});
            this.connection.close();
        } else if (u instanceof TokenExpiredException) {
            log.warn(requestId, "Expired token during operation {}", new Object[]{operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.AuthTokenCheckFailed(requestId, clientReplyStackTrace, WireCommands.AuthTokenCheckFailed.ErrorCode.TOKEN_EXPIRED), failureHandler);
        } else if (u instanceof TokenException) {
            log.warn(requestId, "Token exception encountered during operation {}.", new Object[]{operation, u});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.AuthTokenCheckFailed(requestId, clientReplyStackTrace, WireCommands.AuthTokenCheckFailed.ErrorCode.TOKEN_CHECK_FAILED), failureHandler);
        } else if (u instanceof UnsupportedOperationException) {
            log.warn(requestId, "Unsupported Operation '{}'.", new Object[]{operation, u});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.OperationUnsupported(requestId, operation, clientReplyStackTrace), failureHandler);
        } else if (u instanceof BadOffsetException) {
            BadOffsetException badOffset = (BadOffsetException)u;
            log.info(requestId, "Segment '{}' is truncated and cannot perform operation '{}' at offset '{}'", new Object[]{segment, operation, offset});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.SegmentIsTruncated(requestId, segment, badOffset.getExpectedOffset(), clientReplyStackTrace, offset), failureHandler);
        } else if (u instanceof TableSegmentNotEmptyException) {
            log.warn(requestId, "Table segment '{}' is not empty to perform '{}'.", new Object[]{segment, operation});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.TableSegmentNotEmpty(requestId, segment, clientReplyStackTrace), failureHandler);
        } else if (u instanceof KeyNotExistsException) {
            log.warn(requestId, "Conditional update on Table segment '{}' failed as the key does not exist.", new Object[]{segment});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.TableKeyDoesNotExist(requestId, segment, clientReplyStackTrace), failureHandler);
        } else if (u instanceof BadKeyVersionException) {
            log.warn(requestId, "Conditional update on Table segment '{}' failed due to bad key version.", new Object[]{segment});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.TableKeyBadVersion(requestId, segment, clientReplyStackTrace), failureHandler);
        } else if (this.errorCodeExists(u)) {
            log.warn(requestId, "Operation on segment '{}' failed due to a {}.", new Object[]{segment, u.getClass()});
            Callbacks.invokeSafely(this.connection::send, (Object)new WireCommands.ErrorMessage(requestId, segment, u.getMessage(), WireCommands.ErrorMessage.ErrorCode.valueOf(u.getClass())), failureHandler);
        } else {
            this.logError(requestId, segment, operation, u);
            this.connection.close();
            throw new IllegalStateException("Unknown exception.", u);
        }
        return null;
    }

    private boolean errorCodeExists(Throwable e) {
        WireCommands.ErrorMessage.ErrorCode errorCode = WireCommands.ErrorMessage.ErrorCode.valueOf(e.getClass());
        return errorCode != WireCommands.ErrorMessage.ErrorCode.UNSPECIFIED;
    }

    private void logError(long requestId, String segment, String operation, Throwable u) {
        if (u instanceof IllegalContainerStateException) {
            log.warn(requestId, "Error (Segment = '{}', Operation = '{}'): {}", new Object[]{segment, operation, u.toString()});
        } else {
            log.error(requestId, "Error (Segment = '{}', Operation = '{}')", new Object[]{segment, operation, u});
        }
    }

    private void recordStatForTransaction(MergeStreamSegmentResult mergeResult, String targetSegmentName) {
        try {
            if (mergeResult != null && mergeResult.getAttributes().containsKey(Attributes.CREATION_TIME) && mergeResult.getAttributes().containsKey(Attributes.EVENT_COUNT)) {
                long creationTime = (Long)mergeResult.getAttributes().get(Attributes.CREATION_TIME);
                int numOfEvents = ((Long)mergeResult.getAttributes().get(Attributes.EVENT_COUNT)).intValue();
                long len = mergeResult.getMergedDataLength();
                this.statsRecorder.merge(targetSegmentName, len, numOfEvents, creationTime);
            }
        }
        catch (Exception ex) {
            log.error("exception while computing stats while merging txn into {}", (Object)targetSegmentName, (Object)ex);
        }
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    protected StreamSegmentStore getSegmentStore() {
        return this.segmentStore;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    protected TrackedConnection getConnection() {
        return this.connection;
    }

    private static class DeltaIteratorResult<K, V> {
        @GuardedBy(value="this")
        private DeltaIteratorState state = new DeltaIteratorState();
        @GuardedBy(value="this")
        private final Map<K, V> items = new HashMap();
        @GuardedBy(value="this")
        private int sizeBytes;

        DeltaIteratorResult(int initialSizeBytes) {
            this.sizeBytes = initialSizeBytes;
        }

        synchronized void add(K key, V value, int sizeBytes) {
            this.items.put(key, value);
            this.sizeBytes += sizeBytes;
        }

        synchronized void remove(K item, int sizeBytes) {
            this.items.remove(item);
            this.sizeBytes -= sizeBytes;
        }

        synchronized V getItem(K key) {
            return this.items.get(key);
        }

        synchronized List<V> getItems() {
            return new ArrayList<V>(this.items.values());
        }

        synchronized int getItemCount() {
            return this.items.size();
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public DeltaIteratorState getState() {
            return this.state;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public void setState(DeltaIteratorState state) {
            this.state = state;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getSizeBytes() {
            return this.sizeBytes;
        }
    }

    @ThreadSafe
    private static class IteratorResult<T> {
        @GuardedBy(value="this")
        private final ArrayList<T> items = new ArrayList();
        @GuardedBy(value="this")
        private BufferView continuationToken = BufferView.empty();
        @GuardedBy(value="this")
        private int sizeBytes;

        IteratorResult(int initialSizeBytes) {
            this.sizeBytes = initialSizeBytes;
        }

        synchronized void add(T item, int sizeBytes) {
            this.items.add(item);
            this.sizeBytes += sizeBytes;
        }

        synchronized int getItemCount() {
            return this.items.size();
        }

        synchronized int getSizeBytes() {
            return this.sizeBytes;
        }

        synchronized List<T> getItems() {
            return new ArrayList<T>(this.items);
        }

        synchronized void setContinuationToken(BufferView continuationToken) {
            this.sizeBytes = this.sizeBytes - this.continuationToken.getLength() + continuationToken.getLength();
            this.continuationToken = continuationToken;
        }

        synchronized BufferView getContinuationToken() {
            return this.continuationToken;
        }
    }

    private static class ReadCancellationException
    extends RuntimeException {
        ReadCancellationException(Throwable wrappedException) {
            super("CancellationException during operation Read segment", wrappedException);
        }
    }
}

