/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.attributes;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.TimeoutTimer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.AsyncIterator;
import io.pravega.common.util.BufferView;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.IllegalDataFormatException;
import io.pravega.common.util.Retry;
import io.pravega.common.util.btree.BTreeIndex;
import io.pravega.common.util.btree.PageEntry;
import io.pravega.common.util.btree.Statistics;
import io.pravega.segmentstore.contracts.AttributeId;
import io.pravega.segmentstore.contracts.Attributes;
import io.pravega.segmentstore.contracts.BadOffsetException;
import io.pravega.segmentstore.contracts.SegmentProperties;
import io.pravega.segmentstore.contracts.StreamSegmentExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentNotExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentSealedException;
import io.pravega.segmentstore.contracts.StreamSegmentTruncatedException;
import io.pravega.segmentstore.server.AttributeIndex;
import io.pravega.segmentstore.server.AttributeIterator;
import io.pravega.segmentstore.server.CacheManager;
import io.pravega.segmentstore.server.DataCorruptionException;
import io.pravega.segmentstore.server.SegmentMetadata;
import io.pravega.segmentstore.server.attributes.AttributeIndexConfig;
import io.pravega.segmentstore.storage.SegmentHandle;
import io.pravega.segmentstore.storage.Storage;
import io.pravega.segmentstore.storage.cache.CacheFullException;
import io.pravega.segmentstore.storage.cache.CacheStorage;
import io.pravega.shared.NameUtils;
import java.beans.ConstructorProperties;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import lombok.Generated;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SegmentAttributeBTreeIndex
implements AttributeIndex,
CacheManager.Client,
AutoCloseable {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(SegmentAttributeBTreeIndex.class);
    private static final Retry.RetryAndThrowBase<Exception> UPDATE_RETRY = Retry.withExpBackoff((long)10L, (int)2, (int)10, (long)1000L).retryingOn(BadOffsetException.class).throwingOn(Exception.class);
    private static final Retry.RetryAndThrowBase<Exception> READ_RETRY = Retry.withExpBackoff((long)10L, (int)2, (int)10, (long)1000L).retryingOn(StreamSegmentTruncatedException.class).throwingOn(Exception.class);
    private static final int DEFAULT_KEY_LENGTH = Attributes.ATTRIBUTE_ID_LENGTH.byteCount();
    private static final int VALUE_LENGTH = 8;
    private final SegmentMetadata segmentMetadata;
    private final AtomicReference<SegmentHandle> handle;
    private final Storage storage;
    @GuardedBy(value="cacheEntries")
    private final CacheStorage cacheStorage;
    @GuardedBy(value="cacheEntries")
    private int currentCacheGeneration;
    @GuardedBy(value="cacheEntries")
    private final Map<Long, CacheEntry> cacheEntries;
    @GuardedBy(value="cacheEntries")
    private boolean cacheDisabled;
    @GuardedBy(value="pendingReads")
    private final Map<Long, PendingRead> pendingReads;
    private final BTreeIndex index;
    private final AttributeIndexConfig config;
    private final ScheduledExecutorService executor;
    private final String traceObjectId;
    private final AtomicBoolean closed;
    private final KeySerializer keySerializer;

    SegmentAttributeBTreeIndex(@NonNull SegmentMetadata segmentMetadata, @NonNull Storage storage, @NonNull CacheStorage cacheStorage, @NonNull AttributeIndexConfig config, @NonNull ScheduledExecutorService executor) {
        if (segmentMetadata == null) {
            throw new NullPointerException("segmentMetadata is marked non-null but is null");
        }
        if (storage == null) {
            throw new NullPointerException("storage is marked non-null but is null");
        }
        if (cacheStorage == null) {
            throw new NullPointerException("cacheStorage is marked non-null but is null");
        }
        if (config == null) {
            throw new NullPointerException("config is marked non-null but is null");
        }
        if (executor == null) {
            throw new NullPointerException("executor is marked non-null but is null");
        }
        this.segmentMetadata = segmentMetadata;
        this.storage = storage;
        this.cacheStorage = cacheStorage;
        this.config = config;
        this.executor = executor;
        this.handle = new AtomicReference();
        this.traceObjectId = String.format("AttributeIndex[%d-%d]", this.segmentMetadata.getContainerId(), this.segmentMetadata.getId());
        this.keySerializer = this.getKeySerializer(segmentMetadata);
        this.index = BTreeIndex.builder().keyLength(this.keySerializer.getKeyLength()).valueLength(8).maxPageSize(this.config.getMaxIndexPageSize()).executor((Executor)this.executor).getLength(this::getLength).readPage(this::readPage).writePages(this::writePages).maintainStatistics(this.shouldMaintainStatistics(segmentMetadata)).traceObjectId(this.traceObjectId).build();
        this.cacheEntries = new HashMap<Long, CacheEntry>();
        this.pendingReads = new HashMap<Long, PendingRead>();
        this.closed = new AtomicBoolean();
        this.cacheDisabled = false;
    }

    private KeySerializer getKeySerializer(SegmentMetadata segmentMetadata) {
        long keyLength = segmentMetadata.getAttributeIdLength();
        Preconditions.checkArgument((keyLength <= 256L ? 1 : 0) != 0, (String)"Invalid value %s for attribute `%s` for Segment `%s'. Expected at most %s.", (Object)Attributes.ATTRIBUTE_ID_LENGTH, (Object)segmentMetadata.getName(), (Object)256);
        return keyLength <= 0L ? new UUIDKeySerializer() : new BufferKeySerializer((int)keyLength);
    }

    private boolean shouldMaintainStatistics(SegmentMetadata segmentMetadata) {
        return segmentMetadata.getType().isFixedKeyLengthTableSegment();
    }

    CompletableFuture<Void> initialize(Duration timeout) {
        TimeoutTimer timer = new TimeoutTimer(timeout);
        Preconditions.checkState((!this.index.isInitialized() ? 1 : 0) != 0, (Object)"SegmentAttributeIndex is already initialized.");
        String attributeSegmentName = NameUtils.getAttributeSegmentName((String)this.segmentMetadata.getName());
        return ((CompletableFuture)((CompletableFuture)Futures.exceptionallyExpecting((CompletableFuture)this.storage.openWrite(attributeSegmentName).thenAccept(this.handle::set), ex -> ex instanceof StreamSegmentNotExistsException, null).thenComposeAsync(v -> this.index.initialize(timer.getRemaining()), (Executor)this.executor)).thenRun(() -> log.debug("{}: Initialized.", (Object)this.traceObjectId))).exceptionally(this::handleIndexOperationException);
    }

    static CompletableFuture<Void> delete(String segmentName, Storage storage, Duration timeout) {
        TimeoutTimer timer = new TimeoutTimer(timeout);
        String attributeSegmentName = NameUtils.getAttributeSegmentName((String)segmentName);
        return Futures.exceptionallyExpecting((CompletableFuture)storage.openWrite(attributeSegmentName).thenCompose(handle -> storage.delete(handle, timer.getRemaining())), ex -> ex instanceof StreamSegmentNotExistsException, null);
    }

    @Override
    public void close() {
        if (!this.closed.getAndSet(true)) {
            this.executor.execute(() -> {
                this.removeAllCacheEntries();
                this.cancelPendingReads();
                log.info("{}: Closed.", (Object)this.traceObjectId);
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cancelPendingReads() {
        ArrayList<PendingRead> toCancel;
        Map<Long, PendingRead> map = this.pendingReads;
        synchronized (map) {
            toCancel = new ArrayList<PendingRead>(this.pendingReads.values());
            this.pendingReads.clear();
        }
        toCancel.forEach(f -> f.completion.cancel(true));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void removeAllCacheEntries() {
        ArrayList<CacheEntry> entries;
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            entries = new ArrayList<CacheEntry>(this.cacheEntries.values());
            this.cacheEntries.clear();
        }
        this.removeFromCache(entries);
        if (entries.size() > 0) {
            log.debug("{}: Cleared all cache entries ({}).", (Object)this.traceObjectId, (Object)entries.size());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CacheManager.CacheStatus getCacheStatus() {
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            return CacheManager.CacheStatus.fromGenerations(this.cacheEntries.values().stream().filter(Objects::nonNull).map(CacheEntry::getGeneration).iterator());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean updateGenerations(int currentGeneration, int oldestGeneration, boolean essentialOnly) {
        boolean anyRemoved;
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            this.cacheDisabled = essentialOnly;
            this.currentCacheGeneration = currentGeneration;
            ArrayList<CacheEntry> toRemove = new ArrayList<CacheEntry>();
            for (CacheEntry entry : this.cacheEntries.values()) {
                if (entry.getGeneration() >= oldestGeneration) continue;
                toRemove.add(entry);
            }
            this.removeFromCache(toRemove);
            anyRemoved = !toRemove.isEmpty();
        }
        return anyRemoved;
    }

    @Override
    public CompletableFuture<Long> update(@NonNull Map<AttributeId, Long> values, @NonNull Duration timeout) {
        if (values == null) {
            throw new NullPointerException("values is marked non-null but is null");
        }
        if (timeout == null) {
            throw new NullPointerException("timeout is marked non-null but is null");
        }
        this.ensureInitialized();
        if (values.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        Collection entries = values.entrySet().stream().map(this::serialize).collect(Collectors.toList());
        return this.executeConditionally(tm -> this.index.update(entries, tm), timeout);
    }

    @Override
    public CompletableFuture<Map<AttributeId, Long>> get(@NonNull Collection<AttributeId> keys, @NonNull Duration timeout) {
        if (keys == null) {
            throw new NullPointerException("keys is marked non-null but is null");
        }
        if (timeout == null) {
            throw new NullPointerException("timeout is marked non-null but is null");
        }
        this.ensureInitialized();
        if (keys.isEmpty()) {
            return CompletableFuture.completedFuture(Collections.emptyMap());
        }
        ArrayList<AttributeId> keyList = new ArrayList<AttributeId>(keys.size());
        ArrayList<ByteArraySegment> serializedKeys = new ArrayList<ByteArraySegment>(keys.size());
        for (AttributeId key : keys) {
            keyList.add(key);
            serializedKeys.add(this.keySerializer.serialize(key));
        }
        return ((CompletableFuture)READ_RETRY.runAsync(() -> this.index.get((List)serializedKeys, timeout), this.executor).thenApply(entries -> {
            assert (entries.size() == keys.size()) : "Unexpected number of entries returned by the index search.";
            HashMap<AttributeId, Long> result = new HashMap<AttributeId, Long>();
            for (int i = 0; i < keyList.size(); ++i) {
                ByteArraySegment v = (ByteArraySegment)entries.get(i);
                if (v == null) continue;
                result.put((AttributeId)keyList.get(i), this.deserializeValue(v));
            }
            return result;
        })).exceptionally(this::handleIndexOperationException);
    }

    @Override
    public CompletableFuture<Void> seal(@NonNull Duration timeout) {
        if (timeout == null) {
            throw new NullPointerException("timeout is marked non-null but is null");
        }
        this.ensureInitialized();
        SegmentHandle handle = this.handle.get();
        if (handle == null) {
            return CompletableFuture.completedFuture(null);
        }
        return Futures.exceptionallyExpecting((CompletableFuture)this.storage.seal(handle, timeout).thenRun(() -> log.info("{}: Sealed.", (Object)this.traceObjectId)), ex -> ex instanceof StreamSegmentSealedException, null);
    }

    @Override
    public AttributeIterator iterator(AttributeId fromId, AttributeId toId, Duration fetchTimeout) {
        this.ensureInitialized();
        return new AttributeIteratorImpl(fromId, (id, inclusive) -> this.index.iterator(this.keySerializer.serialize(id), inclusive, this.keySerializer.serialize(toId), true, fetchTimeout));
    }

    @Override
    public long getCount() {
        Statistics s = this.index.getStatistics();
        return s == null ? -1L : s.getEntryCount();
    }

    @VisibleForTesting
    SegmentHandle getAttributeSegmentHandle() {
        return this.handle.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    int getPendingReadCount() {
        Map<Long, PendingRead> map = this.pendingReads;
        synchronized (map) {
            return this.pendingReads.size();
        }
    }

    public String toString() {
        return this.traceObjectId;
    }

    private <T> CompletableFuture<T> createAttributeSegmentIfNecessary(Supplier<CompletableFuture<T>> toRun, Duration timeout) {
        if (this.handle.get() == null) {
            String attributeSegmentName = NameUtils.getAttributeSegmentName((String)this.segmentMetadata.getName());
            return Futures.exceptionallyComposeExpecting((CompletableFuture)this.storage.create(attributeSegmentName, this.config.getAttributeSegmentRollingPolicy(), timeout), ex -> ex instanceof StreamSegmentExistsException, () -> {
                log.info("{}: Attribute Segment did not exist in Storage when initialize() was called, but does now.", (Object)this.traceObjectId);
                return this.storage.openWrite(attributeSegmentName);
            }).thenComposeAsync(handle -> {
                this.handle.set((SegmentHandle)handle);
                return (CompletionStage)toRun.get();
            }, (Executor)this.executor);
        }
        return toRun.get();
    }

    private CompletableFuture<Long> executeConditionally(Function<Duration, CompletableFuture<Long>> indexOperation, Duration timeout) {
        TimeoutTimer timer = new TimeoutTimer(timeout);
        return UPDATE_RETRY.runAsync(() -> this.executeConditionallyOnce(indexOperation, timer), this.executor).exceptionally(this::handleIndexOperationException);
    }

    private CompletableFuture<Long> executeConditionallyOnce(Function<Duration, CompletableFuture<Long>> indexOperation, TimeoutTimer timer) {
        return Futures.exceptionallyCompose(indexOperation.apply(timer.getRemaining()), ex -> {
            BadOffsetException boe;
            if (Exceptions.unwrap((Throwable)ex) instanceof BadOffsetException && (boe = (BadOffsetException)Exceptions.unwrap((Throwable)ex)).getExpectedOffset() != this.index.getIndexLength()) {
                log.warn("{}: Conditional Index Update failed (expected {}, given {}). Reinitializing index.", new Object[]{this.traceObjectId, boe.getExpectedOffset(), boe.getGivenOffset()});
                return this.index.initialize(timer.getRemaining()).thenCompose(v -> Futures.failedFuture((Throwable)ex));
            }
            return Futures.failedFuture((Throwable)ex);
        });
    }

    private PageEntry serialize(Map.Entry<AttributeId, Long> entry) {
        return new PageEntry(this.keySerializer.serialize(entry.getKey()), this.serializeValue(entry.getValue()));
    }

    private ByteArraySegment serializeValue(Long value) {
        if (value == null || value == Long.MIN_VALUE) {
            return null;
        }
        ByteArraySegment result = new ByteArraySegment(new byte[8]);
        result.setLong(0, value.longValue());
        return result;
    }

    private long deserializeValue(ByteArraySegment value) {
        Preconditions.checkArgument((value.getLength() == 8 ? 1 : 0) != 0, (Object)"Unexpected value length.");
        return value.getLong(0);
    }

    private CompletableFuture<BTreeIndex.IndexInfo> getLength(Duration timeout) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        SegmentHandle handle = this.handle.get();
        if (handle == null) {
            return CompletableFuture.completedFuture(BTreeIndex.IndexInfo.EMPTY);
        }
        return this.storage.getStreamSegmentInfo(handle.getSegmentName(), timeout).thenApply(segmentInfo -> {
            long rootPointer = this.getRootPointerIfNeeded((SegmentProperties)segmentInfo);
            return new BTreeIndex.IndexInfo(segmentInfo.getLength(), rootPointer);
        });
    }

    private long getRootPointerIfNeeded(SegmentProperties segmentInfo) {
        long rootPointer = BTreeIndex.IndexInfo.EMPTY.getRootPointer();
        if (this.storage.supportsAtomicWrites()) {
            return rootPointer;
        }
        rootPointer = this.segmentMetadata.getAttributes().getOrDefault(Attributes.ATTRIBUTE_SEGMENT_ROOT_POINTER, rootPointer);
        if (rootPointer != BTreeIndex.IndexInfo.EMPTY.getRootPointer() && rootPointer < segmentInfo.getStartOffset()) {
            log.info("{}: Root Pointer ({}) is below Attribute Segment's StartOffset ({}). Ignoring.", new Object[]{this.traceObjectId, rootPointer, segmentInfo.getStartOffset()});
            rootPointer = BTreeIndex.IndexInfo.EMPTY.getRootPointer();
        }
        return rootPointer;
    }

    private CompletableFuture<ByteArraySegment> readPage(long offset, int length, boolean cacheResult, Duration timeout) {
        byte[] fromCache = this.getFromCache(offset, length);
        if (fromCache != null) {
            return CompletableFuture.completedFuture(new ByteArraySegment(fromCache));
        }
        SegmentHandle handle = this.handle.get();
        if (handle == null) {
            if (offset == 0L && length == 0) {
                return CompletableFuture.completedFuture(new ByteArraySegment(new byte[0]));
            }
            return Futures.failedFuture((Throwable)new ArrayIndexOutOfBoundsException(String.format("Attribute Index Segment has not been created yet. Cannot read %d byte(s) from offset (%d).", length, offset)));
        }
        return this.readPageFromStorage(handle, offset, length, cacheResult, timeout);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    CompletableFuture<ByteArraySegment> readPageFromStorage(SegmentHandle handle, long offset, int length, boolean cacheResult, Duration timeout) {
        PendingRead pr;
        Map<Long, PendingRead> map = this.pendingReads;
        synchronized (map) {
            pr = this.pendingReads.get(offset);
            if (pr == null) {
                pr = new PendingRead(offset, length);
                pr.completion.whenComplete((r, ex) -> this.unregisterPendingRead(offset));
                this.pendingReads.put(offset, pr);
            } else if (pr.length < length) {
                log.warn("{}: Concurrent read at (Offset={}, OldLength={}), but with different length ({}). Not piggybacking.", new Object[]{this.traceObjectId, offset, pr.length, length});
                pr = new PendingRead(offset, length);
            } else {
                log.debug("{}: Concurrent read (Offset={}, Length={}, NewCount={}). Piggybacking.", new Object[]{this.traceObjectId, offset, pr.length, pr.count});
                pr.count.incrementAndGet();
                return pr.completion;
            }
        }
        return this.readPageFromStorage(handle, pr, cacheResult, timeout);
    }

    private CompletableFuture<ByteArraySegment> readPageFromStorage(SegmentHandle handle, PendingRead pr, boolean cacheResult, Duration timeout) {
        byte[] buffer = new byte[pr.length];
        Futures.completeAfter(() -> this.storage.read(handle, pr.offset, buffer, 0, pr.length, timeout).thenApplyAsync(bytesRead -> {
            Preconditions.checkArgument((pr.length == bytesRead ? 1 : 0) != 0, (Object)"Unexpected number of bytes read.");
            if (cacheResult) {
                this.storeInCache(pr.offset, buffer);
            }
            return new ByteArraySegment(buffer);
        }, (Executor)this.executor), pr.completion);
        return pr.completion;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void unregisterPendingRead(long offset) {
        PendingRead pr;
        Map<Long, PendingRead> map = this.pendingReads;
        synchronized (map) {
            pr = this.pendingReads.remove(offset);
        }
        if (pr != null && pr.count.get() > 1) {
            log.debug("{}: Concurrent reads unregistered (Offset={}, Length={}, Count={}).", new Object[]{this.traceObjectId, pr.offset, pr.length, pr.count});
        }
    }

    private CompletableFuture<Long> writePages(List<BTreeIndex.WritePage> pages, Collection<Long> obsoleteOffsets, long truncateOffset, Duration timeout) {
        long writeOffset = pages.get(0).getOffset();
        ArrayList<InputStream> streams = new ArrayList<InputStream>();
        AtomicInteger length = new AtomicInteger();
        for (BTreeIndex.WritePage p : pages) {
            Preconditions.checkArgument((p.getOffset() == writeOffset + (long)length.get() ? 1 : 0) != 0, (Object)"Unexpected page offset.");
            streams.add(p.getContents().getReader());
            length.addAndGet(p.getContents().getLength());
        }
        TimeoutTimer timer = new TimeoutTimer(timeout);
        return this.createAttributeSegmentIfNecessary(() -> this.writeToSegment(streams, writeOffset, length.get(), timer), timer.getRemaining()).thenApplyAsync(v -> {
            Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
            this.truncateAsync(truncateOffset, timer.getRemaining());
            this.storeInCache(pages, obsoleteOffsets);
            return writeOffset + (long)length.get();
        }, (Executor)this.executor);
    }

    private CompletableFuture<Void> writeToSegment(List<InputStream> streams, long writeOffset, int length, TimeoutTimer timer) {
        SequenceInputStream toWrite = new SequenceInputStream(Collections.enumeration(streams));
        return this.storage.write(this.handle.get(), writeOffset, (InputStream)toWrite, length, timer.getRemaining());
    }

    private void truncateAsync(long truncateOffset, Duration timeout) {
        if (this.storage.supportsTruncation() && truncateOffset >= 0L) {
            log.debug("{}: Truncating attribute segment at offset {}.", (Object)this.traceObjectId, (Object)truncateOffset);
            Futures.exceptionListener((CompletableFuture)this.storage.truncate(this.handle.get(), truncateOffset, timeout), ex -> log.warn("{}: Error while performing async truncation at offset {}.", new Object[]{this.traceObjectId, truncateOffset, ex}));
        } else {
            log.debug("{}: Not truncating attribute segment. SupportsTruncation = {}, TruncateOffset = {}.", new Object[]{this.traceObjectId, this.storage.supportsTruncation(), truncateOffset});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private byte[] getFromCache(long offset, int length) {
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            BufferView data;
            CacheEntry entry = this.cacheEntries.getOrDefault(offset, null);
            if (entry != null && (data = this.cacheStorage.get(entry.getCacheAddress())) != null && data.getLength() == length) {
                entry.setGeneration(this.currentCacheGeneration);
                return data.getCopy();
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void storeInCache(long offset, byte[] data) {
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
            this.storeInCache(offset, new ByteArraySegment(data));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void storeInCache(List<BTreeIndex.WritePage> toAdd, Collection<Long> obsoleteOffsets) {
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
            obsoleteOffsets.stream().map(this.cacheEntries::get).filter(Objects::nonNull).forEach(this::removeFromCache);
            for (BTreeIndex.WritePage p : toAdd) {
                if (!p.isCache()) continue;
                this.storeInCache(p.getOffset(), p.getContents());
            }
        }
    }

    @GuardedBy(value="cacheEntries")
    private void storeInCache(long entryOffset, ByteArraySegment data) {
        CacheEntry entry = this.cacheEntries.getOrDefault(entryOffset, null);
        if (entry != null && entry.getSize() == data.getLength()) {
            return;
        }
        if (this.cacheDisabled) {
            log.debug("{}: Cache update skipped for offset {}, length {}. Non-essential cache disabled.", new Object[]{this.traceObjectId, entryOffset, data.getLength()});
            return;
        }
        int entryAddress = entry == null ? -1 : entry.getCacheAddress();
        try {
            entryAddress = entry != null && entry.isStored() ? this.cacheStorage.replace(entry.getCacheAddress(), (BufferView)data) : this.cacheStorage.insert((BufferView)data);
        }
        catch (CacheFullException cfe) {
            log.warn("{}: Cache update failed for offset {}, length {} (existing entry={}). Cache full.", new Object[]{this.traceObjectId, entryOffset, data.getLength(), entry});
        }
        catch (Throwable ex) {
            log.error("{}: Cache update failed for offset {}, length {} (existing entry={}).", new Object[]{this.traceObjectId, entryOffset, data.getLength(), entry, ex});
        }
        entry = new CacheEntry(entryOffset, data.getLength(), this.currentCacheGeneration, entryAddress);
        if (entry.isStored()) {
            this.cacheEntries.put(entryOffset, entry);
        } else {
            this.cacheEntries.remove(entry.getOffset());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeFromCache(Collection<CacheEntry> entries) {
        Map<Long, CacheEntry> map = this.cacheEntries;
        synchronized (map) {
            entries.forEach(this::removeFromCache);
        }
    }

    @GuardedBy(value="cacheEntries")
    private void removeFromCache(CacheEntry e) {
        if (e.isStored()) {
            this.cacheStorage.delete(e.getCacheAddress());
        }
        this.cacheEntries.remove(e.getOffset());
    }

    private void ensureInitialized() {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkState((boolean)this.index.isInitialized(), (Object)"SegmentAttributeIndex is not initialized.");
    }

    private <T> T handleIndexOperationException(Throwable ex) {
        ex = Exceptions.unwrap((Throwable)ex);
        if (ex instanceof IllegalDataFormatException) {
            throw new DataCorruptionException("BTreeIndex operation failed. Index corrupted.", ex, new Object[0]);
        }
        throw ex;
    }

    private static class UUIDKeySerializer
    extends KeySerializer {
        private UUIDKeySerializer() {
        }

        @Override
        int getKeyLength() {
            return DEFAULT_KEY_LENGTH;
        }

        @Override
        ByteArraySegment serialize(AttributeId attributeId) {
            this.checkKeyLength(attributeId.byteCount());
            ByteArraySegment result = new ByteArraySegment(new byte[DEFAULT_KEY_LENGTH]);
            result.setUnsignedLong(0, attributeId.getBitGroup(0));
            result.setUnsignedLong(8, attributeId.getBitGroup(1));
            return result;
        }

        @Override
        AttributeId deserialize(ByteArraySegment serializedId) {
            this.checkKeyLength(serializedId.getLength());
            long msb = serializedId.getUnsignedLong(0);
            long lsb = serializedId.getUnsignedLong(8);
            return AttributeId.uuid((long)msb, (long)lsb);
        }
    }

    private static class BufferKeySerializer
    extends KeySerializer {
        private final int keyLength;

        @Override
        ByteArraySegment serialize(AttributeId attributeId) {
            this.checkKeyLength(attributeId.byteCount());
            return attributeId.toBuffer();
        }

        @Override
        AttributeId deserialize(ByteArraySegment serializedId) {
            this.checkKeyLength(serializedId.getLength());
            return AttributeId.from((byte[])serializedId.getCopy());
        }

        @ConstructorProperties(value={"keyLength"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BufferKeySerializer(int keyLength) {
            this.keyLength = keyLength;
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getKeyLength() {
            return this.keyLength;
        }
    }

    private static abstract class KeySerializer {
        private KeySerializer() {
        }

        abstract int getKeyLength();

        abstract ByteArraySegment serialize(AttributeId var1);

        abstract AttributeId deserialize(ByteArraySegment var1);

        protected void checkKeyLength(int length) {
            Preconditions.checkArgument((length == this.getKeyLength() ? 1 : 0) != 0, (String)"Expected Attribute Id length %s, given %s.", (int)this.getKeyLength(), (int)length);
        }
    }

    private static class PendingRead {
        final long offset;
        final int length;
        final AtomicInteger count = new AtomicInteger(1);
        final CompletableFuture<ByteArraySegment> completion = new CompletableFuture();

        @ConstructorProperties(value={"offset", "length"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public PendingRead(long offset, int length) {
            this.offset = offset;
            this.length = length;
        }
    }

    @FunctionalInterface
    private static interface CreatePageEntryIterator {
        public AsyncIterator<List<PageEntry>> apply(AttributeId var1, boolean var2);
    }

    private class AttributeIteratorImpl
    implements AttributeIterator {
        private final CreatePageEntryIterator getPageEntryIterator;
        private final AtomicReference<AsyncIterator<List<PageEntry>>> pageEntryIterator;
        private final AtomicReference<AttributeId> lastProcessedId;
        private final AtomicBoolean firstInvocation;

        AttributeIteratorImpl(AttributeId firstId, CreatePageEntryIterator getPageEntryIterator) {
            this.getPageEntryIterator = getPageEntryIterator;
            this.pageEntryIterator = new AtomicReference();
            this.lastProcessedId = new AtomicReference<AttributeId>(firstId);
            this.firstInvocation = new AtomicBoolean(true);
            this.reinitialize();
        }

        public CompletableFuture<List<Map.Entry<AttributeId, Long>>> getNext() {
            return ((CompletableFuture)READ_RETRY.runAsync(this::getNextPageEntries, SegmentAttributeBTreeIndex.this.executor).thenApply(pageEntries -> {
                if (pageEntries == null) {
                    return null;
                }
                List result = pageEntries.stream().map(e -> Maps.immutableEntry((Object)SegmentAttributeBTreeIndex.this.keySerializer.deserialize(e.getKey()), (Object)SegmentAttributeBTreeIndex.this.deserializeValue(e.getValue()))).collect(Collectors.toList());
                if (result.size() > 0) {
                    this.lastProcessedId.set((AttributeId)((Map.Entry)result.get(result.size() - 1)).getKey());
                    this.firstInvocation.set(false);
                }
                return result;
            })).exceptionally(x$0 -> (List)SegmentAttributeBTreeIndex.this.handleIndexOperationException((Throwable)x$0));
        }

        private CompletableFuture<List<PageEntry>> getNextPageEntries() {
            Exceptions.checkNotClosed((boolean)SegmentAttributeBTreeIndex.this.closed.get(), (Object)SegmentAttributeBTreeIndex.this);
            return this.pageEntryIterator.get().getNext().exceptionally(ex -> {
                this.reinitialize();
                throw new CompletionException((Throwable)ex);
            });
        }

        private void reinitialize() {
            Exceptions.checkNotClosed((boolean)SegmentAttributeBTreeIndex.this.closed.get(), (Object)SegmentAttributeBTreeIndex.this);
            this.pageEntryIterator.set(this.getPageEntryIterator.apply(this.lastProcessedId.get(), this.firstInvocation.get()));
        }
    }

    private static class CacheEntry {
        static final int NO_ADDRESS = -1;
        private final long offset;
        private final int size;
        @GuardedBy(value="this")
        private int generation;
        private final int cacheAddress;

        synchronized int getGeneration() {
            return this.generation;
        }

        synchronized boolean isStored() {
            return this.cacheAddress >= 0;
        }

        synchronized void setGeneration(int value) {
            this.generation = value;
        }

        public String toString() {
            return String.format("Offset = %s, Length = %s", this.offset, this.size);
        }

        @ConstructorProperties(value={"offset", "size", "generation", "cacheAddress"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public CacheEntry(long offset, int size, int generation, int cacheAddress) {
            this.offset = offset;
            this.size = size;
            this.generation = generation;
            this.cacheAddress = cacheAddress;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public long getOffset() {
            return this.offset;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getSize() {
            return this.size;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getCacheAddress() {
            return this.cacheAddress;
        }
    }
}

