/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.reading;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterators;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.ObjectClosedException;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.AvlTreeIndex;
import io.pravega.common.util.BufferView;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.SortedIndex;
import io.pravega.segmentstore.contracts.ReadResult;
import io.pravega.segmentstore.contracts.ReadResultEntryType;
import io.pravega.segmentstore.contracts.StreamSegmentSealedException;
import io.pravega.segmentstore.server.CacheManager;
import io.pravega.segmentstore.server.SegmentMetadata;
import io.pravega.segmentstore.server.reading.CacheIndexEntry;
import io.pravega.segmentstore.server.reading.CacheReadResultEntry;
import io.pravega.segmentstore.server.reading.CompletableReadResultEntry;
import io.pravega.segmentstore.server.reading.EndOfStreamSegmentReadResultEntry;
import io.pravega.segmentstore.server.reading.FutureReadResultEntry;
import io.pravega.segmentstore.server.reading.FutureReadResultEntryCollection;
import io.pravega.segmentstore.server.reading.MergedIndexEntry;
import io.pravega.segmentstore.server.reading.PendingMerge;
import io.pravega.segmentstore.server.reading.ReadIndexConfig;
import io.pravega.segmentstore.server.reading.ReadIndexEntry;
import io.pravega.segmentstore.server.reading.ReadIndexSummary;
import io.pravega.segmentstore.server.reading.ReadResultEntryBase;
import io.pravega.segmentstore.server.reading.RedirectIndexEntry;
import io.pravega.segmentstore.server.reading.RedirectedReadResultEntry;
import io.pravega.segmentstore.server.reading.StorageReadManager;
import io.pravega.segmentstore.server.reading.StorageReadResultEntry;
import io.pravega.segmentstore.server.reading.StreamSegmentReadResult;
import io.pravega.segmentstore.server.reading.TruncatedReadResultEntry;
import io.pravega.segmentstore.storage.ReadOnlyStorage;
import io.pravega.segmentstore.storage.cache.CacheFullException;
import io.pravega.segmentstore.storage.cache.CacheStorage;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Predicate;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
class StreamSegmentReadIndex
implements CacheManager.Client,
AutoCloseable {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(StreamSegmentReadIndex.class);
    private final String traceObjectId;
    @GuardedBy(value="lock")
    private final SortedIndex<ReadIndexEntry> indexEntries;
    private final ReadIndexConfig config;
    private final CacheStorage cacheStorage;
    private final FutureReadResultEntryCollection futureReads;
    @GuardedBy(value="lock")
    private final HashMap<Long, PendingMerge> pendingMergers;
    private final StorageReadManager storageReadManager;
    @VisibleForTesting
    private final ReadIndexSummary summary;
    private final ScheduledExecutorService executor;
    private SegmentMetadata metadata;
    private final AtomicLong lastAppendedOffset;
    private volatile boolean storageCacheDisabled;
    private boolean recoveryMode;
    private boolean closed;
    private boolean merged;
    private final Object lock = new Object();
    private final int storageReadAlignment;

    StreamSegmentReadIndex(ReadIndexConfig config, SegmentMetadata metadata, CacheStorage cacheStorage, ReadOnlyStorage storage, ScheduledExecutorService executor, boolean recoveryMode) {
        Preconditions.checkNotNull((Object)config, (Object)"config");
        Preconditions.checkNotNull((Object)metadata, (Object)"metadata");
        Preconditions.checkNotNull((Object)cacheStorage, (Object)"cacheStorage");
        Preconditions.checkNotNull((Object)storage, (Object)"storage");
        Preconditions.checkNotNull((Object)executor, (Object)"executor");
        this.traceObjectId = String.format("ReadIndex[%d-%d]", metadata.getContainerId(), metadata.getId());
        this.config = config;
        this.metadata = metadata;
        this.cacheStorage = cacheStorage;
        this.recoveryMode = recoveryMode;
        this.indexEntries = new AvlTreeIndex();
        this.futureReads = new FutureReadResultEntryCollection();
        this.pendingMergers = new HashMap();
        this.lastAppendedOffset = new AtomicLong(-1L);
        this.storageReadManager = new StorageReadManager(metadata, storage, executor);
        this.executor = executor;
        this.summary = new ReadIndexSummary();
        this.storageReadAlignment = this.alignToCacheBlockSize(this.config.getStorageReadAlignment());
        this.storageCacheDisabled = false;
    }

    private int alignToCacheBlockSize(int value) {
        int r = value % this.cacheStorage.getBlockAlignment();
        if (r != 0) {
            value += this.cacheStorage.getBlockAlignment() - r;
        }
        return value;
    }

    @Override
    public void close() {
        this.close(true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void close(boolean cleanCache) {
        if (!this.closed) {
            this.closed = true;
            this.storageReadManager.close();
            ArrayList<Iterator<FutureReadResultEntry>> futureReads = new ArrayList<Iterator<FutureReadResultEntry>>();
            futureReads.add(this.futureReads.close().iterator());
            Object object = this.lock;
            synchronized (object) {
                this.pendingMergers.values().forEach(pm -> futureReads.add(pm.seal().iterator()));
            }
            this.cancelFutureReads(Iterators.concat(futureReads.iterator()));
            if (cleanCache) {
                this.executor.execute(() -> {
                    this.removeAllEntries();
                    log.info("{}: Closed.", (Object)this.traceObjectId);
                });
            } else {
                log.info("{}: Closed (no cache cleanup).", (Object)this.traceObjectId);
            }
        }
    }

    private void cancelFutureReads(Iterator<FutureReadResultEntry> toCancel) {
        CancellationException ce = new CancellationException();
        while (toCancel.hasNext()) {
            FutureReadResultEntry e = toCancel.next();
            if (e.getContent().isDone()) continue;
            e.fail(ce);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeAllEntries() {
        int count;
        Preconditions.checkState((boolean)this.closed, (Object)"Cannot call removeAllEntries unless the ReadIndex is closed.");
        Object object = this.lock;
        synchronized (object) {
            this.indexEntries.forEach(this::deleteData);
            count = this.indexEntries.size();
            this.indexEntries.clear();
        }
        if (count > 0) {
            log.debug("{}: Cleared all cache entries ({}).", (Object)this.traceObjectId, (Object)count);
        }
    }

    @Override
    public CacheManager.CacheStatus getCacheStatus() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        return this.summary.toCacheStatus();
    }

    @Override
    public boolean updateGenerations(int currentGeneration, int oldestGeneration, boolean essentialOnly) {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        this.storageCacheDisabled = essentialOnly;
        this.summary.setCurrentGeneration(currentGeneration);
        return this.evictCacheEntries(entry -> this.isEvictable((ReadIndexEntry)entry, oldestGeneration)) > 0L;
    }

    private boolean isEvictable(ReadIndexEntry entry, int oldestGeneration) {
        long lastOffset = entry.getLastStreamSegmentOffset();
        return entry.isDataEntry() && lastOffset < this.metadata.getStorageLength() && (entry.getGeneration() < oldestGeneration || lastOffset < this.metadata.getStartOffset());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long evictCacheEntries(Predicate<ReadIndexEntry> isEvictable) {
        ArrayList<ReadIndexEntry> toRemove = new ArrayList<ReadIndexEntry>();
        Object object = this.lock;
        synchronized (object) {
            this.indexEntries.forEach(entry -> {
                if (isEvictable.test((ReadIndexEntry)entry)) {
                    toRemove.add((ReadIndexEntry)entry);
                }
            });
            toRemove.forEach(e -> this.indexEntries.remove(e.key()));
        }
        AtomicLong totalSize = new AtomicLong();
        toRemove.forEach(e -> {
            this.deleteData((ReadIndexEntry)e);
            this.summary.removeOne(e.getGeneration());
            totalSize.addAndGet(e.getLength());
        });
        if (!toRemove.isEmpty()) {
            log.debug("{}: Evicted {} entries totalling {} bytes.", new Object[]{this.traceObjectId, toRemove.size(), totalSize});
        }
        return totalSize.get();
    }

    public String toString() {
        return String.format("%s (%s)", this.traceObjectId, this.metadata.getName());
    }

    boolean isMerged() {
        return this.merged;
    }

    void markMerged() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((!this.merged ? 1 : 0) != 0, (String)"StreamSegmentReadIndex %d is already merged.", (long)this.metadata.getId());
        log.debug("{}: Merged.", (Object)this.traceObjectId);
        this.merged = true;
    }

    boolean isActive() {
        return this.metadata.isActive() && !this.metadata.isDeleted();
    }

    long getSegmentLength() {
        return this.metadata.getLength();
    }

    @VisibleForTesting
    int getFutureReadCount() {
        return this.futureReads.size();
    }

    public void exitRecoveryMode(SegmentMetadata newMetadata) {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((boolean)this.recoveryMode, (String)"ReadIndex[%s] is not in recovery mode.", (Object)this.traceObjectId);
        Preconditions.checkNotNull((Object)newMetadata, (Object)"newMetadata");
        Exceptions.checkArgument((newMetadata.getId() == this.metadata.getId() ? 1 : 0) != 0, (String)"newMetadata", (String)"New Metadata StreamSegmentId is different from existing one.", (Object[])new Object[0]);
        Exceptions.checkArgument((newMetadata.getLength() == this.metadata.getLength() ? 1 : 0) != 0, (String)"newMetadata", (String)"New Metadata Length is different from existing one.", (Object[])new Object[0]);
        Exceptions.checkArgument((newMetadata.getStorageLength() == this.metadata.getStorageLength() ? 1 : 0) != 0, (String)"newMetadata", (String)"New Metadata StorageLength is different from existing one.", (Object[])new Object[0]);
        Exceptions.checkArgument((newMetadata.isSealed() == this.metadata.isSealed() ? 1 : 0) != 0, (String)"newMetadata", (String)"New Metadata Sealed Flag is different from existing one.", (Object[])new Object[0]);
        Exceptions.checkArgument((newMetadata.isMerged() == this.metadata.isMerged() ? 1 : 0) != 0, (String)"newMetadata", (String)"New Metadata Merged Flag is different from existing one.", (Object[])new Object[0]);
        Exceptions.checkArgument((newMetadata.isDeleted() == this.metadata.isDeleted() ? 1 : 0) != 0, (String)"newMetadata", (String)"New Metadata Deletion Flag is different from existing one.", (Object[])new Object[0]);
        this.metadata = newMetadata;
        this.recoveryMode = false;
        log.debug("{}: Exit RecoveryMode.", (Object)this.traceObjectId);
    }

    long trimCache() {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((boolean)this.recoveryMode, (String)"ReadIndex[%s] is not in recovery mode.", (Object)this.traceObjectId);
        return this.evictCacheEntries(entry -> this.isEvictable((ReadIndexEntry)entry, Integer.MAX_VALUE));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void append(long offset, BufferView data) {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((!this.isMerged() ? 1 : 0) != 0, (Object)"StreamSegment has been merged into a different one. Cannot append more ReadIndex entries.");
        if (data.getLength() == 0) {
            return;
        }
        long length = this.metadata.getLength();
        long endOffset = offset + (long)data.getLength();
        Exceptions.checkArgument((endOffset <= length ? 1 : 0) != 0, (String)"offset", (String)"The given range of bytes (%d-%d) is beyond the StreamSegment Length (%d).", (Object[])new Object[]{offset, endOffset, length});
        log.debug("{}: Append (Offset = {}, Length = {}).", new Object[]{this.traceObjectId, offset, data.getLength()});
        Preconditions.checkArgument((this.lastAppendedOffset.get() < 0L || offset == this.lastAppendedOffset.get() + 1L ? 1 : 0) != 0, (String)"The given range of bytes (Offset=%s) does not start right after the last appended offset (%s).", (long)offset, (Object)this.lastAppendedOffset);
        int appendLength = 0;
        Object object = this.lock;
        synchronized (object) {
            ReadIndexEntry lastEntry = (ReadIndexEntry)this.indexEntries.getLast();
            if (lastEntry != null && lastEntry.isDataEntry() && lastEntry.getLastStreamSegmentOffset() == this.lastAppendedOffset.get()) {
                appendLength = this.appendToEntry(data, (CacheIndexEntry)lastEntry);
            }
        }
        assert (appendLength <= data.getLength());
        if (appendLength < data.getLength()) {
            data = data.slice(appendLength, data.getLength() - appendLength);
            CacheIndexEntry lastEntry = this.addToCacheAndIndex(data, offset += (long)appendLength, this::appendSingleEntryToCacheAndIndex);
            this.lastAppendedOffset.set(lastEntry.getLastStreamSegmentOffset());
        } else {
            this.lastAppendedOffset.addAndGet(appendLength);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void beginMerge(long offset, StreamSegmentReadIndex sourceStreamSegmentIndex) {
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"beginMerge", (Object[])new Object[]{offset, sourceStreamSegmentIndex.traceObjectId});
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Exceptions.checkArgument((!sourceStreamSegmentIndex.isMerged() ? 1 : 0) != 0, (String)"sourceStreamSegmentIndex", (String)"Given StreamSegmentReadIndex is already merged.", (Object[])new Object[0]);
        SegmentMetadata sourceMetadata = sourceStreamSegmentIndex.metadata;
        Exceptions.checkArgument((boolean)sourceMetadata.isSealed(), (String)"sourceStreamSegmentIndex", (String)"Given StreamSegmentReadIndex refers to a StreamSegment that is not sealed.", (Object[])new Object[0]);
        long sourceLength = sourceStreamSegmentIndex.getSegmentLength();
        RedirectIndexEntry newEntry = new RedirectIndexEntry(offset, sourceStreamSegmentIndex);
        if (sourceLength == 0L) {
            return;
        }
        long endOffset = offset + sourceLength;
        long ourLength = this.getSegmentLength();
        Exceptions.checkArgument((endOffset <= ourLength ? 1 : 0) != 0, (String)"offset", (String)"The given range of bytes(%d-%d) is beyond the StreamSegment Length (%d).", (Object[])new Object[]{offset, endOffset, ourLength});
        log.debug("{}: BeginMerge (Offset = {}, Length = {}).", new Object[]{this.traceObjectId, offset, newEntry.getLength()});
        Object object = this.lock;
        synchronized (object) {
            Exceptions.checkArgument((!this.pendingMergers.containsKey(sourceMetadata.getId()) ? 1 : 0) != 0, (String)"sourceStreamSegmentIndex", (String)"Given StreamSegmentReadIndex is already merged or in the process of being merged into this one.", (Object[])new Object[0]);
            this.pendingMergers.put(sourceMetadata.getId(), new PendingMerge(newEntry.key()));
            try {
                ReadIndexEntry oldEntry = this.addToIndex(newEntry);
                assert (oldEntry == null) : String.format("Added a new entry in the ReadIndex that overrode an existing element. New = %s, Old = %s.", newEntry, oldEntry);
            }
            catch (Exception ex) {
                this.pendingMergers.remove(sourceMetadata.getId());
                throw ex;
            }
        }
        this.lastAppendedOffset.set(newEntry.getLastStreamSegmentOffset());
        LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"beginMerge", (long)traceId, (Object[])new Object[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void completeMerge(SegmentMetadata sourceMetadata) {
        RedirectIndexEntry redirectEntry;
        PendingMerge pendingMerge;
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"completeMerge", (Object[])new Object[]{sourceMetadata.getId()});
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Exceptions.checkArgument((boolean)sourceMetadata.isDeleted(), (String)"sourceSegmentStreamId", (String)"Given StreamSegmentReadIndex refers to a StreamSegment that has not been deleted yet.", (Object[])new Object[0]);
        if (sourceMetadata.getLength() == 0L) {
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            pendingMerge = this.pendingMergers.getOrDefault(sourceMetadata.getId(), null);
            Exceptions.checkArgument((pendingMerge != null ? 1 : 0) != 0, (String)"sourceSegmentStreamId", (String)"Given StreamSegmentReadIndex's merger with this one has not been initiated using beginMerge. Cannot finalize the merger.", (Object[])new Object[0]);
            ReadIndexEntry indexEntry = (ReadIndexEntry)this.indexEntries.get(pendingMerge.getMergeOffset());
            assert (indexEntry != null && !indexEntry.isDataEntry()) : String.format("pendingMergers points to a ReadIndexEntry that does not exist or is of the wrong type. sourceStreamSegmentId = %d, offset = %d, treeEntry = %s.", sourceMetadata.getId(), pendingMerge.getMergeOffset(), indexEntry);
            redirectEntry = (RedirectIndexEntry)indexEntry;
        }
        StreamSegmentReadIndex sourceIndex = redirectEntry.getRedirectReadIndex();
        List<MergedIndexEntry> sourceEntries = sourceIndex.removeAllDataEntries(redirectEntry.getStreamSegmentOffset());
        Object object2 = this.lock;
        synchronized (object2) {
            this.indexEntries.remove(pendingMerge.getMergeOffset());
            this.pendingMergers.remove(sourceMetadata.getId());
            sourceEntries.forEach(this::addToIndex);
        }
        List<FutureReadResultEntry> pendingReads = pendingMerge.seal();
        if (pendingReads.size() > 0) {
            log.debug("{}: triggerFutureReads for Pending Merge (Count = {}, MergeOffset = {}, MergeLength = {}).", new Object[]{this.traceObjectId, pendingReads.size(), pendingMerge.getMergeOffset(), sourceIndex.getSegmentLength()});
            this.triggerFutureReads(pendingReads);
        }
        LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"completeMerge", (long)traceId, (Object[])new Object[0]);
    }

    private void insert(long offset, ByteArraySegment data) {
        if (this.storageCacheDisabled) {
            log.debug("{}: Not inserting (Offset = {}, Length = {}) due to Storage Cache disabled.", new Object[]{this.traceObjectId, offset, data.getLength()});
            return;
        }
        log.debug("{}: Insert (Offset = {}, Length = {}).", new Object[]{this.traceObjectId, offset, data.getLength()});
        Exceptions.checkArgument((offset + (long)data.getLength() <= this.metadata.getStorageLength() ? 1 : 0) != 0, (String)"entry", (String)"The given range of bytes (Offset=%s, Length=%s) does not correspond to the StreamSegment range that is in Storage (%s).", (Object[])new Object[]{offset, data.getLength(), this.metadata.getStorageLength()});
        try {
            this.addToCacheAndIndex((BufferView)data, offset, this::insertEntriesToCacheAndIndex);
        }
        catch (CacheFullException ex) {
            log.warn("{}: Unable to insert Storage Read data (Offset={}, Length={}) into the Cache. {}", new Object[]{this.traceObjectId, offset, data.getLength(), ex.getMessage()});
        }
    }

    @GuardedBy(value="lock")
    private int appendToEntry(BufferView data, CacheIndexEntry entry) {
        int appendLength = this.cacheStorage.getAppendableLength((int)entry.getLength());
        if (appendLength == 0) {
            return appendLength;
        }
        if (data.getLength() > appendLength) {
            data = data.slice(0, appendLength);
        }
        appendLength = this.cacheStorage.append(entry.getCacheAddress(), (int)entry.getLength(), data);
        entry.increaseLength(appendLength);
        entry.setGeneration(this.summary.touchOne(entry.getGeneration()));
        return appendLength;
    }

    private CacheIndexEntry addToCacheAndIndex(BufferView data, long offset, BiFunction<BufferView, Long, CacheIndexEntry> add) {
        int partLength;
        if (data.getLength() <= this.cacheStorage.getMaxEntryLength()) {
            return add.apply(data, offset);
        }
        CacheIndexEntry lastEntry = null;
        for (int bufferOffset = 0; bufferOffset < data.getLength(); bufferOffset += partLength) {
            partLength = Math.min(data.getLength() - bufferOffset, this.cacheStorage.getMaxEntryLength());
            lastEntry = add.apply(data.slice(bufferOffset, partLength), offset + (long)bufferOffset);
        }
        return lastEntry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CacheIndexEntry appendSingleEntryToCacheAndIndex(BufferView data, long segmentOffset) {
        CacheIndexEntry newEntry;
        int dataAddress = this.cacheStorage.insert(data);
        try {
            newEntry = new CacheIndexEntry(segmentOffset, data.getLength(), dataAddress);
            Object object = this.lock;
            synchronized (object) {
                ReadIndexEntry previous = (ReadIndexEntry)this.indexEntries.put((SortedIndex.IndexEntry)newEntry);
                assert (previous == null);
                newEntry.setGeneration(this.summary.addOne());
            }
        }
        catch (Throwable ex) {
            if (!Exceptions.mustRethrow((Throwable)ex)) {
                this.cacheStorage.delete(dataAddress);
            }
            throw ex;
        }
        return newEntry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CacheIndexEntry insertEntriesToCacheAndIndex(BufferView data, long segmentOffset) {
        CacheIndexEntry lastInsertedEntry = null;
        Object object = this.lock;
        synchronized (object) {
            Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
            while (data != null && data.getLength() > 0) {
                long overlapLength;
                ReadIndexEntry existingEntry = (ReadIndexEntry)this.indexEntries.getFloor(segmentOffset);
                if (existingEntry != null && existingEntry.getLastStreamSegmentOffset() >= segmentOffset) {
                    overlapLength = existingEntry.getStreamSegmentOffset() + existingEntry.getLength() - segmentOffset;
                } else {
                    existingEntry = (ReadIndexEntry)this.indexEntries.getCeiling(segmentOffset);
                    long l = overlapLength = existingEntry == null ? (long)data.getLength() : existingEntry.getStreamSegmentOffset() - segmentOffset;
                    assert (overlapLength > 0L) : "indexEntries.getFloor(offset) == null != indexEntries.getCeiling(offset)";
                    BufferView dataToInsert = overlapLength >= (long)data.getLength() ? data : data.slice(0, (int)overlapLength);
                    int dataAddress = 0;
                    try {
                        dataAddress = this.cacheStorage.insert(dataToInsert);
                        CacheIndexEntry newEntry = new CacheIndexEntry(segmentOffset, dataToInsert.getLength(), dataAddress);
                        ReadIndexEntry overriddenEntry = this.addToIndex(newEntry);
                        assert (overriddenEntry == null) : "Insert overrode existing entry; " + segmentOffset + ":" + dataToInsert.getLength();
                        lastInsertedEntry = newEntry;
                    }
                    catch (Throwable ex) {
                        this.cacheStorage.delete(dataAddress);
                        throw ex;
                    }
                }
                assert (overlapLength != 0L) : "unable to make any progress";
                data = overlapLength >= (long)data.getLength() ? null : data.slice((int)overlapLength, data.getLength() - (int)overlapLength);
                segmentOffset += overlapLength;
            }
        }
        return lastInsertedEntry;
    }

    @GuardedBy(value="lock")
    private ReadIndexEntry addToIndex(ReadIndexEntry entry) {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        ReadIndexEntry rejectedEntry = (ReadIndexEntry)this.indexEntries.put((SortedIndex.IndexEntry)entry);
        if (entry.isDataEntry()) {
            if (entry instanceof MergedIndexEntry) {
                this.summary.addOne(entry.getGeneration());
            } else {
                entry.setGeneration(this.summary.addOne());
            }
        }
        if (rejectedEntry != null && rejectedEntry.isDataEntry()) {
            this.summary.removeOne(rejectedEntry.getGeneration());
        }
        return rejectedEntry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void triggerFutureReads() {
        Collection<FutureReadResultEntry> futureReads;
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((!this.recoveryMode ? 1 : 0) != 0, (Object)"StreamSegmentReadIndex is in Recovery Mode.");
        boolean sealed = this.metadata.isSealed();
        if (sealed) {
            futureReads = this.futureReads.pollAll();
            log.debug("{}: triggerFutureReads (Count = {}, Offset = {}, Sealed = True).", new Object[]{this.traceObjectId, futureReads.size(), this.metadata.getLength()});
        } else {
            ReadIndexEntry lastEntry;
            Object object = this.lock;
            synchronized (object) {
                lastEntry = (ReadIndexEntry)this.indexEntries.getLast();
            }
            if (lastEntry == null) {
                return;
            }
            futureReads = this.futureReads.poll(lastEntry.getLastStreamSegmentOffset());
            log.debug("{}: triggerFutureReads (Count = {}, Offset = {}, Sealed = False).", new Object[]{this.traceObjectId, futureReads.size(), lastEntry.getLastStreamSegmentOffset()});
        }
        this.triggerFutureReads(futureReads);
    }

    private void triggerFutureReads(Collection<FutureReadResultEntry> futureReads) {
        for (FutureReadResultEntry r : futureReads) {
            CompletableReadResultEntry entry = this.getSingleReadResultEntry(r.getStreamSegmentOffset(), r.getRequestedReadLength(), false);
            assert (entry != null) : "Serving a FutureReadResultEntry with a null result";
            if (entry instanceof FutureReadResultEntry) {
                assert (entry.getStreamSegmentOffset() == r.getStreamSegmentOffset());
                log.warn("{}: triggerFutureReads (Offset = {}). Serving a FutureReadResultEntry ({}) with another FutureReadResultEntry ({}). Segment Info = [{}].", new Object[]{this.traceObjectId, r.getStreamSegmentOffset(), r, entry, this.metadata.getSnapshot()});
            }
            log.debug("{}: triggerFutureReads (Offset = {}, Type = {}).", new Object[]{this.traceObjectId, r.getStreamSegmentOffset(), entry.getType()});
            if (entry.getType() == ReadResultEntryType.EndOfStreamSegment) {
                r.fail((Throwable)new StreamSegmentSealedException(String.format("StreamSegment has been sealed at offset %d. There can be no more reads beyond this offset.", this.metadata.getLength())));
                continue;
            }
            if (!entry.getContent().isDone()) {
                entry.requestContent(this.config.getStorageReadDefaultTimeout());
            }
            CompletableFuture entryContent = entry.getContent();
            entryContent.thenAccept(r::complete);
            Futures.exceptionListener((CompletableFuture)entryContent, r::fail);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    BufferView readDirect(long startOffset, int length) {
        int entryReadLength;
        CacheReadResultEntry nextEntry;
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((!this.recoveryMode ? 1 : 0) != 0, (Object)"StreamSegmentReadIndex is in Recovery Mode.");
        Preconditions.checkArgument((length >= 0 ? 1 : 0) != 0, (Object)"length must be a non-negative number");
        Preconditions.checkArgument((startOffset >= this.metadata.getStorageLength() ? 1 : 0) != 0, (String)"[%s]: startOffset (%s) must refer to an offset beyond the Segment's StorageLength offset(%s).", (Object)this.traceObjectId, (Object)startOffset, (Object)this.metadata.getStorageLength());
        Preconditions.checkArgument((startOffset + (long)length <= this.metadata.getLength() ? 1 : 0) != 0, (Object)"startOffset+length must be less than the length of the Segment.");
        Preconditions.checkArgument((startOffset >= Math.min(this.metadata.getStartOffset(), this.metadata.getStorageLength()) ? 1 : 0) != 0, (Object)"startOffset is before the Segment's StartOffset.");
        Object object = this.lock;
        synchronized (object) {
            ReadIndexEntry indexEntry = (ReadIndexEntry)this.indexEntries.getFloor(startOffset);
            if (indexEntry == null || startOffset > indexEntry.getLastStreamSegmentOffset() || !indexEntry.isDataEntry()) {
                return null;
            }
            nextEntry = this.createMemoryRead(indexEntry, startOffset, length, false, false);
        }
        assert (Futures.isSuccessful((CompletableFuture)nextEntry.getContent())) : "Found CacheReadResultEntry that is not completed yet: " + nextEntry;
        BufferView entryContents = (BufferView)nextEntry.getContent().join();
        ArrayList<BufferView> contents = new ArrayList<BufferView>();
        contents.add(entryContents);
        for (int readLength = entryContents.getLength(); readLength < length; readLength += entryReadLength) {
            BufferView entryData;
            Object object2 = this.lock;
            synchronized (object2) {
                ReadIndexEntry indexEntry = (ReadIndexEntry)this.indexEntries.get(startOffset + (long)readLength);
                if (!indexEntry.isDataEntry()) {
                    return null;
                }
                entryData = this.cacheStorage.get(indexEntry.getCacheAddress());
            }
            if (entryData == null) {
                return null;
            }
            entryReadLength = Math.min(entryData.getLength(), length - readLength);
            assert (entryReadLength > 0) : "about to have fetched zero bytes from a cache entry";
            contents.add(entryData.slice(0, entryReadLength));
        }
        return BufferView.wrap(contents);
    }

    ReadResult read(long startOffset, int maxLength, Duration timeout) {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        Preconditions.checkState((!this.recoveryMode ? 1 : 0) != 0, (Object)"StreamSegmentReadIndex is in Recovery Mode.");
        Exceptions.checkArgument((startOffset >= 0L ? 1 : 0) != 0, (String)"startOffset", (String)"startOffset must be a non-negative number.", (Object[])new Object[0]);
        Exceptions.checkArgument((maxLength >= 0 ? 1 : 0) != 0, (String)"maxLength", (String)"maxLength must be a non-negative number.", (Object[])new Object[0]);
        Exceptions.checkArgument((this.checkReadAvailability(startOffset, true) != ReadAvailability.BeyondLastOffset ? 1 : 0) != 0, (String)"startOffset", (String)"StreamSegment is sealed and startOffset is beyond the last offset of the StreamSegment.", (Object[])new Object[0]);
        log.debug("{}: Read (Offset = {}, MaxLength = {}).", new Object[]{this.traceObjectId, startOffset, maxLength});
        return new StreamSegmentReadResult(startOffset, maxLength, this::getMultiReadResultEntry, this.traceObjectId);
    }

    private ReadAvailability checkReadAvailability(long offset, boolean lastOffsetInclusive) {
        if (offset < this.metadata.getStartOffset()) {
            return ReadAvailability.BeforeStartOffset;
        }
        if (this.metadata.isSealed()) {
            return offset < this.metadata.getLength() + (long)(lastOffsetInclusive ? 1 : 0) ? ReadAvailability.Available : ReadAvailability.BeyondLastOffset;
        }
        return ReadAvailability.Available;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    CompletableReadResultEntry getSingleReadResultEntry(long resultStartOffset, int maxLength, boolean makeCopy) {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        if (maxLength < 0) {
            return null;
        }
        CompletableReadResultEntry result = null;
        ReadAvailability ra = this.checkReadAvailability(resultStartOffset, false);
        if (ra == ReadAvailability.BeyondLastOffset) {
            result = new EndOfStreamSegmentReadResultEntry(resultStartOffset, maxLength);
        } else if (ra == ReadAvailability.BeforeStartOffset) {
            result = new TruncatedReadResultEntry(resultStartOffset, maxLength, this.metadata.getStartOffset(), this.metadata.getName());
        } else {
            ReadIndexEntry indexEntry;
            boolean redirect = false;
            Object object = this.lock;
            synchronized (object) {
                indexEntry = (ReadIndexEntry)this.indexEntries.getFloor(resultStartOffset);
                if (indexEntry == null) {
                    result = this.createDataNotAvailableRead(resultStartOffset, maxLength);
                } else if (resultStartOffset > indexEntry.getLastStreamSegmentOffset()) {
                    result = this.createDataNotAvailableRead(resultStartOffset, maxLength);
                } else if (indexEntry.isDataEntry()) {
                    result = this.createMemoryRead(indexEntry, resultStartOffset, maxLength, true, makeCopy);
                } else if (indexEntry instanceof RedirectIndexEntry) {
                    assert (!((RedirectIndexEntry)indexEntry).getRedirectReadIndex().closed);
                    redirect = true;
                }
            }
            if (redirect) {
                result = this.createRedirectedRead(resultStartOffset, maxLength, (RedirectIndexEntry)indexEntry, makeCopy);
            }
        }
        assert (result != null) : String.format("Reached the end of getSingleReadResultEntry(id=%d, offset=%d, length=%d) with no plausible result in sight. This means we missed a case.", this.metadata.getId(), resultStartOffset, maxLength);
        return result;
    }

    private CompletableReadResultEntry getMultiReadResultEntry(long resultStartOffset, int maxLength, boolean makeCopy) {
        BufferView entryContents;
        int readLength = 0;
        CompletableReadResultEntry nextEntry = this.getSingleReadResultEntry(resultStartOffset, maxLength, makeCopy);
        if (nextEntry == null || !(nextEntry instanceof CacheReadResultEntry)) {
            return nextEntry;
        }
        ArrayList<BufferView> contents = new ArrayList<BufferView>();
        do {
            assert (Futures.isSuccessful((CompletableFuture)nextEntry.getContent())) : "Found CacheReadResultEntry that is not completed yet: " + nextEntry;
            entryContents = (BufferView)nextEntry.getContent().join();
            contents.add(entryContents);
        } while ((readLength += entryContents.getLength()) < this.config.getMemoryReadMinLength() && readLength < maxLength && (nextEntry = this.getSingleMemoryReadResultEntry(resultStartOffset + (long)readLength, maxLength - readLength, makeCopy)) != null);
        return new CacheReadResultEntry(resultStartOffset, BufferView.wrap(contents));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CacheReadResultEntry getSingleMemoryReadResultEntry(long resultStartOffset, int maxLength, boolean makeCopy) {
        Exceptions.checkNotClosed((boolean)this.closed, (Object)this);
        if (maxLength > 0 && this.checkReadAvailability(resultStartOffset, false) == ReadAvailability.Available) {
            Object object = this.lock;
            synchronized (object) {
                ReadIndexEntry indexEntry = (ReadIndexEntry)this.indexEntries.get(resultStartOffset);
                if (indexEntry != null && indexEntry.isDataEntry()) {
                    return this.createMemoryRead(indexEntry, resultStartOffset, maxLength, true, makeCopy);
                }
            }
        }
        return null;
    }

    private CompletableReadResultEntry createRedirectedRead(long streamSegmentOffset, int maxLength, RedirectIndexEntry entry, boolean makeCopy) {
        StreamSegmentReadIndex redirectedIndex = entry.getRedirectReadIndex();
        long redirectOffset = streamSegmentOffset - entry.getStreamSegmentOffset();
        long entryLength = entry.getLength();
        assert (redirectOffset >= 0L && redirectOffset < entryLength) : String.format("Redirected offset would be outside of the range of the Redirected StreamSegment. StreamSegmentOffset = %d, MaxLength = %d, Entry.StartOffset = %d, Entry.Length = %d, RedirectOffset = %d.", streamSegmentOffset, maxLength, entry.getStreamSegmentOffset(), entryLength, redirectOffset);
        if (entryLength < (long)maxLength) {
            maxLength = (int)entryLength;
        }
        try {
            CompletableReadResultEntry result = redirectedIndex.getSingleReadResultEntry(redirectOffset, maxLength, makeCopy);
            if (result != null) {
                result = new RedirectedReadResultEntry(result, entry.getStreamSegmentOffset(), (rso, ml, sourceSegmentId) -> this.getOrRegisterRedirectedRead(rso, ml, sourceSegmentId, makeCopy), redirectedIndex.metadata.getId());
            }
            return result;
        }
        catch (ObjectClosedException ex) {
            if (!redirectedIndex.closed) {
                throw ex;
            }
            return this.getSingleReadResultEntry(streamSegmentOffset, maxLength, makeCopy);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableReadResultEntry getOrRegisterRedirectedRead(long resultStartOffset, int maxLength, long sourceSegmentId, boolean makeCopy) {
        CompletableReadResultEntry result = this.getSingleReadResultEntry(resultStartOffset, maxLength, makeCopy);
        if (result instanceof RedirectedReadResultEntry) {
            PendingMerge pendingMerge;
            Object object = this.lock;
            synchronized (object) {
                pendingMerge = this.pendingMergers.getOrDefault(sourceSegmentId, null);
            }
            FutureReadResultEntry futureResult = new FutureReadResultEntry(result.getStreamSegmentOffset(), result.getRequestedReadLength());
            if (pendingMerge != null && pendingMerge.register(futureResult)) {
                result = futureResult;
                log.debug("{}: Registered Pending Merge Future Read {}.", (Object)this.traceObjectId, (Object)result);
            } else {
                if (pendingMerge == null) {
                    log.debug("{}: Could not find Pending Merge for Id {} for {}; re-issuing.", new Object[]{this.traceObjectId, sourceSegmentId, result});
                } else {
                    log.debug("{}: Pending Merge for id {} was sealed for {}; re-issuing.", new Object[]{this.traceObjectId, sourceSegmentId, result});
                }
                result = this.getSingleReadResultEntry(resultStartOffset, maxLength, makeCopy);
            }
        }
        return result;
    }

    @GuardedBy(value="lock")
    private ReadResultEntryBase createDataNotAvailableRead(long streamSegmentOffset, int maxLength) {
        maxLength = this.getLengthUntilNextEntry(streamSegmentOffset, maxLength);
        long storageLength = this.metadata.getStorageLength();
        if (streamSegmentOffset < storageLength) {
            long actualReadLength = storageLength - streamSegmentOffset;
            if (actualReadLength > (long)maxLength) {
                actualReadLength = maxLength;
            }
            return this.createStorageRead(streamSegmentOffset, (int)actualReadLength);
        }
        return this.createFutureRead(streamSegmentOffset, maxLength);
    }

    @GuardedBy(value="lock")
    private CacheReadResultEntry createMemoryRead(ReadIndexEntry entry, long streamSegmentOffset, int maxLength, boolean updateStats, boolean makeCopy) {
        assert (streamSegmentOffset >= entry.getStreamSegmentOffset()) : String.format("streamSegmentOffset{%d} < entry.getStreamSegmentOffset{%d}", streamSegmentOffset, entry.getStreamSegmentOffset());
        int entryOffset = (int)(streamSegmentOffset - entry.getStreamSegmentOffset());
        int length = (int)Math.min((long)maxLength, entry.getLength() - (long)entryOffset);
        assert (length > 0) : String.format("length{%d} <= 0. streamSegmentOffset = %d, maxLength = %d, entry.offset = %d, entry.length = %d", length, streamSegmentOffset, maxLength, entry.getStreamSegmentOffset(), entry.getLength());
        BufferView data = this.cacheStorage.get(entry.getCacheAddress());
        assert (data != null) : String.format("No Cache Entry could be retrieved for entry %s", entry);
        if (updateStats) {
            entry.setGeneration(this.summary.touchOne(entry.getGeneration()));
        }
        data = data.slice(entryOffset, length);
        if (makeCopy) {
            data = new ByteArraySegment(data.getCopy());
        }
        return new CacheReadResultEntry(entry.getStreamSegmentOffset() + (long)entryOffset, data);
    }

    private ReadResultEntryBase createStorageRead(long streamSegmentOffset, int readLength) {
        return new StorageReadResultEntry(streamSegmentOffset, readLength, this::queueStorageRead);
    }

    private void queueStorageRead(long offset, int length, Consumer<BufferView> successCallback, Consumer<Throwable> failureCallback, Duration timeout) {
        Consumer<StorageReadManager.Result> doneCallback = result -> {
            try {
                ByteArraySegment data = result.getData();
                successCallback.accept((BufferView)data);
                if (!result.isDerived()) {
                    this.insert(offset, data);
                }
            }
            catch (Exception ex) {
                log.error("{}: Unable to process Storage Read callback. Offset={}, Result=[{}].", new Object[]{this.traceObjectId, offset, result, ex});
            }
        };
        length = this.getReadAlignedLength(offset, length);
        this.storageReadManager.execute(new StorageReadManager.Request(offset, length, doneCallback, failureCallback, timeout));
    }

    @GuardedBy(value="lock")
    private int getLengthUntilNextEntry(long startOffset, int maxLength) {
        ReadIndexEntry ceilingEntry = (ReadIndexEntry)this.indexEntries.getCeiling(startOffset);
        if (ceilingEntry != null) {
            maxLength = (int)Math.min((long)maxLength, ceilingEntry.getStreamSegmentOffset() - startOffset);
        }
        return maxLength;
    }

    private int getReadAlignedLength(long offset, int readLength) {
        int lengthSinceLastMultiple = (int)(offset % (long)this.storageReadAlignment);
        return Math.min(readLength, this.storageReadAlignment - lengthSinceLastMultiple);
    }

    private ReadResultEntryBase createFutureRead(long streamSegmentOffset, int maxLength) {
        FutureReadResultEntry entry = new FutureReadResultEntry(streamSegmentOffset, maxLength);
        this.futureReads.add(entry);
        return entry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<MergedIndexEntry> removeAllDataEntries(long offsetAdjustment) {
        ArrayList<MergedIndexEntry> result;
        Preconditions.checkState((boolean)this.metadata.isDeleted(), (Object)"Cannot fetch entries for a Segment that has not been deleted yet.");
        Exceptions.checkArgument((offsetAdjustment >= 0L ? 1 : 0) != 0, (String)"offsetAdjustment", (String)"offsetAdjustment must be a non-negative number.", (Object[])new Object[0]);
        Object object = this.lock;
        synchronized (object) {
            result = new ArrayList<MergedIndexEntry>(this.indexEntries.size());
            this.indexEntries.forEach(entry -> {
                if (entry.isDataEntry()) {
                    result.add(new MergedIndexEntry(entry.getStreamSegmentOffset() + offsetAdjustment, this.metadata.getId(), (CacheIndexEntry)entry));
                }
            });
            this.indexEntries.clear();
        }
        return result;
    }

    private void deleteData(ReadIndexEntry entry) {
        if (entry.isDataEntry()) {
            this.cacheStorage.delete(entry.getCacheAddress());
        }
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    ReadIndexSummary getSummary() {
        return this.summary;
    }

    private static enum ReadAvailability {
        Available,
        BeyondLastOffset,
        BeforeStartOffset;

    }
}

