/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.server.writer;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.AbstractTimer;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.TimeoutTimer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.BufferView;
import io.pravega.segmentstore.contracts.Attributes;
import io.pravega.segmentstore.contracts.BadOffsetException;
import io.pravega.segmentstore.contracts.SegmentProperties;
import io.pravega.segmentstore.contracts.StreamSegmentExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentInformation;
import io.pravega.segmentstore.contracts.StreamSegmentNotExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentSealedException;
import io.pravega.segmentstore.server.DataCorruptionException;
import io.pravega.segmentstore.server.SegmentMetadata;
import io.pravega.segmentstore.server.SegmentOperation;
import io.pravega.segmentstore.server.ServiceHaltException;
import io.pravega.segmentstore.server.UpdateableSegmentMetadata;
import io.pravega.segmentstore.server.WriterFlushResult;
import io.pravega.segmentstore.server.WriterSegmentProcessor;
import io.pravega.segmentstore.server.logs.operations.CachedStreamSegmentAppendOperation;
import io.pravega.segmentstore.server.logs.operations.DeleteSegmentOperation;
import io.pravega.segmentstore.server.logs.operations.MergeSegmentOperation;
import io.pravega.segmentstore.server.logs.operations.StorageOperation;
import io.pravega.segmentstore.server.logs.operations.StreamSegmentAppendOperation;
import io.pravega.segmentstore.server.logs.operations.StreamSegmentSealOperation;
import io.pravega.segmentstore.server.logs.operations.StreamSegmentTruncateOperation;
import io.pravega.segmentstore.server.writer.AggregatorState;
import io.pravega.segmentstore.server.writer.ReconciliationFailureException;
import io.pravega.segmentstore.server.writer.WriterConfig;
import io.pravega.segmentstore.server.writer.WriterDataSource;
import io.pravega.segmentstore.storage.SegmentHandle;
import io.pravega.segmentstore.storage.SegmentRollingPolicy;
import io.pravega.segmentstore.storage.Storage;
import java.io.InputStream;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SegmentAggregator
implements WriterSegmentProcessor,
AutoCloseable {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(SegmentAggregator.class);
    private final UpdateableSegmentMetadata metadata;
    private final WriterConfig config;
    private final OperationQueue operations;
    private final AbstractTimer timer;
    private final Executor executor;
    private final String traceObjectId;
    private final Storage storage;
    private final AtomicReference<SegmentHandle> handle;
    private final WriterDataSource dataSource;
    private final AtomicInteger mergeTransactionCount;
    private final AtomicInteger truncateCount;
    private final AtomicBoolean hasSealPending;
    private final AtomicBoolean hasDeletePending;
    private final AtomicLong lastAddedOffset;
    private final AtomicReference<Duration> lastFlush;
    private final AtomicReference<AggregatorState> state;
    private final AtomicReference<ReconciliationState> reconciliationState;

    SegmentAggregator(UpdateableSegmentMetadata segmentMetadata, WriterDataSource dataSource, Storage storage, WriterConfig config, AbstractTimer timer, Executor executor) {
        this.metadata = (UpdateableSegmentMetadata)Preconditions.checkNotNull((Object)segmentMetadata, (Object)"segmentMetadata");
        Preconditions.checkArgument((this.metadata.getContainerId() == dataSource.getId() ? 1 : 0) != 0, (Object)"SegmentMetadata.ContainerId is different from WriterDataSource.Id");
        this.traceObjectId = String.format("StorageWriter[%d-%d]", this.metadata.getContainerId(), this.metadata.getId());
        this.config = (WriterConfig)Preconditions.checkNotNull((Object)config, (Object)"config");
        this.storage = (Storage)Preconditions.checkNotNull((Object)storage, (Object)"storage");
        this.dataSource = (WriterDataSource)Preconditions.checkNotNull((Object)dataSource, (Object)"dataSource");
        this.timer = (AbstractTimer)Preconditions.checkNotNull((Object)timer, (Object)"timer");
        this.executor = (Executor)Preconditions.checkNotNull((Object)executor, (Object)"executor");
        this.lastFlush = new AtomicReference<Duration>(timer.getElapsed());
        this.lastAddedOffset = new AtomicLong(-1L);
        this.mergeTransactionCount = new AtomicInteger();
        this.truncateCount = new AtomicInteger();
        this.hasSealPending = new AtomicBoolean();
        this.hasDeletePending = new AtomicBoolean();
        this.operations = new OperationQueue();
        this.state = new AtomicReference<AggregatorState>(AggregatorState.NotInitialized);
        this.reconciliationState = new AtomicReference();
        this.handle = new AtomicReference();
    }

    @Override
    public void close() {
        if (!this.isClosed()) {
            this.setState(AggregatorState.Closed);
        }
    }

    @Override
    public long getLowestUncommittedSequenceNumber() {
        StorageOperation first = this.operations.getFirst();
        return first == null ? Long.MIN_VALUE : first.getSequenceNumber();
    }

    @Override
    public boolean isClosed() {
        return this.state.get() == AggregatorState.Closed;
    }

    SegmentMetadata getMetadata() {
        return this.metadata;
    }

    Duration getElapsedSinceLastFlush() {
        return this.timer.getElapsed().minus(this.lastFlush.get());
    }

    @Override
    public boolean mustFlush() {
        if (this.metadata.isDeletedInStorage()) {
            return false;
        }
        return this.exceedsThresholds() || this.hasDeletePending.get() || this.hasSealPending.get() || this.mergeTransactionCount.get() > 0 || this.truncateCount.get() > 0 || this.operations.size() > 0 && this.isReconciling();
    }

    private boolean exceedsThresholds() {
        boolean isFirstAppend = this.operations.size() > 0 && this.isAppendOperation(this.operations.getFirst());
        long length = isFirstAppend ? this.operations.getFirst().getLength() : 0L;
        return length >= (long)this.config.getFlushThresholdBytes() || length > 0L && this.getElapsedSinceLastFlush().compareTo(this.config.getFlushThresholdTime()) >= 0;
    }

    private boolean isReconciling() {
        AggregatorState currentState = this.state.get();
        return currentState == AggregatorState.ReconciliationNeeded || currentState == AggregatorState.Reconciling;
    }

    public String toString() {
        return String.format("[%d: %s] Count = %d, LastOffset = %s, LUSN = %d, LastFlush = %ds", this.metadata.getId(), this.metadata.getName(), this.operations.size(), this.lastAddedOffset, this.getLowestUncommittedSequenceNumber(), this.getElapsedSinceLastFlush().toMillis() / 1000L);
    }

    CompletableFuture<Void> initialize(Duration timeout) {
        Exceptions.checkNotClosed((boolean)this.isClosed(), (Object)this);
        Preconditions.checkState((this.state.get() == AggregatorState.NotInitialized ? 1 : 0) != 0, (Object)"SegmentAggregator has already been initialized.");
        assert (this.handle.get() == null) : "non-null handle but state == " + (Object)((Object)this.state.get());
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"initialize", (Object[])new Object[0]);
        if (this.metadata.isDeleted()) {
            log.info("{}: Segment '{}' is marked as Deleted in Metadata. Attempting Storage delete.", (Object)this.traceObjectId, (Object)this.metadata.getName());
            return Futures.exceptionallyExpecting((CompletableFuture)this.storage.openWrite(this.metadata.getName()).thenComposeAsync(handle -> this.storage.delete(handle, timeout), this.executor), ex -> ex instanceof StreamSegmentNotExistsException, null).thenRun(() -> {
                this.updateMetadataPostDeletion(this.metadata);
                log.info("{}: Segment '{}' is marked as Deleted in Metadata and has been deleted from Storage. Ignoring all further operations on it.", (Object)this.traceObjectId, (Object)this.metadata.getName());
                this.setState(AggregatorState.Writing);
                LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"initialize", (long)traceId, (Object[])new Object[0]);
            });
        }
        return ((CompletableFuture)this.openWrite(this.metadata.getName(), this.handle, timeout).thenAcceptAsync(segmentInfo -> {
            if (this.metadata.getStorageLength() != segmentInfo.getLength()) {
                if (this.metadata.getStorageLength() >= 0L) {
                    log.info("{}: SegmentMetadata has a StorageLength ({}) that is different than the actual one ({}) - updating metadata.", new Object[]{this.traceObjectId, this.metadata.getStorageLength(), segmentInfo.getLength()});
                }
                this.metadata.setStorageLength(segmentInfo.getLength());
            }
            if (segmentInfo.isSealed()) {
                if (!this.metadata.isSealed()) {
                    throw new CompletionException((Throwable)((Object)new DataCorruptionException(String.format("Segment '%s' is sealed in Storage but not in the metadata.", this.metadata.getName()), new Object[0])));
                }
                if (!this.metadata.isSealedInStorage()) {
                    this.metadata.markSealedInStorage();
                    log.info("{}: Segment is sealed in Storage but metadata does not reflect that - updating metadata.", (Object)this.traceObjectId);
                }
            }
            log.info("{}: Initialized. StorageLength = {}, Sealed = {}.", new Object[]{this.traceObjectId, segmentInfo.getLength(), segmentInfo.isSealed()});
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"initialize", (long)traceId, (Object[])new Object[0]);
            this.setState(AggregatorState.Writing);
        }, this.executor)).exceptionally(ex -> {
            if ((ex = Exceptions.unwrap((Throwable)ex)) instanceof StreamSegmentNotExistsException) {
                if (this.metadata.getStorageLength() == 0L && !this.metadata.isDeletedInStorage()) {
                    this.handle.set(null);
                    log.info("{}: Initialized. Segment does not exist in Storage but Metadata indicates it should be empty.", (Object)this.traceObjectId);
                    if (this.metadata.isSealed() && this.metadata.getLength() == 0L) {
                        this.metadata.markSealedInStorage();
                        log.info("{}: Segment does not exist in Storage, but Metadata indicates it is empty and sealed - marking as sealed in storage.", (Object)this.traceObjectId);
                    }
                } else {
                    this.updateMetadataPostDeletion(this.metadata);
                    log.info("{}: Segment '{}' does not exist in Storage. Ignoring all further operations on it.", (Object)this.traceObjectId, (Object)this.metadata.getName());
                }
            } else {
                throw new CompletionException((Throwable)ex);
            }
            this.setState(AggregatorState.Writing);
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"initialize", (long)traceId, (Object[])new Object[0]);
            return null;
        });
    }

    @Override
    public void add(SegmentOperation operation) throws ServiceHaltException {
        this.ensureInitializedAndNotClosed();
        if (!(operation instanceof StorageOperation)) {
            return;
        }
        StorageOperation storageOp = (StorageOperation)operation;
        this.checkValidOperation(storageOp);
        boolean isDelete = this.isDeleteOperation(storageOp);
        if (isDelete) {
            this.addDeleteOperation((DeleteSegmentOperation)storageOp);
            log.debug("{}: Add {}.", (Object)this.traceObjectId, (Object)storageOp);
        } else if (!this.metadata.isDeleted()) {
            this.addStorageOperation(storageOp);
            log.debug("{}: Add {}; OpCount={}, MergeCount={}, Seal={}.", new Object[]{this.traceObjectId, storageOp, this.operations.size(), this.mergeTransactionCount, this.hasSealPending});
        }
    }

    private void addDeleteOperation(DeleteSegmentOperation operation) {
        this.operations.add(operation);
        this.hasDeletePending.set(true);
    }

    private void addStorageOperation(StorageOperation operation) throws ServiceHaltException {
        boolean processOp;
        this.checkValidStorageOperation(operation);
        long lastOffset = operation.getLastStreamSegmentOffset();
        boolean isTruncate = this.isTruncateOperation(operation);
        boolean isMerge = operation instanceof MergeSegmentOperation;
        boolean bl = processOp = lastOffset > this.metadata.getStorageLength() || isTruncate || !this.metadata.isSealedInStorage() && operation instanceof StreamSegmentSealOperation || isMerge && operation.getLength() == 0L && lastOffset == this.metadata.getStorageLength();
        if (processOp) {
            this.processNewOperation(operation);
        } else {
            this.acknowledgeAlreadyProcessedOperation(operation);
        }
        if (!isTruncate) {
            this.lastAddedOffset.set(lastOffset);
        }
    }

    private void processNewOperation(StorageOperation operation) {
        if (operation instanceof MergeSegmentOperation) {
            this.operations.add(operation);
            this.mergeTransactionCount.incrementAndGet();
        } else if (operation instanceof StreamSegmentSealOperation) {
            this.operations.add(operation);
            this.hasSealPending.set(true);
        } else if (operation instanceof StreamSegmentTruncateOperation) {
            this.operations.add(operation);
            this.truncateCount.incrementAndGet();
        } else if (operation instanceof CachedStreamSegmentAppendOperation) {
            AggregatedAppendOperation aggregatedAppend = this.getOrCreateAggregatedAppend(operation.getStreamSegmentOffset(), operation.getSequenceNumber());
            this.aggregateAppendOperation((CachedStreamSegmentAppendOperation)operation, aggregatedAppend);
        }
    }

    private void acknowledgeAlreadyProcessedOperation(SegmentOperation operation) throws ServiceHaltException {
        if (operation instanceof MergeSegmentOperation) {
            MergeSegmentOperation mergeOp = (MergeSegmentOperation)operation;
            try {
                this.updateMetadataForTransactionPostMerger(this.dataSource.getStreamSegmentMetadata(mergeOp.getSourceSegmentId()), mergeOp.getStreamSegmentId());
            }
            catch (Throwable ex) {
                throw new ServiceHaltException(String.format("Unable to acknowledge already processed operation '%s'.", operation), ex, new Object[0]);
            }
        }
    }

    private void aggregateAppendOperation(CachedStreamSegmentAppendOperation operation, AggregatedAppendOperation aggregatedAppend) {
        long remainingLength = operation.getLength();
        if (operation.getStreamSegmentOffset() < aggregatedAppend.getLastStreamSegmentOffset()) {
            long delta = aggregatedAppend.getLastStreamSegmentOffset() - operation.getStreamSegmentOffset();
            remainingLength -= delta;
            log.debug("{}: Skipping {} bytes from the beginning of '{}' since it has already been partially written to Storage.", new Object[]{this.traceObjectId, delta, operation});
        }
        while (remainingLength > 0L) {
            int lengthToAdd = (int)Math.min((long)this.config.getMaxFlushSizeBytes() - aggregatedAppend.getLength(), remainingLength);
            aggregatedAppend.increaseLength(lengthToAdd);
            if ((remainingLength -= (long)lengthToAdd) <= 0L) continue;
            aggregatedAppend = new AggregatedAppendOperation(this.metadata.getId(), aggregatedAppend.getLastStreamSegmentOffset(), operation.getSequenceNumber());
            this.operations.add(aggregatedAppend);
        }
    }

    private AggregatedAppendOperation getOrCreateAggregatedAppend(long operationOffset, long operationSequenceNumber) {
        StorageOperation last;
        AggregatedAppendOperation aggregatedAppend = null;
        if (this.operations.size() > 0 && (last = this.operations.getLast()).getLength() < (long)this.config.getMaxFlushSizeBytes() && this.isAppendOperation(last) && (aggregatedAppend = (AggregatedAppendOperation)last).isSealed()) {
            aggregatedAppend = null;
        }
        if (aggregatedAppend == null) {
            long offset = Math.max(operationOffset, this.metadata.getStorageLength());
            aggregatedAppend = new AggregatedAppendOperation(this.metadata.getId(), offset, operationSequenceNumber);
            this.operations.add(aggregatedAppend);
        }
        return aggregatedAppend;
    }

    @Override
    public CompletableFuture<WriterFlushResult> flush(boolean force, Duration timeout) {
        CompletableFuture result;
        this.ensureInitializedAndNotClosed();
        if (this.metadata.isDeletedInStorage()) {
            return CompletableFuture.completedFuture(new WriterFlushResult());
        }
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"flush", (Object[])new Object[0]);
        TimeoutTimer timer = new TimeoutTimer(timeout);
        try {
            switch (this.state.get()) {
                case Writing: {
                    result = this.flushNormally(force, timer);
                    break;
                }
                case ReconciliationNeeded: {
                    result = this.beginReconciliation(timer).thenComposeAsync(v -> this.reconcile(timer), this.executor);
                    break;
                }
                case Reconciling: {
                    result = this.reconcile(timer);
                    break;
                }
                default: {
                    result = Futures.failedFuture((Throwable)new IllegalStateException(String.format("Unexpected state for SegmentAggregator (%s) for segment '%s'.", this.state, this.metadata.getName())));
                    break;
                }
            }
        }
        catch (Exception ex) {
            result = Futures.failedFuture((Throwable)ex);
        }
        return result.thenApply(r -> {
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"flush", (long)traceId, (Object[])new Object[]{r});
            return r;
        });
    }

    private CompletableFuture<WriterFlushResult> flushNormally(boolean force, TimeoutTimer timer) {
        assert (this.state.get() == AggregatorState.Writing) : "flushNormally cannot be called if state == " + this.state;
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"flushNormally", (Object[])new Object[]{force, this.operations.size()});
        WriterFlushResult result = new WriterFlushResult();
        AtomicBoolean canContinue = new AtomicBoolean(true);
        return Futures.loop(canContinue::get, () -> this.flushOnce(force, timer), partialResult -> {
            canContinue.set(partialResult.getFlushedBytes() + partialResult.getMergedBytes() > 0L);
            result.withFlushResult((WriterFlushResult)partialResult);
        }, (Executor)this.executor).thenApply(v -> {
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"flushNormally", (long)traceId, (Object[])new Object[]{result});
            return result;
        });
    }

    private CompletableFuture<WriterFlushResult> flushOnce(boolean force, TimeoutTimer timer) {
        CompletionStage<WriterFlushResult> result;
        boolean hasDelete = this.hasDeletePending.get();
        boolean hasMerge = this.mergeTransactionCount.get() > 0;
        boolean hasSeal = this.hasSealPending.get();
        boolean hasTruncate = this.truncateCount.get() > 0;
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"flushOnce", (Object[])new Object[]{this.operations.size(), this.mergeTransactionCount, hasSeal, hasTruncate, hasDelete});
        if (hasDelete) {
            result = this.deleteSegment(timer);
        } else if (hasSeal || hasMerge || hasTruncate) {
            result = this.flushFully(timer);
            if (hasMerge) {
                result = result.thenComposeAsync(flushResult -> this.mergeIfNecessary((WriterFlushResult)flushResult, timer), this.executor);
            }
            if (hasSeal) {
                result = result.thenComposeAsync(flushResult -> this.sealIfNecessary((WriterFlushResult)flushResult, timer), this.executor);
            }
        } else {
            CompletableFuture<WriterFlushResult> completableFuture = result = force ? this.flushFully(timer) : this.flushExcess(timer);
        }
        if (log.isTraceEnabled()) {
            result = result.thenApply(r -> {
                LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"flushOnce", (long)traceId, (Object[])new Object[]{r});
                return r;
            });
        }
        return result;
    }

    private CompletableFuture<WriterFlushResult> flushFully(TimeoutTimer timer) {
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"flushFully", (Object[])new Object[0]);
        WriterFlushResult result = new WriterFlushResult();
        return Futures.loop(this::canContinueFlushingFully, () -> this.flushPendingAppends(timer.getRemaining()).thenCompose(flushResult -> this.flushPendingTruncate((WriterFlushResult)flushResult, timer.getRemaining())), result::withFlushResult, (Executor)this.executor).thenApply(v -> {
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"flushFully", (long)traceId, (Object[])new Object[]{result});
            return result;
        });
    }

    private boolean canContinueFlushingFully() {
        if (this.metadata.isDeleted()) {
            return false;
        }
        StorageOperation next = this.operations.getFirst();
        return this.isAppendOperation(next) || this.isTruncateOperation(next);
    }

    private CompletableFuture<WriterFlushResult> flushExcess(TimeoutTimer timer) {
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"flushExcess", (Object[])new Object[0]);
        WriterFlushResult result = new WriterFlushResult();
        return Futures.loop(this::canContinueFlushingExcess, () -> this.flushPendingAppends(timer.getRemaining()).thenCompose(flushResult -> this.flushPendingTruncate((WriterFlushResult)flushResult, timer.getRemaining())), result::withFlushResult, (Executor)this.executor).thenApply(v -> {
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"flushExcess", (long)traceId, (Object[])new Object[]{result});
            return result;
        });
    }

    private boolean canContinueFlushingExcess() {
        return !this.metadata.isDeleted() && this.exceedsThresholds() || this.isTruncateOperation(this.operations.getFirst());
    }

    private CompletableFuture<WriterFlushResult> flushPendingTruncate(WriterFlushResult flushResult, Duration timeout) {
        CompletableFuture truncateTask;
        StorageOperation op = this.operations.getFirst();
        if (!this.isTruncateOperation(op) || !this.storage.supportsTruncation()) {
            return CompletableFuture.completedFuture(flushResult);
        }
        if (this.handle.get() == null) {
            assert (this.metadata.getStorageLength() == 0L) : "handle is null but Metadata.getStorageLength is non-zero";
            truncateTask = CompletableFuture.completedFuture(null);
        } else {
            long truncateOffset = Math.min(this.metadata.getStorageLength(), op.getStreamSegmentOffset());
            truncateTask = this.storage.truncate(this.handle.get(), truncateOffset, timeout);
        }
        return truncateTask.thenApplyAsync(v -> {
            this.updateStatePostTruncate();
            return flushResult;
        }, this.executor);
    }

    private CompletableFuture<WriterFlushResult> flushPendingAppends(Duration timeout) {
        BufferView flushData;
        try {
            flushData = this.getFlushData();
        }
        catch (DataCorruptionException ex2) {
            return Futures.failedFuture((Throwable)((Object)ex2));
        }
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"flushPendingAppends", (Object[])new Object[0]);
        TimeoutTimer timer = new TimeoutTimer(timeout);
        CompletableFuture<Object> flush = flushData == null || flushData.getLength() == 0 ? CompletableFuture.completedFuture(null) : this.createSegmentIfNecessary(() -> this.storage.write(this.handle.get(), this.metadata.getStorageLength(), flushData.getReader(), flushData.getLength(), timer.getRemaining()), timer.getRemaining());
        return ((CompletableFuture)flush.thenApplyAsync(v -> {
            WriterFlushResult result = this.updateStatePostFlush(flushData);
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"flushPendingAppends", (long)traceId, (Object[])new Object[]{result});
            return result;
        }, this.executor)).exceptionally(ex -> {
            if (Exceptions.unwrap((Throwable)ex) instanceof BadOffsetException) {
                this.setState(AggregatorState.ReconciliationNeeded);
            }
            throw new CompletionException((Throwable)ex);
        });
    }

    @Nullable
    private BufferView getFlushData() throws DataCorruptionException {
        StorageOperation first = this.operations.getFirst();
        if (!(first instanceof AggregatedAppendOperation)) {
            return null;
        }
        AggregatedAppendOperation appendOp = (AggregatedAppendOperation)first;
        int length = (int)appendOp.getLength();
        BufferView data = null;
        if (length > 0 && (data = this.dataSource.getAppendData(appendOp.getStreamSegmentId(), appendOp.getStreamSegmentOffset(), length)) == null) {
            if (this.metadata.isDeleted()) {
                return null;
            }
            throw new DataCorruptionException(String.format("Unable to retrieve CacheContents for '%s'.", appendOp), new Object[0]);
        }
        appendOp.seal();
        return data;
    }

    private CompletableFuture<WriterFlushResult> mergeIfNecessary(WriterFlushResult flushResult, TimeoutTimer timer) {
        this.ensureInitializedAndNotClosed();
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"mergeIfNecessary", (Object[])new Object[0]);
        StorageOperation first = this.operations.getFirst();
        if (first == null || !(first instanceof MergeSegmentOperation)) {
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"mergeIfNecessary", (long)traceId, (Object[])new Object[]{flushResult});
            return CompletableFuture.completedFuture(flushResult);
        }
        MergeSegmentOperation mergeSegmentOperation = (MergeSegmentOperation)first;
        UpdateableSegmentMetadata transactionMetadata = this.dataSource.getStreamSegmentMetadata(mergeSegmentOperation.getSourceSegmentId());
        return this.mergeWith(transactionMetadata, mergeSegmentOperation, timer).thenApply(mergeResult -> {
            flushResult.withFlushResult((WriterFlushResult)mergeResult);
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"mergeIfNecessary", (long)traceId, (Object[])new Object[]{flushResult});
            return flushResult;
        });
    }

    private CompletableFuture<WriterFlushResult> mergeWith(UpdateableSegmentMetadata transactionMetadata, MergeSegmentOperation mergeOp, TimeoutTimer timer) {
        CompletableFuture<UpdateableSegmentMetadata> merge;
        boolean emptySourceSegment;
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"mergeWith", (Object[])new Object[]{transactionMetadata.getId(), transactionMetadata.getName(), transactionMetadata.isSealedInStorage()});
        boolean bl = emptySourceSegment = transactionMetadata.getLength() == 0L;
        if (transactionMetadata.isDeleted() && !emptySourceSegment) {
            this.setState(AggregatorState.ReconciliationNeeded);
            return Futures.failedFuture((Throwable)new StreamSegmentNotExistsException(transactionMetadata.getName()));
        }
        WriterFlushResult result = new WriterFlushResult();
        if (emptySourceSegment) {
            log.warn("{}: Not applying '{}' because source segment is missing or empty.", (Object)this.traceObjectId, (Object)mergeOp);
            merge = CompletableFuture.completedFuture(this.metadata);
        } else {
            if (!transactionMetadata.isSealedInStorage() || transactionMetadata.getLength() > transactionMetadata.getStorageLength()) {
                LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"mergeWith", (long)traceId, (Object[])new Object[]{result});
                return CompletableFuture.completedFuture(result);
            }
            merge = this.mergeInStorage(transactionMetadata, mergeOp, timer);
        }
        return ((CompletableFuture)((CompletableFuture)((CompletableFuture)merge.thenAcceptAsync(segmentProperties -> this.mergeCompleted((SegmentProperties)segmentProperties, transactionMetadata, mergeOp), this.executor)).thenComposeAsync(v -> this.dataSource.deleteAllAttributes(transactionMetadata, timer.getRemaining()), this.executor)).thenApply(v -> {
            this.lastFlush.set(this.timer.getElapsed());
            result.withMergedBytes(mergeOp.getLength());
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"mergeWith", (long)traceId, (Object[])new Object[]{result});
            return result;
        })).exceptionally(ex -> {
            Throwable realEx = Exceptions.unwrap((Throwable)ex);
            if (realEx instanceof BadOffsetException || realEx instanceof StreamSegmentNotExistsException) {
                this.setState(AggregatorState.ReconciliationNeeded);
            }
            throw new CompletionException((Throwable)ex);
        });
    }

    private CompletableFuture<SegmentProperties> mergeInStorage(SegmentMetadata transactionMetadata, MergeSegmentOperation mergeOp, TimeoutTimer timer) {
        return ((CompletableFuture)((CompletableFuture)((CompletableFuture)this.storage.getStreamSegmentInfo(transactionMetadata.getName(), timer.getRemaining()).thenAcceptAsync(transProperties -> {
            if (transProperties.getLength() != transactionMetadata.getStorageLength()) {
                throw new CompletionException((Throwable)((Object)new DataCorruptionException(String.format("Transaction Segment '%s' cannot be merged into parent '%s' because its metadata disagrees with the Storage. Metadata.StorageLength=%d, Storage.StorageLength=%d", transactionMetadata.getName(), this.metadata.getName(), transactionMetadata.getStorageLength(), transProperties.getLength()), new Object[0])));
            }
            if (transProperties.getLength() != mergeOp.getLength()) {
                throw new CompletionException((Throwable)((Object)new DataCorruptionException(String.format("Transaction Segment '%s' cannot be merged into parent '%s' because the declared length in the operation disagrees with the Storage. Operation.Length=%d, Storage.StorageLength=%d", transactionMetadata.getName(), this.metadata.getName(), mergeOp.getLength(), transProperties.getLength()), new Object[0])));
            }
        }, this.executor)).thenComposeAsync(v -> this.createSegmentIfNecessary(() -> this.storage.concat(this.handle.get(), mergeOp.getStreamSegmentOffset(), transactionMetadata.getName(), timer.getRemaining()), timer.getRemaining()), this.executor)).exceptionally(ex -> {
            ex = Exceptions.unwrap((Throwable)ex);
            if (transactionMetadata.getLength() == 0L && ex instanceof StreamSegmentNotExistsException && ((StreamSegmentNotExistsException)ex).getStreamSegmentName().equals(transactionMetadata.getName())) {
                log.warn("{}: Not applying '{}' because source segment is missing (storage) and had no data.", (Object)this.traceObjectId, (Object)mergeOp);
                return null;
            }
            throw new CompletionException((Throwable)ex);
        })).thenComposeAsync(v -> this.storage.getStreamSegmentInfo(this.metadata.getName(), timer.getRemaining()), this.executor);
    }

    private void mergeCompleted(SegmentProperties segmentProperties, UpdateableSegmentMetadata transactionMetadata, MergeSegmentOperation mergeOp) {
        StorageOperation processedOperation = this.operations.removeFirst();
        assert (processedOperation != null && processedOperation instanceof MergeSegmentOperation) : "First outstanding operation was not a MergeSegmentOperation";
        MergeSegmentOperation mop = (MergeSegmentOperation)processedOperation;
        assert (mop.getSourceSegmentId() == transactionMetadata.getId()) : "First outstanding operation was a MergeSegmentOperation for the wrong Transaction id.";
        int newCount = this.mergeTransactionCount.decrementAndGet();
        assert (newCount >= 0) : "Negative value for mergeTransactionCount";
        long expectedNewLength = this.metadata.getStorageLength() + mergeOp.getLength();
        if (segmentProperties.getLength() != expectedNewLength) {
            throw new CompletionException((Throwable)((Object)new DataCorruptionException(String.format("Transaction Segment '%s' was merged into parent '%s' but the parent segment has an unexpected StorageLength after the merger. Previous=%d, MergeLength=%d, Expected=%d, Actual=%d", transactionMetadata.getName(), this.metadata.getName(), segmentProperties.getLength(), mergeOp.getLength(), expectedNewLength, segmentProperties.getLength()), new Object[0])));
        }
        this.updateMetadata(segmentProperties);
        this.updateMetadataForTransactionPostMerger(transactionMetadata, mop.getStreamSegmentId());
    }

    private CompletableFuture<WriterFlushResult> sealIfNecessary(WriterFlushResult flushResult, TimeoutTimer timer) {
        CompletableFuture sealTask;
        if (!this.hasSealPending.get() || !(this.operations.getFirst() instanceof StreamSegmentSealOperation)) {
            return CompletableFuture.completedFuture(flushResult);
        }
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"sealIfNecessary", (Object[])new Object[0]);
        if (this.handle.get() == null) {
            assert (this.metadata.getStorageLength() == 0L) : "handle is null but Metadata.StorageLength is non-zero";
            sealTask = CompletableFuture.completedFuture(null);
        } else {
            sealTask = this.storage.seal(this.handle.get(), timer.getRemaining());
        }
        return sealTask.handleAsync((v, ex) -> {
            if ((ex = Exceptions.unwrap((Throwable)ex)) != null && !(ex instanceof StreamSegmentSealedException)) {
                if (ex instanceof StreamSegmentNotExistsException) {
                    this.setState(AggregatorState.ReconciliationNeeded);
                }
                throw new CompletionException((Throwable)ex);
            }
            this.updateStatePostSeal();
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"sealIfNecessary", (long)traceId, (Object[])new Object[]{flushResult});
            return flushResult;
        }, this.executor);
    }

    private CompletableFuture<WriterFlushResult> deleteSegment(TimeoutTimer timer) {
        CompletableFuture<Void> deleteFuture = this.handle.get() == null ? this.dataSource.deleteAllAttributes(this.metadata, timer.getRemaining()) : this.deleteSegmentAndAttributes(this.handle.get(), this.metadata, timer);
        return ((CompletableFuture)deleteFuture.thenComposeAsync(v -> this.deleteUnmergedSourceSegments(timer), this.executor)).thenApplyAsync(v -> {
            this.updateMetadataPostDeletion(this.metadata);
            this.hasSealPending.set(false);
            this.hasDeletePending.set(false);
            this.truncateCount.set(0);
            this.mergeTransactionCount.set(0);
            this.operations.clear();
            return new WriterFlushResult();
        }, this.executor);
    }

    private CompletableFuture<Void> deleteSegmentAndAttributes(SegmentHandle handle, SegmentMetadata metadata, TimeoutTimer timer) {
        assert (handle.getSegmentName().equals(metadata.getName()));
        return CompletableFuture.allOf(Futures.exceptionallyExpecting((CompletableFuture)this.storage.delete(handle, timer.getRemaining()), ex -> ex instanceof StreamSegmentNotExistsException, null), this.dataSource.deleteAllAttributes(metadata, timer.getRemaining()));
    }

    private CompletableFuture<Void> deleteUnmergedSourceSegments(TimeoutTimer timer) {
        if (this.mergeTransactionCount.get() == 0) {
            return CompletableFuture.completedFuture(null);
        }
        List toDelete = this.operations.filter(op -> op instanceof MergeSegmentOperation).stream().map(op -> {
            UpdateableSegmentMetadata m = this.dataSource.getStreamSegmentMetadata(((MergeSegmentOperation)op).getSourceSegmentId());
            return Futures.exceptionallyExpecting((CompletableFuture)((CompletableFuture)this.storage.openWrite(m.getName()).thenCompose(handle -> this.deleteSegmentAndAttributes((SegmentHandle)handle, m, timer))).thenAcceptAsync(v -> this.updateMetadataPostDeletion(m), this.executor), ex -> ex instanceof StreamSegmentNotExistsException, null);
        }).collect(Collectors.toList());
        return Futures.allOf(toDelete);
    }

    private CompletableFuture<Void> createSegmentIfNecessary(Supplier<CompletableFuture<Void>> toRun, Duration timeout) {
        if (this.handle.get() == null) {
            assert (this.metadata.getStorageLength() == 0L) : "no handle yet but metadata indicates Storage Segment not empty";
            return Futures.exceptionallyComposeExpecting((CompletableFuture)this.storage.create(this.metadata.getName(), new SegmentRollingPolicy(this.getRolloverSize()), timeout), ex -> ex instanceof StreamSegmentExistsException, () -> {
                log.info("{}: Segment did not exist in Storage when initialize() was called, but does now.", (Object)this.traceObjectId);
                return this.storage.openWrite(this.metadata.getName());
            }).thenComposeAsync(handle -> {
                this.handle.set((SegmentHandle)handle);
                return (CompletionStage)toRun.get();
            }, this.executor);
        }
        return toRun.get();
    }

    private long getRolloverSize() {
        long rolloverSize = this.metadata.getAttributes().getOrDefault(Attributes.ROLLOVER_SIZE, SegmentRollingPolicy.NO_ROLLING.getMaxLength());
        rolloverSize = rolloverSize == 0L ? SegmentRollingPolicy.NO_ROLLING.getMaxLength() : rolloverSize;
        return Math.min(rolloverSize, this.config.getMaxRolloverSize());
    }

    private CompletableFuture<Void> beginReconciliation(TimeoutTimer timer) {
        assert (this.state.get() == AggregatorState.ReconciliationNeeded) : "beginReconciliation cannot be called if state == " + this.state;
        return ((CompletableFuture)this.storage.getStreamSegmentInfo(this.metadata.getName(), timer.getRemaining()).thenAcceptAsync(sp -> {
            if (sp.getLength() > this.metadata.getLength()) {
                throw new CompletionException((Throwable)((Object)new ReconciliationFailureException("Actual Segment length in Storage is larger than the Metadata Length.", this.metadata, (SegmentProperties)sp)));
            }
            if (sp.getLength() < this.metadata.getStorageLength()) {
                throw new CompletionException((Throwable)((Object)new ReconciliationFailureException("Actual Segment length in Storage is smaller than the Metadata StorageLength.", this.metadata, (SegmentProperties)sp)));
            }
            if (sp.getLength() == this.metadata.getStorageLength() && sp.isSealed() == this.metadata.isSealedInStorage()) {
                this.setState(AggregatorState.Writing);
                return;
            }
            this.reconciliationState.set(new ReconciliationState(this.metadata, (SegmentProperties)sp));
            this.setState(AggregatorState.Reconciling);
        }, this.executor)).exceptionally(ex -> {
            if ((ex = Exceptions.unwrap((Throwable)ex)) instanceof StreamSegmentNotExistsException) {
                if (this.metadata.isMerged() || this.metadata.isDeleted()) {
                    this.updateMetadataPostDeletion(this.metadata);
                    log.info("{}: Segment '{}' does not exist in Storage (reconciliation). Ignoring all further operations on it.", (Object)this.traceObjectId, (Object)this.metadata.getName());
                    this.reconciliationState.set(null);
                    this.setState(AggregatorState.Reconciling);
                } else {
                    if (this.metadata.getStorageLength() > 0L) {
                        throw new CompletionException((Throwable)((Object)new ReconciliationFailureException("Segment does not exist in Storage, but Metadata StorageLength is non-zero.", this.metadata, (SegmentProperties)StreamSegmentInformation.builder().name(this.metadata.getName()).deleted(true).build())));
                    }
                    this.reconciliationState.set(new ReconciliationState(this.metadata, (SegmentProperties)StreamSegmentInformation.builder().name(this.metadata.getName()).build()));
                    this.setState(AggregatorState.Reconciling);
                }
            } else {
                throw new CompletionException((Throwable)ex);
            }
            return null;
        });
    }

    private CompletableFuture<WriterFlushResult> reconcile(TimeoutTimer timer) {
        ReconciliationState rc = this.reconciliationState.get();
        WriterFlushResult result = new WriterFlushResult();
        if (rc == null) {
            this.setState(AggregatorState.Writing);
            return CompletableFuture.completedFuture(result);
        }
        if (this.hasDeletePending.get()) {
            this.setState(AggregatorState.Writing);
            return this.deleteSegment(timer);
        }
        SegmentProperties storageInfo = rc.getStorageInfo();
        long traceId = LoggerHelpers.traceEnterWithContext((Logger)log, (String)this.traceObjectId, (String)"reconcile", (Object[])new Object[]{rc});
        AtomicBoolean exceededStorageLength = new AtomicBoolean(false);
        return Futures.loop(() -> this.operations.size() > 0 && !exceededStorageLength.get(), () -> {
            StorageOperation op = this.operations.getFirst();
            return this.reconcileOperation(op, storageInfo, timer).thenApply(partialFlushResult -> {
                if (op.getLastStreamSegmentOffset() >= storageInfo.getLength()) {
                    exceededStorageLength.set(true);
                }
                log.info("{}: Reconciled {} ({}).", new Object[]{this.traceObjectId, op, partialFlushResult});
                return partialFlushResult;
            });
        }, result::withFlushResult, (Executor)this.executor).thenApply(v -> {
            this.updateMetadata(storageInfo);
            this.reconciliationState.set(null);
            this.setState(AggregatorState.Writing);
            LoggerHelpers.traceLeave((Logger)log, (String)this.traceObjectId, (String)"reconcile", (long)traceId, (Object[])new Object[]{result});
            return result;
        });
    }

    private CompletableFuture<WriterFlushResult> reconcileOperation(StorageOperation op, SegmentProperties storageInfo, TimeoutTimer timer) {
        CompletableFuture result;
        if (this.isAppendOperation(op)) {
            result = this.reconcileAppendOperation((AggregatedAppendOperation)op, storageInfo, timer);
        } else if (op instanceof MergeSegmentOperation) {
            result = this.reconcileMergeOperation((MergeSegmentOperation)op, storageInfo, timer);
        } else if (op instanceof StreamSegmentSealOperation) {
            result = this.reconcileSealOperation(storageInfo, timer.getRemaining());
        } else if (this.isTruncateOperation(op)) {
            this.updateStatePostTruncate();
            result = CompletableFuture.completedFuture(new WriterFlushResult());
        } else {
            result = Futures.failedFuture((Throwable)((Object)new ReconciliationFailureException(String.format("Operation '%s' is not supported for reconciliation.", op), this.metadata, storageInfo)));
        }
        return result;
    }

    private CompletableFuture<WriterFlushResult> reconcileAppendOperation(AggregatedAppendOperation op, SegmentProperties storageInfo, TimeoutTimer timer) {
        CompletableFuture<Integer> reconcileResult = op.getLength() > 0L ? this.reconcileData(op, storageInfo, timer) : CompletableFuture.completedFuture(0);
        return reconcileResult.thenApplyAsync(reconciledBytes -> {
            op.reconcileComplete((int)reconciledBytes);
            if (op.getLength() == 0L) {
                StorageOperation removedOp = this.operations.removeFirst();
                assert (op == removedOp) : "Reconciled operation is not the same as removed operation";
            }
            return new WriterFlushResult().withFlushedBytes(reconciledBytes.intValue());
        }, this.executor);
    }

    private CompletableFuture<Integer> reconcileData(AggregatedAppendOperation op, SegmentProperties storageInfo, TimeoutTimer timer) {
        BufferView appendData = this.dataSource.getAppendData(op.getStreamSegmentId(), op.getStreamSegmentOffset(), (int)op.getLength());
        if (appendData == null) {
            return Futures.failedFuture((Throwable)((Object)new ReconciliationFailureException(String.format("Unable to reconcile operation '%s' because no append data is associated with it.", op), this.metadata, storageInfo)));
        }
        long readLength = Math.min(op.getLastStreamSegmentOffset(), storageInfo.getLength()) - op.getStreamSegmentOffset();
        assert (readLength > 0L) : "Append Operation to be reconciled is beyond the Segment's StorageLength (" + storageInfo.getLength() + "): " + op;
        byte[] storageData = new byte[(int)readLength];
        AtomicInteger reconciledBytes = new AtomicInteger();
        return Futures.loop(() -> (long)reconciledBytes.get() < readLength, () -> this.storage.read(this.handle.get(), op.getStreamSegmentOffset() + (long)reconciledBytes.get(), storageData, reconciledBytes.get(), (int)readLength - reconciledBytes.get(), timer.getRemaining()), bytesRead -> {
            assert (bytesRead > 0) : String.format("Unable to make any read progress when reconciling operation '%s' after reading %s bytes.", op, reconciledBytes);
            reconciledBytes.addAndGet((int)bytesRead);
        }, (Executor)this.executor).thenApplyAsync(v -> {
            this.verifySame(appendData, storageData, op, storageInfo);
            return reconciledBytes.get();
        }, this.executor);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifySame(BufferView appendData, byte[] storageData, StorageOperation op, SegmentProperties storageInfo) {
        InputStream appendStream = appendData.getReader();
        try {
            for (int i = 0; i < storageData.length; ++i) {
                if ((byte)appendStream.read() == storageData[i]) continue;
                throw new ReconciliationFailureException(String.format("Unable to reconcile operation '%s' because of data differences at SegmentOffset %d.", op, op.getStreamSegmentOffset() + (long)i), this.metadata, storageInfo);
            }
        }
        finally {
            if (Collections.singletonList(appendStream).get(0) != null) {
                appendStream.close();
            }
        }
    }

    private CompletableFuture<WriterFlushResult> reconcileMergeOperation(MergeSegmentOperation op, SegmentProperties storageInfo, TimeoutTimer timer) {
        UpdateableSegmentMetadata transactionMeta = this.dataSource.getStreamSegmentMetadata(op.getSourceSegmentId());
        if (transactionMeta == null) {
            return Futures.failedFuture((Throwable)((Object)new ReconciliationFailureException(String.format("Cannot reconcile operation '%s' because the source segment is missing from the metadata.", op), this.metadata, storageInfo)));
        }
        if (op.getLastStreamSegmentOffset() > storageInfo.getLength()) {
            return Futures.failedFuture((Throwable)((Object)new ReconciliationFailureException(String.format("Cannot reconcile operation '%s' because the source segment is not fully merged into the target.", op), this.metadata, storageInfo)));
        }
        return ((CompletableFuture)this.storage.exists(transactionMeta.getName(), timer.getRemaining()).thenComposeAsync(exists -> {
            if (exists.booleanValue()) {
                return Futures.failedFuture((Throwable)((Object)new ReconciliationFailureException(String.format("Cannot reconcile operation '%s' because the transaction segment still exists in Storage.", op), this.metadata, storageInfo)));
            }
            return this.dataSource.deleteAllAttributes(transactionMeta, timer.getRemaining());
        }, this.executor)).thenApplyAsync(v -> {
            StorageOperation processedOperation = this.operations.removeFirst();
            assert (processedOperation != null && processedOperation instanceof MergeSegmentOperation) : "First outstanding operation was not a MergeSegmentOperation";
            int newCount = this.mergeTransactionCount.decrementAndGet();
            assert (newCount >= 0) : "Negative value for mergeTransactionCount";
            long minStorageLength = processedOperation.getLastStreamSegmentOffset();
            if (this.metadata.getStorageLength() < minStorageLength) {
                this.metadata.setStorageLength(minStorageLength);
            }
            this.updateMetadataForTransactionPostMerger(transactionMeta, processedOperation.getStreamSegmentId());
            return new WriterFlushResult().withMergedBytes(op.getLength());
        }, this.executor);
    }

    private CompletableFuture<WriterFlushResult> reconcileSealOperation(SegmentProperties storageInfo, Duration timeout) {
        if (storageInfo.isSealed() || storageInfo.getLength() == 0L) {
            return CompletableFuture.supplyAsync(() -> {
                this.updateStatePostSeal();
                return new WriterFlushResult();
            }, this.executor);
        }
        return Futures.failedFuture((Throwable)((Object)new ReconciliationFailureException("Segment was supposed to be sealed in storage but it is not.", this.metadata, storageInfo)));
    }

    private void checkValidOperation(StorageOperation operation) throws DataCorruptionException {
        Preconditions.checkArgument((operation.getStreamSegmentId() == this.metadata.getId() ? 1 : 0) != 0, (String)"Operation '%s' refers to a different Segment than this one (%s).", (Object)operation, (long)this.metadata.getId());
        if (this.hasSealPending.get() && !this.isTruncateOperation(operation) && !this.isDeleteOperation(operation)) {
            throw new DataCorruptionException(String.format("Illegal operation for a sealed Segment; received '%s'.", operation), new Object[0]);
        }
    }

    private void checkValidStorageOperation(StorageOperation operation) throws DataCorruptionException {
        Preconditions.checkArgument((!(operation instanceof StreamSegmentAppendOperation) ? 1 : 0) != 0, (Object)"SegmentAggregator cannot process StreamSegmentAppendOperations.");
        long offset = operation.getStreamSegmentOffset();
        long length = operation.getLength();
        Preconditions.checkArgument((offset >= 0L ? 1 : 0) != 0, (String)"Operation '%s' has an invalid offset (%s).", (Object)operation, (long)operation.getStreamSegmentOffset());
        Preconditions.checkArgument((length >= 0L ? 1 : 0) != 0, (String)"Operation '%s' has an invalid length (%s).", (Object)operation, (long)operation.getLength());
        if (this.isTruncateOperation(operation)) {
            if (this.metadata.getStartOffset() < operation.getStreamSegmentOffset()) {
                throw new DataCorruptionException(String.format("StreamSegmentTruncateOperation '%s' has a truncation offset beyond the one in the Segment's Metadata. Expected: at most %d, actual: %d.", operation, this.metadata.getStartOffset(), offset), new Object[0]);
            }
        } else {
            long lastOffset = this.lastAddedOffset.get();
            if (lastOffset >= 0L && offset != lastOffset) {
                throw new DataCorruptionException(String.format("Wrong offset for Operation '%s'. Expected: %s, actual: %d.", operation, this.lastAddedOffset, offset), new Object[0]);
            }
        }
        if (offset + length > this.metadata.getLength()) {
            throw new DataCorruptionException(String.format("Operation '%s' has at least one byte beyond its Length. Offset = %d, Length = %d, Length = %d.", operation, offset, length, this.metadata.getLength()), new Object[0]);
        }
        if (operation instanceof StreamSegmentSealOperation) {
            if (this.metadata.getLength() != offset) {
                throw new DataCorruptionException(String.format("Wrong offset for Operation '%s'. Expected: %d (Length), actual: %d.", operation, this.metadata.getLength(), offset), new Object[0]);
            }
            if (!this.metadata.isSealed()) {
                throw new DataCorruptionException(String.format("Received Operation '%s' for a non-sealed segment.", operation), new Object[0]);
            }
        }
    }

    private WriterFlushResult updateStatePostFlush(BufferView flushData) {
        int flushLength;
        long newLength = this.metadata.getStorageLength();
        int n = flushLength = flushData == null ? 0 : flushData.getLength();
        if (flushLength > 0) {
            this.metadata.setStorageLength(newLength += (long)flushData.getLength());
        }
        boolean reachedEnd = false;
        while (this.operations.size() > 0 && !reachedEnd) {
            StorageOperation first = this.operations.getFirst();
            long lastOffset = first.getLastStreamSegmentOffset();
            boolean bl = reachedEnd = lastOffset >= newLength;
            if (!this.isAppendOperation(first)) {
                reachedEnd = true;
                continue;
            }
            if (lastOffset > newLength) continue;
            this.operations.removeFirst();
        }
        this.lastFlush.set(this.timer.getElapsed());
        return new WriterFlushResult().withFlushedBytes(flushLength);
    }

    private void updateStatePostSeal() {
        this.metadata.markSealedInStorage();
        this.operations.removeFirst();
        assert (this.operations.size() - this.truncateCount.get() == 0) : "there are outstanding non-truncate operations after a Seal";
        this.hasSealPending.set(false);
    }

    private void updateStatePostTruncate() {
        this.operations.removeFirst();
        this.truncateCount.decrementAndGet();
    }

    private void updateMetadata(SegmentProperties segmentProperties) {
        this.metadata.setStorageLength(segmentProperties.getLength());
        if (segmentProperties.isSealed() && !this.metadata.isSealedInStorage()) {
            this.metadata.markSealed();
            this.metadata.markSealedInStorage();
        }
    }

    private void updateMetadataForTransactionPostMerger(UpdateableSegmentMetadata transactionMetadata, long targetSegmentId) {
        this.updateMetadataPostDeletion(transactionMetadata);
        this.dataSource.completeMerge(targetSegmentId, transactionMetadata.getId());
    }

    private void updateMetadataPostDeletion(UpdateableSegmentMetadata metadata) {
        metadata.markDeleted();
        metadata.markDeletedInStorage();
    }

    private boolean isAppendOperation(StorageOperation op) {
        return op instanceof AggregatedAppendOperation;
    }

    private boolean isTruncateOperation(StorageOperation operation) {
        return operation instanceof StreamSegmentTruncateOperation;
    }

    private boolean isDeleteOperation(StorageOperation operation) {
        return operation instanceof DeleteSegmentOperation;
    }

    private void ensureInitializedAndNotClosed() {
        Exceptions.checkNotClosed((boolean)this.isClosed(), (Object)this);
        Preconditions.checkState((this.state.get() != AggregatorState.NotInitialized ? 1 : 0) != 0, (Object)"SegmentAggregator is not initialized. Cannot execute this operation.");
    }

    private void setState(AggregatorState newState) {
        AggregatorState oldState = this.state.get();
        if (newState != oldState) {
            log.info("{}: State changed from {} to {}.", new Object[]{this.traceObjectId, oldState, newState});
        }
        this.state.set(newState);
    }

    private CompletableFuture<SegmentProperties> openWrite(String segmentName, AtomicReference<SegmentHandle> handleRef, Duration timeout) {
        return this.storage.openWrite(segmentName).thenComposeAsync(handle -> {
            handleRef.set((SegmentHandle)handle);
            return this.storage.getStreamSegmentInfo(segmentName, timeout);
        }, this.executor);
    }

    @ThreadSafe
    private static class OperationQueue {
        @GuardedBy(value="this")
        private final ArrayDeque<StorageOperation> queue = new ArrayDeque();

        private OperationQueue() {
        }

        synchronized boolean add(StorageOperation operation) {
            return this.queue.add(operation);
        }

        synchronized StorageOperation getLast() {
            return this.queue.peekLast();
        }

        synchronized StorageOperation getFirst() {
            return this.queue.peekFirst();
        }

        synchronized StorageOperation removeFirst() {
            return this.queue.pollFirst();
        }

        synchronized int size() {
            return this.queue.size();
        }

        synchronized void clear() {
            this.queue.clear();
        }

        synchronized List<StorageOperation> filter(Predicate<StorageOperation> test) {
            return this.queue.stream().filter(test).collect(Collectors.toList());
        }
    }

    private static class AggregatedAppendOperation
    extends StorageOperation {
        private final AtomicLong streamSegmentOffset;
        private final AtomicInteger length;
        private final AtomicBoolean sealed;

        AggregatedAppendOperation(long streamSegmentId, long streamSegmentOffset, long sequenceNumber) {
            super(streamSegmentId);
            this.streamSegmentOffset = new AtomicLong(streamSegmentOffset);
            this.setSequenceNumber(sequenceNumber);
            this.length = new AtomicInteger();
            this.sealed = new AtomicBoolean();
        }

        void increaseLength(int amount) {
            Preconditions.checkArgument((amount > 0 ? 1 : 0) != 0, (Object)"amount must be a positive integer.");
            this.length.addAndGet(amount);
        }

        void seal() {
            this.sealed.set(true);
        }

        boolean isSealed() {
            return this.sealed.get();
        }

        void reconcileComplete(int reconciledBytes) {
            this.streamSegmentOffset.addAndGet(reconciledBytes);
            this.length.addAndGet(-reconciledBytes);
        }

        @Override
        public long getStreamSegmentOffset() {
            return this.streamSegmentOffset.get();
        }

        @Override
        public long getLength() {
            return this.length.get();
        }

        @Override
        public String toString() {
            return String.format("AggregatedAppend: SegmentId = %s, Offsets = [%s-%s), SeqNo = %s", this.getStreamSegmentId(), this.getStreamSegmentOffset(), this.getStreamSegmentOffset() + this.getLength(), this.getSequenceNumber());
        }
    }

    private static class ReconciliationState {
        private final SegmentProperties storageInfo;
        private final long initialStorageLength;

        ReconciliationState(SegmentMetadata segmentMetadata, SegmentProperties storageInfo) {
            Preconditions.checkNotNull((Object)storageInfo, (Object)"storageInfo");
            this.storageInfo = storageInfo;
            this.initialStorageLength = segmentMetadata.getStorageLength();
        }

        public String toString() {
            return String.format("Metadata.StorageLength = %d, Storage.Length = %d", this.initialStorageLength, this.storageInfo.getLength());
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public SegmentProperties getStorageInfo() {
            return this.storageInfo;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public long getInitialStorageLength() {
            return this.initialStorageLength;
        }
    }
}

