/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.storage.filesystem;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.Timer;
import io.pravega.common.util.BufferView;
import io.pravega.common.util.ImmutableDate;
import io.pravega.segmentstore.contracts.BadOffsetException;
import io.pravega.segmentstore.contracts.SegmentProperties;
import io.pravega.segmentstore.contracts.StreamSegmentException;
import io.pravega.segmentstore.contracts.StreamSegmentExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentInformation;
import io.pravega.segmentstore.contracts.StreamSegmentNotExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentSealedException;
import io.pravega.segmentstore.storage.SegmentHandle;
import io.pravega.segmentstore.storage.SyncStorage;
import io.pravega.storage.filesystem.FileSystemMetrics;
import io.pravega.storage.filesystem.FileSystemSegmentHandle;
import io.pravega.storage.filesystem.FileSystemStorageConfig;
import io.pravega.storage.filesystem.FileSystemWrapper;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.NonWritableChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.AccessControlException;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicBoolean;
import lombok.Generated;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemStorage
implements SyncStorage {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(FileSystemStorage.class);
    private final FileSystemStorageConfig config;
    private final AtomicBoolean closed;

    public FileSystemStorage(FileSystemStorageConfig config) {
        this.config = (FileSystemStorageConfig)Preconditions.checkNotNull((Object)config, (Object)"config");
        this.closed = new AtomicBoolean(false);
    }

    public void initialize(long containerEpoch) {
    }

    public SegmentHandle openRead(String streamSegmentName) throws StreamSegmentException {
        return this.execute(streamSegmentName, () -> this.doOpenRead(streamSegmentName));
    }

    public int read(SegmentHandle handle, long offset, byte[] buffer, int bufferOffset, int length) throws StreamSegmentException {
        return this.execute(handle.getSegmentName(), () -> this.doRead(handle, offset, buffer, bufferOffset, length));
    }

    public SegmentProperties getStreamSegmentInfo(String streamSegmentName) throws StreamSegmentException {
        return this.execute(streamSegmentName, () -> this.doGetStreamSegmentInfo(streamSegmentName));
    }

    public boolean exists(String streamSegmentName) {
        return this.execute(streamSegmentName, () -> this.doExists(streamSegmentName));
    }

    public SegmentHandle openWrite(String streamSegmentName) throws StreamSegmentException {
        return this.execute(streamSegmentName, () -> this.doOpenWrite(streamSegmentName));
    }

    public SegmentHandle create(String streamSegmentName) throws StreamSegmentException {
        return this.execute(streamSegmentName, () -> this.doCreate(streamSegmentName));
    }

    public void write(SegmentHandle handle, long offset, InputStream data, int length) throws StreamSegmentException {
        this.execute(handle.getSegmentName(), () -> this.doWrite(handle, offset, data, length));
    }

    public void seal(SegmentHandle handle) throws StreamSegmentException {
        this.execute(handle.getSegmentName(), () -> this.doSeal(handle));
    }

    public void unseal(SegmentHandle handle) throws StreamSegmentException {
        this.execute(handle.getSegmentName(), () -> this.doUnseal(handle));
    }

    public void concat(SegmentHandle targetHandle, long offset, String sourceSegment) throws StreamSegmentException {
        this.execute(targetHandle.getSegmentName(), () -> this.doConcat(targetHandle, offset, sourceSegment));
    }

    public void delete(SegmentHandle handle) throws StreamSegmentException {
        this.execute(handle.getSegmentName(), () -> this.doDelete(handle));
    }

    public void truncate(SegmentHandle handle, long offset) {
        throw new UnsupportedOperationException(this.getClass().getName() + " does not support Segment truncation.");
    }

    public boolean supportsTruncation() {
        return false;
    }

    @SuppressFBWarnings(value={"OS_OPEN_STREAM"}, justification="Rare operation. The leaked object is collected by GC. In case of a iterator in a for loop this would be fast.")
    public Iterator<SegmentProperties> listSegments() throws IOException {
        return Files.find(Paths.get(this.config.getRoot(), new String[0]), Integer.MAX_VALUE, (filePath, fileAttr) -> fileAttr.isRegularFile(), new FileVisitOption[0]).map(path -> this.getStreamSegmentInformation(this.config.getRoot(), (Path)path)).iterator();
    }

    private StreamSegmentInformation getStreamSegmentInformation(String root, Path path) {
        PosixFileAttributes attrs = Files.readAttributes(path.toAbsolutePath(), PosixFileAttributes.class, new LinkOption[0]);
        return StreamSegmentInformation.builder().name(Paths.get(root, new String[0]).relativize(path).toString()).length(attrs.size()).sealed(!attrs.permissions().contains((Object)PosixFilePermission.OWNER_WRITE)).lastModified(new ImmutableDate(attrs.creationTime().toMillis())).build();
    }

    public SyncStorage withReplaceSupport() {
        return this.config.isReplaceEnabled() ? new FileSystemStorageWithReplace(this.config) : this;
    }

    public void close() {
        this.closed.set(true);
    }

    @VisibleForTesting
    protected FileChannel getFileChannel(Path path, StandardOpenOption openOption) throws IOException {
        return FileChannel.open(path, openOption);
    }

    @VisibleForTesting
    protected long getFileSize(Path path) throws IOException {
        return Files.size(path);
    }

    protected SegmentHandle doOpenRead(String streamSegmentName) throws StreamSegmentNotExistsException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"openRead", (Object[])new Object[]{streamSegmentName});
        Path path = this.getPath(streamSegmentName);
        if (!Files.exists(path, new LinkOption[0])) {
            throw new StreamSegmentNotExistsException(streamSegmentName);
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"openRead", (long)traceId, (Object[])new Object[]{streamSegmentName});
        return FileSystemSegmentHandle.readHandle(streamSegmentName);
    }

    protected SegmentHandle doOpenWrite(String streamSegmentName) throws StreamSegmentNotExistsException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"openWrite", (Object[])new Object[]{streamSegmentName});
        Path path = this.getPath(streamSegmentName);
        if (!Files.exists(path, new LinkOption[0])) {
            throw new StreamSegmentNotExistsException(streamSegmentName);
        }
        if (Files.isWritable(path)) {
            LoggerHelpers.traceLeave((Logger)log, (String)"openWrite", (long)traceId, (Object[])new Object[0]);
            return FileSystemSegmentHandle.writeHandle(streamSegmentName);
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"openWrite", (long)traceId, (Object[])new Object[0]);
        return FileSystemSegmentHandle.readHandle(streamSegmentName);
    }

    private int doRead(SegmentHandle handle, long offset, byte[] buffer, int bufferOffset, int length) throws IOException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"read", (Object[])new Object[]{handle.getSegmentName(), offset, bufferOffset, length});
        Timer timer = new Timer();
        Path path = this.getPath(handle.getSegmentName());
        long fileSize = this.getFileSize(path);
        if (fileSize < offset) {
            throw new IllegalArgumentException(String.format("Reading at offset (%d) which is beyond the current size of segment (%d).", offset, fileSize));
        }
        try (FileChannel channel = this.getFileChannel(path, StandardOpenOption.READ);){
            int bytesRead;
            int totalBytesRead = 0;
            long readOffset = offset;
            do {
                ByteBuffer readBuffer = ByteBuffer.wrap(buffer, bufferOffset, length);
                bytesRead = channel.read(readBuffer, readOffset);
                bufferOffset += bytesRead;
                totalBytesRead += bytesRead;
                readOffset += (long)bytesRead;
            } while ((length -= bytesRead) != 0);
            FileSystemMetrics.READ_LATENCY.reportSuccessEvent(timer.getElapsed());
            FileSystemMetrics.READ_BYTES.add((long)totalBytesRead);
            LoggerHelpers.traceLeave((Logger)log, (String)"read", (long)traceId, (Object[])new Object[]{totalBytesRead});
            int n = totalBytesRead;
            return n;
        }
    }

    protected SegmentProperties doGetStreamSegmentInfo(String streamSegmentName) throws IOException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"getStreamSegmentInfo", (Object[])new Object[]{streamSegmentName});
        PosixFileAttributes attrs = Files.readAttributes(Paths.get(this.config.getRoot(), streamSegmentName), PosixFileAttributes.class, new LinkOption[0]);
        StreamSegmentInformation information = StreamSegmentInformation.builder().name(streamSegmentName).length(attrs.size()).sealed(!attrs.permissions().contains((Object)PosixFilePermission.OWNER_WRITE)).lastModified(new ImmutableDate(attrs.creationTime().toMillis())).build();
        LoggerHelpers.traceLeave((Logger)log, (String)"getStreamSegmentInfo", (long)traceId, (Object[])new Object[]{streamSegmentName});
        return information;
    }

    protected boolean doExists(String streamSegmentName) {
        return Files.exists(Paths.get(this.config.getRoot(), streamSegmentName), new LinkOption[0]);
    }

    protected SegmentHandle doCreate(String streamSegmentName) throws IOException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"create", (Object[])new Object[]{streamSegmentName});
        FileAttribute<Set<PosixFilePermission>> fileAttributes = PosixFilePermissions.asFileAttribute(FileSystemWrapper.READ_WRITE_PERMISSION);
        Path path = Paths.get(this.config.getRoot(), streamSegmentName);
        Path parent = path.getParent();
        assert (parent != null);
        Files.createDirectories(parent, new FileAttribute[0]);
        Files.createFile(path, fileAttributes);
        LoggerHelpers.traceLeave((Logger)log, (String)"create", (long)traceId, (Object[])new Object[0]);
        FileSystemMetrics.CREATE_COUNT.inc();
        return FileSystemSegmentHandle.writeHandle(streamSegmentName);
    }

    private Void doWrite(SegmentHandle handle, long offset, InputStream data, int length) throws IOException, StreamSegmentException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"write", (Object[])new Object[]{handle.getSegmentName(), offset, length});
        Timer timer = new Timer();
        if (handle.isReadOnly()) {
            throw new IllegalArgumentException("Write called on a readonly handle of segment " + handle.getSegmentName());
        }
        Path path = this.getPath(handle.getSegmentName());
        if (!this.isWritableFile(path)) {
            throw new StreamSegmentSealedException(handle.getSegmentName());
        }
        long totalBytesWritten = 0L;
        try (FileChannel channel = this.getFileChannel(path, StandardOpenOption.WRITE);){
            long fileSize = channel.size();
            if (fileSize != offset) {
                throw new BadOffsetException(handle.getSegmentName(), fileSize, offset);
            }
            ReadableByteChannel sourceChannel = Channels.newChannel(data);
            while (length != 0) {
                long bytesWritten = channel.transferFrom(sourceChannel, offset, length);
                assert (bytesWritten > 0L) : "Unable to make any progress transferring data.";
                offset += bytesWritten;
                totalBytesWritten += bytesWritten;
                length = (int)((long)length - bytesWritten);
            }
            channel.force(false);
        }
        FileSystemMetrics.WRITE_LATENCY.reportSuccessEvent(timer.getElapsed());
        FileSystemMetrics.WRITE_BYTES.add(totalBytesWritten);
        LoggerHelpers.traceLeave((Logger)log, (String)"write", (long)traceId, (Object[])new Object[0]);
        return null;
    }

    private boolean isWritableFile(Path path) throws IOException {
        PosixFileAttributes attrs = Files.readAttributes(path, PosixFileAttributes.class, new LinkOption[0]);
        return attrs.permissions().contains((Object)PosixFilePermission.OWNER_WRITE);
    }

    protected Void doSeal(SegmentHandle handle) throws IOException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"seal", (Object[])new Object[]{handle.getSegmentName()});
        if (handle.isReadOnly()) {
            throw new IllegalArgumentException(handle.getSegmentName());
        }
        Files.setPosixFilePermissions(this.getPath(handle.getSegmentName()), FileSystemWrapper.READ_ONLY_PERMISSION);
        LoggerHelpers.traceLeave((Logger)log, (String)"seal", (long)traceId, (Object[])new Object[0]);
        return null;
    }

    private Void doUnseal(SegmentHandle handle) throws IOException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"unseal", (Object[])new Object[]{handle.getSegmentName()});
        Files.setPosixFilePermissions(this.getPath(handle.getSegmentName()), FileSystemWrapper.READ_WRITE_PERMISSION);
        LoggerHelpers.traceLeave((Logger)log, (String)"unseal", (long)traceId, (Object[])new Object[0]);
        return null;
    }

    protected Void doConcat(SegmentHandle targetHandle, long offset, String sourceSegment) throws IOException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"concat", (Object[])new Object[]{targetHandle.getSegmentName(), offset, sourceSegment});
        Path sourcePath = this.getPath(sourceSegment);
        Path targetPath = this.getPath(targetHandle.getSegmentName());
        try (FileChannel targetChannel = FileChannel.open(targetPath, StandardOpenOption.WRITE);){
            Void void_;
            try (RandomAccessFile sourceFile = new RandomAccessFile(String.valueOf(sourcePath), "r");){
                long bytesTransferred;
                if (this.isWritableFile(sourcePath)) {
                    throw new IllegalStateException(String.format("Source segment (%s) is not sealed.", sourceSegment));
                }
                for (long length = this.getFileSize(sourcePath); length > 0L; length -= bytesTransferred) {
                    bytesTransferred = targetChannel.transferFrom(sourceFile.getChannel(), offset, length);
                    offset += bytesTransferred;
                }
                targetChannel.force(false);
                Files.delete(sourcePath);
                LoggerHelpers.traceLeave((Logger)log, (String)"concat", (long)traceId, (Object[])new Object[0]);
                void_ = null;
            }
            return void_;
        }
    }

    protected Void doDelete(SegmentHandle handle) throws IOException {
        Files.delete(this.getPath(handle.getSegmentName()));
        return null;
    }

    protected <R> R execute(String segmentName, Callable<R> operation) throws StreamSegmentException {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        try {
            return operation.call();
        }
        catch (Exception e) {
            return (R)this.throwException(segmentName, e);
        }
    }

    private <T> T throwException(String segmentName, Exception e) throws StreamSegmentException {
        if (this.isFileNotFoundException(e)) {
            throw new StreamSegmentNotExistsException(segmentName);
        }
        if (e instanceof FileAlreadyExistsException) {
            throw new StreamSegmentExistsException(segmentName);
        }
        if (e instanceof IndexOutOfBoundsException) {
            throw new IllegalArgumentException(e.getMessage());
        }
        if (e instanceof AccessControlException || e instanceof AccessDeniedException || e instanceof NonWritableChannelException) {
            throw new StreamSegmentSealedException(segmentName, (Throwable)e);
        }
        throw Exceptions.sneakyThrow((Throwable)e);
    }

    protected boolean isFileNotFoundException(Exception e) {
        return e instanceof NoSuchFileException || e instanceof FileNotFoundException || e instanceof StreamSegmentNotExistsException;
    }

    protected Path getPath(String segmentName) {
        return Paths.get(this.config.getRoot(), segmentName);
    }

    @VisibleForTesting
    static class FileSystemStorageWithReplace
    extends FileSystemStorage {
        @VisibleForTesting
        static final String TEMP_SUFFIX = ".replace.tmp";

        private FileSystemStorageWithReplace(FileSystemStorageConfig config) {
            super(config);
        }

        @Override
        protected SegmentHandle doCreate(String streamSegmentName) throws IOException {
            String tempSegmentName = this.getTempSegmentName(streamSegmentName);
            if (!super.doExists(streamSegmentName) && super.doExists(tempSegmentName)) {
                this.finalizeRename(this.getTempSegmentName(streamSegmentName), streamSegmentName);
                throw new FileAlreadyExistsException(streamSegmentName);
            }
            return super.doCreate(streamSegmentName);
        }

        @Override
        protected Void doDelete(SegmentHandle handle) throws IOException {
            String tempSegmentName = this.getTempSegmentName(handle.getSegmentName());
            if (super.doExists(tempSegmentName)) {
                Files.delete(this.getPath(tempSegmentName));
                try {
                    return super.doDelete(handle);
                }
                catch (IOException ex) {
                    return null;
                }
            }
            return super.doDelete(handle);
        }

        @Override
        protected SegmentHandle doOpenWrite(String streamSegmentName) throws StreamSegmentNotExistsException {
            return this.withRecovery(streamSegmentName, x$0 -> super.doOpenWrite(x$0));
        }

        @Override
        protected SegmentHandle doOpenRead(String streamSegmentName) throws StreamSegmentNotExistsException {
            return this.withRecovery(streamSegmentName, x$0 -> super.doOpenRead(x$0));
        }

        @Override
        protected SegmentProperties doGetStreamSegmentInfo(String streamSegmentName) throws IOException {
            return this.withRecovery(streamSegmentName, x$0 -> super.doGetStreamSegmentInfo(x$0));
        }

        @Override
        protected Void doConcat(SegmentHandle targetHandle, long offset, String sourceSegment) throws IOException {
            return this.withRecovery(sourceSegment, source -> super.doConcat(targetHandle, offset, source));
        }

        @Override
        protected boolean doExists(String streamSegmentName) {
            return super.doExists(streamSegmentName) || super.doExists(this.getTempSegmentName(streamSegmentName));
        }

        public boolean supportsReplace() {
            return true;
        }

        public void replace(@NonNull SegmentHandle segment, @NonNull BufferView contents) throws StreamSegmentException {
            if (segment == null) {
                throw new NullPointerException("segment is marked non-null but is null");
            }
            if (contents == null) {
                throw new NullPointerException("contents is marked non-null but is null");
            }
            String segmentName = segment.getSegmentName();
            this.execute(segment.getSegmentName(), () -> this.replaceExistingFile(segmentName, contents));
        }

        @Override
        public SyncStorage withReplaceSupport() {
            return this;
        }

        private <T, TEx extends Exception> T withRecovery(String segmentName, RecoverableAction<T, TEx> toExecute) throws TEx {
            try {
                return toExecute.apply(segmentName);
            }
            catch (Exception ex) {
                String tmpName = this.getTempSegmentName(segmentName);
                if (this.isFileNotFoundException(ex) && super.doExists(tmpName)) {
                    log.info("Incomplete replace operation detected for '{}'. Finalizing.", (Object)segmentName);
                    this.finalizeRename(tmpName, segmentName);
                    log.debug("Replace finalized for '{}'. Retrying operation.", (Object)segmentName);
                    return toExecute.apply(segmentName);
                }
                throw ex;
            }
        }

        private Void replaceExistingFile(String segmentName, BufferView contents) throws IOException, StreamSegmentException {
            boolean baseExists = super.doExists(segmentName);
            boolean shouldReseal = baseExists && super.getStreamSegmentInfo(segmentName).isSealed();
            String tmpSegmentName = this.getTempSegmentName(segmentName);
            if (super.doExists(tmpSegmentName)) {
                if (baseExists) {
                    log.info("Incomplete replace operation detected for '{}'. Deleting temp file before new replace attempt.", (Object)segmentName);
                    super.doDelete(super.doOpenWrite(tmpSegmentName));
                } else {
                    log.info("Incomplete replace operation detected for '{}'. Finalizing before new replace attempt.", (Object)segmentName);
                    this.finalizeRename(tmpSegmentName, segmentName);
                }
            } else if (!baseExists) {
                throw new StreamSegmentNotExistsException(segmentName);
            }
            SegmentHandle tmpHandle = super.doCreate(tmpSegmentName);
            try {
                super.doWrite(tmpHandle, 0L, contents.getReader(), contents.getLength());
                if (shouldReseal) {
                    super.doSeal(tmpHandle);
                }
            }
            catch (Exception ex) {
                log.warn("Unable to write to temporary file when attempting to replace '{}'. Original file has not been touched. Cleaning up.", (Object)segmentName, (Object)ex);
                super.doDelete(tmpHandle);
                throw ex;
            }
            assert (super.doGetStreamSegmentInfo(tmpSegmentName).getLength() == (long)contents.getLength());
            this.finalizeRename(tmpSegmentName, segmentName);
            assert (super.doGetStreamSegmentInfo(segmentName).getLength() == (long)contents.getLength());
            return null;
        }

        private void finalizeRename(String fromSegmentName, String toSegmentName) {
            File fromFile = this.getPath(fromSegmentName).toFile().getAbsoluteFile();
            File toFile = this.getPath(toSegmentName).toFile().getAbsoluteFile();
            boolean renamed = fromFile.renameTo(toFile);
            if (!renamed) {
                log.debug("File.renameTo unsuccessful for '{}' to '{}'. Attempting two-step replace.", (Object)fromSegmentName, (Object)toSegmentName);
                if (!toFile.delete() || !fromFile.renameTo(toFile)) {
                    throw new StreamSegmentReplaceException(fromSegmentName, toSegmentName);
                }
            }
            assert (!super.doExists(fromSegmentName));
            log.debug("Renamed '{}' to '{}'.", (Object)fromSegmentName, (Object)toSegmentName);
        }

        private String getTempSegmentName(String segmentName) {
            assert (!segmentName.endsWith(TEMP_SUFFIX));
            return segmentName + TEMP_SUFFIX;
        }

        private static class StreamSegmentReplaceException
        extends StreamSegmentException {
            StreamSegmentReplaceException(String fromSegmentName, String toSegmentName) {
                super(toSegmentName, String.format("Could not rename temporary Segment '%s' to '%s'.", fromSegmentName, toSegmentName));
            }
        }

        private static interface RecoverableAction<T, TEx extends Exception> {
            public T apply(String var1) throws TEx;
        }
    }
}

