/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.storage.hdfs;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.segmentstore.storage.chunklayer.BaseChunkStorage;
import io.pravega.segmentstore.storage.chunklayer.ChunkAlreadyExistsException;
import io.pravega.segmentstore.storage.chunklayer.ChunkHandle;
import io.pravega.segmentstore.storage.chunklayer.ChunkInfo;
import io.pravega.segmentstore.storage.chunklayer.ChunkNotFoundException;
import io.pravega.segmentstore.storage.chunklayer.ChunkStorageException;
import io.pravega.segmentstore.storage.chunklayer.ConcatArgument;
import io.pravega.segmentstore.storage.chunklayer.InvalidOffsetException;
import io.pravega.storage.hdfs.HDFSStorageConfig;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import lombok.Generated;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.ipc.RemoteException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HDFSChunkStorage
extends BaseChunkStorage {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(HDFSChunkStorage.class);
    private static final FsPermission READWRITE_PERMISSION = new FsPermission(FsAction.READ_WRITE, FsAction.NONE, FsAction.NONE);
    private static final FsPermission READONLY_PERMISSION = new FsPermission(FsAction.READ, FsAction.READ, FsAction.READ);
    private final HDFSStorageConfig config;
    private FileSystem fileSystem;
    private final AtomicBoolean closed;

    HDFSChunkStorage(HDFSStorageConfig config, Executor executor) {
        super(executor);
        Preconditions.checkNotNull((Object)config, (Object)"config");
        this.config = config;
        this.closed = new AtomicBoolean(false);
        this.initialize();
    }

    public boolean supportsConcat() {
        return true;
    }

    public boolean supportsAppend() {
        return true;
    }

    public boolean supportsTruncation() {
        return false;
    }

    public void close() {
        if (!this.closed.getAndSet(true) && this.fileSystem != null) {
            try {
                this.fileSystem.close();
                this.fileSystem = null;
            }
            catch (IOException e) {
                log.warn("Could not close the HDFS filesystem.", (Throwable)e);
            }
        }
        super.close();
    }

    protected ChunkInfo doGetInfo(String chunkName) throws ChunkStorageException {
        this.ensureInitializedAndNotClosed();
        try {
            FileStatus status = this.fileSystem.getFileStatus(this.getFilePath(chunkName));
            return ChunkInfo.builder().name(chunkName).length(status.getLen()).build();
        }
        catch (IOException e) {
            throw this.convertException(chunkName, "doGetInfo", e);
        }
    }

    protected ChunkHandle doCreate(String chunkName) throws ChunkStorageException {
        this.ensureInitializedAndNotClosed();
        try {
            Path fullPath = this.getFilePath(chunkName);
            this.fileSystem.create(fullPath, READWRITE_PERMISSION, false, 0, this.config.getReplication(), this.config.getBlockSize(), null).close();
            log.debug("Created '{}'.", (Object)fullPath);
            return ChunkHandle.writeHandle((String)chunkName);
        }
        catch (FileAlreadyExistsException e) {
            throw new ChunkAlreadyExistsException(chunkName, "HDFSChunkStorage::doCreate");
        }
        catch (IOException e) {
            throw this.convertException(chunkName, "doCreate", e);
        }
    }

    protected boolean checkExists(String chunkName) throws ChunkStorageException {
        this.ensureInitializedAndNotClosed();
        try {
            this.fileSystem.getFileStatus(this.getFilePath(chunkName));
            return true;
        }
        catch (FileNotFoundException e) {
            return false;
        }
        catch (IOException e) {
            throw this.convertException(chunkName, "checkExists", e);
        }
    }

    protected void doDelete(ChunkHandle handle) throws ChunkStorageException {
        this.ensureInitializedAndNotClosed();
        try {
            Path path = this.getFilePath(handle.getChunkName());
            if (!this.fileSystem.delete(path, true)) {
                this.checkFileExists(handle.getChunkName(), "doDelete");
            }
        }
        catch (IOException e) {
            throw this.convertException(handle.getChunkName(), "doDelete", e);
        }
    }

    protected ChunkHandle doOpenRead(String chunkName) throws ChunkStorageException {
        this.ensureInitializedAndNotClosed();
        this.checkFileExists(chunkName, "doOpenRead");
        return ChunkHandle.readHandle((String)chunkName);
    }

    protected ChunkHandle doOpenWrite(String chunkName) throws ChunkStorageException {
        this.ensureInitializedAndNotClosed();
        try {
            FileStatus status = this.fileSystem.getFileStatus(this.getFilePath(chunkName));
            if (status.getPermission().getUserAction() == FsAction.READ) {
                return ChunkHandle.readHandle((String)chunkName);
            }
            return ChunkHandle.writeHandle((String)chunkName);
        }
        catch (IOException e) {
            throw this.convertException(chunkName, "doOpenWrite", e);
        }
    }

    protected int doRead(ChunkHandle handle, long fromOffset, int length, byte[] buffer, int bufferOffset) throws ChunkStorageException, NullPointerException, IndexOutOfBoundsException {
        this.ensureInitializedAndNotClosed();
        try (FSDataInputStream stream = this.fileSystem.open(this.getFilePath(handle.getChunkName()));){
            stream.readFully(fromOffset, buffer, bufferOffset, length);
        }
        catch (EOFException e) {
            throw new IllegalArgumentException(String.format("Reading at offset (%d) which is beyond the current size of chunk.", fromOffset));
        }
        catch (IOException e) {
            throw this.convertException(handle.getChunkName(), "doRead", e);
        }
        return length;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected int doWrite(ChunkHandle handle, long offset, int length, InputStream data) throws ChunkStorageException {
        this.ensureInitializedAndNotClosed();
        try (FSDataOutputStream stream = this.fileSystem.append(this.getFilePath(handle.getChunkName()));){
            if (stream.getPos() != offset) {
                throw new InvalidOffsetException(handle.getChunkName(), stream.getPos(), offset, "doWrite");
            }
            if (length == 0) {
                int n = 0;
                return n;
            }
            IOUtils.copyBytes((InputStream)data, (OutputStream)stream, (long)length, (boolean)false);
            stream.flush();
            return length;
        }
        catch (IOException e) {
            throw this.convertException(handle.getChunkName(), "doWrite", e);
        }
    }

    public int doConcat(ConcatArgument[] chunks) throws ChunkStorageException {
        this.ensureInitializedAndNotClosed();
        int length = 0;
        try {
            Path[] sources = new Path[chunks.length - 1];
            this.fileSystem.truncate(this.getFilePath(chunks[0].getName()), chunks[0].getLength());
            for (int i = 1; i < chunks.length; ++i) {
                long chunkLength = chunks[i].getLength();
                this.fileSystem.truncate(this.getFilePath(chunks[i].getName()), chunkLength);
                sources[i - 1] = this.getFilePath(chunks[i].getName());
                length = (int)((long)length + chunkLength);
            }
            this.fileSystem.concat(this.getFilePath(chunks[0].getName()), sources);
        }
        catch (IOException e) {
            throw this.convertException(chunks[0].getName(), "doConcat", e);
        }
        return length;
    }

    protected void doSetReadOnly(ChunkHandle handle, boolean isReadOnly) throws ChunkStorageException {
        try {
            this.fileSystem.setPermission(this.getFilePath(handle.getChunkName()), isReadOnly ? READONLY_PERMISSION : READWRITE_PERMISSION);
        }
        catch (IOException e) {
            throw this.convertException(handle.getChunkName(), "doSetReadOnly", e);
        }
    }

    public void initialize() {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)((Object)this));
        Preconditions.checkState((this.fileSystem == null ? 1 : 0) != 0, (Object)"HDFSStorage has already been initialized.");
        Configuration conf = new Configuration();
        conf.set("fs.default.name", this.config.getHdfsHostURL());
        conf.set("fs.default.fs", this.config.getHdfsHostURL());
        conf.set("fs.hdfs.impl", DistributedFileSystem.class.getName());
        conf.set("fs.hdfs.impl.disable.cache", "true");
        if (!this.config.isReplaceDataNodesOnFailure()) {
            conf.set("dfs.client.block.write.replace-datanode-on-failure.policy", "NEVER");
        }
        this.fileSystem = this.openFileSystem(conf);
        log.info("Initialized (HDFSHost = '{}'", (Object)this.config.getHdfsHostURL());
    }

    private FileSystem openFileSystem(Configuration conf) throws IOException {
        return FileSystem.get((Configuration)conf);
    }

    private void ensureInitializedAndNotClosed() {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)((Object)this));
        Preconditions.checkState((this.fileSystem != null ? 1 : 0) != 0, (Object)"HDFSStorage is not initialized.");
    }

    private String getPathPrefix(String chunkName) {
        return this.config.getHdfsRoot() + "/" + chunkName;
    }

    private Path getFilePath(String chunkName) {
        Preconditions.checkState((chunkName != null && chunkName.length() > 0 ? 1 : 0) != 0, (Object)"chunkName must be non-null and non-empty");
        return new Path(this.getPathPrefix(chunkName));
    }

    private void checkFileExists(String chunkName, String message) throws ChunkStorageException {
        try {
            this.fileSystem.getFileStatus(this.getFilePath(chunkName));
        }
        catch (IOException e) {
            throw this.convertException(chunkName, message, e);
        }
    }

    private ChunkStorageException convertException(String chunkName, String message, IOException e) {
        if (e instanceof ChunkStorageException) {
            return (ChunkStorageException)e;
        }
        if (e instanceof RemoteException) {
            e = ((RemoteException)e).unwrapRemoteException();
        }
        if (e instanceof FileNotFoundException) {
            return new ChunkNotFoundException(chunkName, message, (Throwable)e);
        }
        return new ChunkStorageException(chunkName, message, (Throwable)e);
    }
}

