/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.common.util.btree;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.TimeoutTimer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.AsyncIterator;
import io.pravega.common.util.BufferView;
import io.pravega.common.util.BufferViewComparator;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.IllegalDataFormatException;
import io.pravega.common.util.btree.BTreePage;
import io.pravega.common.util.btree.EntryIterator;
import io.pravega.common.util.btree.PageCollection;
import io.pravega.common.util.btree.PageEntry;
import io.pravega.common.util.btree.PagePointer;
import io.pravega.common.util.btree.PageWrapper;
import io.pravega.common.util.btree.SearchResult;
import io.pravega.common.util.btree.Statistics;
import io.pravega.common.util.btree.UpdateablePageCollection;
import java.beans.ConstructorProperties;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.annotation.concurrent.NotThreadSafe;
import lombok.Generated;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class BTreeIndex {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(BTreeIndex.class);
    private static final int INDEX_VALUE_LENGTH = 18;
    private static final int FOOTER_LENGTH = 12;
    private static final BufferViewComparator KEY_COMPARATOR = BufferViewComparator.create();
    private final BTreePage.Config indexPageConfig;
    private final BTreePage.Config leafPageConfig;
    private final ReadPage read;
    private final WritePages write;
    private final GetLength getLength;
    private volatile IndexState state;
    private volatile boolean maintainStatistics;
    private volatile Statistics statistics;
    private final Executor executor;
    private final String traceObjectId;

    public BTreeIndex(int maxPageSize, int keyLength, int valueLength, @NonNull ReadPage readPage, @NonNull WritePages writePages, @NonNull GetLength getLength, boolean maintainStatistics, @NonNull Executor executor, String traceObjectId) {
        if (readPage == null) {
            throw new NullPointerException("readPage is marked non-null but is null");
        }
        if (writePages == null) {
            throw new NullPointerException("writePages is marked non-null but is null");
        }
        if (getLength == null) {
            throw new NullPointerException("getLength is marked non-null but is null");
        }
        if (executor == null) {
            throw new NullPointerException("executor is marked non-null but is null");
        }
        this.read = readPage;
        this.write = writePages;
        this.getLength = getLength;
        this.maintainStatistics = maintainStatistics;
        this.executor = executor;
        this.traceObjectId = traceObjectId;
        this.indexPageConfig = new BTreePage.Config(keyLength, 18, maxPageSize, true);
        this.leafPageConfig = new BTreePage.Config(keyLength, valueLength, maxPageSize, false);
        this.state = null;
    }

    public boolean isInitialized() {
        return this.state != null;
    }

    public long getIndexLength() {
        IndexState s = this.state;
        return s == null ? -1L : s.length;
    }

    public CompletableFuture<Void> initialize(Duration timeout) {
        if (this.isInitialized()) {
            log.warn("{}: Reinitializing.", (Object)this.traceObjectId);
        }
        TimeoutTimer timer = new TimeoutTimer(timeout);
        return this.getLength.apply(timer.getRemaining()).thenCompose(indexInfo -> {
            if (indexInfo.getIndexLength() <= 12L) {
                this.setState(indexInfo.getIndexLength(), -1L, 0);
                this.statistics = this.maintainStatistics ? Statistics.EMPTY : null;
                return CompletableFuture.completedFuture(null);
            }
            long footerOffset = indexInfo.getRootPointer() >= 0L ? indexInfo.getRootPointer() : this.getFooterOffset(indexInfo.getIndexLength());
            return ((CompletableFuture)((CompletableFuture)this.read.apply(footerOffset, 12, false, timer.getRemaining()).thenAcceptAsync(footer -> this.initialize((ByteArraySegment)footer, footerOffset, indexInfo.getIndexLength()), this.executor)).thenCompose(v -> this.loadStatistics(timer.getRemaining()))).thenRun(() -> log.info("{}: Initialized. State = {}, Stats = {}.", new Object[]{this.traceObjectId, this.state, this.statistics}));
        });
    }

    private void initialize(ByteArraySegment footer, long footerOffset, long indexLength) {
        int rootPageLength;
        if (footer.getLength() != 12) {
            throw new IllegalDataFormatException(String.format("[%s] Wrong footer length. Expected %s, actual %s.", this.traceObjectId, 12, footer.getLength()), new Object[0]);
        }
        long rootPageOffset = this.getRootPageOffset(footer);
        if (rootPageOffset + (long)(rootPageLength = this.getRootPageLength(footer)) > footerOffset) {
            throw new IllegalDataFormatException(String.format("[%s] Wrong footer information. RootPage Offset (%s) + Length (%s) exceeds Footer Offset (%s).", this.traceObjectId, rootPageOffset, rootPageLength, footerOffset), new Object[0]);
        }
        this.setState(indexLength, rootPageOffset, rootPageLength);
    }

    private CompletableFuture<Void> loadStatistics(Duration timeout) {
        if (!this.maintainStatistics) {
            return CompletableFuture.completedFuture(null);
        }
        IndexState s = this.state;
        if (s.rootPageOffset == -1L) {
            this.statistics = Statistics.EMPTY;
            log.debug("{}: Resetting stats due to index empty.", (Object)this.traceObjectId);
            return CompletableFuture.completedFuture(null);
        }
        long statsOffset = s.rootPageOffset + (long)s.rootPageLength;
        int statsLength = (int)Math.min(s.length - 12L - statsOffset, Integer.MAX_VALUE);
        if (statsLength <= 0) {
            this.maintainStatistics = false;
            this.statistics = null;
            log.debug("{}: Not loading stats due to legacy index not supporting stats.", (Object)this.traceObjectId);
            return CompletableFuture.completedFuture(null);
        }
        return this.read.apply(statsOffset, statsLength, false, timeout).thenAccept(data -> {
            try {
                this.statistics = (Statistics)Statistics.SERIALIZER.deserialize((BufferView)data);
            }
            catch (IOException ex) {
                throw new CompletionException(ex);
            }
        });
    }

    public CompletableFuture<ByteArraySegment> get(@NonNull ByteArraySegment key, @NonNull Duration timeout) {
        if (key == null) {
            throw new NullPointerException("key is marked non-null but is null");
        }
        if (timeout == null) {
            throw new NullPointerException("timeout is marked non-null but is null");
        }
        this.ensureInitialized();
        TimeoutTimer timer = new TimeoutTimer(timeout);
        PageCollection pageCollection = new PageCollection(this.state.length);
        return this.locatePage(key, pageCollection, timer).thenApplyAsync(page -> page.getPage().searchExact(key), this.executor);
    }

    public CompletableFuture<List<ByteArraySegment>> get(@NonNull List<ByteArraySegment> keys, @NonNull Duration timeout) {
        if (keys == null) {
            throw new NullPointerException("keys is marked non-null but is null");
        }
        if (timeout == null) {
            throw new NullPointerException("timeout is marked non-null but is null");
        }
        if (keys.size() == 1) {
            return this.get(keys.get(0), timeout).thenApply(Collections::singletonList);
        }
        this.ensureInitialized();
        TimeoutTimer timer = new TimeoutTimer(timeout);
        PageCollection pageCollection = new PageCollection(this.state.length);
        List gets = keys.stream().map(key -> this.locatePage((ByteArraySegment)key, pageCollection, timer).thenApplyAsync(page -> page.getPage().searchExact((ByteArraySegment)key), this.executor)).collect(Collectors.toList());
        return Futures.allOfWithResults(gets);
    }

    public CompletableFuture<Long> update(@NonNull Collection<PageEntry> entries, @NonNull Duration timeout) {
        if (entries == null) {
            throw new NullPointerException("entries is marked non-null but is null");
        }
        if (timeout == null) {
            throw new NullPointerException("timeout is marked non-null but is null");
        }
        this.ensureInitialized();
        TimeoutTimer timer = new TimeoutTimer(timeout);
        Iterator<PageEntry> toUpdate = entries.stream().sorted((e1, e2) -> KEY_COMPARATOR.compare((ArrayView)e1.getKey(), (ArrayView)e2.getKey())).iterator();
        return this.applyUpdates(toUpdate, timer).thenComposeAsync(pageCollection -> ((CompletableFuture)this.loadSmallestOffsetPage((PageCollection)pageCollection, timer).thenRun(() -> this.processModifiedPages((UpdateablePageCollection)pageCollection))).thenComposeAsync(v -> this.writePages((UpdateablePageCollection)pageCollection, timer.getRemaining()), this.executor), this.executor);
    }

    public AsyncIterator<List<PageEntry>> iterator(@NonNull ByteArraySegment firstKey, boolean firstKeyInclusive, @NonNull ByteArraySegment lastKey, boolean lastKeyInclusive, Duration fetchTimeout) {
        if (firstKey == null) {
            throw new NullPointerException("firstKey is marked non-null but is null");
        }
        if (lastKey == null) {
            throw new NullPointerException("lastKey is marked non-null but is null");
        }
        this.ensureInitialized();
        return new EntryIterator(firstKey, firstKeyInclusive, lastKey, lastKeyInclusive, this::locatePage, this.state.length, fetchTimeout);
    }

    private CompletableFuture<UpdateablePageCollection> applyUpdates(Iterator<PageEntry> updates, TimeoutTimer timer) {
        UpdateablePageCollection pageCollection = new UpdateablePageCollection(this.state.length);
        AtomicReference<Object> lastPage = new AtomicReference<Object>(null);
        ArrayList lastPageUpdates = new ArrayList();
        return Futures.loop(updates::hasNext, () -> {
            PageEntry next = (PageEntry)updates.next();
            return this.locatePage(next.getKey(), pageCollection, timer).thenAccept(page -> {
                PageWrapper last = (PageWrapper)lastPage.get();
                if (page != last) {
                    if (last != null) {
                        last.setEntryCountDelta(last.getPage().update(lastPageUpdates));
                    }
                    lastPage.set(page);
                    lastPageUpdates.clear();
                }
                lastPageUpdates.add(next);
            });
        }, (Executor)this.executor).thenApplyAsync(v -> {
            PageWrapper last = (PageWrapper)lastPage.get();
            if (last != null) {
                last.setEntryCountDelta(last.getPage().update(lastPageUpdates));
            }
            return pageCollection;
        }, this.executor);
    }

    private CompletableFuture<?> loadSmallestOffsetPage(PageCollection pageCollection, TimeoutTimer timer) {
        if (pageCollection.getCount() <= 1) {
            return CompletableFuture.completedFuture(null);
        }
        long minOffset = this.calculateMinOffset(pageCollection.getRootPage());
        return this.locatePage(page -> this.getPagePointer(minOffset, (BTreePage)page), page -> !page.isIndexPage() || page.getOffset() == minOffset, pageCollection, timer);
    }

    private void processModifiedPages(UpdateablePageCollection pageCollection) {
        ArrayList<PageWrapper> currentBatch = new ArrayList<PageWrapper>();
        pageCollection.collectLeafPages(currentBatch);
        while (!currentBatch.isEmpty()) {
            HashSet<Long> parents = new HashSet<Long>();
            for (PageWrapper p : currentBatch) {
                List<BTreePage> splitResult;
                PageModificationContext context = new PageModificationContext(p, pageCollection);
                if (p.needsFirstKeyUpdate()) {
                    this.updateFirstKey(context);
                }
                if ((splitResult = p.getPage().splitIfNecessary()) != null) {
                    this.processSplitPage(splitResult, context);
                } else {
                    this.processModifiedPage(context);
                }
                PageWrapper parentPage = p.getParent();
                if (parentPage == null && splitResult != null) {
                    parentPage = PageWrapper.wrapNew(this.createEmptyIndexPage(), null, null);
                    pageCollection.insert(parentPage);
                }
                if (parentPage == null) continue;
                this.processParentPage(parentPage, context);
                parents.add(parentPage.getOffset());
            }
            currentBatch.clear();
            pageCollection.collectPages(parents, currentBatch);
        }
    }

    private void updateFirstKey(PageModificationContext context) {
        BTreePage page = context.getPageWrapper().getPage();
        assert (page.getConfig().isIndexPage()) : "expected index page";
        if (page.getCount() > 0) {
            page.setFirstKey(this.generateMinKey());
        }
    }

    private void processSplitPage(List<BTreePage> splitResult, PageModificationContext context) {
        PageWrapper originalPage = context.getPageWrapper();
        for (int i = 0; i < splitResult.size(); ++i) {
            PageWrapper processedPage;
            ByteArraySegment newPageKey;
            BTreePage page = splitResult.get(i);
            if (i == 0) {
                originalPage.setPage(page);
                newPageKey = originalPage.getPageKey();
                context.getPageCollection().complete(originalPage);
                processedPage = originalPage;
            } else {
                newPageKey = page.getKeyAt(0);
                processedPage = PageWrapper.wrapNew(page, originalPage.getParent(), new PagePointer(newPageKey, -1L, page.getLength()));
                context.getPageCollection().insert(processedPage);
                context.getPageCollection().complete(processedPage);
            }
            long newOffset = processedPage.getOffset();
            long minOffset = this.calculateMinOffset(processedPage);
            processedPage.setMinOffset(minOffset);
            context.updatePagePointer(new PagePointer(newPageKey, newOffset, page.getLength(), minOffset));
        }
    }

    private void processModifiedPage(PageModificationContext context) {
        PageWrapper page = context.getPageWrapper();
        boolean emptyPage = page.getPage().getCount() == 0;
        ByteArraySegment pageKey = page.getPageKey();
        if (emptyPage && page.getParent() != null) {
            context.getPageCollection().remove(page);
            context.setDeletedPageKey(pageKey);
        } else {
            if (emptyPage && page.getPage().getConfig().isIndexPage()) {
                page.setPage(this.createEmptyLeafPage());
            }
            context.pageCollection.complete(page);
            page.setMinOffset(this.calculateMinOffset(page));
            context.updatePagePointer(new PagePointer(pageKey, page.getOffset(), page.getPage().getLength(), page.getMinOffset()));
        }
    }

    private long calculateMinOffset(PageWrapper pageWrapper) {
        long min = pageWrapper.getOffset();
        if (!pageWrapper.isIndexPage()) {
            return min;
        }
        BTreePage page = pageWrapper.getPage();
        int count = page.getCount();
        for (int pos = 0; pos < count; ++pos) {
            long ppMinOffset = this.deserializePointerMinOffset(page.getValueAt(pos));
            min = Math.min(min, ppMinOffset);
        }
        return min;
    }

    private void processParentPage(PageWrapper parentPage, PageModificationContext context) {
        if (context.getDeletedPageKey() != null) {
            parentPage.getPage().update(Collections.singletonList(PageEntry.noValue(context.getDeletedPageKey())));
            parentPage.markNeedsFirstKeyUpdate();
        } else {
            List<PageEntry> toUpdate = context.getUpdatedPagePointers().stream().map(pp -> new PageEntry(pp.getKey(), this.serializePointer((PagePointer)pp))).collect(Collectors.toList());
            parentPage.getPage().update(toUpdate);
        }
    }

    private CompletableFuture<PageWrapper> locatePage(ByteArraySegment key, PageCollection pageCollection, TimeoutTimer timer) {
        Preconditions.checkArgument((key.getLength() == this.leafPageConfig.getKeyLength() ? 1 : 0) != 0, (Object)"Invalid key length.");
        Preconditions.checkArgument((pageCollection.getIndexLength() <= this.state.length ? 1 : 0) != 0, (Object)"Unexpected PageCollection.IndexLength.");
        if (this.state.rootPageOffset == -1L && pageCollection.getCount() == 0) {
            return CompletableFuture.completedFuture(pageCollection.insert(PageWrapper.wrapNew(this.createEmptyLeafPage(), null, null)));
        }
        return this.locatePage(page -> this.getPagePointer(key, (BTreePage)page), page -> !page.isIndexPage(), pageCollection, timer);
    }

    private CompletableFuture<PageWrapper> locatePage(Function<BTreePage, PagePointer> getChildPointer, Predicate<PageWrapper> found, PageCollection pageCollection, TimeoutTimer timer) {
        AtomicReference<PagePointer> pagePointer = new AtomicReference<PagePointer>(new PagePointer(null, this.state.rootPageOffset, this.state.rootPageLength));
        CompletableFuture<PageWrapper> result = new CompletableFuture<PageWrapper>();
        AtomicReference<Object> parentPage = new AtomicReference<Object>(null);
        Futures.loop(() -> !result.isDone(), () -> this.fetchPage((PagePointer)pagePointer.get(), (PageWrapper)parentPage.get(), pageCollection, timer.getRemaining()).thenAccept(page -> {
            if (found.test((PageWrapper)page)) {
                result.complete((PageWrapper)page);
            } else {
                PagePointer childPointer = (PagePointer)getChildPointer.apply(page.getPage());
                pagePointer.set(childPointer);
                parentPage.set(page);
            }
        }), (Executor)this.executor).exceptionally(ex -> {
            result.completeExceptionally((Throwable)ex);
            return null;
        });
        return result;
    }

    private CompletableFuture<PageWrapper> fetchPage(PagePointer pagePointer, PageWrapper parentPage, PageCollection pageCollection, Duration timeout) {
        PageWrapper fromCache = pageCollection.get(pagePointer.getOffset());
        if (fromCache != null) {
            return CompletableFuture.completedFuture(fromCache);
        }
        return this.readPage(pagePointer.getOffset(), pagePointer.getLength(), timeout).thenApply(data -> {
            if (data.getLength() != pagePointer.getLength()) {
                throw new IllegalDataFormatException(String.format("Requested page of length %s from offset %s, got a page of length %s.", pagePointer.getLength(), pagePointer.getOffset(), data.getLength()), new Object[0]);
            }
            BTreePage.Config pageConfig = BTreePage.isIndexPage(data) ? this.indexPageConfig : this.leafPageConfig;
            return pageCollection.insert(PageWrapper.wrapExisting(new BTreePage(pageConfig, (ByteArraySegment)data), parentPage, pagePointer));
        });
    }

    private BTreePage createEmptyLeafPage() {
        return new BTreePage(this.leafPageConfig);
    }

    private BTreePage createEmptyIndexPage() {
        return new BTreePage(this.indexPageConfig);
    }

    private PagePointer getPagePointer(ByteArraySegment key, BTreePage page) {
        int pos;
        SearchResult searchResult = page.search(key, 0);
        int n = pos = searchResult.isExactMatch() ? searchResult.getPosition() : searchResult.getPosition() - 1;
        assert (pos >= 0) : "negative pos";
        return this.deserializePointer(page.getValueAt(pos), page.getKeyAt(pos));
    }

    private PagePointer getPagePointer(long minOffset, BTreePage page) {
        int count = page.getCount();
        for (int pos = 0; pos < count; ++pos) {
            long ppMinOffset = this.deserializePointerMinOffset(page.getValueAt(pos));
            if (ppMinOffset != minOffset) continue;
            return this.deserializePointer(page.getValueAt(pos), page.getKeyAt(pos));
        }
        return null;
    }

    private ByteArraySegment serializePointer(PagePointer pointer) {
        assert (pointer.getLength() <= Short.MAX_VALUE) : "PagePointer.length too large";
        ByteArraySegment result = new ByteArraySegment(new byte[this.indexPageConfig.getValueLength()]);
        result.setLong(0, pointer.getOffset());
        result.setShort(8, (short)pointer.getLength());
        result.setLong(10, pointer.getMinOffset());
        return result;
    }

    private PagePointer deserializePointer(ByteArraySegment serialization, ByteArraySegment pageKey) {
        long pageOffset = serialization.getLong(0);
        short pageLength = serialization.getShort(8);
        long minOffset = this.deserializePointerMinOffset(serialization);
        return new PagePointer(pageKey, pageOffset, pageLength, minOffset);
    }

    private long deserializePointerMinOffset(ByteArraySegment serialization) {
        return serialization.getLong(10);
    }

    private ByteArraySegment generateMinKey() {
        byte[] result = new byte[this.indexPageConfig.getKeyLength()];
        return new ByteArraySegment(result);
    }

    private CompletableFuture<ByteArraySegment> readPage(long offset, int length, Duration timeout) {
        return this.read.apply(offset, length, true, timeout);
    }

    private CompletableFuture<Long> writePages(UpdateablePageCollection pageCollection, Duration timeout) {
        Statistics newStats;
        IndexState state = this.state;
        Preconditions.checkState((state != null ? 1 : 0) != 0, (Object)"Cannot write without fetching the state first.");
        ArrayList<WritePage> pages = new ArrayList<WritePage>();
        ArrayList<Long> oldOffsets = new ArrayList<Long>();
        long offset = state.length;
        PageWrapper lastPage = null;
        for (PageWrapper p : pageCollection.getPagesSortedByOffset()) {
            if (offset >= 0L) {
                Preconditions.checkArgument((p.getOffset() == offset ? 1 : 0) != 0, (String)"Expecting Page offset %s, found %s.", (long)offset, (long)p.getOffset());
            }
            pages.add(new WritePage(offset, p.getPage().getContents(), true));
            if (p.getPointer() != null && p.getPointer().getOffset() >= 0L) {
                oldOffsets.add(p.getPointer().getOffset());
                if (this.maintainStatistics && p.getParent() == null) {
                    oldOffsets.add(p.getPointer().getOffset() + (long)p.getPointer().getLength());
                }
            }
            offset = p.getOffset() + (long)p.getPage().getLength();
            lastPage = p;
        }
        Preconditions.checkArgument((lastPage != null && lastPage.getParent() == null ? 1 : 0) != 0, (Object)"Last page to be written is not the root page");
        Preconditions.checkArgument((pageCollection.getIndexLength() == offset ? 1 : 0) != 0, (Object)"IndexLength mismatch.");
        if (this.maintainStatistics) {
            newStats = this.statistics.update(pageCollection.getEntryCountDelta(), pageCollection.getPageCountDelta());
            WritePage sp = new WritePage(offset, Statistics.SERIALIZER.serialize(newStats), false);
            pages.add(sp);
            offset += (long)sp.getContents().getLength();
        } else {
            newStats = null;
        }
        long footerOffset = offset;
        pages.add(new WritePage(footerOffset, this.getFooter(lastPage.getOffset(), lastPage.getPage().getLength()), false));
        long oldFooterOffset = this.getFooterOffset(state.length);
        if (oldFooterOffset >= 0L) {
            oldOffsets.add(oldFooterOffset);
        }
        pageCollection.collectRemovedPageOffsets(oldOffsets);
        long rootOffset = lastPage.getOffset();
        int rootLength = lastPage.getPage().getContents().getLength();
        long rootMinOffset = lastPage.getMinOffset();
        assert (rootMinOffset >= 0L) : "root.MinOffset not set";
        return this.write.apply(pages, oldOffsets, rootMinOffset, timeout).thenApply(indexLength -> {
            this.statistics = newStats;
            this.setState((long)indexLength, rootOffset, rootLength);
            assert (footerOffset == this.getFooterOffset((long)indexLength));
            return footerOffset;
        });
    }

    private void setState(long length, long rootPageOffset, int rootPageLength) {
        this.state = new IndexState(length, rootPageOffset, rootPageLength);
        log.debug("{}: IndexState: {}, Stats: {}.", new Object[]{this.traceObjectId, this.state, this.statistics});
    }

    private long getFooterOffset(long indexLength) {
        return indexLength - 12L;
    }

    private ByteArraySegment getFooter(long rootPageOffset, int rootPageLength) {
        ByteArraySegment result = new ByteArraySegment(new byte[12]);
        result.setLong(0, rootPageOffset);
        result.setInt(8, rootPageLength);
        return result;
    }

    private long getRootPageOffset(ByteArraySegment footer) {
        return footer.getLong(0);
    }

    private int getRootPageLength(ByteArraySegment footer) {
        return footer.getInt(8);
    }

    private void ensureInitialized() {
        Preconditions.checkArgument((boolean)this.isInitialized(), (Object)"BTreeIndex is not initialized.");
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public static BTreeIndexBuilder builder() {
        return new BTreeIndexBuilder();
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public Statistics getStatistics() {
        return this.statistics;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public static class BTreeIndexBuilder {
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private int maxPageSize;
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private int keyLength;
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private int valueLength;
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private ReadPage readPage;
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private WritePages writePages;
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private GetLength getLength;
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private boolean maintainStatistics;
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private Executor executor;
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private String traceObjectId;

        @SuppressFBWarnings(justification="generated code")
        @Generated
        BTreeIndexBuilder() {
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndexBuilder maxPageSize(int maxPageSize) {
            this.maxPageSize = maxPageSize;
            return this;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndexBuilder keyLength(int keyLength) {
            this.keyLength = keyLength;
            return this;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndexBuilder valueLength(int valueLength) {
            this.valueLength = valueLength;
            return this;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndexBuilder readPage(@NonNull ReadPage readPage) {
            if (readPage == null) {
                throw new NullPointerException("readPage is marked non-null but is null");
            }
            this.readPage = readPage;
            return this;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndexBuilder writePages(@NonNull WritePages writePages) {
            if (writePages == null) {
                throw new NullPointerException("writePages is marked non-null but is null");
            }
            this.writePages = writePages;
            return this;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndexBuilder getLength(@NonNull GetLength getLength) {
            if (getLength == null) {
                throw new NullPointerException("getLength is marked non-null but is null");
            }
            this.getLength = getLength;
            return this;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndexBuilder maintainStatistics(boolean maintainStatistics) {
            this.maintainStatistics = maintainStatistics;
            return this;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndexBuilder executor(@NonNull Executor executor) {
            if (executor == null) {
                throw new NullPointerException("executor is marked non-null but is null");
            }
            this.executor = executor;
            return this;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndexBuilder traceObjectId(String traceObjectId) {
            this.traceObjectId = traceObjectId;
            return this;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BTreeIndex build() {
            return new BTreeIndex(this.maxPageSize, this.keyLength, this.valueLength, this.readPage, this.writePages, this.getLength, this.maintainStatistics, this.executor, this.traceObjectId);
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String toString() {
            return "BTreeIndex.BTreeIndexBuilder(maxPageSize=" + this.maxPageSize + ", keyLength=" + this.keyLength + ", valueLength=" + this.valueLength + ", readPage=" + this.readPage + ", writePages=" + this.writePages + ", getLength=" + this.getLength + ", maintainStatistics=" + this.maintainStatistics + ", executor=" + this.executor + ", traceObjectId=" + this.traceObjectId + ")";
        }
    }

    public static class WritePage {
        private final long offset;
        private final ByteArraySegment contents;
        private final boolean cache;

        @ConstructorProperties(value={"offset", "contents", "cache"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public WritePage(long offset, ByteArraySegment contents, boolean cache) {
            this.offset = offset;
            this.contents = contents;
            this.cache = cache;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public long getOffset() {
            return this.offset;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public ByteArraySegment getContents() {
            return this.contents;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean isCache() {
            return this.cache;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof WritePage)) {
                return false;
            }
            WritePage other = (WritePage)o;
            if (!other.canEqual(this)) {
                return false;
            }
            if (this.getOffset() != other.getOffset()) {
                return false;
            }
            ByteArraySegment this$contents = this.getContents();
            ByteArraySegment other$contents = other.getContents();
            if (this$contents == null ? other$contents != null : !this$contents.equals(other$contents)) {
                return false;
            }
            return this.isCache() == other.isCache();
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        protected boolean canEqual(Object other) {
            return other instanceof WritePage;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            long $offset = this.getOffset();
            result = result * 59 + (int)($offset >>> 32 ^ $offset);
            ByteArraySegment $contents = this.getContents();
            result = result * 59 + ($contents == null ? 43 : $contents.hashCode());
            result = result * 59 + (this.isCache() ? 79 : 97);
            return result;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String toString() {
            return "BTreeIndex.WritePage(offset=" + this.getOffset() + ", contents=" + this.getContents() + ", cache=" + this.isCache() + ")";
        }
    }

    @FunctionalInterface
    public static interface WritePages {
        public CompletableFuture<Long> apply(List<WritePage> var1, Collection<Long> var2, long var3, Duration var5);
    }

    @FunctionalInterface
    public static interface ReadPage {
        public CompletableFuture<ByteArraySegment> apply(long var1, int var3, boolean var4, Duration var5);
    }

    public static class IndexInfo {
        public static final IndexInfo EMPTY = new IndexInfo(0L, -1L);
        private final long indexLength;
        private final long rootPointer;

        public String toString() {
            return String.format("IndexLength = %d, RootPointer = %d", this.indexLength, this.rootPointer);
        }

        @ConstructorProperties(value={"indexLength", "rootPointer"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public IndexInfo(long indexLength, long rootPointer) {
            this.indexLength = indexLength;
            this.rootPointer = rootPointer;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public long getIndexLength() {
            return this.indexLength;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public long getRootPointer() {
            return this.rootPointer;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof IndexInfo)) {
                return false;
            }
            IndexInfo other = (IndexInfo)o;
            if (!other.canEqual(this)) {
                return false;
            }
            if (this.getIndexLength() != other.getIndexLength()) {
                return false;
            }
            return this.getRootPointer() == other.getRootPointer();
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        protected boolean canEqual(Object other) {
            return other instanceof IndexInfo;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            long $indexLength = this.getIndexLength();
            result = result * 59 + (int)($indexLength >>> 32 ^ $indexLength);
            long $rootPointer = this.getRootPointer();
            result = result * 59 + (int)($rootPointer >>> 32 ^ $rootPointer);
            return result;
        }
    }

    @FunctionalInterface
    public static interface GetLength {
        public CompletableFuture<IndexInfo> apply(Duration var1);
    }

    private static class IndexState {
        private final long length;
        private final long rootPageOffset;
        private final int rootPageLength;

        public String toString() {
            return String.format("Length = %s, RootOffset = %s, RootLength = %s", this.length, this.rootPageOffset, this.rootPageLength);
        }

        @ConstructorProperties(value={"length", "rootPageOffset", "rootPageLength"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public IndexState(long length, long rootPageOffset, int rootPageLength) {
            this.length = length;
            this.rootPageOffset = rootPageOffset;
            this.rootPageLength = rootPageLength;
        }
    }

    private static class PageModificationContext {
        private final PageWrapper pageWrapper;
        private final UpdateablePageCollection pageCollection;
        private final List<PagePointer> updatedPagePointers = new ArrayList<PagePointer>();
        private ByteArraySegment deletedPageKey;

        void updatePagePointer(PagePointer pp) {
            this.updatedPagePointers.add(pp);
        }

        @ConstructorProperties(value={"pageWrapper", "pageCollection"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public PageModificationContext(PageWrapper pageWrapper, UpdateablePageCollection pageCollection) {
            this.pageWrapper = pageWrapper;
            this.pageCollection = pageCollection;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public PageWrapper getPageWrapper() {
            return this.pageWrapper;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public UpdateablePageCollection getPageCollection() {
            return this.pageCollection;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public List<PagePointer> getUpdatedPagePointers() {
            return this.updatedPagePointers;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public ByteArraySegment getDeletedPageKey() {
            return this.deletedPageKey;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public void setDeletedPageKey(ByteArraySegment deletedPageKey) {
            this.deletedPageKey = deletedPageKey;
        }
    }
}

