/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.common.util.btree;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.BufferViewComparator;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.IllegalDataFormatException;
import io.pravega.common.util.btree.PageEntry;
import io.pravega.common.util.btree.SearchResult;
import java.beans.ConstructorProperties;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import javax.annotation.concurrent.NotThreadSafe;
import lombok.Generated;
import lombok.NonNull;

@NotThreadSafe
class BTreePage {
    private static final byte CURRENT_VERSION = 0;
    private static final int VERSION_OFFSET = 0;
    private static final int VERSION_LENGTH = 1;
    private static final int FLAGS_OFFSET = 1;
    private static final int FLAGS_LENGTH = 1;
    private static final byte FLAG_NONE = 0;
    private static final byte FLAG_INDEX_PAGE = 1;
    private static final int ID_OFFSET = 2;
    private static final int ID_LENGTH = 4;
    private static final int COUNT_OFFSET = 6;
    private static final int COUNT_LENGTH = 4;
    private static final int DATA_OFFSET = 10;
    private static final int FOOTER_LENGTH = 4;
    private static final BufferViewComparator KEY_COMPARATOR = BufferViewComparator.create();
    private static final Random ID_GENERATOR = new Random();
    private ByteArraySegment contents;
    private ByteArraySegment header;
    private ByteArraySegment data;
    private ByteArraySegment footer;
    private final Config config;
    private int count;

    BTreePage(Config config) {
        this(config, new ByteArraySegment(new byte[14]), false);
        this.formatHeaderAndFooter(this.count, ID_GENERATOR.nextInt());
    }

    BTreePage(Config config, ByteArraySegment contents) {
        this(config, contents, true);
    }

    private BTreePage(Config config, int count, ByteArraySegment data) {
        this(config, new ByteArraySegment(new byte[10 + data.getLength() + 4]), false);
        Preconditions.checkArgument((count * config.entryLength == data.getLength() ? 1 : 0) != 0, (Object)"Unexpected data length given the count.");
        this.formatHeaderAndFooter(count, ID_GENERATOR.nextInt());
        this.data.copyFrom((ArrayView)data, 0, data.getLength());
    }

    private BTreePage(@NonNull Config config, @NonNull ByteArraySegment contents, boolean validate) {
        int footerId;
        int headerId;
        if (config == null) {
            throw new NullPointerException("config is marked non-null but is null");
        }
        if (contents == null) {
            throw new NullPointerException("contents is marked non-null but is null");
        }
        this.config = config;
        this.contents = contents;
        this.header = contents.slice(0, 10);
        this.data = contents.slice(10, contents.getLength() - 10 - 4);
        this.footer = contents.slice(contents.getLength() - 4, 4);
        if (validate && (headerId = this.getHeaderId()) != (footerId = this.getFooterId())) {
            throw new IllegalDataFormatException("Invalid Page Format (id mismatch). HeaderId=%s, FooterId=%s.", new Object[]{headerId, footerId});
        }
        this.count = this.header.getInt(6);
    }

    private void formatHeaderAndFooter(int itemCount, int id) {
        this.header.set(0, (byte)0);
        this.header.set(1, this.getFlags(this.config.isIndexPage ? (byte)1 : 0));
        this.setHeaderId(id);
        this.setCount(itemCount);
        this.setFooterId(id);
    }

    static boolean isIndexPage(@NonNull ByteArraySegment pageContents) {
        int footerId;
        if (pageContents == null) {
            throw new NullPointerException("pageContents is marked non-null but is null");
        }
        int headerId = pageContents.getInt(2);
        if (headerId != (footerId = pageContents.getInt(pageContents.getLength() - 4))) {
            throw new IllegalDataFormatException("Invalid Page Format (id mismatch). HeaderId=%s, FooterId=%s.", new Object[]{headerId, footerId});
        }
        byte flags = pageContents.get(1);
        return (flags & 1) == 1;
    }

    int getLength() {
        return this.contents.getLength();
    }

    ByteArraySegment getValueAt(int pos) {
        Preconditions.checkElementIndex((int)pos, (int)this.getCount(), (String)"pos must be non-negative and smaller than the number of items.");
        return this.data.slice(pos * this.config.entryLength + this.config.keyLength, this.config.valueLength);
    }

    ByteArraySegment getKeyAt(int pos) {
        Preconditions.checkElementIndex((int)pos, (int)this.getCount(), (String)"pos must be non-negative and smaller than the number of items.");
        return this.data.slice(pos * this.config.entryLength, this.config.keyLength);
    }

    void setFirstKey(ByteArraySegment newKey) {
        Preconditions.checkState((this.getCount() > 0 ? 1 : 0) != 0, (Object)"BTreePage is empty. Cannot set first key.");
        Preconditions.checkArgument((newKey.getLength() == this.config.getKeyLength() ? 1 : 0) != 0, (Object)"Incorrect key length.");
        Preconditions.checkArgument((KEY_COMPARATOR.compare((ArrayView)newKey, (ArrayView)this.getKeyAt(0)) <= 0 ? 1 : 0) != 0, (Object)"Replacement first Key must be smaller than or equal to the existing first key.");
        this.data.copyFrom((ArrayView)newKey, 0, newKey.getLength());
    }

    PageEntry getEntryAt(int pos) {
        Preconditions.checkElementIndex((int)pos, (int)this.getCount(), (String)"pos must be non-negative and smaller than the number of items.");
        return new PageEntry(this.data.slice(pos * this.config.entryLength, this.config.keyLength), this.data.slice(pos * this.config.entryLength + this.config.keyLength, this.config.valueLength));
    }

    List<PageEntry> getEntries(int firstIndex, int lastIndex) {
        Preconditions.checkArgument((firstIndex <= lastIndex ? 1 : 0) != 0, (Object)"firstIndex must be smaller than or equal to lastIndex.");
        ArrayList<PageEntry> result = new ArrayList<PageEntry>();
        for (int i = firstIndex; i <= lastIndex; ++i) {
            result.add(this.getEntryAt(i));
        }
        return result;
    }

    List<BTreePage> splitIfNecessary() {
        if (this.contents.getLength() <= this.config.getMaxPageSize()) {
            return null;
        }
        int maxDataLength = (this.config.getMaxPageSize() - this.header.getLength() - this.footer.getLength()) / this.config.entryLength * this.config.entryLength;
        int remainingPageCount = (int)Math.ceil((double)this.data.getLength() / (double)maxDataLength);
        ArrayList<BTreePage> result = new ArrayList<BTreePage>(remainingPageCount);
        int readIndex = 0;
        int remainingItems = this.getCount();
        while (remainingPageCount > 0) {
            int itemsPerPage = remainingItems / remainingPageCount;
            ByteArraySegment splitPageData = this.data.slice(readIndex, itemsPerPage * this.config.entryLength);
            result.add(new BTreePage(this.config, itemsPerPage, splitPageData));
            readIndex += splitPageData.getLength();
            --remainingPageCount;
            remainingItems -= itemsPerPage;
        }
        assert (readIndex == this.data.getLength()) : "did not copy everything";
        return result;
    }

    int update(@NonNull List<PageEntry> entries) {
        if (entries == null) {
            throw new NullPointerException("entries is marked non-null but is null");
        }
        if (entries.isEmpty()) {
            return 0;
        }
        ChangeInfo ci = this.applyUpdates(entries);
        if (ci.changes.isEmpty()) {
            return 0;
        }
        BTreePage newPage = this.applyInsertsAndRemovals(ci);
        this.header = newPage.header;
        this.data = newPage.data;
        this.contents = newPage.contents;
        this.footer = newPage.footer;
        int delta = newPage.count - this.count;
        this.count = newPage.count;
        return delta;
    }

    ByteArraySegment searchExact(@NonNull ByteArraySegment key) {
        if (key == null) {
            throw new NullPointerException("key is marked non-null but is null");
        }
        SearchResult pos = this.search(key, 0);
        if (!pos.isExactMatch()) {
            return null;
        }
        return this.getValueAt(pos.getPosition());
    }

    SearchResult search(@NonNull ByteArraySegment key, int startPos) {
        if (key == null) {
            throw new NullPointerException("key is marked non-null but is null");
        }
        int endPos = this.getCount();
        Preconditions.checkArgument((startPos <= endPos ? 1 : 0) != 0, (Object)"startPos must be non-negative and smaller than the number of items.");
        while (startPos < endPos) {
            int midPos = startPos + (endPos - startPos) / 2;
            int c = KEY_COMPARATOR.compare(key.array(), key.arrayOffset(), this.data.array(), this.data.arrayOffset() + midPos * this.config.entryLength, this.config.keyLength);
            if (c == 0) {
                return new SearchResult(midPos, true);
            }
            if (c < 0) {
                endPos = midPos;
                continue;
            }
            startPos = midPos + 1;
        }
        return new SearchResult(startPos, false);
    }

    private ChangeInfo applyUpdates(List<PageEntry> entries) {
        ArrayList<Map.Entry<Integer, PageEntry>> changes = new ArrayList<Map.Entry<Integer, PageEntry>>();
        int removeCount = 0;
        int lastPos = 0;
        ByteArraySegment lastKey = null;
        for (PageEntry e : entries) {
            SearchResult searchResult;
            if (e.getKey().getLength() != this.config.keyLength || e.hasValue() && e.getValue().getLength() != this.config.valueLength) {
                throw new IllegalDataFormatException("Found an entry with unexpected Key or Value length.", new Object[0]);
            }
            if (lastKey != null) {
                Preconditions.checkArgument((KEY_COMPARATOR.compare((ArrayView)lastKey, (ArrayView)e.getKey()) < 0 ? 1 : 0) != 0, (Object)"Entries must be sorted by key and no duplicates are allowed.");
            }
            if ((searchResult = this.search(e.getKey(), lastPos)).isExactMatch()) {
                if (e.hasValue()) {
                    this.setValueAtPosition(searchResult.getPosition(), e.getValue());
                } else {
                    changes.add(new AbstractMap.SimpleImmutableEntry<Integer, Object>(searchResult.getPosition(), null));
                    ++removeCount;
                }
            } else if (e.hasValue()) {
                changes.add(new AbstractMap.SimpleImmutableEntry<Integer, PageEntry>(searchResult.getPosition(), e));
            }
            lastPos = searchResult.getPosition();
            lastKey = e.getKey();
        }
        return new ChangeInfo(changes, changes.size() - removeCount, removeCount);
    }

    private BTreePage applyInsertsAndRemovals(ChangeInfo ci) {
        int newCount = this.getCount() + ci.insertCount - ci.deleteCount;
        BTreePage newPage = new BTreePage(this.config, new ByteArraySegment(new byte[10 + newCount * this.config.entryLength + 4]), false);
        newPage.formatHeaderAndFooter(newCount, this.getHeaderId());
        int readIndex = 0;
        int writeIndex = 0;
        for (Map.Entry<Integer, PageEntry> e : ci.changes) {
            int entryIndex = e.getKey() * this.config.entryLength;
            if (entryIndex > readIndex) {
                int length = entryIndex - readIndex;
                assert (length % this.config.entryLength == 0);
                newPage.data.copyFrom((ArrayView)this.data, readIndex, writeIndex, length);
                writeIndex += length;
            }
            PageEntry entryContents = e.getValue();
            readIndex = entryIndex;
            if (entryContents != null) {
                newPage.setEntryAtIndex(writeIndex, entryContents);
                writeIndex += this.config.entryLength;
                continue;
            }
            readIndex += this.config.getEntryLength();
        }
        if (readIndex < this.data.getLength()) {
            int length = this.data.getLength() - readIndex;
            newPage.data.copyFrom((ArrayView)this.data, readIndex, writeIndex, length);
        }
        return newPage;
    }

    private void setCount(int itemCount) {
        this.header.setInt(6, itemCount);
        this.count = itemCount;
    }

    private void setValueAtPosition(int pos, ByteArraySegment value) {
        Preconditions.checkElementIndex((int)pos, (int)this.getCount(), (String)"pos must be non-negative and smaller than the number of items.");
        Preconditions.checkArgument((value.getLength() == this.config.valueLength ? 1 : 0) != 0, (Object)"Given value has incorrect length.");
        this.data.copyFrom((ArrayView)value, pos * this.config.entryLength + this.config.keyLength, value.getLength());
    }

    private void setEntryAtIndex(int dataIndex, PageEntry entry) {
        Preconditions.checkElementIndex((int)dataIndex, (int)this.data.getLength(), (String)"dataIndex must be non-negative and smaller than the size of the data.");
        Preconditions.checkArgument((entry.getKey().getLength() == this.config.keyLength ? 1 : 0) != 0, (Object)"Given entry key has incorrect length.");
        Preconditions.checkArgument((entry.getValue().getLength() == this.config.valueLength ? 1 : 0) != 0, (Object)"Given entry value has incorrect length.");
        this.data.copyFrom((ArrayView)entry.getKey(), dataIndex, entry.getKey().getLength());
        this.data.copyFrom((ArrayView)entry.getValue(), dataIndex + this.config.keyLength, entry.getValue().getLength());
    }

    private byte getFlags(byte ... flags) {
        byte result = 0;
        for (byte f : flags) {
            result = (byte)(result | f);
        }
        return result;
    }

    int getHeaderId() {
        return this.header.getInt(2);
    }

    private int getFooterId() {
        return this.footer.getInt(0);
    }

    private void setHeaderId(int id) {
        this.header.setInt(2, id);
    }

    private void setFooterId(int id) {
        this.footer.setInt(0, id);
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public ByteArraySegment getContents() {
        return this.contents;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public Config getConfig() {
        return this.config;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public int getCount() {
        return this.count;
    }

    static class Config {
        private static final int MAX_PAGE_SIZE = Short.MAX_VALUE;
        private final int keyLength;
        private final int valueLength;
        private final int entryLength;
        private final int maxPageSize;
        private final boolean isIndexPage;

        Config(int keyLength, int valueLength, int maxPageSize, boolean isIndexPage) {
            Preconditions.checkArgument((maxPageSize <= Short.MAX_VALUE ? 1 : 0) != 0, (String)"maxPageSize must be at most %s, given %s.", (int)Short.MAX_VALUE, (int)maxPageSize);
            Preconditions.checkArgument((keyLength > 0 ? 1 : 0) != 0, (Object)"keyLength must be a positive integer.");
            Preconditions.checkArgument((valueLength > 0 ? 1 : 0) != 0, (Object)"valueLength must be a positive integer.");
            Preconditions.checkArgument((keyLength + valueLength + 10 + 4 <= maxPageSize ? 1 : 0) != 0, (Object)"maxPageSize must be able to fit at least one entry.");
            this.keyLength = keyLength;
            this.valueLength = valueLength;
            this.entryLength = this.keyLength + this.valueLength;
            this.maxPageSize = maxPageSize;
            this.isIndexPage = isIndexPage;
        }

        @ConstructorProperties(value={"keyLength", "valueLength", "entryLength", "maxPageSize", "isIndexPage"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public Config(int keyLength, int valueLength, int entryLength, int maxPageSize, boolean isIndexPage) {
            this.keyLength = keyLength;
            this.valueLength = valueLength;
            this.entryLength = entryLength;
            this.maxPageSize = maxPageSize;
            this.isIndexPage = isIndexPage;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getKeyLength() {
            return this.keyLength;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getValueLength() {
            return this.valueLength;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getEntryLength() {
            return this.entryLength;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getMaxPageSize() {
            return this.maxPageSize;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean isIndexPage() {
            return this.isIndexPage;
        }
    }

    private static class ChangeInfo {
        private final List<Map.Entry<Integer, PageEntry>> changes;
        private final int insertCount;
        private final int deleteCount;

        @ConstructorProperties(value={"changes", "insertCount", "deleteCount"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public ChangeInfo(List<Map.Entry<Integer, PageEntry>> changes, int insertCount, int deleteCount) {
            this.changes = changes;
            this.insertCount = insertCount;
            this.deleteCount = deleteCount;
        }
    }
}

