/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.storage.cache;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.pravega.common.Exceptions;
import io.pravega.common.util.BufferView;
import io.pravega.segmentstore.storage.cache.CacheCorruptedException;
import io.pravega.segmentstore.storage.cache.CacheLayout;
import io.pravega.segmentstore.storage.cache.IncorrectCacheEntryLengthException;
import java.beans.ConstructorProperties;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Stack;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import lombok.Generated;
import lombok.NonNull;

@ThreadSafe
class DirectMemoryBuffer
implements AutoCloseable {
    private final int id;
    private final CacheLayout layout;
    private final ByteBufAllocator allocator;
    @GuardedBy(value="this")
    private ByteBuf buf;
    @GuardedBy(value="this")
    private int usedBlockCount;

    DirectMemoryBuffer(int bufferId, @NonNull ByteBufAllocator allocator, @NonNull CacheLayout layout) {
        if (allocator == null) {
            throw new NullPointerException("allocator is marked non-null but is null");
        }
        if (layout == null) {
            throw new NullPointerException("layout is marked non-null but is null");
        }
        Preconditions.checkArgument((bufferId >= 0 && bufferId < layout.maxBufferCount() ? 1 : 0) != 0);
        this.allocator = allocator;
        this.layout = layout;
        this.id = bufferId;
        this.usedBlockCount = 1;
    }

    @Override
    public synchronized void close() {
        if (this.buf != null && this.buf.refCnt() > 0) {
            this.buf.release();
            this.buf = null;
        }
        this.usedBlockCount = -1;
    }

    synchronized int getUsedBlockCount() {
        return this.usedBlockCount;
    }

    synchronized boolean isAllocated() {
        return this.buf != null;
    }

    synchronized boolean hasCapacity() {
        return this.usedBlockCount > 0 && this.usedBlockCount < this.layout.blocksPerBuffer();
    }

    public synchronized String toString() {
        return String.format("Id=%d, UsedBlockCount=%d", this.id, this.usedBlockCount);
    }

    synchronized WriteResult write(BufferView data, int predecessorAddress) {
        if (this.usedBlockCount >= this.layout.blocksPerBuffer()) {
            return null;
        }
        ByteBuf metadataBuf = this.getMetadataBlock();
        long blockMetadata = metadataBuf.getLong(0);
        int blockId = this.layout.getNextFreeBlockId(blockMetadata);
        assert (blockId != 0);
        int firstBlockId = blockId;
        int dataOffset = 0;
        try {
            do {
                int bufIndex = blockId * this.layout.blockMetadataSize();
                blockMetadata = metadataBuf.getLong(bufIndex);
                assert (!this.layout.isUsedBlock(blockMetadata));
                int blockLength = Math.min(data.getLength() - dataOffset, this.layout.blockSize());
                if (blockLength > 0) {
                    data.slice(dataOffset, blockLength).copyTo(this.getWriteableBlock(blockId, 0));
                    dataOffset += blockLength;
                }
                long metadata = this.layout.newBlockMetadata(0, blockLength, predecessorAddress);
                metadataBuf.setLong(bufIndex, metadata);
                ++this.usedBlockCount;
                predecessorAddress = this.layout.calculateAddress(this.id, blockId);
            } while ((blockId = this.layout.getNextFreeBlockId(blockMetadata)) != 0 && dataOffset < data.getLength());
        }
        catch (Throwable ex) {
            if (!Exceptions.mustRethrow((Throwable)ex)) {
                this.rollbackWrite(this.layout.getBlockId(predecessorAddress), blockId);
            }
            throw ex;
        }
        blockMetadata = metadataBuf.getLong(0);
        blockMetadata = this.layout.setNextFreeBlockId(blockMetadata, blockId);
        metadataBuf.setLong(0, blockMetadata);
        return new WriteResult(dataOffset, predecessorAddress, firstBlockId);
    }

    synchronized int tryAppend(int blockId, int expectedLength, BufferView data) {
        this.validateBlockId(blockId, false);
        ByteBuf metadataBuf = this.getMetadataBlock();
        int bufIndex = blockId * this.layout.blockMetadataSize();
        long blockMetadata = metadataBuf.getLong(bufIndex);
        Preconditions.checkArgument((blockId != 0 && this.layout.isUsedBlock(blockMetadata) ? 1 : 0) != 0, (Object)"Given blockId is not allocated.");
        int blockLength = this.layout.getLength(blockMetadata);
        if (blockLength != expectedLength) {
            throw new IncorrectCacheEntryLengthException(String.format("Incorrect last block length. Expected %s, given %s.", blockLength, expectedLength));
        }
        int maxLength = this.layout.blockSize() - blockLength;
        if (maxLength < data.getLength()) {
            data = data.slice(0, maxLength);
        }
        data.copyTo(this.getWriteableBlock(blockId, blockLength));
        blockMetadata = this.layout.setLength(blockMetadata, blockLength += data.getLength());
        metadataBuf.setLong(bufIndex, blockMetadata);
        return data.getLength();
    }

    synchronized int read(int blockId, List<ByteBuf> readBuffers) {
        this.validateBlockId(blockId, true);
        ByteBuf metadataBuf = this.getMetadataBlock();
        while (blockId != 0) {
            int bufIndex = blockId * this.layout.blockMetadataSize();
            long blockMetadata = metadataBuf.getLong(bufIndex);
            if (this.layout.isUsedBlock(blockMetadata)) {
                int blockLength = this.layout.getLength(blockMetadata);
                if (!readBuffers.isEmpty() && blockLength < this.layout.blockSize()) {
                    throw new CacheCorruptedException(String.format("Buffer %s, Block %s: Non-full, non-terminal block (length=%s).", this.id, blockId, blockLength));
                }
                int predecessorAddress = this.layout.getPredecessorAddress(blockMetadata);
                readBuffers.add(this.getReadOnlyDataBlock(blockId, Math.min(blockLength, this.layout.blockSize())));
                if (predecessorAddress == 0 || this.layout.getBufferId(predecessorAddress) != this.id) {
                    return predecessorAddress;
                }
                blockId = this.layout.getBlockId(predecessorAddress);
                assert (blockId >= 1 && blockId < this.layout.blocksPerBuffer());
                continue;
            }
            if (readBuffers.isEmpty()) {
                return 0;
            }
            throw new CacheCorruptedException(String.format("Buffer %s, Block %s: Unallocated.", this.id, blockId));
        }
        return 0;
    }

    synchronized DeleteResult delete(int blockId) {
        this.validateBlockId(blockId, false);
        ByteBuf metadataBuf = this.getMetadataBlock();
        int deletedLength = 0;
        int predecessorAddress = 0;
        Stack<Integer> freedBlocks = new Stack<Integer>();
        while (blockId != 0) {
            long blockMetadata = metadataBuf.getLong(blockId * this.layout.blockMetadataSize());
            if (this.layout.isUsedBlock(blockMetadata)) {
                freedBlocks.push(blockId);
                predecessorAddress = this.layout.getPredecessorAddress(blockMetadata);
                deletedLength += this.layout.getLength(blockMetadata);
                if (predecessorAddress == 0 || this.layout.getBufferId(predecessorAddress) != this.id) break;
                blockId = this.layout.getBlockId(predecessorAddress);
                assert (blockId >= 1 && blockId < this.layout.blocksPerBuffer());
                continue;
            }
            blockId = 0;
        }
        this.deallocateBlocks(freedBlocks, metadataBuf);
        return new DeleteResult(deletedLength, predecessorAddress);
    }

    @GuardedBy(value="this")
    private void rollbackWrite(int lastWrittenBlockId, int nextFreeBlockId) {
        ByteBuf metadataBuf = this.getMetadataBlock();
        int blockId = lastWrittenBlockId;
        while (blockId != 0) {
            int bufIndex = blockId * this.layout.blockMetadataSize();
            long blockMetadata = metadataBuf.getLong(bufIndex);
            int predecessorAddress = this.layout.getPredecessorAddress(blockMetadata);
            blockMetadata = this.layout.setNextFreeBlockId(this.layout.emptyBlockMetadata(), nextFreeBlockId);
            metadataBuf.setLong(bufIndex, blockMetadata);
            nextFreeBlockId = blockId;
            --this.usedBlockCount;
            blockId = this.layout.getBlockId(predecessorAddress);
            if (this.layout.getBufferId(predecessorAddress) == this.id) continue;
            break;
        }
    }

    @GuardedBy(value="this")
    private void deallocateBlocks(Stack<Integer> blocks, ByteBuf metadataBuf) {
        long prevMetadata;
        int freedBlockId;
        if (blocks.size() == 0) {
            return;
        }
        int prevBlockId = freedBlockId = blocks.pop().intValue();
        do {
            prevMetadata = metadataBuf.getLong(--prevBlockId * this.layout.blockMetadataSize());
        } while (prevBlockId > 0 && this.layout.isUsedBlock(prevMetadata));
        while (freedBlockId != 0) {
            int prevNextFreeBlockId = this.layout.getNextFreeBlockId(prevMetadata);
            while (prevNextFreeBlockId != 0 && prevNextFreeBlockId < freedBlockId) {
                prevBlockId = prevNextFreeBlockId;
                prevMetadata = metadataBuf.getLong(prevBlockId * this.layout.blockMetadataSize());
                prevNextFreeBlockId = this.layout.getNextFreeBlockId(prevMetadata);
            }
            long blockMetadata = this.layout.setNextFreeBlockId(this.layout.emptyBlockMetadata(), prevNextFreeBlockId);
            prevMetadata = this.layout.setNextFreeBlockId(prevMetadata, freedBlockId);
            metadataBuf.setLong(prevBlockId * this.layout.blockMetadataSize(), prevMetadata);
            metadataBuf.setLong(freedBlockId * this.layout.blockMetadataSize(), blockMetadata);
            freedBlockId = blocks.size() == 0 ? 0 : blocks.pop();
            --this.usedBlockCount;
        }
    }

    @GuardedBy(value="this")
    private ByteBuf getMetadataBlock() {
        return this.getBuf().slice(0, this.layout.blockSize());
    }

    @GuardedBy(value="this")
    private ByteBuffer getWriteableBlock(int blockIndex, int blockOffset) {
        assert (blockOffset >= 0 && blockOffset <= this.layout.blockSize());
        return this.getBuf().nioBuffer(blockIndex * this.layout.blockSize() + blockOffset, this.layout.blockSize() - blockOffset);
    }

    @GuardedBy(value="this")
    private ByteBuf getReadOnlyDataBlock(int blockIndex, int blockLength) {
        assert (blockLength <= this.layout.blockSize());
        return this.getBuf().slice(blockIndex * this.layout.blockSize(), blockLength).asReadOnly();
    }

    @GuardedBy(value="this")
    private ByteBuf getBuf() {
        if (this.buf == null) {
            Exceptions.checkNotClosed((this.usedBlockCount < 0 ? 1 : 0) != 0, (Object)this);
            assert (this.usedBlockCount == 1);
            this.buf = this.allocator.directBuffer(this.layout.bufferSize(), this.layout.bufferSize());
            ByteBuf metadataBuf = this.getMetadataBlock();
            metadataBuf.writerIndex(0);
            for (int blockId = 0; blockId < this.layout.blocksPerBuffer(); ++blockId) {
                long m = this.layout.emptyBlockMetadata();
                m = this.layout.setNextFreeBlockId(m, blockId == this.layout.blocksPerBuffer() - 1 ? 0 : blockId + 1);
                metadataBuf.writeLong(m);
            }
        }
        return this.buf;
    }

    @GuardedBy(value="this")
    private void validateBlockId(int blockId, boolean canBeEmpty) {
        Preconditions.checkState((canBeEmpty || this.usedBlockCount > 1 ? 1 : 0) != 0, (Object)"Empty buffer.");
        Preconditions.checkArgument((blockId >= 1 && blockId < this.layout.blocksPerBuffer() ? 1 : 0) != 0, (String)"blockId must be a number in the interval [1, %s).", (int)this.layout.blocksPerBuffer());
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public int getId() {
        return this.id;
    }

    class WriteResult {
        private final int writtenLength;
        private final int lastBlockAddress;
        @VisibleForTesting
        private final int firstBlockId;

        public String toString() {
            return String.format("FirstBlockId=%d, LastBlockId=%d, WrittenLength=%d", this.firstBlockId, DirectMemoryBuffer.this.layout.getBlockId(this.lastBlockAddress), this.writtenLength);
        }

        @ConstructorProperties(value={"writtenLength", "lastBlockAddress", "firstBlockId"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private WriteResult(int writtenLength, int lastBlockAddress, int firstBlockId) {
            this.writtenLength = writtenLength;
            this.lastBlockAddress = lastBlockAddress;
            this.firstBlockId = firstBlockId;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getWrittenLength() {
            return this.writtenLength;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getLastBlockAddress() {
            return this.lastBlockAddress;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getFirstBlockId() {
            return this.firstBlockId;
        }
    }

    class DeleteResult {
        private final int deletedLength;
        private final int predecessorAddress;

        public String toString() {
            return String.format("DeletedLength = %d, Previous = %s", this.deletedLength, DirectMemoryBuffer.this.layout.getAddressString(this.predecessorAddress));
        }

        @ConstructorProperties(value={"deletedLength", "predecessorAddress"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private DeleteResult(int deletedLength, int predecessorAddress) {
            this.deletedLength = deletedLength;
            this.predecessorAddress = predecessorAddress;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getDeletedLength() {
            return this.deletedLength;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getPredecessorAddress() {
            return this.predecessorAddress;
        }
    }
}

