/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.storage.cache;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.CompositeByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;
import io.pravega.common.Exceptions;
import io.pravega.common.util.BufferView;
import io.pravega.segmentstore.storage.cache.CacheFullException;
import io.pravega.segmentstore.storage.cache.CacheLayout;
import io.pravega.segmentstore.storage.cache.CacheMetrics;
import io.pravega.segmentstore.storage.cache.CacheState;
import io.pravega.segmentstore.storage.cache.CacheStorage;
import io.pravega.segmentstore.storage.cache.DirectMemoryBuffer;
import io.pravega.shared.protocol.netty.ByteBufWrapper;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import lombok.NonNull;

@ThreadSafe
public class DirectMemoryCache
implements CacheStorage {
    @VisibleForTesting
    static final int MAX_CLEANUP_ATTEMPTS = 5;
    private final CacheLayout layout;
    private final DirectMemoryBuffer[] buffers;
    @GuardedBy(value="availableBufferIds")
    private final ArrayDeque<Integer> availableBufferIds;
    @GuardedBy(value="availableBufferIds")
    private final ArrayDeque<Integer> unallocatedBufferIds;
    private final AtomicBoolean closed;
    private final AtomicLong storedBytes;
    private final AtomicReference<Supplier<Boolean>> tryCleanup;
    private final AtomicInteger retryDelayBaseMillis;
    private final CacheMetrics metrics = new CacheMetrics();

    public DirectMemoryCache(long maxSizeBytes) {
        this(new CacheLayout.DefaultLayout(), maxSizeBytes);
    }

    @VisibleForTesting
    DirectMemoryCache(@NonNull CacheLayout layout, long maxSizeBytes) {
        if (layout == null) {
            throw new NullPointerException("layout is marked non-null but is null");
        }
        Preconditions.checkArgument((maxSizeBytes > 0L && maxSizeBytes <= 0x4000000000L ? 1 : 0) != 0, (String)"maxSizeBytes must be a positive number less than %s.", (long)0x4000000000L);
        maxSizeBytes = this.adjustMaxSizeIfNeeded(maxSizeBytes, layout);
        this.layout = layout;
        this.tryCleanup = new AtomicReference<Object>(null);
        this.retryDelayBaseMillis = new AtomicInteger(0);
        this.storedBytes = new AtomicLong(0L);
        this.closed = new AtomicBoolean(false);
        this.buffers = new DirectMemoryBuffer[(int)(maxSizeBytes / (long)this.layout.bufferSize())];
        this.availableBufferIds = new ArrayDeque(this.buffers.length);
        this.unallocatedBufferIds = new ArrayDeque(this.buffers.length);
        this.createBuffers();
    }

    @GuardedBy(value="availableBufferIds")
    private void createBuffers() {
        ByteBufAllocator allocator = this.createAllocator();
        for (int i = 0; i < this.buffers.length; ++i) {
            this.unallocatedBufferIds.addLast(i);
            this.buffers[i] = new DirectMemoryBuffer(i, allocator, this.layout);
        }
    }

    @VisibleForTesting
    protected ByteBufAllocator createAllocator() {
        return new UnpooledByteBufAllocator(true, true);
    }

    private long adjustMaxSizeIfNeeded(long maxSize, CacheLayout layout) {
        long r = maxSize % (long)layout.bufferSize();
        if (r != 0L) {
            maxSize = maxSize - r + (long)layout.bufferSize();
        }
        return maxSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void close() {
        if (this.closed.getAndSet(true)) return;
        DirectMemoryBuffer[] directMemoryBufferArray = this.availableBufferIds;
        synchronized (this.availableBufferIds) {
            this.availableBufferIds.clear();
            this.unallocatedBufferIds.clear();
            // ** MonitorExit[var1_1] (shouldn't be in output)
            for (DirectMemoryBuffer b : this.buffers) {
                b.close();
            }
            this.metrics.close();
            return;
        }
    }

    @Override
    public int getBlockAlignment() {
        return this.layout.blockSize();
    }

    @Override
    public int getMaxEntryLength() {
        return 0x3FFFFFF;
    }

    @Override
    public int insert(BufferView data) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkArgument((data.getLength() <= 0x3FFFFFF ? 1 : 0) != 0, (String)"Entry too long. Expected max %s, given %s.", (int)0x3FFFFFF, (int)data.getLength());
        int lastBlockAddress = 0;
        int remainingLength = data.getLength();
        try {
            while (remainingLength > 0 || lastBlockAddress == 0) {
                BufferView slice;
                DirectMemoryBuffer buffer = this.getNextAvailableBuffer();
                DirectMemoryBuffer.WriteResult writeResult = buffer.write(slice = data.slice(data.getLength() - remainingLength, remainingLength), lastBlockAddress);
                if (writeResult == null) continue;
                assert (writeResult.getWrittenLength() >= 0 && writeResult.getWrittenLength() <= remainingLength) : writeResult.getWrittenLength();
                remainingLength -= writeResult.getWrittenLength();
                this.storedBytes.addAndGet(writeResult.getWrittenLength());
                lastBlockAddress = writeResult.getLastBlockAddress();
            }
        }
        catch (Throwable ex) {
            if (!Exceptions.mustRethrow((Throwable)ex) && lastBlockAddress != 0) {
                this.delete(lastBlockAddress);
            }
            throw ex;
        }
        this.metrics.insert(data.getLength());
        return lastBlockAddress;
    }

    @Override
    public int replace(int address, BufferView data) {
        int newAddress = this.insert(data);
        this.delete(address);
        return newAddress;
    }

    @Override
    public int getAppendableLength(int currentLength) {
        int lastBlockLength = currentLength % this.layout.blockSize();
        return currentLength == 0 ? this.layout.blockSize() : (lastBlockLength == 0 ? 0 : this.layout.blockSize() - lastBlockLength);
    }

    @Override
    public int append(int address, int expectedLength, BufferView data) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        Preconditions.checkArgument((address != 0 ? 1 : 0) != 0, (Object)"Invalid address.");
        int appendedBytes = 0;
        int expectedLastBlockLength = this.layout.blockSize() - this.getAppendableLength(expectedLength);
        Preconditions.checkArgument((expectedLastBlockLength + data.getLength() <= this.layout.blockSize() ? 1 : 0) != 0, (Object)"data is too long; use getAppendableLength() to determine how much data can be appended.");
        int bufferId = this.layout.getBufferId(address);
        int blockId = this.layout.getBlockId(address);
        appendedBytes = this.buffers[bufferId].tryAppend(blockId, expectedLastBlockLength, data);
        this.storedBytes.addAndGet(appendedBytes);
        this.metrics.append(appendedBytes);
        return appendedBytes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void delete(int address) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        int deletedLength = 0;
        while (address != 0) {
            int bufferId = this.layout.getBufferId(address);
            int blockId = this.layout.getBlockId(address);
            DirectMemoryBuffer b = this.buffers[bufferId];
            boolean wasFull = !b.hasCapacity();
            DirectMemoryBuffer.DeleteResult result = b.delete(blockId);
            address = result.getPredecessorAddress();
            deletedLength += result.getDeletedLength();
            if (!wasFull || !b.hasCapacity()) continue;
            ArrayDeque<Integer> arrayDeque = this.availableBufferIds;
            synchronized (arrayDeque) {
                this.availableBufferIds.addLast(b.getId());
            }
        }
        this.storedBytes.addAndGet(-deletedLength);
        this.metrics.delete(deletedLength);
    }

    @Override
    public BufferView get(int address) {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        ArrayList<ByteBuf> readBuffers = new ArrayList<ByteBuf>();
        while (address != 0) {
            int bufferId = this.layout.getBufferId(address);
            int blockId = this.layout.getBlockId(address);
            DirectMemoryBuffer b = this.buffers[bufferId];
            address = b.read(blockId, readBuffers);
        }
        if (readBuffers.isEmpty()) {
            return null;
        }
        ByteBuf first = (ByteBuf)readBuffers.get(0);
        ByteBuf result = readBuffers.size() == 1 ? first : new CompositeByteBuf(first.alloc(), false, readBuffers.size(), (Iterable)Lists.reverse(readBuffers));
        this.metrics.get(result.readableBytes());
        return new NonReleaseableByteBufWrapper(result);
    }

    @Override
    public CacheState getState() {
        Exceptions.checkNotClosed((boolean)this.closed.get(), (Object)this);
        int allocatedBufferCount = 0;
        int blockCount = 0;
        for (DirectMemoryBuffer b : this.buffers) {
            if (!b.isAllocated()) continue;
            ++allocatedBufferCount;
            blockCount += b.getUsedBlockCount();
        }
        return new CacheState(this.storedBytes.get(), (long)blockCount * (long)this.layout.blockSize(), (long)allocatedBufferCount * (long)this.layout.blockSize(), (long)allocatedBufferCount * (long)this.layout.bufferSize(), (long)this.buffers.length * (long)this.layout.bufferSize());
    }

    @Override
    public void setCacheFullCallback(Supplier<Boolean> cacheFullCallback, int retryDelayBaseMillis) {
        this.tryCleanup.set(cacheFullCallback);
        this.retryDelayBaseMillis.set(retryDelayBaseMillis);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private DirectMemoryBuffer getNextAvailableBuffer() {
        int attempts = 0;
        while (attempts < 5) {
            ArrayDeque<Integer> arrayDeque = this.availableBufferIds;
            synchronized (arrayDeque) {
                while (!this.availableBufferIds.isEmpty() || !this.unallocatedBufferIds.isEmpty()) {
                    while (!this.availableBufferIds.isEmpty()) {
                        DirectMemoryBuffer b = this.buffers[this.availableBufferIds.peekFirst()];
                        if (b.hasCapacity()) {
                            return b;
                        }
                        this.availableBufferIds.removeFirst();
                    }
                    if (this.unallocatedBufferIds.isEmpty()) continue;
                    this.availableBufferIds.addLast(this.unallocatedBufferIds.removeFirst());
                }
            }
            this.tryCleanup(++attempts);
        }
        throw new CacheFullException(String.format("%s full: %s.", DirectMemoryCache.class.getSimpleName(), this.getState()));
    }

    private void tryCleanup(int attempts) {
        int sleepMillis;
        Supplier<Boolean> c = this.tryCleanup.get();
        if (c != null && !c.get().booleanValue() && (sleepMillis = attempts * this.retryDelayBaseMillis.get()) > 0 && attempts < 5) {
            Exceptions.handleInterrupted(() -> Thread.sleep(sleepMillis));
        }
    }

    private static class NonReleaseableByteBufWrapper
    extends ByteBufWrapper {
        NonReleaseableByteBufWrapper(@NonNull ByteBuf buf) {
            super(buf);
            if (buf == null) {
                throw new NullPointerException("buf is marked non-null but is null");
            }
        }

        public void retain() {
        }

        public void release() {
        }
    }
}

