/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.storage.chunklayer;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.Timer;
import io.pravega.common.concurrent.Futures;
import io.pravega.segmentstore.contracts.StreamSegmentTruncatedException;
import io.pravega.segmentstore.storage.SegmentHandle;
import io.pravega.segmentstore.storage.chunklayer.ChunkHandle;
import io.pravega.segmentstore.storage.chunklayer.ChunkNameOffsetPair;
import io.pravega.segmentstore.storage.chunklayer.ChunkStorageMetrics;
import io.pravega.segmentstore.storage.chunklayer.ChunkedSegmentStorage;
import io.pravega.segmentstore.storage.metadata.ChunkMetadata;
import io.pravega.segmentstore.storage.metadata.MetadataTransaction;
import io.pravega.segmentstore.storage.metadata.ReadIndexBlockMetadata;
import io.pravega.segmentstore.storage.metadata.SegmentMetadata;
import io.pravega.shared.NameUtils;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ReadOperation
implements Callable<CompletableFuture<Integer>> {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(ReadOperation.class);
    private final SegmentHandle handle;
    private final long offset;
    private final byte[] buffer;
    private final int bufferOffset;
    private final int length;
    private final ChunkedSegmentStorage chunkedSegmentStorage;
    private final long traceId;
    private final Timer timer;
    private volatile SegmentMetadata segmentMetadata;
    private final AtomicInteger bytesRemaining = new AtomicInteger();
    private final AtomicInteger currentBufferOffset = new AtomicInteger();
    private final AtomicLong currentOffset = new AtomicLong();
    private final AtomicInteger totalBytesRead = new AtomicInteger();
    private final AtomicLong startOffsetForCurrentChunk = new AtomicLong();
    private volatile String currentChunkName;
    private volatile ChunkMetadata chunkToReadFrom = null;
    private volatile boolean isLoopExited;
    private final AtomicInteger cntScanned = new AtomicInteger();
    private volatile int bytesToRead;
    private final AtomicInteger cntChunksRead = new AtomicInteger();

    ReadOperation(ChunkedSegmentStorage chunkedSegmentStorage, SegmentHandle handle, long offset, byte[] buffer, int bufferOffset, int length) {
        this.handle = handle;
        this.offset = offset;
        this.buffer = buffer;
        this.bufferOffset = bufferOffset;
        this.length = length;
        this.chunkedSegmentStorage = chunkedSegmentStorage;
        this.traceId = LoggerHelpers.traceEnter((Logger)log, (String)"read", (Object[])new Object[]{handle, offset, length});
        this.timer = new Timer();
    }

    @Override
    public CompletableFuture<Integer> call() {
        this.checkPreconditions();
        log.debug("{} read - started op={}, segment={}, offset={}, length={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.offset, this.length});
        String streamSegmentName = this.handle.getSegmentName();
        return ChunkedSegmentStorage.tryWith(this.chunkedSegmentStorage.getMetadataStore().beginTransaction(true, streamSegmentName), txn -> txn.get(streamSegmentName).thenComposeAsync(storageMetadata -> {
            this.segmentMetadata = (SegmentMetadata)storageMetadata;
            this.checkState();
            if (this.length == 0) {
                return CompletableFuture.completedFuture(0);
            }
            return ((CompletableFuture)((CompletableFuture)this.findChunkForOffset((MetadataTransaction)txn).thenComposeAsync(v -> this.readData((MetadataTransaction)txn), this.chunkedSegmentStorage.getExecutor())).exceptionally(ex -> {
                log.debug("{} read - exception op={}, segment={}, offset={}, bytesRead={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.offset, this.totalBytesRead});
                if (ex instanceof CompletionException) {
                    throw (CompletionException)ex;
                }
                throw new CompletionException((Throwable)ex);
            })).thenApplyAsync(v -> {
                this.logEnd();
                return this.totalBytesRead.get();
            }, this.chunkedSegmentStorage.getExecutor());
        }, this.chunkedSegmentStorage.getExecutor()), this.chunkedSegmentStorage.getExecutor());
    }

    private void logEnd() {
        Duration elapsed = this.timer.getElapsed();
        ChunkStorageMetrics.SLTS_READ_LATENCY.reportSuccessEvent(elapsed);
        ChunkStorageMetrics.SLTS_NUM_CHUNKS_READ.reportSuccessValue((long)this.cntChunksRead.get());
        ChunkStorageMetrics.SLTS_READ_BYTES.add((long)this.length);
        if (this.segmentMetadata.isStorageSystemSegment()) {
            ChunkStorageMetrics.SLTS_SYSTEM_READ_LATENCY.reportSuccessEvent(elapsed);
            ChunkStorageMetrics.SLTS_SYSTEM_NUM_CHUNKS_READ.reportSuccessValue((long)this.cntChunksRead.get());
            ChunkStorageMetrics.SLTS_SYSTEM_READ_BYTES.add((long)this.length);
        }
        if (elapsed.toMillis() > 0L) {
            long bytesPerSecond = 1000L * (long)this.length / elapsed.toMillis();
            ChunkStorageMetrics.SLTS_READ_INSTANT_TPUT.reportSuccessValue(bytesPerSecond);
        }
        if ((long)this.chunkedSegmentStorage.getConfig().getLateWarningThresholdInMillis() < elapsed.toMillis()) {
            log.warn("{} read - late op={}, segment={}, offset={}, bytesRead={}, latency={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.offset, this.totalBytesRead, elapsed.toMillis()});
        } else {
            log.debug("{} read - finished op={}, segment={}, offset={}, bytesRead={}, latency={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.offset, this.totalBytesRead, elapsed.toMillis()});
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"read", (long)this.traceId, (Object[])new Object[]{this.handle, this.offset, this.totalBytesRead});
    }

    private CompletableFuture<Void> readData(MetadataTransaction txn) {
        List chunkReadFutures = Collections.synchronizedList(new ArrayList());
        return Futures.loop(() -> this.bytesRemaining.get() > 0 && null != this.currentChunkName, () -> {
            Preconditions.checkState((null != this.chunkToReadFrom ? 1 : 0) != 0, (String)"chunkToReadFrom is null. currentChunkName=%s Segment=%s", (Object)this.currentChunkName, (Object)this.segmentMetadata.getName());
            this.bytesToRead = Math.toIntExact(Math.min((long)this.bytesRemaining.get(), this.chunkToReadFrom.getLength() - (this.currentOffset.get() - this.startOffsetForCurrentChunk.get())));
            if (this.currentOffset.get() >= this.startOffsetForCurrentChunk.get() + this.chunkToReadFrom.getLength()) {
                this.currentChunkName = this.chunkToReadFrom.getNextChunk();
                if (null != this.currentChunkName) {
                    this.startOffsetForCurrentChunk.addAndGet(this.chunkToReadFrom.getLength());
                    return txn.get(this.currentChunkName).thenAcceptAsync(storageMetadata -> {
                        this.chunkToReadFrom = (ChunkMetadata)storageMetadata;
                        Preconditions.checkState((null != this.chunkToReadFrom ? 1 : 0) != 0, (String)"chunkToReadFrom is null. currentChunkName=%s Segment=%s", (Object)this.currentChunkName, (Object)this.segmentMetadata.getName());
                        log.debug("{} read - reading from next chunk - op={}, segment={}, chunk={}", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.chunkToReadFrom});
                    }, this.chunkedSegmentStorage.getExecutor());
                }
            } else {
                Preconditions.checkState((this.bytesToRead != 0 ? 1 : 0) != 0, (String)"bytesToRead is 0. Segment=%s", (Object)this.segmentMetadata.getName());
                return CompletableFuture.runAsync(() -> {
                    chunkReadFutures.add(this.readChunk(this.chunkToReadFrom.getName(), this.currentOffset.get() - this.startOffsetForCurrentChunk.get(), this.bytesToRead, this.currentBufferOffset.get()));
                    log.trace("{} read - reading chunk - op={}, segment={}, chunk={} offset={} length={} bufferOffset={}", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.chunkToReadFrom.getName(), this.currentOffset.get() - this.startOffsetForCurrentChunk.get(), this.bytesToRead, this.currentBufferOffset.get()});
                    this.cntChunksRead.incrementAndGet();
                    this.bytesRemaining.addAndGet(-this.bytesToRead);
                    this.currentOffset.addAndGet(this.bytesToRead);
                    this.currentBufferOffset.addAndGet(this.bytesToRead);
                    this.totalBytesRead.addAndGet(this.bytesToRead);
                }, this.chunkedSegmentStorage.getExecutor());
            }
            return CompletableFuture.completedFuture(null);
        }, (Executor)this.chunkedSegmentStorage.getExecutor()).thenCompose(v -> Futures.allOf((Collection)chunkReadFutures));
    }

    private CompletableFuture<Void> readChunk(String chunkName, long fromOffset, int bytesToRead, int bufferOffset) {
        AtomicInteger chunkBytesRemaining = new AtomicInteger(bytesToRead);
        AtomicLong chunkFromOffset = new AtomicLong(fromOffset);
        AtomicInteger chunkBufferOffset = new AtomicInteger(bufferOffset);
        ChunkHandle chunkHandle = ChunkHandle.readHandle(chunkName);
        return Futures.loop(() -> chunkBytesRemaining.get() > 0, () -> this.chunkedSegmentStorage.getChunkStorage().read(chunkHandle, chunkFromOffset.get(), chunkBytesRemaining.get(), this.buffer, chunkBufferOffset.get()).thenAccept(n -> {
            Preconditions.checkState((n != 0 ? 1 : 0) != 0, (String)"Zero bytes read chunk=%s, fromOffset=%d", (Object)chunkName, (long)fromOffset);
            chunkBytesRemaining.addAndGet(-n.intValue());
            chunkFromOffset.addAndGet(n.intValue());
            chunkBufferOffset.addAndGet((int)n);
        }), (Executor)this.chunkedSegmentStorage.getExecutor());
    }

    private CompletableFuture<Void> findChunkForOffset(MetadataTransaction txn) {
        CompletionStage<Object> f;
        boolean shouldOnlyReadLastChunk;
        this.currentChunkName = this.segmentMetadata.getFirstChunk();
        this.chunkToReadFrom = null;
        Preconditions.checkState((null != this.currentChunkName ? 1 : 0) != 0, (String)"currentChunkName must not be null. Segment=%s", (Object)this.segmentMetadata.getName());
        this.bytesRemaining.set(this.length);
        this.currentBufferOffset.set(this.bufferOffset);
        this.currentOffset.set(this.offset);
        this.totalBytesRead.set(0);
        this.startOffsetForCurrentChunk.set(this.segmentMetadata.getFirstChunkStartOffset());
        boolean bl = shouldOnlyReadLastChunk = this.offset >= this.segmentMetadata.getLastChunkStartOffset();
        if (shouldOnlyReadLastChunk) {
            this.startOffsetForCurrentChunk.set(this.segmentMetadata.getLastChunkStartOffset());
            this.currentChunkName = this.segmentMetadata.getLastChunk();
        } else {
            ChunkNameOffsetPair floorEntry = this.chunkedSegmentStorage.getReadIndexCache().findFloor(this.handle.getSegmentName(), this.offset);
            if (null != floorEntry && this.startOffsetForCurrentChunk.get() < floorEntry.getOffset() && null != floorEntry.getChunkName()) {
                this.startOffsetForCurrentChunk.set(floorEntry.getOffset());
                this.currentChunkName = floorEntry.getChunkName();
            }
        }
        long floorBlockStartOffset = this.getFloorBlockStartOffset(this.offset);
        if (!shouldOnlyReadLastChunk && !this.segmentMetadata.isStorageSystemSegment() && this.startOffsetForCurrentChunk.get() < floorBlockStartOffset) {
            Timer indexLookupTimer = new Timer();
            f = txn.get(NameUtils.getSegmentReadIndexBlockName((String)this.segmentMetadata.getName(), (long)floorBlockStartOffset)).thenAcceptAsync(storageMetadata -> {
                if (null != storageMetadata) {
                    ReadIndexBlockMetadata blockMetadata = (ReadIndexBlockMetadata)storageMetadata;
                    if (blockMetadata.getStartOffset() <= this.offset) {
                        this.startOffsetForCurrentChunk.set(blockMetadata.getStartOffset());
                        this.currentChunkName = blockMetadata.getChunkName();
                        log.debug("{} read - found block index to start scanning - op={}, segment={}, chunk={}, startOffset={}, offset={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.currentChunkName, this.startOffsetForCurrentChunk.get(), this.offset});
                        long nextBlock = this.getFloorBlockStartOffset(this.offset + (long)this.length);
                        if (nextBlock > floorBlockStartOffset + this.chunkedSegmentStorage.getConfig().getIndexBlockSize()) {
                            txn.get(NameUtils.getSegmentReadIndexBlockName((String)this.segmentMetadata.getName(), (long)nextBlock));
                        } else {
                            txn.get(NameUtils.getSegmentReadIndexBlockName((String)this.segmentMetadata.getName(), (long)(floorBlockStartOffset + this.chunkedSegmentStorage.getConfig().getIndexBlockSize())));
                        }
                    } else {
                        log.warn("{} read - block entry offset must be floor to requested offset. op={} segment={} offset={} length={} block={}", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.segmentMetadata, this.offset, this.length, blockMetadata});
                    }
                }
                if (this.segmentMetadata.isStorageSystemSegment()) {
                    ChunkStorageMetrics.SLTS_SYS_READ_INDEX_BLOCK_LOOKUP_LATENCY.reportSuccessEvent(indexLookupTimer.getElapsed());
                } else {
                    ChunkStorageMetrics.SLTS_READ_INDEX_BLOCK_LOOKUP_LATENCY.reportSuccessEvent(indexLookupTimer.getElapsed());
                }
            }, this.chunkedSegmentStorage.getExecutor());
        } else {
            f = CompletableFuture.completedFuture(null);
        }
        Timer readIndexTimer = new Timer();
        return f.thenComposeAsync(vv -> Futures.loop(() -> this.currentChunkName != null && !this.isLoopExited, () -> txn.get(this.currentChunkName).thenAcceptAsync(storageMetadata -> {
            this.chunkToReadFrom = (ChunkMetadata)storageMetadata;
            Preconditions.checkState((null != this.chunkToReadFrom ? 1 : 0) != 0, (String)"chunkToReadFrom is null. currentChunkName=%s Segment=%s", (Object)this.currentChunkName, (Object)this.segmentMetadata.getName());
            if (this.startOffsetForCurrentChunk.get() <= this.currentOffset.get() && this.startOffsetForCurrentChunk.get() + this.chunkToReadFrom.getLength() > this.currentOffset.get()) {
                log.debug("{} read - found chunk to read - op={}, segment={}, chunk={}, startOffset={}, length={}, readOffset={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.chunkToReadFrom, this.startOffsetForCurrentChunk.get(), this.chunkToReadFrom.getLength(), this.currentOffset});
                this.isLoopExited = true;
                return;
            }
            this.currentChunkName = this.chunkToReadFrom.getNextChunk();
            this.startOffsetForCurrentChunk.addAndGet(this.chunkToReadFrom.getLength());
            if (null != this.currentChunkName) {
                this.chunkedSegmentStorage.getReadIndexCache().addIndexEntry(this.handle.getSegmentName(), this.currentChunkName, this.startOffsetForCurrentChunk.get());
            }
            this.cntScanned.incrementAndGet();
        }, this.chunkedSegmentStorage.getExecutor()), (Executor)this.chunkedSegmentStorage.getExecutor()).thenAcceptAsync(v -> {
            Duration elapsed = readIndexTimer.getElapsed();
            if (this.segmentMetadata.isStorageSystemSegment()) {
                ChunkStorageMetrics.SLTS_SYS_READ_INDEX_SCAN_LATENCY.reportSuccessEvent(elapsed);
                ChunkStorageMetrics.SLTS_SYS_READ_INDEX_NUM_SCANNED.reportSuccessValue((long)this.cntScanned.get());
            } else {
                ChunkStorageMetrics.SLTS_READ_INDEX_SCAN_LATENCY.reportSuccessEvent(elapsed);
                ChunkStorageMetrics.SLTS_READ_INDEX_NUM_SCANNED.reportSuccessValue((long)this.cntScanned.get());
            }
            if (this.chunkToReadFrom.getNextChunk() != null) {
                txn.get(this.chunkToReadFrom.getNextChunk());
            }
            log.debug("{} read - chunk lookup - op={}, segment={}, offset={}, scanned={}, latency={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.offset, this.cntScanned.get(), elapsed.toMillis()});
        }, this.chunkedSegmentStorage.getExecutor()), this.chunkedSegmentStorage.getExecutor());
    }

    private long getFloorBlockStartOffset(long offsetToRead) {
        long floorBlock = offsetToRead / this.chunkedSegmentStorage.getConfig().getIndexBlockSize();
        long floorBlockStartOffset = floorBlock * this.chunkedSegmentStorage.getConfig().getIndexBlockSize();
        return floorBlockStartOffset;
    }

    private void checkState() {
        this.chunkedSegmentStorage.checkSegmentExists(this.handle.getSegmentName(), this.segmentMetadata);
        this.segmentMetadata.checkInvariants();
        Preconditions.checkArgument((this.offset < this.segmentMetadata.getLength() ? 1 : 0) != 0, (String)"Offset %s is beyond the last offset %s of the segment %s.", (Object)this.offset, (Object)this.segmentMetadata.getLength(), (Object)this.handle.getSegmentName());
        if (this.offset < this.segmentMetadata.getStartOffset()) {
            throw new CompletionException((Throwable)new StreamSegmentTruncatedException(this.handle.getSegmentName(), this.segmentMetadata.getStartOffset(), this.offset));
        }
    }

    private void checkPreconditions() {
        Preconditions.checkNotNull((Object)this.handle, (Object)"handle");
        Preconditions.checkNotNull((Object)this.buffer, (Object)"buffer");
        Preconditions.checkNotNull((Object)this.handle.getSegmentName(), (Object)"streamSegmentName");
        Exceptions.checkArrayRange((long)this.bufferOffset, (int)this.length, (long)this.buffer.length, (String)"bufferOffset", (String)"length");
        if (this.offset < 0L) {
            throw new ArrayIndexOutOfBoundsException(String.format("Offset (%s) must be non-negative, and bufferOffset (%s) and length (%s) must be valid indices into buffer of size %s.", this.offset, this.bufferOffset, this.length, this.buffer.length));
        }
    }
}

