/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.storage.chunklayer;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.ObjectBuilder;
import io.pravega.common.Timer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.concurrent.MultiKeySequentialProcessor;
import io.pravega.common.io.serialization.RevisionDataInput;
import io.pravega.common.io.serialization.RevisionDataOutput;
import io.pravega.common.io.serialization.VersionedSerializer;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.segmentstore.storage.chunklayer.ChunkHandle;
import io.pravega.segmentstore.storage.chunklayer.ChunkNotFoundException;
import io.pravega.segmentstore.storage.chunklayer.ChunkStorage;
import io.pravega.segmentstore.storage.chunklayer.ChunkStorageException;
import io.pravega.segmentstore.storage.chunklayer.ChunkedSegmentStorageConfig;
import io.pravega.segmentstore.storage.chunklayer.GarbageCollector;
import io.pravega.segmentstore.storage.chunklayer.InvalidOffsetException;
import io.pravega.segmentstore.storage.chunklayer.SnapshotInfo;
import io.pravega.segmentstore.storage.chunklayer.SnapshotInfoStore;
import io.pravega.segmentstore.storage.metadata.ChunkMetadata;
import io.pravega.segmentstore.storage.metadata.ChunkMetadataStore;
import io.pravega.segmentstore.storage.metadata.MetadataTransaction;
import io.pravega.segmentstore.storage.metadata.SegmentMetadata;
import io.pravega.segmentstore.storage.metadata.StorageMetadata;
import io.pravega.shared.NameUtils;
import java.beans.ConstructorProperties;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import lombok.Generated;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SystemJournal {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(SystemJournal.class);
    private static final String LOCK_KEY_NAME = "SingleThreadedLock";
    private static final SystemJournalRecordBatch.SystemJournalRecordBatchSerializer BATCH_SERIALIZER = new SystemJournalRecordBatch.SystemJournalRecordBatchSerializer();
    private static final SystemSnapshotRecord.Serializer SYSTEM_SNAPSHOT_SERIALIZER = new SystemSnapshotRecord.Serializer();
    private final ChunkStorage chunkStorage;
    private final ChunkMetadataStore metadataStore;
    private volatile long epoch;
    private final int containerId;
    private final AtomicInteger currentFileIndex = new AtomicInteger();
    private final AtomicLong currentSnapshotIndex = new AtomicLong();
    private final AtomicBoolean newChunkRequired = new AtomicBoolean(true);
    private final AtomicReference<SystemSnapshotRecord> lastSavedSystemSnapshot = new AtomicReference();
    private final AtomicLong lastSavedSystemSnapshotId = new AtomicLong();
    private final AtomicReference<SnapshotInfo> lastSavedSnapshotInfo = new AtomicReference();
    private final AtomicLong lastSavedSnapshotTime = new AtomicLong();
    private final AtomicInteger recordsSinceSnapshot = new AtomicInteger();
    private volatile SnapshotInfoStore snapshotInfoStore;
    private final String systemSegmentsPrefix;
    private final String[] systemSegments;
    private final AtomicLong systemJournalOffset = new AtomicLong();
    private final AtomicReference<ChunkHandle> currentHandle = new AtomicReference();
    private final List<String> pendingGarbageChunks = Collections.synchronizedList(new ArrayList());
    private final ChunkedSegmentStorageConfig config;
    private final GarbageCollector garbageCollector;
    private final Supplier<Long> currentTimeSupplier;
    private final AtomicBoolean reentryGuard = new AtomicBoolean();
    private final Executor executor;
    private final MultiKeySequentialProcessor<String> taskProcessor;

    public SystemJournal(int containerId, ChunkStorage chunkStorage, ChunkMetadataStore metadataStore, GarbageCollector garbageCollector, Supplier<Long> currentTimeSupplier, ChunkedSegmentStorageConfig config, Executor executor) {
        this.chunkStorage = (ChunkStorage)Preconditions.checkNotNull((Object)chunkStorage, (Object)"chunkStorage");
        this.metadataStore = (ChunkMetadataStore)Preconditions.checkNotNull((Object)metadataStore, (Object)"metadataStore");
        this.config = (ChunkedSegmentStorageConfig)Preconditions.checkNotNull((Object)config, (Object)"config");
        this.garbageCollector = (GarbageCollector)Preconditions.checkNotNull((Object)garbageCollector, (Object)"garbageCollector");
        this.containerId = containerId;
        this.systemSegments = SystemJournal.getChunkStorageSystemSegments(containerId);
        this.systemSegmentsPrefix = "_system/containers/";
        this.currentTimeSupplier = (Supplier)Preconditions.checkNotNull(currentTimeSupplier, (Object)"currentTimeSupplier");
        this.executor = (Executor)Preconditions.checkNotNull((Object)executor, (Object)"executor");
        this.taskProcessor = new MultiKeySequentialProcessor(this.executor);
    }

    public SystemJournal(int containerId, ChunkStorage chunkStorage, ChunkMetadataStore metadataStore, GarbageCollector garbageCollector, ChunkedSegmentStorageConfig config, Executor executor) {
        this(containerId, chunkStorage, metadataStore, garbageCollector, System::currentTimeMillis, config, executor);
    }

    public void initialize() throws Exception {
        if (this.chunkStorage.supportsAppend()) {
            this.chunkStorage.create(this.getSystemJournalChunkName()).get();
        }
    }

    public CompletableFuture<Void> bootstrap(long epoch, SnapshotInfoStore snapshotInfoStore) {
        this.epoch = epoch;
        this.snapshotInfoStore = (SnapshotInfoStore)Preconditions.checkNotNull((Object)snapshotInfoStore, (Object)"snapshotInfoStore");
        Preconditions.checkState((!this.reentryGuard.getAndSet(true) ? 1 : 0) != 0, (Object)"bootstrap called multiple times.");
        log.debug("SystemJournal[{}] BOOT started.", (Object)this.containerId);
        Timer t = new Timer();
        MetadataTransaction txn = this.metadataStore.beginTransaction(false, this.getSystemSegments());
        BootstrapState state = new BootstrapState();
        return ((CompletableFuture)((CompletableFuture)((CompletableFuture)((CompletableFuture)((CompletableFuture)((CompletableFuture)this.findLatestSnapshot().thenComposeAsync(snapshot -> this.applySystemSnapshotRecord(txn, state, (SystemSnapshotRecord)snapshot), this.executor)).thenComposeAsync(latestSnapshot -> this.applySystemLogOperations(txn, state, (SystemSnapshotRecord)latestSnapshot), this.executor)).thenComposeAsync(v -> {
            if (this.config.isLazyCommitEnabled()) {
                return this.adjustLastChunkLengths(txn);
            }
            return CompletableFuture.completedFuture(null);
        }, this.executor)).thenComposeAsync(v -> this.applyFinalTruncateOffsets(txn, state), this.executor)).thenComposeAsync(v -> {
            if (this.config.isSelfCheckEnabled()) {
                Preconditions.checkState((this.currentFileIndex.get() == 0 ? 1 : 0) != 0, (Object)"currentFileIndex must be zero");
                Preconditions.checkState((this.systemJournalOffset.get() == 0L ? 1 : 0) != 0, (Object)"systemJournalOffset must be zero");
                Preconditions.checkState((boolean)this.newChunkRequired.get(), (Object)"newChunkRequired must be true");
            }
            return ((CompletableFuture)this.createSystemSnapshotRecord(txn, true, this.config.isSelfCheckEnabled()).thenComposeAsync(systemSnapshotRecord -> this.writeRecordBatch(Collections.singletonList(systemSnapshotRecord)), this.executor)).thenRunAsync(() -> this.newChunkRequired.set(true), this.executor);
        }, this.executor)).thenComposeAsync(v -> txn.commit(true, true), this.executor)).whenCompleteAsync((v, e) -> {
            txn.close();
            log.info("SystemJournal[{}] BOOT complete - applied {} records in {} journals. Total time = {} ms.", new Object[]{this.containerId, state.recordsProcessedCount.get(), state.filesProcessedCount.get(), t.getElapsedMillis()});
        }, this.executor);
    }

    private CompletableFuture<Void> checkSnapshotFileExists(long snapshotId) {
        if (this.getConfig().isSelfCheckEnabled()) {
            String snapshotFileName = NameUtils.getSystemJournalSnapshotFileName((int)this.containerId, (long)this.epoch, (long)snapshotId);
            return this.chunkStorage.exists(snapshotFileName).thenAcceptAsync(exists -> Preconditions.checkState((boolean)exists, (Object)"Snapshot chunk must exist"), this.executor);
        }
        return CompletableFuture.completedFuture(null);
    }

    public CompletableFuture<Void> commitRecord(SystemJournalRecord record) {
        Preconditions.checkArgument((null != record ? 1 : 0) != 0, (Object)"record must not be null");
        return this.commitRecords(Collections.singletonList(record));
    }

    public CompletableFuture<Void> commitRecords(Collection<SystemJournalRecord> records) {
        Preconditions.checkArgument((null != records ? 1 : 0) != 0, (Object)"records must not be null");
        Preconditions.checkArgument((records.size() > 0 ? 1 : 0) != 0, (Object)"records must not be empty");
        return this.executeSerialized(() -> ((CompletableFuture)this.generateSnapshotIfRequired().thenComposeAsync(v -> this.writeSnapshotInfoIfRequired(), this.executor)).thenComposeAsync(v -> this.writeRecordBatch(records), this.executor));
    }

    private CompletableFuture<Void> writeRecordBatch(Collection<SystemJournalRecord> records) {
        ByteArraySegment bytes;
        SystemJournalRecordBatch batch = SystemJournalRecordBatch.builder().systemJournalRecords(records).build();
        try {
            bytes = BATCH_SERIALIZER.serialize(batch);
        }
        catch (IOException e) {
            return CompletableFuture.failedFuture(new ChunkStorageException(this.getSystemJournalChunkName(), "Unable to serialize", e));
        }
        AtomicInteger attempt = new AtomicInteger();
        AtomicBoolean done = new AtomicBoolean();
        return Futures.loop(() -> !done.get() && attempt.get() < this.config.getMaxJournalWriteAttempts(), () -> ((CompletableFuture)((CompletableFuture)this.writeToJournal(bytes).thenAcceptAsync(v -> {
            log.trace("SystemJournal[{}] Logging system log records - journal={}, batch={}.", new Object[]{this.containerId, this.currentHandle.get().getChunkName(), batch});
            this.recordsSinceSnapshot.incrementAndGet();
            done.set(true);
        }, this.executor)).handleAsync((v, e) -> {
            attempt.incrementAndGet();
            if (e != null) {
                Throwable ex = Exceptions.unwrap((Throwable)e);
                if (attempt.get() >= this.config.getMaxJournalWriteAttempts()) {
                    throw new CompletionException(ex);
                }
                log.warn("SystemJournal[{}] Error while writing journal {}. Attempt#{}", new Object[]{this.containerId, this.getSystemJournalChunkName(this.containerId, this.epoch, this.currentFileIndex.get()), attempt.get(), e});
                if (ex instanceof InvalidOffsetException) {
                    return null;
                }
                if (ex instanceof ChunkStorageException) {
                    return null;
                }
                throw new CompletionException(ex);
            }
            return v;
        }, this.executor)).thenAcceptAsync(v -> {
            if (!(this.chunkStorage.supportsAppend() && this.config.isAppendEnabled() && done.get())) {
                this.newChunkRequired.set(true);
            }
        }, this.executor), (Executor)this.executor);
    }

    private CompletableFuture<Void> generateSnapshotIfRequired() {
        boolean shouldGenerate = true;
        if (this.lastSavedSystemSnapshot.get() == null) {
            log.debug("SystemJournal[{}] Generating first snapshot.", (Object)this.containerId);
        } else if (this.recordsSinceSnapshot.get() > this.config.getMaxJournalUpdatesPerSnapshot()) {
            log.debug("SystemJournal[{}] Generating snapshot based on update threshold. {} updates since last snapshot.", (Object)this.containerId, (Object)this.recordsSinceSnapshot.get());
        } else if (this.currentTimeSupplier.get() - this.lastSavedSnapshotTime.get() > this.config.getJournalSnapshotInfoUpdateFrequency().toMillis()) {
            log.debug("SystemJournal[{}] Generating snapshot based on time threshold. current time={} last saved ={}.", new Object[]{this.containerId, this.currentTimeSupplier.get(), this.lastSavedSnapshotTime.get()});
        } else {
            shouldGenerate = false;
        }
        if (shouldGenerate) {
            MetadataTransaction txn = this.metadataStore.beginTransaction(true, this.getSystemSegments());
            return ((CompletableFuture)this.validateAndSaveSnapshot(txn, true, this.config.isSelfCheckEnabled()).thenAcceptAsync(saved -> {
                txn.close();
                if (saved.booleanValue()) {
                    this.lastSavedSnapshotTime.set(this.currentTimeSupplier.get());
                    this.recordsSinceSnapshot.set(0);
                    this.newChunkRequired.set(true);
                }
            }, this.executor)).exceptionally(e -> {
                log.error("SystemJournal[{}] Error while creating snapshot", (Object)this.containerId, e);
                return null;
            });
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Void> writeSnapshotInfoIfRequired() {
        if (this.lastSavedSystemSnapshot.get() != null) {
            boolean shouldSave = true;
            if (this.lastSavedSnapshotInfo.get() == null) {
                log.debug("SystemJournal[{}] Saving first snapshot info new={}.", (Object)this.containerId, (Object)this.lastSavedSystemSnapshotId.get());
            } else if (this.lastSavedSnapshotInfo.get().getSnapshotId() < this.lastSavedSystemSnapshotId.get()) {
                log.debug("SystemJournal[{}] Saving new snapshot info new={} old={}.", new Object[]{this.containerId, this.lastSavedSystemSnapshotId.get(), this.lastSavedSnapshotInfo.get().getSnapshotId()});
            } else {
                shouldSave = false;
            }
            if (shouldSave) {
                return this.writeSnapshotInfo(this.lastSavedSystemSnapshotId.get());
            }
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Void> writeSnapshotInfo(long snapshotId) {
        return this.checkSnapshotFileExists(snapshotId).thenComposeAsync(v -> {
            SnapshotInfo info = SnapshotInfo.builder().snapshotId(snapshotId).epoch(this.epoch).build();
            return ((CompletableFuture)this.snapshotInfoStore.writeSnapshotInfo(info).thenAcceptAsync(v1 -> {
                SnapshotInfo oldSnapshotInfo = this.lastSavedSnapshotInfo.get();
                log.info("SystemJournal[{}] Snapshot info saved.{}", (Object)this.containerId, (Object)info);
                this.lastSavedSnapshotInfo.set(info);
                if (null != oldSnapshotInfo) {
                    String oldSnapshotFile = NameUtils.getSystemJournalSnapshotFileName((int)this.containerId, (long)this.epoch, (long)oldSnapshotInfo.getSnapshotId());
                    this.pendingGarbageChunks.add(oldSnapshotFile);
                }
                this.garbageCollector.addChunksToGarbage(-1L, this.pendingGarbageChunks);
                this.pendingGarbageChunks.clear();
            }, this.executor)).exceptionally(e -> {
                log.error("Unable to persist snapshot info.{}", (Object)this.currentSnapshotIndex, e);
                return null;
            });
        }, this.executor);
    }

    private CompletableFuture<SystemSnapshotRecord> findLatestSnapshot() {
        return this.snapshotInfoStore.readSnapshotInfo().thenComposeAsync(snapshotInfo -> {
            if (null != snapshotInfo) {
                String snapshotFileName = NameUtils.getSystemJournalSnapshotFileName((int)this.containerId, (long)snapshotInfo.getEpoch(), (long)snapshotInfo.getSnapshotId());
                log.debug("SystemJournal[{}] Snapshot info read. {} pointing to {}", new Object[]{this.containerId, snapshotInfo, snapshotFileName});
                return ((CompletableFuture)this.checkSnapshotExists(snapshotFileName).thenComposeAsync(v -> this.getContents(snapshotFileName), this.executor)).thenApplyAsync(contents -> this.readSnapshotRecord((SnapshotInfo)snapshotInfo, (byte[])contents), this.executor);
            }
            log.info("SystemJournal[{}] No Snapshot info available. This is ok if this is new installation", (Object)this.containerId);
            return CompletableFuture.completedFuture(null);
        }, this.executor);
    }

    private CompletableFuture<Void> checkSnapshotExists(String snapshotFileName) {
        if (this.getConfig().isSelfCheckEnabled()) {
            return this.chunkStorage.exists(snapshotFileName).thenAcceptAsync(exists -> Preconditions.checkState((boolean)exists, (Object)"Chunk pointed by SnapshotInfo must exist"), this.executor);
        }
        return CompletableFuture.completedFuture(null);
    }

    private SystemSnapshotRecord readSnapshotRecord(SnapshotInfo snapshotInfo, byte[] snapshotContents) {
        try {
            SystemSnapshotRecord systemSnapshot = (SystemSnapshotRecord)SYSTEM_SNAPSHOT_SERIALIZER.deserialize(snapshotContents);
            log.info("SystemJournal[{}] Done finding snapshots. Snapshot found and parsed. {}", (Object)this.containerId, (Object)snapshotInfo);
            return systemSnapshot;
        }
        catch (Exception e) {
            Throwable ex = Exceptions.unwrap((Throwable)e);
            if (ex instanceof EOFException) {
                log.error("SystemJournal[{}] Incomplete snapshot found, skipping {}.", new Object[]{this.containerId, snapshotInfo, e});
                throw new CompletionException(e);
            }
            if (!(ex instanceof ChunkNotFoundException)) {
                log.error("SystemJournal[{}] Error with snapshot, skipping {}.", new Object[]{this.containerId, snapshotInfo, e});
                throw new CompletionException(e);
            }
            log.warn("SystemJournal[{}] Missing snapshot, skipping {}.", new Object[]{this.containerId, snapshotInfo, e});
            return null;
        }
    }

    private CompletableFuture<SystemSnapshotRecord> applySystemSnapshotRecord(MetadataTransaction txn, BootstrapState state, SystemSnapshotRecord systemSnapshot) {
        if (null != systemSnapshot) {
            systemSnapshot.checkInvariants();
            txn.getData().clear();
            state.finalTruncateOffsets.clear();
            state.finalFirstChunkStartsAtOffsets.clear();
            state.chunkStartOffsets.clear();
            log.debug("SystemJournal[{}] Applying snapshot that includes journals up to epoch={} journal index={}", new Object[]{this.containerId, systemSnapshot.epoch, systemSnapshot.fileIndex});
            log.trace("SystemJournal[{}] Processing system log snapshot {}.", (Object)this.containerId, (Object)systemSnapshot);
            for (SegmentSnapshotRecord segmentSnapshot : systemSnapshot.segmentSnapshotRecords) {
                segmentSnapshot.segmentMetadata.setActive(true).setOwnershipChanged(true).setStorageSystemSegment(true);
                segmentSnapshot.segmentMetadata.setOwnerEpoch(this.epoch);
                txn.create(segmentSnapshot.segmentMetadata);
                txn.markPinned(segmentSnapshot.segmentMetadata);
                long offset = segmentSnapshot.segmentMetadata.getFirstChunkStartOffset();
                for (ChunkMetadata metadata : segmentSnapshot.chunkMetadataCollection) {
                    txn.create(metadata);
                    txn.markPinned(metadata);
                    state.chunkStartOffsets.put(metadata.getName(), offset);
                    offset += metadata.getLength();
                }
            }
        } else {
            log.debug("SystemJournal[{}] No previous snapshot present.", (Object)this.containerId);
            for (String systemSegment : this.systemSegments) {
                SegmentMetadata segmentMetadata = SegmentMetadata.builder().name(systemSegment).ownerEpoch(this.epoch).maxRollinglength(this.config.getStorageMetadataRollingPolicy().getMaxLength()).build();
                segmentMetadata.setActive(true).setOwnershipChanged(true).setStorageSystemSegment(true);
                segmentMetadata.checkInvariants();
                txn.create(segmentMetadata);
                txn.markPinned(segmentMetadata);
            }
        }
        return this.validateSystemSnapshotExistsInTxn(txn, systemSnapshot).thenApplyAsync(v -> {
            log.debug("SystemJournal[{}] Done applying snapshots.", (Object)this.containerId);
            return systemSnapshot;
        }, this.executor);
    }

    private CompletableFuture<Void> validateSystemSnapshotExistsInTxn(MetadataTransaction txn, SystemSnapshotRecord systemSnapshot) {
        if (null == systemSnapshot) {
            return CompletableFuture.completedFuture(null);
        }
        return Futures.loop(systemSnapshot.getSegmentSnapshotRecords(), segmentSnapshot -> ((CompletableFuture)((CompletableFuture)txn.get(segmentSnapshot.segmentMetadata.getKey()).thenComposeAsync(m -> this.validateChunksInSegmentSnapshot(txn, (SegmentSnapshotRecord)segmentSnapshot), this.executor)).thenComposeAsync(vv -> this.validateSegment(txn, segmentSnapshot.segmentMetadata.getKey()), this.executor)).thenApplyAsync(v -> true, this.executor), (Executor)this.executor);
    }

    private CompletableFuture<Void> validateChunksInSegmentSnapshot(MetadataTransaction txn, SegmentSnapshotRecord segmentSnapshot) {
        return Futures.loop(segmentSnapshot.getChunkMetadataCollection(), m -> txn.get(m.getKey()).thenApplyAsync(mm -> {
            Preconditions.checkState((null != mm ? 1 : 0) != 0, (Object)"Chunk metadata must not be null.");
            return true;
        }, this.executor), (Executor)this.executor);
    }

    private CompletableFuture<Void> validateSegment(MetadataTransaction txn, String segmentName) {
        return txn.get(segmentName).thenComposeAsync(m -> {
            SegmentMetadata segmentMetadata = (SegmentMetadata)m;
            Preconditions.checkState((null != segmentMetadata ? 1 : 0) != 0, (Object)"Segment metadata must not be null.");
            AtomicReference<String> chunkName = new AtomicReference<String>(segmentMetadata.getFirstChunk());
            return Futures.loop(() -> chunkName.get() != null, () -> txn.get((String)chunkName.get()).thenAcceptAsync(mm -> {
                Preconditions.checkState((null != mm ? 1 : 0) != 0, (Object)"Chunk metadata must not be null.");
                ChunkMetadata chunkMetadata = (ChunkMetadata)mm;
                chunkName.set(chunkMetadata.getNextChunk());
            }, this.executor), (Executor)this.executor);
        }, this.executor);
    }

    private CompletableFuture<byte[]> getContents(String chunkPath) {
        return this.getContents(chunkPath, false);
    }

    private CompletableFuture<byte[]> getContents(String chunkPath, boolean supressExceptionWarning) {
        AtomicBoolean isReadDone = new AtomicBoolean();
        AtomicBoolean shouldBreak = new AtomicBoolean();
        AtomicInteger attempt = new AtomicInteger();
        AtomicReference lastException = new AtomicReference();
        AtomicReference retValue = new AtomicReference();
        return ((CompletableFuture)Futures.loop(() -> attempt.get() < this.config.getMaxJournalReadAttempts() && !isReadDone.get() && !shouldBreak.get(), () -> this.readFully(chunkPath, retValue).handleAsync((v, e) -> {
            attempt.incrementAndGet();
            if (e != null) {
                lastException.set(e);
                Throwable ex = Exceptions.unwrap((Throwable)e);
                boolean shouldLog = true;
                if (!this.shouldRetry(ex)) {
                    shouldBreak.set(true);
                    boolean bl = shouldLog = !supressExceptionWarning;
                }
                if (shouldLog) {
                    log.warn("SystemJournal[{}] Error while reading journal {}. Attempt#{}", new Object[]{this.containerId, chunkPath, attempt.get(), lastException.get()});
                }
                return null;
            }
            isReadDone.set(true);
            return v;
        }, this.executor), (Executor)this.executor).handleAsync((v, e) -> {
            if (shouldBreak.get() || !isReadDone.get() && lastException.get() != null) {
                throw new CompletionException((Throwable)lastException.get());
            }
            return v;
        }, this.executor)).thenApplyAsync(v -> (byte[])retValue.get(), this.executor);
    }

    private boolean shouldRetry(Throwable ex) {
        return !(ex instanceof ChunkNotFoundException);
    }

    private CompletableFuture<Void> readFully(String chunkPath, AtomicReference<byte[]> retValue) {
        return this.chunkStorage.getInfo(chunkPath).thenComposeAsync(info -> {
            ChunkHandle h = ChunkHandle.readHandle(chunkPath);
            retValue.set(new byte[Math.toIntExact(info.getLength())]);
            if (info.getLength() == 0L) {
                log.warn("SystemJournal[{}] journal {} is empty.", (Object)this.containerId, (Object)chunkPath);
                return CompletableFuture.completedFuture(null);
            }
            AtomicLong fromOffset = new AtomicLong();
            AtomicInteger remaining = new AtomicInteger(((byte[])retValue.get()).length);
            return Futures.loop(() -> remaining.get() > 0, () -> this.chunkStorage.read(h, fromOffset.get(), remaining.get(), (byte[])retValue.get(), Math.toIntExact(fromOffset.get())), bytesRead -> {
                Preconditions.checkState((0 != bytesRead ? 1 : 0) != 0, (Object)"bytesRead must not be 0");
                remaining.addAndGet(-bytesRead.intValue());
                fromOffset.addAndGet(bytesRead.intValue());
            }, (Executor)this.executor);
        }, this.executor);
    }

    private CompletableFuture<Void> applySystemLogOperations(MetadataTransaction txn, BootstrapState state, SystemSnapshotRecord systemSnapshotRecord) {
        AtomicLong epochToStartScanning = new AtomicLong();
        AtomicInteger fileIndexToRecover = new AtomicInteger(1);
        List journalsProcessed = Collections.synchronizedList(new ArrayList());
        if (null != systemSnapshotRecord) {
            epochToStartScanning.set(systemSnapshotRecord.epoch);
            fileIndexToRecover.set(systemSnapshotRecord.fileIndex + 1);
        }
        log.debug("SystemJournal[{}] Applying journal operations. Starting at epoch={}  journal index={}", new Object[]{this.containerId, epochToStartScanning.get(), fileIndexToRecover.get()});
        AtomicLong epochToRecover = new AtomicLong(epochToStartScanning.get());
        return Futures.loop(() -> epochToRecover.get() < this.epoch, () -> {
            if (epochToRecover.get() > epochToStartScanning.get()) {
                fileIndexToRecover.set(1);
            }
            AtomicInteger scanAhead = new AtomicInteger();
            AtomicBoolean isScanDone = new AtomicBoolean();
            return Futures.loop(() -> !isScanDone.get(), () -> {
                String systemLogName = this.getSystemJournalChunkName(this.containerId, epochToRecover.get(), fileIndexToRecover.get());
                return ((CompletableFuture)((CompletableFuture)this.getContents(systemLogName, true).thenApplyAsync(contents -> {
                    journalsProcessed.add(systemLogName);
                    scanAhead.set(0);
                    return contents;
                }, this.executor)).thenComposeAsync(contents -> this.processJournalContents(txn, state, systemLogName, new ByteArrayInputStream((byte[])contents)), this.executor)).handleAsync((v, e) -> {
                    if (null != e) {
                        Throwable ex = Exceptions.unwrap((Throwable)e);
                        if (ex instanceof ChunkNotFoundException) {
                            log.debug("SystemJournal[{}] Journal does not exist for epoch={}. Last journal index={}", new Object[]{this.containerId, epochToRecover.get(), fileIndexToRecover.get()});
                            if (scanAhead.incrementAndGet() > this.config.getMaxJournalWriteAttempts()) {
                                isScanDone.set(true);
                                log.debug("SystemJournal[{}] Done applying journal operations for epoch={}. Last journal index={}", new Object[]{this.containerId, epochToRecover.get(), fileIndexToRecover.get()});
                                return null;
                            }
                        } else {
                            throw new CompletionException((Throwable)e);
                        }
                    }
                    fileIndexToRecover.incrementAndGet();
                    state.filesProcessedCount.incrementAndGet();
                    return v;
                }, this.executor);
            }, (Executor)this.executor);
        }, v -> epochToRecover.incrementAndGet(), (Executor)this.executor).thenRunAsync(() -> this.pendingGarbageChunks.addAll(journalsProcessed), this.executor);
    }

    private CompletableFuture<Void> processJournalContents(MetadataTransaction txn, BootstrapState state, String systemLogName, ByteArrayInputStream input) {
        AtomicBoolean isBatchDone = new AtomicBoolean();
        return Futures.loop(() -> !isBatchDone.get(), () -> {
            try {
                log.debug("SystemJournal[{}] Processing journal {}.", (Object)this.containerId, (Object)systemLogName);
                SystemJournalRecordBatch batch = (SystemJournalRecordBatch)BATCH_SERIALIZER.deserialize(input);
                return Futures.loop(batch.getSystemJournalRecords(), record -> this.applyRecord(txn, state, (SystemJournalRecord)record).thenApply(r -> true), (Executor)this.executor);
            }
            catch (EOFException e) {
                log.debug("SystemJournal[{}] Done processing journal {}.", (Object)this.containerId, (Object)systemLogName);
                isBatchDone.set(true);
            }
            catch (Exception e) {
                log.error("SystemJournal[{}] Error while processing journal {}.", new Object[]{this.containerId, systemLogName, e});
                throw new CompletionException(e);
            }
            return CompletableFuture.completedFuture(null);
        }, (Executor)this.executor);
    }

    private CompletableFuture<Void> applyRecord(MetadataTransaction txn, BootstrapState state, SystemJournalRecord record) {
        log.trace("SystemJournal[{}] Processing system log record ={}.", (Object)this.epoch, (Object)record);
        if (state.visitedRecords.contains(record)) {
            return CompletableFuture.completedFuture(null);
        }
        state.visitedRecords.add(record);
        state.recordsProcessedCount.incrementAndGet();
        CompletableFuture<Void> retValue = null;
        if (record instanceof ChunkAddedRecord) {
            ChunkAddedRecord chunkAddedRecord = (ChunkAddedRecord)record;
            retValue = this.applyChunkAddition(txn, state.chunkStartOffsets, chunkAddedRecord.getSegmentName(), Strings.nullToEmpty((String)chunkAddedRecord.getOldChunkName()), chunkAddedRecord.getNewChunkName(), chunkAddedRecord.getOffset());
        } else if (record instanceof TruncationRecord) {
            TruncationRecord truncationRecord = (TruncationRecord)record;
            state.finalTruncateOffsets.put(truncationRecord.getSegmentName(), truncationRecord.getOffset());
            state.finalFirstChunkStartsAtOffsets.put(truncationRecord.getSegmentName(), truncationRecord.getStartOffset());
            retValue = CompletableFuture.completedFuture(null);
        } else if (record instanceof SystemSnapshotRecord) {
            SystemSnapshotRecord snapshotRecord = (SystemSnapshotRecord)record;
            retValue = Futures.toVoid(this.applySystemSnapshotRecord(txn, state, snapshotRecord));
        }
        if (null == retValue) {
            retValue = CompletableFuture.failedFuture(new IllegalStateException(String.format("Unknown record type encountered. record = %s", record)));
        }
        return retValue;
    }

    private CompletableFuture<Void> adjustLastChunkLengths(MetadataTransaction txn) {
        ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
        for (String systemSegment : this.systemSegments) {
            CompletionStage f = ((CompletableFuture)txn.get(systemSegment).thenComposeAsync(m -> {
                SegmentMetadata segmentMetadata = (SegmentMetadata)m;
                segmentMetadata.checkInvariants();
                CompletionStage<Object> ff = null != segmentMetadata.getLastChunk() ? this.chunkStorage.getInfo(segmentMetadata.getLastChunk()).thenComposeAsync(chunkInfo -> {
                    long length = chunkInfo.getLength();
                    return txn.get(segmentMetadata.getLastChunk()).thenAcceptAsync(mm -> {
                        ChunkMetadata lastChunk = (ChunkMetadata)mm;
                        Preconditions.checkState((null != lastChunk ? 1 : 0) != 0, (String)"lastChunk must not be null. Segment=%s", (Object)segmentMetadata);
                        lastChunk.setLength(length);
                        txn.update(lastChunk);
                        long newLength = segmentMetadata.getLastChunkStartOffset() + length;
                        segmentMetadata.setLength(newLength);
                        log.debug("SystemJournal[{}] Adjusting length of last chunk segment. segment={}, length={} chunk={}, chunk length={}", new Object[]{this.containerId, segmentMetadata.getName(), length, lastChunk.getName(), newLength});
                    }, this.executor);
                }, this.executor) : CompletableFuture.completedFuture(null);
                return ff.thenApplyAsync(v -> {
                    Preconditions.checkState((boolean)segmentMetadata.isOwnershipChanged(), (String)"ownershipChanged must be true. Segment=%s", (Object)segmentMetadata);
                    segmentMetadata.checkInvariants();
                    return segmentMetadata;
                }, this.executor);
            }, this.executor)).thenAcceptAsync(segmentMetadata -> txn.update((StorageMetadata)segmentMetadata), this.executor);
            futures.add(f);
        }
        return Futures.allOf(futures);
    }

    private CompletableFuture<Void> applyFinalTruncateOffsets(MetadataTransaction txn, BootstrapState state) {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        for (String systemSegment : this.systemSegments) {
            if (!state.finalTruncateOffsets.containsKey(systemSegment)) continue;
            Long truncateAt = state.finalTruncateOffsets.get(systemSegment);
            Long firstChunkStartsAt = state.finalFirstChunkStartsAtOffsets.get(systemSegment);
            futures.add(this.applyTruncate(txn, systemSegment, truncateAt, firstChunkStartsAt));
        }
        return Futures.allOf(futures);
    }

    private CompletableFuture<Void> applyChunkAddition(MetadataTransaction txn, Map<String, Long> chunkStartOffsets, String segmentName, String oldChunkName, String newChunkName, long offset) {
        Preconditions.checkState((null != oldChunkName ? 1 : 0) != 0, (Object)"oldChunkName must not be null");
        Preconditions.checkState((null != newChunkName && !newChunkName.isEmpty() ? 1 : 0) != 0, (Object)"newChunkName must not be null or empty");
        return txn.get(segmentName).thenComposeAsync(m -> {
            CompletionStage<Object> f;
            SegmentMetadata segmentMetadata = (SegmentMetadata)m;
            segmentMetadata.checkInvariants();
            this.validateSegment(txn, segmentName);
            segmentMetadata.setLength(offset);
            ChunkMetadata newChunkMetadata = ChunkMetadata.builder().name(newChunkName).build();
            newChunkMetadata.setActive(true);
            txn.create(newChunkMetadata);
            txn.markPinned(newChunkMetadata);
            chunkStartOffsets.put(newChunkName, offset);
            if (!oldChunkName.isEmpty()) {
                Preconditions.checkState((boolean)txn.getData().containsKey(oldChunkName), (String)"Txn must contain old key", (Object)oldChunkName);
                f = txn.get(oldChunkName).thenComposeAsync(mm -> {
                    ChunkMetadata oldChunk = (ChunkMetadata)mm;
                    Preconditions.checkState((null != oldChunk ? 1 : 0) != 0, (String)"oldChunk must not be null. oldChunkName=%s", (Object)oldChunkName);
                    AtomicReference<String> toDelete = new AtomicReference<String>(oldChunk.getNextChunk());
                    return Futures.loop(() -> toDelete.get() != null, () -> txn.get((String)toDelete.get()).thenAcceptAsync(mmm -> {
                        ChunkMetadata chunkToDelete = (ChunkMetadata)mmm;
                        txn.delete((String)toDelete.get());
                        segmentMetadata.setChunkCount(segmentMetadata.getChunkCount() - 1);
                        toDelete.set(chunkToDelete.getNextChunk());
                    }, this.executor), (Executor)this.executor).thenAcceptAsync(v -> {
                        oldChunk.setNextChunk(newChunkName);
                        Long oldLength = (Long)chunkStartOffsets.get(oldChunkName);
                        oldChunk.setLength(offset - oldLength);
                        txn.update(oldChunk);
                    }, this.executor);
                }, this.executor);
            } else {
                segmentMetadata.setFirstChunk(newChunkName);
                segmentMetadata.setStartOffset(offset);
                Preconditions.checkState((segmentMetadata.getChunkCount() == 0 ? 1 : 0) != 0, (String)"Chunk count must be 0. %s", (Object)segmentMetadata);
                f = CompletableFuture.completedFuture(null);
            }
            return f.thenComposeAsync(v -> {
                segmentMetadata.setLastChunk(newChunkName);
                segmentMetadata.setLastChunkStartOffset(offset);
                segmentMetadata.setChunkCount(segmentMetadata.getChunkCount() + 1);
                segmentMetadata.checkInvariants();
                txn.update(segmentMetadata);
                if (this.config.isSelfCheckEnabled()) {
                    return this.validateSegment(txn, segmentName);
                }
                return CompletableFuture.completedFuture(null);
            }, this.executor);
        }, this.executor);
    }

    private CompletableFuture<Void> applyTruncate(MetadataTransaction txn, String segmentName, long truncateAt, long firstChunkStartsAt) {
        return txn.get(segmentName).thenComposeAsync(metadata -> {
            SegmentMetadata segmentMetadata = (SegmentMetadata)metadata;
            segmentMetadata.checkInvariants();
            AtomicReference<String> currentChunkName = new AtomicReference<String>(segmentMetadata.getFirstChunk());
            AtomicReference currentMetadata = new AtomicReference();
            AtomicLong startOffset = new AtomicLong(segmentMetadata.getFirstChunkStartOffset());
            AtomicBoolean shouldBreak = new AtomicBoolean();
            return Futures.loop(() -> null != currentChunkName.get() && !shouldBreak.get(), () -> txn.get((String)currentChunkName.get()).thenAcceptAsync(m -> {
                currentMetadata.set((ChunkMetadata)m);
                if (startOffset.get() <= truncateAt && startOffset.get() + ((ChunkMetadata)currentMetadata.get()).getLength() > truncateAt) {
                    shouldBreak.set(true);
                } else {
                    startOffset.addAndGet(((ChunkMetadata)currentMetadata.get()).getLength());
                    currentChunkName.set(((ChunkMetadata)currentMetadata.get()).getNextChunk());
                    txn.delete(((ChunkMetadata)currentMetadata.get()).getName());
                    segmentMetadata.setChunkCount(segmentMetadata.getChunkCount() - 1);
                }
            }, this.executor), (Executor)this.executor).thenAcceptAsync(v -> {
                Preconditions.checkState((firstChunkStartsAt == startOffset.get() ? 1 : 0) != 0, (String)"firstChunkStartsAt (%s) must be equal to startOffset (%s)", (long)firstChunkStartsAt, (Object)startOffset);
                segmentMetadata.setFirstChunk((String)currentChunkName.get());
                if (null == currentChunkName.get()) {
                    segmentMetadata.setLastChunk(null);
                    segmentMetadata.setLastChunkStartOffset(firstChunkStartsAt);
                }
                segmentMetadata.setStartOffset(truncateAt);
                segmentMetadata.setFirstChunkStartOffset(firstChunkStartsAt);
                segmentMetadata.checkInvariants();
            }, this.executor);
        }, this.executor);
    }

    CompletableFuture<Boolean> validateAndSaveSnapshot(MetadataTransaction txn, boolean validateSegment, boolean validateChunks) {
        return this.createSystemSnapshotRecord(txn, validateSegment, validateChunks).thenComposeAsync(this::writeSystemSnapshotRecord, this.executor);
    }

    private CompletableFuture<SystemSnapshotRecord> createSystemSnapshotRecord(MetadataTransaction txn, boolean validateSegment, boolean validateChunks) {
        SystemSnapshotRecord systemSnapshot = SystemSnapshotRecord.builder().epoch(this.epoch).fileIndex(this.currentFileIndex.get()).segmentSnapshotRecords(new ArrayList<SegmentSnapshotRecord>()).build();
        List<CompletionStage> futures = Collections.synchronizedList(new ArrayList());
        for (String systemSegment : this.systemSegments) {
            CompletionStage future = txn.get(systemSegment).thenComposeAsync(metadata -> {
                SegmentMetadata segmentMetadata = (SegmentMetadata)metadata;
                segmentMetadata.checkInvariants();
                SegmentSnapshotRecord segmentSnapshot = SegmentSnapshotRecord.builder().segmentMetadata(segmentMetadata).chunkMetadataCollection(new ArrayList<ChunkMetadata>()).build();
                AtomicReference<String> currentChunkName = new AtomicReference<String>(segmentMetadata.getFirstChunk());
                AtomicLong dataSize = new AtomicLong();
                AtomicLong chunkCount = new AtomicLong();
                return Futures.loop(() -> null != currentChunkName.get(), () -> txn.get((String)currentChunkName.get()).thenComposeAsync(m -> {
                    ChunkMetadata currentChunkMetadata = (ChunkMetadata)m;
                    Preconditions.checkState((null != currentChunkMetadata ? 1 : 0) != 0, (Object)"currentChunkMetadata must not be null");
                    CompletionStage<Object> f = validateChunks ? this.chunkStorage.getInfo((String)currentChunkName.get()).thenAcceptAsync(chunkInfo -> Preconditions.checkState((chunkInfo.getLength() >= currentChunkMetadata.getLength() ? 1 : 0) != 0, (String)"Wrong chunk length chunkInfo=%d, currentMetadata=%d.", (long)chunkInfo.getLength(), (long)currentChunkMetadata.getLength()), this.executor) : CompletableFuture.completedFuture(null);
                    return f.thenAcceptAsync(v -> {
                        chunkCount.getAndIncrement();
                        dataSize.addAndGet(currentChunkMetadata.getLength());
                        segmentSnapshot.chunkMetadataCollection.add(currentChunkMetadata);
                        currentChunkName.set(currentChunkMetadata.getNextChunk());
                    }, this.executor);
                }, this.executor), (Executor)this.executor).thenAcceptAsync(v -> {
                    if (validateSegment) {
                        Preconditions.checkState((chunkCount.get() == (long)segmentMetadata.getChunkCount() ? 1 : 0) != 0, (String)"Wrong chunk count. Segment=%s", (Object)segmentMetadata);
                        Preconditions.checkState((dataSize.get() == segmentMetadata.getLength() - segmentMetadata.getFirstChunkStartOffset() ? 1 : 0) != 0, (String)"Data size does not match dataSize (%s). Segment=%s", (long)dataSize.get(), (Object)segmentMetadata);
                    }
                    SystemSnapshotRecord systemSnapshotRecord = systemSnapshot;
                    synchronized (systemSnapshotRecord) {
                        systemSnapshot.segmentSnapshotRecords.add(segmentSnapshot);
                    }
                }, this.executor);
            }, this.executor);
            futures.add(future);
        }
        return Futures.allOf(futures).thenApplyAsync(vv -> {
            systemSnapshot.checkInvariants();
            return systemSnapshot;
        }, this.executor);
    }

    private CompletableFuture<Boolean> writeSystemSnapshotRecord(SystemSnapshotRecord systemSnapshot) {
        ByteArraySegment bytes;
        try {
            bytes = SYSTEM_SNAPSHOT_SERIALIZER.serialize(systemSnapshot);
        }
        catch (IOException e2) {
            log.error("SystemJournal[{}] Error while creating snapshot {}", (Object)this.containerId, (Object)e2);
            return CompletableFuture.completedFuture(false);
        }
        AtomicBoolean isWritten = new AtomicBoolean();
        AtomicInteger attempt = new AtomicInteger();
        AtomicReference lastException = new AtomicReference();
        return ((CompletableFuture)Futures.loop(() -> attempt.get() < this.config.getMaxJournalWriteAttempts() && !isWritten.get(), () -> {
            this.currentSnapshotIndex.incrementAndGet();
            String snapshotFile = NameUtils.getSystemJournalSnapshotFileName((int)this.containerId, (long)this.epoch, (long)this.currentSnapshotIndex.get());
            return ((CompletableFuture)this.chunkStorage.createWithContent(snapshotFile, bytes.getLength(), new ByteArrayInputStream(bytes.array(), bytes.arrayOffset(), bytes.getLength())).thenComposeAsync(v -> this.getContents(snapshotFile).thenAcceptAsync(contents -> {
                try {
                    SystemSnapshotRecord snapshotReadback = (SystemSnapshotRecord)SYSTEM_SNAPSHOT_SERIALIZER.deserialize((byte[])contents);
                    if (this.config.isSelfCheckEnabled()) {
                        snapshotReadback.checkInvariants();
                    }
                    Preconditions.checkState((boolean)systemSnapshot.equals(snapshotReadback), (String)"Records do not match %s != %s", (Object)snapshotReadback, (Object)systemSnapshot);
                    this.lastSavedSystemSnapshot.set(systemSnapshot);
                    this.lastSavedSystemSnapshotId.set(this.currentSnapshotIndex.get());
                    isWritten.set(true);
                }
                catch (Exception e1) {
                    throw new CompletionException(Exceptions.unwrap((Throwable)e1));
                }
            }, this.executor), this.executor)).handleAsync((v, e) -> {
                this.newChunkRequired.set(true);
                attempt.incrementAndGet();
                if (e != null) {
                    lastException.set(Exceptions.unwrap((Throwable)e));
                    this.pendingGarbageChunks.add(snapshotFile);
                    return null;
                }
                return v;
            }, this.executor);
        }, (Executor)this.executor).thenApplyAsync(v -> isWritten.get(), this.executor)).whenCompleteAsync((v, e) -> {
            if (!isWritten.get() && null != lastException.get()) {
                throw new CompletionException((Throwable)lastException.get());
            }
        }, this.executor);
    }

    private CompletableFuture<Void> writeToJournal(ByteArraySegment bytes) {
        if (this.newChunkRequired.get()) {
            this.currentFileIndex.incrementAndGet();
            this.systemJournalOffset.set(0L);
            return this.chunkStorage.createWithContent(this.getSystemJournalChunkName(this.containerId, this.epoch, this.currentFileIndex.get()), bytes.getLength(), new ByteArrayInputStream(bytes.array(), bytes.arrayOffset(), bytes.getLength())).thenAcceptAsync(h -> {
                this.currentHandle.set((ChunkHandle)h);
                this.pendingGarbageChunks.add(h.getChunkName());
                this.systemJournalOffset.addAndGet(bytes.getLength());
                this.newChunkRequired.set(false);
            }, this.executor);
        }
        Preconditions.checkState((this.chunkStorage.supportsAppend() && this.config.isAppendEnabled() ? 1 : 0) != 0, (Object)"Append mode not enabled or chunk storage does not support appends.");
        return this.chunkStorage.write(this.currentHandle.get(), this.systemJournalOffset.get(), bytes.getLength(), new ByteArrayInputStream(bytes.array(), bytes.arrayOffset(), bytes.getLength())).thenAcceptAsync(bytesWritten -> {
            Preconditions.checkState((bytesWritten.intValue() == bytes.getLength() ? 1 : 0) != 0, (String)"Bytes written do not match expected length. Actual=%d, expected=%d", (Object)bytesWritten, (int)bytes.getLength());
            this.systemJournalOffset.addAndGet(bytesWritten.intValue());
        }, this.executor);
    }

    public boolean isStorageSystemSegment(String segmentName) {
        if (segmentName.startsWith(this.systemSegmentsPrefix)) {
            for (String systemSegment : this.systemSegments) {
                if (!segmentName.equals(systemSegment)) continue;
                return true;
            }
        }
        return false;
    }

    public static String[] getChunkStorageSystemSegments(int containerId) {
        return new String[]{NameUtils.getStorageMetadataSegmentName((int)containerId), NameUtils.getAttributeSegmentName((String)NameUtils.getStorageMetadataSegmentName((int)containerId)), NameUtils.getMetadataSegmentName((int)containerId), NameUtils.getAttributeSegmentName((String)NameUtils.getMetadataSegmentName((int)containerId))};
    }

    private String getSystemJournalChunkName() {
        return this.getSystemJournalChunkName(this.containerId, this.epoch, this.currentFileIndex.get());
    }

    private String getSystemJournalChunkName(int containerId, long epoch, long currentFileIndex) {
        return NameUtils.getSystemJournalFileName((int)containerId, (long)epoch, (long)currentFileIndex);
    }

    private <R> CompletableFuture<R> executeSerialized(Callable<CompletableFuture<R>> operation) {
        return this.taskProcessor.add(Collections.singletonList(LOCK_KEY_NAME), () -> this.executeExclusive(operation));
    }

    private <R> CompletableFuture<R> executeExclusive(Callable<CompletableFuture<R>> operation) {
        return CompletableFuture.completedFuture(null).thenComposeAsync(v -> {
            try {
                return (CompletionStage)operation.call();
            }
            catch (CompletionException e) {
                throw new CompletionException(Exceptions.unwrap((Throwable)e));
            }
            catch (Exception e) {
                throw new CompletionException(e);
            }
        }, this.executor);
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public ChunkStorage getChunkStorage() {
        return this.chunkStorage;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public ChunkMetadataStore getMetadataStore() {
        return this.metadataStore;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public long getEpoch() {
        return this.epoch;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public int getContainerId() {
        return this.containerId;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public AtomicInteger getCurrentFileIndex() {
        return this.currentFileIndex;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public String getSystemSegmentsPrefix() {
        return this.systemSegmentsPrefix;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public String[] getSystemSegments() {
        return this.systemSegments;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public ChunkedSegmentStorageConfig getConfig() {
        return this.config;
    }

    static class SystemSnapshotRecord
    extends SystemJournalRecord {
        private final long epoch;
        private final int fileIndex;
        @NonNull
        private final Collection<SegmentSnapshotRecord> segmentSnapshotRecords;

        public void checkInvariants() {
            for (SegmentSnapshotRecord segmentSnapshot : this.getSegmentSnapshotRecords()) {
                segmentSnapshot.checkInvariants();
            }
        }

        @ConstructorProperties(value={"epoch", "fileIndex", "segmentSnapshotRecords"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        SystemSnapshotRecord(long epoch, int fileIndex, @NonNull Collection<SegmentSnapshotRecord> segmentSnapshotRecords) {
            if (segmentSnapshotRecords == null) {
                throw new NullPointerException("segmentSnapshotRecords is marked non-null but is null");
            }
            this.epoch = epoch;
            this.fileIndex = fileIndex;
            this.segmentSnapshotRecords = segmentSnapshotRecords;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public static SystemSnapshotRecordBuilder builder() {
            return new SystemSnapshotRecordBuilder();
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public SystemSnapshotRecordBuilder toBuilder() {
            return new SystemSnapshotRecordBuilder().epoch(this.epoch).fileIndex(this.fileIndex).segmentSnapshotRecords(this.segmentSnapshotRecords);
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public long getEpoch() {
            return this.epoch;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int getFileIndex() {
            return this.fileIndex;
        }

        @NonNull
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public Collection<SegmentSnapshotRecord> getSegmentSnapshotRecords() {
            return this.segmentSnapshotRecords;
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String toString() {
            return "SystemJournal.SystemSnapshotRecord(epoch=" + this.getEpoch() + ", fileIndex=" + this.getFileIndex() + ", segmentSnapshotRecords=" + this.getSegmentSnapshotRecords() + ")";
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof SystemSnapshotRecord)) {
                return false;
            }
            SystemSnapshotRecord other = (SystemSnapshotRecord)o;
            if (!other.canEqual(this)) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            if (this.getEpoch() != other.getEpoch()) {
                return false;
            }
            if (this.getFileIndex() != other.getFileIndex()) {
                return false;
            }
            Collection<SegmentSnapshotRecord> this$segmentSnapshotRecords = this.getSegmentSnapshotRecords();
            Collection<SegmentSnapshotRecord> other$segmentSnapshotRecords = other.getSegmentSnapshotRecords();
            return !(this$segmentSnapshotRecords == null ? other$segmentSnapshotRecords != null : !((Object)this$segmentSnapshotRecords).equals(other$segmentSnapshotRecords));
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        protected boolean canEqual(Object other) {
            return other instanceof SystemSnapshotRecord;
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int hashCode() {
            int PRIME = 59;
            int result = super.hashCode();
            long $epoch = this.getEpoch();
            result = result * 59 + (int)($epoch >>> 32 ^ $epoch);
            result = result * 59 + this.getFileIndex();
            Collection<SegmentSnapshotRecord> $segmentSnapshotRecords = this.getSegmentSnapshotRecords();
            result = result * 59 + ($segmentSnapshotRecords == null ? 43 : ((Object)$segmentSnapshotRecords).hashCode());
            return result;
        }

        public static class Serializer
        extends VersionedSerializer.WithBuilder<SystemSnapshotRecord, SystemSnapshotRecordBuilder> {
            private static final SegmentSnapshotRecord.Serializer CHUNK_METADATA_SERIALIZER = new SegmentSnapshotRecord.Serializer();
            private static final RevisionDataOutput.ElementSerializer<SegmentSnapshotRecord> ELEMENT_SERIALIZER = (arg_0, arg_1) -> ((SegmentSnapshotRecord.Serializer)CHUNK_METADATA_SERIALIZER).serialize(arg_0, arg_1);
            private static final RevisionDataInput.ElementDeserializer<SegmentSnapshotRecord> ELEMENT_DESERIALIZER = dataInput -> (SegmentSnapshotRecord)CHUNK_METADATA_SERIALIZER.deserialize(dataInput.getBaseStream());

            protected SystemSnapshotRecordBuilder newBuilder() {
                return SystemSnapshotRecord.builder();
            }

            protected byte getWriteVersion() {
                return 0;
            }

            protected void declareVersions() {
                this.version(0).revision(0, this::write00, this::read00);
            }

            private void write00(SystemSnapshotRecord object, RevisionDataOutput output) throws IOException {
                output.writeCompactLong(object.epoch);
                output.writeCompactInt(object.fileIndex);
                output.writeCollection(object.segmentSnapshotRecords, ELEMENT_SERIALIZER);
            }

            private void read00(RevisionDataInput input, SystemSnapshotRecordBuilder b) throws IOException {
                b.epoch(input.readCompactLong());
                b.fileIndex(input.readCompactInt());
                b.segmentSnapshotRecords(input.readCollection(ELEMENT_DESERIALIZER));
            }
        }

        public static class SystemSnapshotRecordBuilder
        implements ObjectBuilder<SystemSnapshotRecord> {
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private long epoch;
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private int fileIndex;
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private Collection<SegmentSnapshotRecord> segmentSnapshotRecords;

            @SuppressFBWarnings(justification="generated code")
            @Generated
            SystemSnapshotRecordBuilder() {
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public SystemSnapshotRecordBuilder epoch(long epoch) {
                this.epoch = epoch;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public SystemSnapshotRecordBuilder fileIndex(int fileIndex) {
                this.fileIndex = fileIndex;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public SystemSnapshotRecordBuilder segmentSnapshotRecords(@NonNull Collection<SegmentSnapshotRecord> segmentSnapshotRecords) {
                if (segmentSnapshotRecords == null) {
                    throw new NullPointerException("segmentSnapshotRecords is marked non-null but is null");
                }
                this.segmentSnapshotRecords = segmentSnapshotRecords;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public SystemSnapshotRecord build() {
                return new SystemSnapshotRecord(this.epoch, this.fileIndex, this.segmentSnapshotRecords);
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public String toString() {
                return "SystemJournal.SystemSnapshotRecord.SystemSnapshotRecordBuilder(epoch=" + this.epoch + ", fileIndex=" + this.fileIndex + ", segmentSnapshotRecords=" + this.segmentSnapshotRecords + ")";
            }
        }
    }

    static class SegmentSnapshotRecord
    extends SystemJournalRecord {
        @NonNull
        private final SegmentMetadata segmentMetadata;
        @NonNull
        private final Collection<ChunkMetadata> chunkMetadataCollection;

        public void checkInvariants() {
            this.segmentMetadata.checkInvariants();
            Preconditions.checkState((boolean)this.segmentMetadata.isStorageSystemSegment(), (String)"Segment must be storage segment. Segment snapshot= %s", (Object)this);
            Preconditions.checkState((this.segmentMetadata.getChunkCount() == this.chunkMetadataCollection.size() ? 1 : 0) != 0, (String)"Chunk count must match. Segment snapshot= %s", (Object)this);
            long dataSize = 0L;
            ChunkMetadata previous = null;
            ChunkMetadata firstChunk = null;
            for (ChunkMetadata metadata : this.getChunkMetadataCollection()) {
                dataSize += metadata.getLength();
                if (previous != null) {
                    Preconditions.checkState((boolean)previous.getNextChunk().equals(metadata.getName()), (String)"In correct link . chunk %s must point to chunk %s. Segment snapshot= %s", (Object)previous.getName(), (Object)metadata.getName(), (Object)this);
                } else {
                    firstChunk = metadata;
                }
                previous = metadata;
            }
            Preconditions.checkState((dataSize == this.segmentMetadata.getLength() - this.segmentMetadata.getFirstChunkStartOffset() ? 1 : 0) != 0, (String)"Data size does not match dataSize (%s). Segment=%s", (long)dataSize, (Object)this.segmentMetadata);
            if (this.chunkMetadataCollection.size() > 0) {
                Preconditions.checkState((boolean)this.segmentMetadata.getFirstChunk().equals(firstChunk.getName()), (String)"First chunk name is wrong. Segment snapshot= %s", (Object)this);
                Preconditions.checkState((boolean)this.segmentMetadata.getLastChunk().equals(previous.getName()), (String)"Last chunk name is wrong. Segment snapshot= %s", (Object)this);
                Preconditions.checkState((previous.getNextChunk() == null ? 1 : 0) != 0, (String)"Invalid last chunk Segment snapshot= %s", (Object)this);
                Preconditions.checkState((this.segmentMetadata.getLength() == this.segmentMetadata.getLastChunkStartOffset() + previous.getLength() ? 1 : 0) != 0, (String)"Last chunk start offset is wrong. snapshot= %s", (Object)this);
            }
        }

        @ConstructorProperties(value={"segmentMetadata", "chunkMetadataCollection"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        SegmentSnapshotRecord(@NonNull SegmentMetadata segmentMetadata, @NonNull Collection<ChunkMetadata> chunkMetadataCollection) {
            if (segmentMetadata == null) {
                throw new NullPointerException("segmentMetadata is marked non-null but is null");
            }
            if (chunkMetadataCollection == null) {
                throw new NullPointerException("chunkMetadataCollection is marked non-null but is null");
            }
            this.segmentMetadata = segmentMetadata;
            this.chunkMetadataCollection = chunkMetadataCollection;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public static SegmentSnapshotRecordBuilder builder() {
            return new SegmentSnapshotRecordBuilder();
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public SegmentSnapshotRecordBuilder toBuilder() {
            return new SegmentSnapshotRecordBuilder().segmentMetadata(this.segmentMetadata).chunkMetadataCollection(this.chunkMetadataCollection);
        }

        @NonNull
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public SegmentMetadata getSegmentMetadata() {
            return this.segmentMetadata;
        }

        @NonNull
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public Collection<ChunkMetadata> getChunkMetadataCollection() {
            return this.chunkMetadataCollection;
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String toString() {
            return "SystemJournal.SegmentSnapshotRecord(segmentMetadata=" + this.getSegmentMetadata() + ", chunkMetadataCollection=" + this.getChunkMetadataCollection() + ")";
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof SegmentSnapshotRecord)) {
                return false;
            }
            SegmentSnapshotRecord other = (SegmentSnapshotRecord)o;
            if (!other.canEqual(this)) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            SegmentMetadata this$segmentMetadata = this.getSegmentMetadata();
            SegmentMetadata other$segmentMetadata = other.getSegmentMetadata();
            if (this$segmentMetadata == null ? other$segmentMetadata != null : !((Object)this$segmentMetadata).equals(other$segmentMetadata)) {
                return false;
            }
            Collection<ChunkMetadata> this$chunkMetadataCollection = this.getChunkMetadataCollection();
            Collection<ChunkMetadata> other$chunkMetadataCollection = other.getChunkMetadataCollection();
            return !(this$chunkMetadataCollection == null ? other$chunkMetadataCollection != null : !((Object)this$chunkMetadataCollection).equals(other$chunkMetadataCollection));
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        protected boolean canEqual(Object other) {
            return other instanceof SegmentSnapshotRecord;
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int hashCode() {
            int PRIME = 59;
            int result = super.hashCode();
            SegmentMetadata $segmentMetadata = this.getSegmentMetadata();
            result = result * 59 + ($segmentMetadata == null ? 43 : ((Object)$segmentMetadata).hashCode());
            Collection<ChunkMetadata> $chunkMetadataCollection = this.getChunkMetadataCollection();
            result = result * 59 + ($chunkMetadataCollection == null ? 43 : ((Object)$chunkMetadataCollection).hashCode());
            return result;
        }

        public static class Serializer
        extends VersionedSerializer.WithBuilder<SegmentSnapshotRecord, SegmentSnapshotRecordBuilder> {
            private static final StorageMetadata.StorageMetadataSerializer SEGMENT_METADATA_SERIALIZER = new StorageMetadata.StorageMetadataSerializer();
            private static final StorageMetadata.StorageMetadataSerializer CHUNK_METADATA_SERIALIZER = new StorageMetadata.StorageMetadataSerializer();
            private static final RevisionDataOutput.ElementSerializer<ChunkMetadata> ELEMENT_SERIALIZER = (arg_0, arg_1) -> ((StorageMetadata.StorageMetadataSerializer)CHUNK_METADATA_SERIALIZER).serialize(arg_0, arg_1);
            private static final RevisionDataInput.ElementDeserializer<ChunkMetadata> ELEMENT_DESERIALIZER = dataInput -> (ChunkMetadata)CHUNK_METADATA_SERIALIZER.deserialize(dataInput.getBaseStream());

            protected SegmentSnapshotRecordBuilder newBuilder() {
                return SegmentSnapshotRecord.builder();
            }

            protected byte getWriteVersion() {
                return 0;
            }

            protected void declareVersions() {
                this.version(0).revision(0, this::write00, this::read00);
            }

            private void write00(SegmentSnapshotRecord object, RevisionDataOutput output) throws IOException {
                SEGMENT_METADATA_SERIALIZER.serialize(output, object.segmentMetadata);
                output.writeCollection(object.chunkMetadataCollection, ELEMENT_SERIALIZER);
            }

            private void read00(RevisionDataInput input, SegmentSnapshotRecordBuilder b) throws IOException {
                b.segmentMetadata((SegmentMetadata)SEGMENT_METADATA_SERIALIZER.deserialize(input.getBaseStream()));
                b.chunkMetadataCollection(input.readCollection(ELEMENT_DESERIALIZER));
            }
        }

        public static class SegmentSnapshotRecordBuilder
        implements ObjectBuilder<SegmentSnapshotRecord> {
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private SegmentMetadata segmentMetadata;
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private Collection<ChunkMetadata> chunkMetadataCollection;

            @SuppressFBWarnings(justification="generated code")
            @Generated
            SegmentSnapshotRecordBuilder() {
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public SegmentSnapshotRecordBuilder segmentMetadata(@NonNull SegmentMetadata segmentMetadata) {
                if (segmentMetadata == null) {
                    throw new NullPointerException("segmentMetadata is marked non-null but is null");
                }
                this.segmentMetadata = segmentMetadata;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public SegmentSnapshotRecordBuilder chunkMetadataCollection(@NonNull Collection<ChunkMetadata> chunkMetadataCollection) {
                if (chunkMetadataCollection == null) {
                    throw new NullPointerException("chunkMetadataCollection is marked non-null but is null");
                }
                this.chunkMetadataCollection = chunkMetadataCollection;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public SegmentSnapshotRecord build() {
                return new SegmentSnapshotRecord(this.segmentMetadata, this.chunkMetadataCollection);
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public String toString() {
                return "SystemJournal.SegmentSnapshotRecord.SegmentSnapshotRecordBuilder(segmentMetadata=" + this.segmentMetadata + ", chunkMetadataCollection=" + this.chunkMetadataCollection + ")";
            }
        }
    }

    static class TruncationRecord
    extends SystemJournalRecord {
        @NonNull
        private final String segmentName;
        private final long offset;
        @NonNull
        private final String firstChunkName;
        private final long startOffset;

        @ConstructorProperties(value={"segmentName", "offset", "firstChunkName", "startOffset"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        TruncationRecord(@NonNull String segmentName, long offset, @NonNull String firstChunkName, long startOffset) {
            if (segmentName == null) {
                throw new NullPointerException("segmentName is marked non-null but is null");
            }
            if (firstChunkName == null) {
                throw new NullPointerException("firstChunkName is marked non-null but is null");
            }
            this.segmentName = segmentName;
            this.offset = offset;
            this.firstChunkName = firstChunkName;
            this.startOffset = startOffset;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public static TruncationRecordBuilder builder() {
            return new TruncationRecordBuilder();
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public TruncationRecordBuilder toBuilder() {
            return new TruncationRecordBuilder().segmentName(this.segmentName).offset(this.offset).firstChunkName(this.firstChunkName).startOffset(this.startOffset);
        }

        @NonNull
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String getSegmentName() {
            return this.segmentName;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public long getOffset() {
            return this.offset;
        }

        @NonNull
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String getFirstChunkName() {
            return this.firstChunkName;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public long getStartOffset() {
            return this.startOffset;
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String toString() {
            return "SystemJournal.TruncationRecord(segmentName=" + this.getSegmentName() + ", offset=" + this.getOffset() + ", firstChunkName=" + this.getFirstChunkName() + ", startOffset=" + this.getStartOffset() + ")";
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof TruncationRecord)) {
                return false;
            }
            TruncationRecord other = (TruncationRecord)o;
            if (!other.canEqual(this)) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            String this$segmentName = this.getSegmentName();
            String other$segmentName = other.getSegmentName();
            if (this$segmentName == null ? other$segmentName != null : !this$segmentName.equals(other$segmentName)) {
                return false;
            }
            if (this.getOffset() != other.getOffset()) {
                return false;
            }
            String this$firstChunkName = this.getFirstChunkName();
            String other$firstChunkName = other.getFirstChunkName();
            if (this$firstChunkName == null ? other$firstChunkName != null : !this$firstChunkName.equals(other$firstChunkName)) {
                return false;
            }
            return this.getStartOffset() == other.getStartOffset();
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        protected boolean canEqual(Object other) {
            return other instanceof TruncationRecord;
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int hashCode() {
            int PRIME = 59;
            int result = super.hashCode();
            String $segmentName = this.getSegmentName();
            result = result * 59 + ($segmentName == null ? 43 : $segmentName.hashCode());
            long $offset = this.getOffset();
            result = result * 59 + (int)($offset >>> 32 ^ $offset);
            String $firstChunkName = this.getFirstChunkName();
            result = result * 59 + ($firstChunkName == null ? 43 : $firstChunkName.hashCode());
            long $startOffset = this.getStartOffset();
            result = result * 59 + (int)($startOffset >>> 32 ^ $startOffset);
            return result;
        }

        public static class Serializer
        extends VersionedSerializer.WithBuilder<TruncationRecord, TruncationRecordBuilder> {
            protected TruncationRecordBuilder newBuilder() {
                return TruncationRecord.builder();
            }

            protected byte getWriteVersion() {
                return 0;
            }

            protected void declareVersions() {
                this.version(0).revision(0, this::write00, this::read00);
            }

            private void write00(TruncationRecord object, RevisionDataOutput output) throws IOException {
                output.writeUTF(object.segmentName);
                output.writeCompactLong(object.offset);
                output.writeUTF(object.firstChunkName);
                output.writeCompactLong(object.startOffset);
            }

            private void read00(RevisionDataInput input, TruncationRecordBuilder b) throws IOException {
                b.segmentName(input.readUTF());
                b.offset(input.readCompactLong());
                b.firstChunkName(input.readUTF());
                b.startOffset(input.readCompactLong());
            }
        }

        public static class TruncationRecordBuilder
        implements ObjectBuilder<TruncationRecord> {
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private String segmentName;
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private long offset;
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private String firstChunkName;
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private long startOffset;

            @SuppressFBWarnings(justification="generated code")
            @Generated
            TruncationRecordBuilder() {
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public TruncationRecordBuilder segmentName(@NonNull String segmentName) {
                if (segmentName == null) {
                    throw new NullPointerException("segmentName is marked non-null but is null");
                }
                this.segmentName = segmentName;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public TruncationRecordBuilder offset(long offset) {
                this.offset = offset;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public TruncationRecordBuilder firstChunkName(@NonNull String firstChunkName) {
                if (firstChunkName == null) {
                    throw new NullPointerException("firstChunkName is marked non-null but is null");
                }
                this.firstChunkName = firstChunkName;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public TruncationRecordBuilder startOffset(long startOffset) {
                this.startOffset = startOffset;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public TruncationRecord build() {
                return new TruncationRecord(this.segmentName, this.offset, this.firstChunkName, this.startOffset);
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public String toString() {
                return "SystemJournal.TruncationRecord.TruncationRecordBuilder(segmentName=" + this.segmentName + ", offset=" + this.offset + ", firstChunkName=" + this.firstChunkName + ", startOffset=" + this.startOffset + ")";
            }
        }
    }

    static class ChunkAddedRecord
    extends SystemJournalRecord {
        @NonNull
        private final String segmentName;
        private final long offset;
        private final String oldChunkName;
        @NonNull
        private final String newChunkName;

        @ConstructorProperties(value={"segmentName", "offset", "oldChunkName", "newChunkName"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        ChunkAddedRecord(@NonNull String segmentName, long offset, String oldChunkName, @NonNull String newChunkName) {
            if (segmentName == null) {
                throw new NullPointerException("segmentName is marked non-null but is null");
            }
            if (newChunkName == null) {
                throw new NullPointerException("newChunkName is marked non-null but is null");
            }
            this.segmentName = segmentName;
            this.offset = offset;
            this.oldChunkName = oldChunkName;
            this.newChunkName = newChunkName;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public static ChunkAddedRecordBuilder builder() {
            return new ChunkAddedRecordBuilder();
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public ChunkAddedRecordBuilder toBuilder() {
            return new ChunkAddedRecordBuilder().segmentName(this.segmentName).offset(this.offset).oldChunkName(this.oldChunkName).newChunkName(this.newChunkName);
        }

        @NonNull
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String getSegmentName() {
            return this.segmentName;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public long getOffset() {
            return this.offset;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String getOldChunkName() {
            return this.oldChunkName;
        }

        @NonNull
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String getNewChunkName() {
            return this.newChunkName;
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String toString() {
            return "SystemJournal.ChunkAddedRecord(segmentName=" + this.getSegmentName() + ", offset=" + this.getOffset() + ", oldChunkName=" + this.getOldChunkName() + ", newChunkName=" + this.getNewChunkName() + ")";
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof ChunkAddedRecord)) {
                return false;
            }
            ChunkAddedRecord other = (ChunkAddedRecord)o;
            if (!other.canEqual(this)) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            String this$segmentName = this.getSegmentName();
            String other$segmentName = other.getSegmentName();
            if (this$segmentName == null ? other$segmentName != null : !this$segmentName.equals(other$segmentName)) {
                return false;
            }
            if (this.getOffset() != other.getOffset()) {
                return false;
            }
            String this$oldChunkName = this.getOldChunkName();
            String other$oldChunkName = other.getOldChunkName();
            if (this$oldChunkName == null ? other$oldChunkName != null : !this$oldChunkName.equals(other$oldChunkName)) {
                return false;
            }
            String this$newChunkName = this.getNewChunkName();
            String other$newChunkName = other.getNewChunkName();
            return !(this$newChunkName == null ? other$newChunkName != null : !this$newChunkName.equals(other$newChunkName));
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        protected boolean canEqual(Object other) {
            return other instanceof ChunkAddedRecord;
        }

        @Override
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int hashCode() {
            int PRIME = 59;
            int result = super.hashCode();
            String $segmentName = this.getSegmentName();
            result = result * 59 + ($segmentName == null ? 43 : $segmentName.hashCode());
            long $offset = this.getOffset();
            result = result * 59 + (int)($offset >>> 32 ^ $offset);
            String $oldChunkName = this.getOldChunkName();
            result = result * 59 + ($oldChunkName == null ? 43 : $oldChunkName.hashCode());
            String $newChunkName = this.getNewChunkName();
            result = result * 59 + ($newChunkName == null ? 43 : $newChunkName.hashCode());
            return result;
        }

        public static class Serializer
        extends VersionedSerializer.WithBuilder<ChunkAddedRecord, ChunkAddedRecordBuilder> {
            protected ChunkAddedRecordBuilder newBuilder() {
                return ChunkAddedRecord.builder();
            }

            protected byte getWriteVersion() {
                return 0;
            }

            protected void declareVersions() {
                this.version(0).revision(0, this::write00, this::read00);
            }

            private void write00(ChunkAddedRecord object, RevisionDataOutput output) throws IOException {
                output.writeUTF(object.segmentName);
                output.writeUTF(Strings.nullToEmpty((String)object.newChunkName));
                output.writeUTF(Strings.nullToEmpty((String)object.oldChunkName));
                output.writeCompactLong(object.offset);
            }

            private void read00(RevisionDataInput input, ChunkAddedRecordBuilder b) throws IOException {
                b.segmentName(input.readUTF());
                b.newChunkName(Strings.emptyToNull((String)input.readUTF()));
                b.oldChunkName(Strings.emptyToNull((String)input.readUTF()));
                b.offset(input.readCompactLong());
            }
        }

        public static class ChunkAddedRecordBuilder
        implements ObjectBuilder<ChunkAddedRecord> {
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private String segmentName;
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private long offset;
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private String oldChunkName;
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private String newChunkName;

            @SuppressFBWarnings(justification="generated code")
            @Generated
            ChunkAddedRecordBuilder() {
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public ChunkAddedRecordBuilder segmentName(@NonNull String segmentName) {
                if (segmentName == null) {
                    throw new NullPointerException("segmentName is marked non-null but is null");
                }
                this.segmentName = segmentName;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public ChunkAddedRecordBuilder offset(long offset) {
                this.offset = offset;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public ChunkAddedRecordBuilder oldChunkName(String oldChunkName) {
                this.oldChunkName = oldChunkName;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public ChunkAddedRecordBuilder newChunkName(@NonNull String newChunkName) {
                if (newChunkName == null) {
                    throw new NullPointerException("newChunkName is marked non-null but is null");
                }
                this.newChunkName = newChunkName;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public ChunkAddedRecord build() {
                return new ChunkAddedRecord(this.segmentName, this.offset, this.oldChunkName, this.newChunkName);
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public String toString() {
                return "SystemJournal.ChunkAddedRecord.ChunkAddedRecordBuilder(segmentName=" + this.segmentName + ", offset=" + this.offset + ", oldChunkName=" + this.oldChunkName + ", newChunkName=" + this.newChunkName + ")";
            }
        }
    }

    static class SystemJournalRecordBatch {
        @NonNull
        private final Collection<SystemJournalRecord> systemJournalRecords;

        @ConstructorProperties(value={"systemJournalRecords"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        SystemJournalRecordBatch(@NonNull Collection<SystemJournalRecord> systemJournalRecords) {
            if (systemJournalRecords == null) {
                throw new NullPointerException("systemJournalRecords is marked non-null but is null");
            }
            this.systemJournalRecords = systemJournalRecords;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public static SystemJournalRecordBatchBuilder builder() {
            return new SystemJournalRecordBatchBuilder();
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public SystemJournalRecordBatchBuilder toBuilder() {
            return new SystemJournalRecordBatchBuilder().systemJournalRecords(this.systemJournalRecords);
        }

        @NonNull
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public Collection<SystemJournalRecord> getSystemJournalRecords() {
            return this.systemJournalRecords;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String toString() {
            return "SystemJournal.SystemJournalRecordBatch(systemJournalRecords=" + this.getSystemJournalRecords() + ")";
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof SystemJournalRecordBatch)) {
                return false;
            }
            SystemJournalRecordBatch other = (SystemJournalRecordBatch)o;
            if (!other.canEqual(this)) {
                return false;
            }
            Collection<SystemJournalRecord> this$systemJournalRecords = this.getSystemJournalRecords();
            Collection<SystemJournalRecord> other$systemJournalRecords = other.getSystemJournalRecords();
            return !(this$systemJournalRecords == null ? other$systemJournalRecords != null : !((Object)this$systemJournalRecords).equals(other$systemJournalRecords));
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        protected boolean canEqual(Object other) {
            return other instanceof SystemJournalRecordBatch;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            Collection<SystemJournalRecord> $systemJournalRecords = this.getSystemJournalRecords();
            result = result * 59 + ($systemJournalRecords == null ? 43 : ((Object)$systemJournalRecords).hashCode());
            return result;
        }

        public static class SystemJournalRecordBatchSerializer
        extends VersionedSerializer.WithBuilder<SystemJournalRecordBatch, SystemJournalRecordBatchBuilder> {
            private static final SystemJournalRecord.SystemJournalRecordSerializer SERIALIZER = new SystemJournalRecord.SystemJournalRecordSerializer();
            private static final RevisionDataOutput.ElementSerializer<SystemJournalRecord> ELEMENT_SERIALIZER = (arg_0, arg_1) -> ((SystemJournalRecord.SystemJournalRecordSerializer)SERIALIZER).serialize(arg_0, arg_1);
            private static final RevisionDataInput.ElementDeserializer<SystemJournalRecord> ELEMENT_DESERIALIZER = dataInput -> (SystemJournalRecord)SERIALIZER.deserialize(dataInput.getBaseStream());

            protected SystemJournalRecordBatchBuilder newBuilder() {
                return SystemJournalRecordBatch.builder();
            }

            protected byte getWriteVersion() {
                return 0;
            }

            protected void declareVersions() {
                this.version(0).revision(0, this::write00, this::read00);
            }

            private void read00(RevisionDataInput input, SystemJournalRecordBatchBuilder b) throws IOException {
                b.systemJournalRecords(input.readCollection(ELEMENT_DESERIALIZER));
            }

            private void write00(SystemJournalRecordBatch object, RevisionDataOutput output) throws IOException {
                output.writeCollection(object.systemJournalRecords, ELEMENT_SERIALIZER);
            }
        }

        public static class SystemJournalRecordBatchBuilder
        implements ObjectBuilder<SystemJournalRecordBatch> {
            @SuppressFBWarnings(justification="generated code")
            @Generated
            private Collection<SystemJournalRecord> systemJournalRecords;

            @SuppressFBWarnings(justification="generated code")
            @Generated
            SystemJournalRecordBatchBuilder() {
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public SystemJournalRecordBatchBuilder systemJournalRecords(@NonNull Collection<SystemJournalRecord> systemJournalRecords) {
                if (systemJournalRecords == null) {
                    throw new NullPointerException("systemJournalRecords is marked non-null but is null");
                }
                this.systemJournalRecords = systemJournalRecords;
                return this;
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public SystemJournalRecordBatch build() {
                return new SystemJournalRecordBatch(this.systemJournalRecords);
            }

            @SuppressFBWarnings(justification="generated code")
            @Generated
            public String toString() {
                return "SystemJournal.SystemJournalRecordBatch.SystemJournalRecordBatchBuilder(systemJournalRecords=" + this.systemJournalRecords + ")";
            }
        }
    }

    public static class SystemJournalRecord {
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public SystemJournalRecord() {
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof SystemJournalRecord)) {
                return false;
            }
            SystemJournalRecord other = (SystemJournalRecord)o;
            return other.canEqual(this);
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        protected boolean canEqual(Object other) {
            return other instanceof SystemJournalRecord;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int hashCode() {
            boolean result = true;
            return 1;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String toString() {
            return "SystemJournal.SystemJournalRecord()";
        }

        public static class SystemJournalRecordSerializer
        extends VersionedSerializer.MultiType<SystemJournalRecord> {
            protected void declareSerializers(VersionedSerializer.MultiType.Builder builder) {
                builder.serializer(ChunkAddedRecord.class, 1, (VersionedSerializer.WithBuilder)new ChunkAddedRecord.Serializer()).serializer(TruncationRecord.class, 2, (VersionedSerializer.WithBuilder)new TruncationRecord.Serializer()).serializer(SystemSnapshotRecord.class, 3, (VersionedSerializer.WithBuilder)new SystemSnapshotRecord.Serializer()).serializer(SegmentSnapshotRecord.class, 4, (VersionedSerializer.WithBuilder)new SegmentSnapshotRecord.Serializer());
            }
        }
    }

    private static class BootstrapState {
        private final Map<String, Long> chunkStartOffsets = Collections.synchronizedMap(new HashMap());
        private final Map<String, Long> finalTruncateOffsets = Collections.synchronizedMap(new HashMap());
        private final Map<String, Long> finalFirstChunkStartsAtOffsets = Collections.synchronizedMap(new HashMap());
        private final Set<SystemJournalRecord> visitedRecords = Collections.synchronizedSet(new HashSet());
        private final AtomicInteger filesProcessedCount = new AtomicInteger();
        private final AtomicInteger recordsProcessedCount = new AtomicInteger();

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public BootstrapState() {
        }
    }
}

