/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.storage.chunklayer;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.Timer;
import io.pravega.common.concurrent.Futures;
import io.pravega.segmentstore.storage.SegmentHandle;
import io.pravega.segmentstore.storage.StorageNotPrimaryException;
import io.pravega.segmentstore.storage.chunklayer.ChunkStorageMetrics;
import io.pravega.segmentstore.storage.chunklayer.ChunkedSegmentStorage;
import io.pravega.segmentstore.storage.chunklayer.SystemJournal;
import io.pravega.segmentstore.storage.metadata.ChunkMetadata;
import io.pravega.segmentstore.storage.metadata.MetadataTransaction;
import io.pravega.segmentstore.storage.metadata.SegmentMetadata;
import io.pravega.segmentstore.storage.metadata.StorageMetadata;
import io.pravega.segmentstore.storage.metadata.StorageMetadataWritesFencedOutException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TruncateOperation
implements Callable<CompletableFuture<Void>> {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(TruncateOperation.class);
    private final SegmentHandle handle;
    private final long offset;
    private final ChunkedSegmentStorage chunkedSegmentStorage;
    private final List<String> chunksToDelete = Collections.synchronizedList(new ArrayList());
    private final long traceId;
    private final Timer timer;
    private volatile String currentChunkName;
    private volatile ChunkMetadata currentMetadata;
    private volatile long oldLength;
    private final AtomicLong startOffset = new AtomicLong();
    private volatile SegmentMetadata segmentMetadata;
    private volatile boolean isLoopExited;

    TruncateOperation(ChunkedSegmentStorage chunkedSegmentStorage, SegmentHandle handle, long offset) {
        this.handle = handle;
        this.offset = offset;
        this.chunkedSegmentStorage = chunkedSegmentStorage;
        this.traceId = LoggerHelpers.traceEnter((Logger)log, (String)"truncate", (Object[])new Object[]{handle, offset});
        this.timer = new Timer();
    }

    @Override
    public CompletableFuture<Void> call() {
        this.checkPreconditions();
        log.debug("{} truncate - started op={}, segment={}, offset={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.offset});
        String streamSegmentName = this.handle.getSegmentName();
        return ChunkedSegmentStorage.tryWith(this.chunkedSegmentStorage.getMetadataStore().beginTransaction(false, streamSegmentName), txn -> txn.get(streamSegmentName).thenComposeAsync(storageMetadata -> {
            this.segmentMetadata = (SegmentMetadata)storageMetadata;
            this.checkPreconditions(streamSegmentName, this.segmentMetadata);
            if (this.segmentMetadata.getStartOffset() >= this.offset) {
                this.logEnd();
                return CompletableFuture.completedFuture(null);
            }
            int oldChunkCount = this.segmentMetadata.getChunkCount();
            long oldStartOffset = this.segmentMetadata.getStartOffset();
            return this.updateFirstChunk((MetadataTransaction)txn).thenComposeAsync(v -> this.deleteChunks((MetadataTransaction)txn).thenComposeAsync(vvv -> {
                txn.update(this.segmentMetadata);
                this.segmentMetadata.checkInvariants();
                Preconditions.checkState((this.segmentMetadata.getLength() == this.oldLength ? 1 : 0) != 0, (String)"truncate should not change segment length. oldLength=%s Segment=%s", (long)this.oldLength, (Object)this.segmentMetadata);
                Preconditions.checkState((oldChunkCount - this.chunksToDelete.size() == this.segmentMetadata.getChunkCount() ? 1 : 0) != 0, (String)"Number of chunks do not match. old value (%s) - number of chunks deleted (%s) must match current chunk count(%s)", (Object)oldChunkCount, (Object)this.chunksToDelete.size(), (Object)this.segmentMetadata.getChunkCount());
                if (null != this.currentMetadata && null != this.segmentMetadata.getFirstChunk()) {
                    Preconditions.checkState((boolean)this.segmentMetadata.getFirstChunk().equals(this.currentMetadata.getName()), (String)"First chunk name must match current metadata. Expected = %s Actual = %s", (Object)this.segmentMetadata.getFirstChunk(), (Object)this.currentMetadata.getName());
                    Preconditions.checkState((this.segmentMetadata.getStartOffset() <= this.segmentMetadata.getFirstChunkStartOffset() + this.currentMetadata.getLength() ? 1 : 0) != 0, (String)"segment start offset (%s) must be less than or equal to first chunk start offset (%s)+ first chunk length (%s)", (Object)this.segmentMetadata.getStartOffset(), (Object)this.segmentMetadata.getFirstChunkStartOffset(), (Object)this.currentMetadata.getLength());
                    if (this.segmentMetadata.getChunkCount() == 1) {
                        Preconditions.checkState((this.segmentMetadata.getLength() - this.segmentMetadata.getFirstChunkStartOffset() == this.currentMetadata.getLength() ? 1 : 0) != 0, (String)"Length of first chunk (%s) must match segment length (%s) - first chunk start offset (%s) when there is only one chunk", (Object)this.currentMetadata.getLength(), (Object)this.segmentMetadata.getLength(), (Object)this.segmentMetadata.getFirstChunkStartOffset());
                    }
                }
                if (!this.segmentMetadata.isStorageSystemSegment()) {
                    this.chunkedSegmentStorage.deleteBlockIndexEntriesForChunk((MetadataTransaction)txn, streamSegmentName, oldStartOffset, this.segmentMetadata.getStartOffset());
                }
                return this.chunkedSegmentStorage.getGarbageCollector().addChunksToGarbage(txn.getVersion(), this.chunksToDelete).thenComposeAsync(vv -> ((CompletableFuture)this.commit((MetadataTransaction)txn).handleAsync(this::handleException, this.chunkedSegmentStorage.getExecutor())).thenRunAsync(this::postCommit, this.chunkedSegmentStorage.getExecutor()), this.chunkedSegmentStorage.getExecutor());
            }, this.chunkedSegmentStorage.getExecutor()), this.chunkedSegmentStorage.getExecutor());
        }, this.chunkedSegmentStorage.getExecutor()), this.chunkedSegmentStorage.getExecutor());
    }

    private void postCommit() {
        this.chunkedSegmentStorage.getReadIndexCache().truncateReadIndex(this.handle.getSegmentName(), this.segmentMetadata.getStartOffset());
        this.logEnd();
    }

    private void logEnd() {
        Duration elapsed = this.timer.getElapsed();
        ChunkStorageMetrics.SLTS_TRUNCATE_LATENCY.reportSuccessEvent(elapsed);
        ChunkStorageMetrics.SLTS_TRUNCATE_COUNT.inc();
        if (this.segmentMetadata.isStorageSystemSegment()) {
            ChunkStorageMetrics.SLTS_SYSTEM_TRUNCATE_COUNT.inc();
            this.chunkedSegmentStorage.reportMetricsForSystemSegment(this.segmentMetadata);
        }
        if ((long)this.chunkedSegmentStorage.getConfig().getLateWarningThresholdInMillis() < elapsed.toMillis()) {
            log.warn("{} truncate - late op={}, segment={}, offset={}, latency={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.offset, elapsed.toMillis()});
        } else {
            log.debug("{} truncate - finished op={}, segment={}, offset={}, latency={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.offset, elapsed.toMillis()});
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"truncate", (long)this.traceId, (Object[])new Object[]{this.handle, this.offset});
    }

    private Void handleException(Void value, Throwable e) {
        if (null != e) {
            log.debug("{} truncate - exception op={}, segment={}, offset={}.", new Object[]{this.chunkedSegmentStorage.getLogPrefix(), System.identityHashCode(this), this.handle.getSegmentName(), this.offset});
            Throwable ex = Exceptions.unwrap((Throwable)e);
            if (ex instanceof StorageMetadataWritesFencedOutException) {
                throw new CompletionException((Throwable)((Object)new StorageNotPrimaryException(this.handle.getSegmentName(), ex)));
            }
            throw new CompletionException(ex);
        }
        return value;
    }

    private CompletableFuture<Void> commit(MetadataTransaction txn) {
        if (this.chunkedSegmentStorage.isStorageSystemSegment(this.segmentMetadata)) {
            txn.setExternalCommitStep(() -> this.chunkedSegmentStorage.getSystemJournal().commitRecord(SystemJournal.TruncationRecord.builder().segmentName(this.handle.getSegmentName()).offset(this.offset).firstChunkName(this.segmentMetadata.getFirstChunk()).startOffset(this.startOffset.get()).build()));
        }
        return txn.commit();
    }

    private CompletableFuture<Void> updateFirstChunk(MetadataTransaction txn) {
        this.currentChunkName = this.segmentMetadata.getFirstChunk();
        this.oldLength = this.segmentMetadata.getLength();
        this.startOffset.set(this.segmentMetadata.getFirstChunkStartOffset());
        return Futures.loop(() -> this.currentChunkName != null && !this.isLoopExited, () -> txn.get(this.currentChunkName).thenAcceptAsync(storageMetadata -> {
            this.currentMetadata = (ChunkMetadata)storageMetadata;
            Preconditions.checkState((null != this.currentMetadata ? 1 : 0) != 0, (String)"currentMetadata is null. Segment=%s currentChunkName=%s", (Object)this.segmentMetadata, (Object)this.currentChunkName);
            if (this.startOffset.get() <= this.offset && this.startOffset.get() + this.currentMetadata.getLength() > this.offset) {
                this.isLoopExited = true;
                return;
            }
            this.startOffset.addAndGet(this.currentMetadata.getLength());
            this.chunksToDelete.add(this.currentMetadata.getName());
            this.segmentMetadata.setChunkCount(this.segmentMetadata.getChunkCount() - 1);
            this.currentChunkName = this.currentMetadata.getNextChunk();
        }, this.chunkedSegmentStorage.getExecutor()), (Executor)this.chunkedSegmentStorage.getExecutor()).thenAcceptAsync(v -> {
            this.segmentMetadata.setFirstChunk(this.currentChunkName);
            this.segmentMetadata.setStartOffset(this.offset);
            this.segmentMetadata.setFirstChunkStartOffset(this.startOffset.get());
        }, this.chunkedSegmentStorage.getExecutor());
    }

    private CompletableFuture<Void> deleteChunks(MetadataTransaction txn) {
        ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
        for (String toDelete : this.chunksToDelete) {
            futures.add(txn.get(toDelete).thenAcceptAsync(metadata -> {
                ((ChunkMetadata)metadata).setActive(false);
                txn.update((StorageMetadata)metadata);
            }, this.chunkedSegmentStorage.getExecutor()));
            if (!toDelete.equals(this.segmentMetadata.getLastChunk())) continue;
            this.segmentMetadata.setLastChunkStartOffset(this.segmentMetadata.getLength());
            this.segmentMetadata.setLastChunk(null);
        }
        return Futures.allOf(futures);
    }

    private void checkPreconditions(String streamSegmentName, SegmentMetadata segmentMetadata) {
        this.chunkedSegmentStorage.checkSegmentExists(streamSegmentName, segmentMetadata);
        this.chunkedSegmentStorage.checkOwnership(streamSegmentName, segmentMetadata);
        if (segmentMetadata.getLength() < this.offset) {
            throw new IllegalArgumentException(String.format("offset %d is outside of valid range [%d, %d) for segment %s", this.offset, segmentMetadata.getStartOffset(), segmentMetadata.getLength(), streamSegmentName));
        }
    }

    private void checkPreconditions() {
        Preconditions.checkArgument((!this.handle.isReadOnly() ? 1 : 0) != 0, (String)"handle must not be read only. Segment = %s", (Object)this.handle.getSegmentName());
        Preconditions.checkArgument((this.offset >= 0L ? 1 : 0) != 0, (String)"offset must be non-negative. Segment = %s offset = %s", (Object)this.handle.getSegmentName(), (long)this.offset);
    }
}

