/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.segmentstore.storage.rolling;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.Exceptions;
import io.pravega.common.LoggerHelpers;
import io.pravega.common.io.BoundedInputStream;
import io.pravega.common.util.BufferView;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.CollectionHelpers;
import io.pravega.segmentstore.contracts.BadOffsetException;
import io.pravega.segmentstore.contracts.SegmentProperties;
import io.pravega.segmentstore.contracts.StreamSegmentException;
import io.pravega.segmentstore.contracts.StreamSegmentExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentInformation;
import io.pravega.segmentstore.contracts.StreamSegmentNotExistsException;
import io.pravega.segmentstore.contracts.StreamSegmentSealedException;
import io.pravega.segmentstore.contracts.StreamSegmentTruncatedException;
import io.pravega.segmentstore.contracts.StreamingException;
import io.pravega.segmentstore.storage.SegmentHandle;
import io.pravega.segmentstore.storage.SegmentRollingPolicy;
import io.pravega.segmentstore.storage.StorageNotPrimaryException;
import io.pravega.segmentstore.storage.SyncStorage;
import io.pravega.segmentstore.storage.rolling.HandleSerializer;
import io.pravega.segmentstore.storage.rolling.RollingSegmentHandle;
import io.pravega.segmentstore.storage.rolling.SegmentChunk;
import io.pravega.shared.NameUtils;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Spliterators;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RollingStorage
implements SyncStorage {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(RollingStorage.class);
    private final SyncStorage baseStorage;
    private final SyncStorage headerStorage;
    private final SegmentRollingPolicy defaultRollingPolicy;
    private final AtomicBoolean closed;

    public RollingStorage(SyncStorage baseStorage) {
        this(baseStorage, SegmentRollingPolicy.NO_ROLLING);
    }

    public RollingStorage(SyncStorage baseStorage, SegmentRollingPolicy defaultRollingPolicy) {
        this.baseStorage = (SyncStorage)Preconditions.checkNotNull((Object)baseStorage, (Object)"baseStorage");
        this.headerStorage = this.baseStorage.withReplaceSupport();
        if (!this.headerStorage.supportsReplace()) {
            log.info("Base Storage does not support replace. Will not be able to trim header chunks after truncation.");
        }
        this.defaultRollingPolicy = (SegmentRollingPolicy)Preconditions.checkNotNull((Object)defaultRollingPolicy, (Object)"defaultRollingPolicy");
        this.closed = new AtomicBoolean();
    }

    @Override
    public void close() {
        if (!this.closed.getAndSet(true)) {
            this.baseStorage.close();
            log.info("Closed");
        }
    }

    @Override
    public void initialize(long containerEpoch) {
        this.baseStorage.initialize(containerEpoch);
    }

    @Override
    public SegmentHandle openRead(String segmentName) throws StreamSegmentException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"openRead", (Object[])new Object[]{segmentName});
        RollingSegmentHandle handle = this.openHandle(segmentName, true);
        LoggerHelpers.traceLeave((Logger)log, (String)"openRead", (long)traceId, (Object[])new Object[]{handle});
        return handle;
    }

    @Override
    public int read(SegmentHandle handle, long offset, byte[] buffer, int bufferOffset, int length) throws StreamSegmentException {
        RollingSegmentHandle h = this.getHandle(handle);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"read", (Object[])new Object[]{handle, offset, length});
        this.ensureNotDeleted(h);
        Exceptions.checkArrayRange((long)bufferOffset, (int)length, (long)buffer.length, (String)"bufferOffset", (String)"length");
        if (offset < 0L || bufferOffset < 0 || length < 0 || buffer.length < bufferOffset + length) {
            throw new ArrayIndexOutOfBoundsException(String.format("Offset (%s) must be non-negative, and bufferOffset (%s) and length (%s) must be valid indices into buffer of size %s.", offset, bufferOffset, length, buffer.length));
        }
        if (!h.isSealed() && offset + (long)length > h.length()) {
            RollingSegmentHandle newHandle = (RollingSegmentHandle)this.openRead(handle.getSegmentName());
            h.refresh(newHandle);
            log.debug("Handle refreshed: {}.", (Object)h);
        }
        Preconditions.checkArgument((offset + (long)length <= h.length() ? 1 : 0) != 0, (String)"Offset %s + length %s is beyond the last offset %s of the segment.", (Object)offset, (Object)length, (Object)h.length());
        List<SegmentChunk> chunks = h.chunks();
        int currentIndex = CollectionHelpers.binarySearch(chunks, s -> offset < s.getStartOffset() ? -1 : (offset >= s.getLastOffset() ? 1 : 0));
        assert (currentIndex >= 0) : "unable to locate first SegmentChunk index.";
        try {
            int bytesRead = 0;
            while (bytesRead < length && currentIndex < chunks.size()) {
                SegmentChunk current = chunks.get(currentIndex);
                this.checkTruncatedSegment(null, h, current);
                if (current.getLength() == 0L) continue;
                long readOffset = offset + (long)bytesRead - current.getStartOffset();
                int readLength = (int)Math.min((long)(length - bytesRead), current.getLength() - readOffset);
                assert (readOffset >= 0L && readLength >= 0) : "negative readOffset or readLength";
                try {
                    SegmentHandle sh = this.baseStorage.openRead(current.getName());
                    int count = this.baseStorage.read(sh, readOffset, buffer, bufferOffset + bytesRead, readLength);
                    bytesRead += count;
                    if (readOffset + (long)count < current.getLength()) continue;
                    ++currentIndex;
                }
                catch (StreamSegmentNotExistsException ex) {
                    log.debug("SegmentChunk '{}' does not exist anymore ({}).", (Object)current, (Object)h);
                    this.checkTruncatedSegment((StreamingException)ex, h, current);
                }
            }
            LoggerHelpers.traceLeave((Logger)log, (String)"read", (long)traceId, (Object[])new Object[]{handle, offset, bytesRead});
            return bytesRead;
        }
        catch (StreamSegmentTruncatedException ex) {
            RollingSegmentHandle newHandle = (RollingSegmentHandle)this.openRead(handle.getSegmentName());
            h.refresh(newHandle);
            if (h.isDeleted()) {
                log.debug("Segment '{}' has been deleted. Cannot read anymore.", (Object)h);
                throw new StreamSegmentNotExistsException(handle.getSegmentName(), (Throwable)ex);
            }
            throw ex;
        }
    }

    @Override
    public SegmentProperties getStreamSegmentInfo(String segmentName) throws StreamSegmentException {
        RollingSegmentHandle handle = (RollingSegmentHandle)this.openRead(segmentName);
        return StreamSegmentInformation.builder().name(handle.getSegmentName()).sealed(handle.isSealed()).length(handle.length()).build();
    }

    @Override
    public boolean exists(String segmentName) {
        try {
            this.openRead(segmentName);
            return true;
        }
        catch (StreamSegmentNotExistsException ex) {
            return false;
        }
    }

    @Override
    public SegmentHandle create(String streamSegmentName) throws StreamSegmentException {
        return this.create(streamSegmentName, this.defaultRollingPolicy);
    }

    @Override
    public SegmentHandle create(String segmentName, SegmentRollingPolicy rollingPolicy) throws StreamSegmentException {
        RollingSegmentHandle retValue;
        Preconditions.checkNotNull((Object)rollingPolicy, (Object)"rollingPolicy");
        String headerName = NameUtils.getHeaderSegmentName((String)segmentName);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"create", (Object[])new Object[]{segmentName, rollingPolicy});
        if (this.baseStorage.exists(segmentName)) {
            throw new StreamSegmentExistsException(segmentName);
        }
        SegmentHandle headerHandle = null;
        try {
            try {
                headerHandle = this.headerStorage.create(headerName);
            }
            catch (StreamSegmentExistsException ex) {
                this.checkIfEmptyAndNotSealed(ex, headerName, this.headerStorage);
                headerHandle = this.headerStorage.openWrite(headerName);
                log.debug("Empty Segment Header found for '{}'; treating as inexistent.", (Object)segmentName);
            }
            retValue = new RollingSegmentHandle(headerHandle, rollingPolicy, new ArrayList<SegmentChunk>());
            this.serializeHandle(retValue);
        }
        catch (StreamSegmentExistsException ex) {
            throw ex;
        }
        catch (Exception ex) {
            if (!Exceptions.mustRethrow((Throwable)ex) && headerHandle != null) {
                try {
                    log.warn("Could not write Header Segment for '{}', rolling back.", (Object)segmentName, (Object)ex);
                    this.headerStorage.delete(headerHandle);
                }
                catch (Exception ex2) {
                    ex.addSuppressed(ex2);
                }
            }
            throw ex;
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"create", (long)traceId, (Object[])new Object[]{segmentName});
        return retValue;
    }

    @Override
    public SegmentHandle openWrite(String segmentName) throws StreamSegmentException {
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"openWrite", (Object[])new Object[]{segmentName});
        RollingSegmentHandle handle = this.openHandle(segmentName, false);
        SegmentChunk last = handle.lastChunk();
        if (last != null && !last.isSealed()) {
            SegmentHandle activeHandle = this.baseStorage.openWrite(last.getName());
            handle.setActiveChunkHandle(activeHandle);
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"openWrite", (long)traceId, (Object[])new Object[]{handle});
        return handle;
    }

    @Override
    public void write(SegmentHandle handle, long offset, InputStream data, int length) throws StreamSegmentException {
        int bytesWritten;
        int writeLength;
        RollingSegmentHandle h = this.getHandle(handle);
        this.ensureNotDeleted(h);
        this.ensureNotSealed(h);
        this.ensureWritable(h);
        this.ensureOffset(h, offset);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"write", (Object[])new Object[]{handle, offset, length});
        for (bytesWritten = 0; bytesWritten < length; bytesWritten += writeLength) {
            if (h.getActiveChunkHandle() == null || h.lastChunk().getLength() >= h.getRollingPolicy().getMaxLength()) {
                this.rollover(h);
            }
            SegmentChunk last = h.lastChunk();
            writeLength = (int)Math.min((long)(length - bytesWritten), h.getRollingPolicy().getMaxLength() - last.getLength());
            assert (writeLength > 0) : "non-positive write length";
            long chunkOffset = offset + (long)bytesWritten - last.getStartOffset();
            try (BoundedInputStream bis = new BoundedInputStream(data, writeLength);){
                this.baseStorage.write(h.getActiveChunkHandle(), chunkOffset, (InputStream)bis, writeLength);
            }
            last.increaseLength(writeLength);
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"write", (long)traceId, (Object[])new Object[]{handle, offset, bytesWritten});
    }

    @Override
    public void seal(SegmentHandle handle) throws StreamSegmentException {
        RollingSegmentHandle h = this.getHandle(handle);
        this.ensureNotDeleted(h);
        if (h.isReadOnly() && h.isSealed()) {
            log.debug("Segment already sealed: '{}'.", (Object)h.getSegmentName());
            return;
        }
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"seal", (Object[])new Object[]{handle});
        this.sealActiveChunk(h);
        SegmentHandle headerHandle = h.getHeaderHandle();
        if (headerHandle != null) {
            this.headerStorage.seal(headerHandle);
        }
        h.markSealed();
        log.debug("Sealed Header for '{}'.", (Object)h.getSegmentName());
        LoggerHelpers.traceLeave((Logger)log, (String)"seal", (long)traceId, (Object[])new Object[]{handle});
    }

    @Override
    public void unseal(SegmentHandle handle) {
        throw new UnsupportedOperationException("RollingStorage does not support unseal().");
    }

    @Override
    public void concat(SegmentHandle targetHandle, long targetOffset, String sourceSegment) throws StreamSegmentException {
        RollingSegmentHandle target = this.getHandle(targetHandle);
        this.ensureOffset(target, targetOffset);
        this.ensureNotDeleted(target);
        this.ensureNotSealed(target);
        this.ensureWritable(target);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"concat", (Object[])new Object[]{target, targetOffset, sourceSegment});
        RollingSegmentHandle source = (RollingSegmentHandle)this.openWrite(sourceSegment);
        Preconditions.checkState((boolean)source.isSealed(), (String)"Cannot concat segment '%s' into '%s' because it is not sealed.", (Object)sourceSegment, (Object)target.getSegmentName());
        if (source.length() == 0L) {
            log.debug("Concat source '{}' is empty. Deleting instead of concatenating.", (Object)source);
            this.delete(source);
            return;
        }
        this.refreshChunkExistence(source);
        Preconditions.checkState((source.chunks().stream().allMatch(SegmentChunk::exists) && source.chunks().get(0).getStartOffset() == 0L ? 1 : 0) != 0, (String)"Cannot use Segment '%s' as concat source because it is truncated.", (Object)source.getSegmentName());
        if (this.shouldConcatNatively(source, target)) {
            log.debug("Concat '{}' into '{}' using native method.", (Object)source, (Object)target);
            SegmentChunk lastTarget = target.lastChunk();
            if (lastTarget == null || lastTarget.isSealed()) {
                this.rollover(target);
            }
            SegmentChunk lastSource = source.lastChunk();
            this.baseStorage.concat(target.getActiveChunkHandle(), target.lastChunk().getLength(), lastSource.getName());
            target.lastChunk().increaseLength(lastSource.getLength());
            if (source.getHeaderHandle() != null) {
                try {
                    this.headerStorage.delete(source.getHeaderHandle());
                }
                catch (StreamSegmentNotExistsException ex) {
                    log.warn("Attempted to delete concat source Header '{}' but it doesn't exist.", (Object)source.getHeaderHandle().getSegmentName(), (Object)ex);
                }
            }
        } else {
            log.debug("Concat '{}' into '{}' using header merge method.", (Object)source, (Object)target);
            if (target.getHeaderHandle() == null) {
                this.createHeader(target);
            }
            List<SegmentChunk> newSegmentChunks = this.rebase(source.chunks(), target.length());
            this.sealActiveChunk(target);
            this.serializeBeginConcat(target, source);
            this.headerStorage.concat(target.getHeaderHandle(), target.getHeaderLength(), source.getHeaderHandle().getSegmentName());
            target.increaseHeaderLength(source.getHeaderLength());
            target.addChunks(newSegmentChunks);
            this.unsealLastChunkIfNecessary(target);
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"concat", (long)traceId, (Object[])new Object[]{target, targetOffset, sourceSegment});
    }

    @Override
    public void delete(SegmentHandle handle) throws StreamSegmentException {
        RollingSegmentHandle h = this.getHandle(handle);
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"delete", (Object[])new Object[]{handle});
        SegmentHandle headerHandle = h.getHeaderHandle();
        if (headerHandle == null) {
            SegmentHandle subHandle = this.baseStorage.openWrite(h.lastChunk().getName());
            try {
                this.baseStorage.delete(subHandle);
                h.lastChunk().markInexistent();
                h.markDeleted();
            }
            catch (StreamSegmentNotExistsException ex) {
                h.lastChunk().markInexistent();
                h.markDeleted();
                throw ex;
            }
        }
        if (!h.isSealed()) {
            RollingSegmentHandle writeHandle = h.isReadOnly() ? (RollingSegmentHandle)this.openWrite(handle.getSegmentName()) : h;
            this.seal(writeHandle);
        }
        this.deleteChunks(h, s -> true);
        try {
            this.headerStorage.delete(headerHandle);
            h.markDeleted();
        }
        catch (StreamSegmentNotExistsException ex) {
            h.markDeleted();
            throw ex;
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"delete", (long)traceId, (Object[])new Object[]{handle});
    }

    @Override
    public void truncate(SegmentHandle handle, long truncationOffset) throws StreamSegmentException {
        boolean chunksDeleted;
        RollingSegmentHandle h = this.getHandle(handle);
        this.ensureNotDeleted(h);
        Preconditions.checkArgument((h.isSealed() || !h.isReadOnly() ? 1 : 0) != 0, (Object)"Can only truncate with a read-only handle if the Segment is Sealed.");
        if (h.getHeaderHandle() == null) {
            return;
        }
        long traceId = LoggerHelpers.traceEnter((Logger)log, (String)"truncate", (Object[])new Object[]{h, truncationOffset});
        Preconditions.checkArgument((truncationOffset >= 0L && truncationOffset <= h.length() ? 1 : 0) != 0, (Object)"truncationOffset must be non-negative and at most the length of the Segment.");
        SegmentChunk last = h.lastChunk();
        if (last != null && this.canTruncate(last, truncationOffset) && !h.isSealed()) {
            this.rollover(h);
            chunksDeleted = this.deleteChunks(h, s -> this.canTruncate((SegmentChunk)s, truncationOffset));
        } else {
            chunksDeleted = this.deleteChunks(h, s -> this.canTruncate((SegmentChunk)s, truncationOffset) && s.getLastOffset() < h.length());
        }
        if (chunksDeleted && this.headerStorage.supportsReplace()) {
            this.truncateHandle(h);
        }
        LoggerHelpers.traceLeave((Logger)log, (String)"truncate", (long)traceId, (Object[])new Object[]{h, truncationOffset});
    }

    @Override
    public boolean supportsTruncation() {
        return true;
    }

    @Override
    public Iterator<SegmentProperties> listSegments() throws IOException {
        return new RollingStorageSegmentIterator(this, this.baseStorage.listSegments(), props -> NameUtils.isHeaderSegment((String)props.getName()));
    }

    private void rollover(RollingSegmentHandle handle) throws StreamSegmentException {
        Preconditions.checkArgument((handle.getHeaderHandle() != null ? 1 : 0) != 0, (Object)"Cannot rollover a Segment with no header.");
        Preconditions.checkArgument((!handle.isReadOnly() ? 1 : 0) != 0, (Object)"Cannot rollover using a read-only handle.");
        Preconditions.checkArgument((!handle.isSealed() ? 1 : 0) != 0, (Object)"Cannot rollover a Sealed Segment.");
        log.debug("Rolling over '{}'.", (Object)handle);
        this.sealActiveChunk(handle);
        try {
            this.createChunk(handle);
        }
        catch (StreamSegmentExistsException ex) {
            int chunkCount = handle.chunks().size();
            handle.refresh(this.openHandle(handle.getSegmentName(), false));
            if (chunkCount == handle.chunks().size()) {
                throw ex;
            }
            log.warn("Aborted rollover due to concurrent rollover detected ('{}').", (Object)handle);
        }
    }

    private void sealActiveChunk(RollingSegmentHandle handle) throws StreamSegmentException {
        SegmentHandle activeChunk = handle.getActiveChunkHandle();
        SegmentChunk last = handle.lastChunk();
        if (activeChunk != null && !last.isSealed()) {
            this.baseStorage.seal(activeChunk);
            handle.setActiveChunkHandle(null);
            last.markSealed();
            log.debug("Sealed active SegmentChunk '{}' for '{}'.", (Object)activeChunk.getSegmentName(), (Object)handle.getSegmentName());
        }
    }

    private void unsealLastChunkIfNecessary(RollingSegmentHandle handle) throws StreamSegmentException {
        boolean needsHandleUpdate;
        SegmentChunk last = handle.lastChunk();
        if (last == null || !last.isSealed()) {
            return;
        }
        SegmentHandle activeChunk = handle.getActiveChunkHandle();
        boolean bl = needsHandleUpdate = activeChunk == null;
        if (needsHandleUpdate) {
            activeChunk = this.baseStorage.openWrite(last.getName());
        }
        try {
            this.baseStorage.unseal(activeChunk);
        }
        catch (UnsupportedOperationException e) {
            log.warn("Unable to unseal SegmentChunk '{}' since base storage does not support unsealing.", (Object)last);
            return;
        }
        last.markUnsealed();
        if (needsHandleUpdate) {
            activeChunk = this.baseStorage.openWrite(last.getName());
            handle.setActiveChunkHandle(activeChunk);
        }
        log.debug("Unsealed active SegmentChunk '{}' for '{}'.", (Object)activeChunk.getSegmentName(), (Object)handle.getSegmentName());
    }

    private void createChunk(RollingSegmentHandle handle) throws StreamSegmentException {
        long segmentLength = handle.length();
        SegmentChunk newSegmentChunk = SegmentChunk.forSegment(handle.getSegmentName(), segmentLength);
        try {
            this.baseStorage.create(newSegmentChunk.getName());
        }
        catch (StreamSegmentExistsException ex) {
            this.checkIfEmptyAndNotSealed(ex, newSegmentChunk.getName());
        }
        this.serializeNewChunk(handle, newSegmentChunk);
        SegmentHandle activeHandle = this.baseStorage.openWrite(newSegmentChunk.getName());
        handle.addChunk(newSegmentChunk, activeHandle);
        log.debug("Created new SegmentChunk '{}' for '{}'.", (Object)newSegmentChunk, (Object)handle);
    }

    private boolean deleteChunks(RollingSegmentHandle handle, Predicate<SegmentChunk> canDelete) throws StreamSegmentException {
        boolean anyDeleted = false;
        for (SegmentChunk s : handle.chunks()) {
            if (!s.exists() || !canDelete.test(s)) continue;
            anyDeleted = true;
            try {
                SegmentHandle subHandle = this.baseStorage.openWrite(s.getName());
                this.baseStorage.delete(subHandle);
                s.markInexistent();
                log.debug("Deleted SegmentChunk '{}' for '{}'.", (Object)s, (Object)handle);
            }
            catch (StreamSegmentNotExistsException ex) {
                s.markInexistent();
            }
        }
        return anyDeleted;
    }

    private boolean canTruncate(SegmentChunk segmentChunk, long truncationOffset) {
        return segmentChunk.getStartOffset() < truncationOffset && segmentChunk.getLastOffset() <= truncationOffset;
    }

    private void refreshChunkExistence(RollingSegmentHandle handle) {
        for (SegmentChunk s : handle.chunks()) {
            if (!s.exists() || this.baseStorage.exists(s.getName())) continue;
            s.markInexistent();
        }
    }

    private void createHeader(RollingSegmentHandle handle) throws StreamSegmentException {
        Preconditions.checkArgument((handle.getHeaderHandle() == null ? 1 : 0) != 0, (Object)"handle already has a header.");
        String headerName = NameUtils.getHeaderSegmentName((String)handle.getSegmentName());
        this.headerStorage.create(headerName);
        SegmentHandle headerHandle = this.headerStorage.openWrite(headerName);
        RollingSegmentHandle newHandle = new RollingSegmentHandle(headerHandle, handle.getRollingPolicy(), handle.chunks());
        this.serializeHandle(newHandle);
        handle.refresh(newHandle);
    }

    private boolean shouldConcatNatively(RollingSegmentHandle source, RollingSegmentHandle target) {
        if (source.getHeaderHandle() == null) {
            return true;
        }
        SegmentChunk lastSource = source.lastChunk();
        SegmentChunk lastTarget = target.lastChunk();
        return lastSource != null && lastSource.getStartOffset() == 0L && lastTarget != null && !lastTarget.isSealed() && lastTarget.getLength() + lastSource.getLength() <= target.getRollingPolicy().getMaxLength();
    }

    private RollingSegmentHandle openHandle(String segmentName, boolean readOnly) throws StreamSegmentException {
        RollingSegmentHandle handle;
        try {
            SegmentProperties headerInfo = this.getHeaderInfo(segmentName);
            SegmentHandle headerHandle = readOnly ? this.headerStorage.openRead(headerInfo.getName()) : this.headerStorage.openWrite(headerInfo.getName());
            handle = this.readHeader(headerInfo, headerHandle);
        }
        catch (StreamSegmentNotExistsException ex) {
            SegmentHandle segmentHandle = readOnly ? this.baseStorage.openRead(segmentName) : this.baseStorage.openWrite(segmentName);
            handle = new RollingSegmentHandle(segmentHandle);
        }
        SegmentChunk last = null;
        for (SegmentChunk s : handle.chunks()) {
            if (last != null) {
                last.setLength(s.getStartOffset() - last.getStartOffset());
                last.markSealed();
            }
            last = s;
        }
        if (last != null) {
            SegmentProperties si = this.baseStorage.getStreamSegmentInfo(last.getName());
            last.setLength(si.getLength());
            if (si.isSealed()) {
                last.markSealed();
                if (handle.getHeaderHandle() == null) {
                    handle.markSealed();
                }
            }
        }
        return handle;
    }

    private SegmentProperties getHeaderInfo(String segmentName) throws StreamSegmentException {
        String headerSegment = NameUtils.getHeaderSegmentName((String)segmentName);
        SegmentProperties headerInfo = this.headerStorage.getStreamSegmentInfo(headerSegment);
        if (headerInfo.getLength() == 0L) {
            throw new StreamSegmentNotExistsException(segmentName);
        }
        return headerInfo;
    }

    private RollingSegmentHandle readHeader(SegmentProperties headerInfo, SegmentHandle headerHandle) throws StreamSegmentException {
        byte[] readBuffer = new byte[(int)headerInfo.getLength()];
        this.headerStorage.read(headerHandle, 0L, readBuffer, 0, readBuffer.length);
        RollingSegmentHandle handle = HandleSerializer.deserialize(readBuffer, headerHandle);
        if (headerInfo.isSealed()) {
            handle.markSealed();
        }
        return handle;
    }

    private void serializeHandle(RollingSegmentHandle handle) throws StreamSegmentException {
        ByteArraySegment handleData = HandleSerializer.serialize(handle);
        try {
            this.headerStorage.write(handle.getHeaderHandle(), 0L, handleData.getReader(), handleData.getLength());
            handle.setHeaderLength(handleData.getLength());
            log.debug("Header for '{}' fully serialized to '{}'.", (Object)handle.getSegmentName(), (Object)handle.getHeaderHandle().getSegmentName());
        }
        catch (BadOffsetException ex) {
            throw new StorageNotPrimaryException(handle.getSegmentName(), ex);
        }
    }

    private void truncateHandle(RollingSegmentHandle handle) throws StreamSegmentException {
        handle.excludeInexistentChunks();
        ByteArraySegment handleData = HandleSerializer.serialize(handle);
        this.headerStorage.replace(handle.getHeaderHandle(), (BufferView)handleData);
        handle.setHeaderLength(handleData.getLength());
        log.debug("Header for '{}' fully serialized (replaced) to '{}'.", (Object)handle.getSegmentName(), (Object)handle.getHeaderHandle().getSegmentName());
    }

    private void serializeNewChunk(RollingSegmentHandle handle, SegmentChunk newSegmentChunk) throws StreamSegmentException {
        this.updateHandle(handle, HandleSerializer.serializeChunk(newSegmentChunk));
    }

    private void serializeBeginConcat(RollingSegmentHandle targetHandle, RollingSegmentHandle sourceHandle) throws StreamSegmentException {
        byte[] updateData = HandleSerializer.serializeConcat(sourceHandle.chunks().size(), targetHandle.length());
        this.updateHandle(targetHandle, updateData);
    }

    private void updateHandle(RollingSegmentHandle handle, byte[] data) throws StreamSegmentException {
        try {
            this.headerStorage.write(handle.getHeaderHandle(), handle.getHeaderLength(), new ByteArrayInputStream(data), data.length);
            handle.increaseHeaderLength(data.length);
            log.debug("Header for '{}' updated with {} bytes for a length of {}.", new Object[]{handle.getSegmentName(), data.length, handle.getHeaderLength()});
        }
        catch (BadOffsetException ex) {
            throw new StorageNotPrimaryException(handle.getSegmentName(), ex);
        }
    }

    private List<SegmentChunk> rebase(List<SegmentChunk> segmentChunks, long newStartOffset) {
        AtomicLong segmentOffset = new AtomicLong(newStartOffset);
        return segmentChunks.stream().map(s -> s.withNewOffset(segmentOffset.getAndAdd(s.getLength()))).collect(Collectors.toList());
    }

    private void checkTruncatedSegment(StreamingException ex, RollingSegmentHandle handle, SegmentChunk segmentChunk) {
        if (ex != null && Exceptions.unwrap((Throwable)ex) instanceof StreamSegmentNotExistsException || !segmentChunk.exists()) {
            segmentChunk.markInexistent();
            String message = String.format("Offsets %d-%d have been deleted.", segmentChunk.getStartOffset(), segmentChunk.getLastOffset());
            ex = new StreamSegmentTruncatedException(handle.getSegmentName(), message, ex);
        }
        if (ex != null) {
            throw ex;
        }
    }

    private void checkIfEmptyAndNotSealed(StreamSegmentExistsException ex, String chunkName) throws StreamSegmentException {
        this.checkIfEmptyAndNotSealed(ex, chunkName, this.baseStorage);
    }

    private void checkIfEmptyAndNotSealed(StreamSegmentExistsException ex, String chunkName, SyncStorage storage) throws StreamSegmentException {
        try {
            SegmentProperties si = storage.getStreamSegmentInfo(chunkName);
            if (si.getLength() > 0L || si.isSealed()) {
                throw ex;
            }
        }
        catch (StreamSegmentNotExistsException streamSegmentNotExistsException) {
            // empty catch block
        }
    }

    private RollingSegmentHandle getHandle(SegmentHandle handle) {
        Preconditions.checkArgument((boolean)(handle instanceof RollingSegmentHandle), (Object)"handle must be of type RollingSegmentHandle.");
        return (RollingSegmentHandle)handle;
    }

    private void ensureWritable(RollingSegmentHandle handle) {
        Preconditions.checkArgument((!handle.isReadOnly() ? 1 : 0) != 0, (String)"handle must not be read-only (%s).", (Object)handle.getSegmentName());
    }

    private void ensureNotDeleted(RollingSegmentHandle handle) throws StreamSegmentNotExistsException {
        if (handle.isDeleted()) {
            throw new StreamSegmentNotExistsException(handle.getSegmentName());
        }
    }

    private void ensureNotSealed(RollingSegmentHandle handle) throws StreamSegmentSealedException {
        if (handle.isSealed()) {
            throw new StreamSegmentSealedException(handle.getSegmentName());
        }
    }

    private void ensureOffset(RollingSegmentHandle handle, long offset) throws StreamSegmentException {
        if (offset != handle.length()) {
            RollingSegmentHandle refreshedHandle = this.openHandle(handle.getSegmentName(), true);
            handle.refresh(refreshedHandle);
            log.debug("Handle refreshed: {}.", (Object)handle);
            if (offset != handle.length()) {
                throw new BadOffsetException(handle.getSegmentName(), handle.length(), offset);
            }
        }
    }

    private static class RollingStorageSegmentIterator
    implements Iterator<SegmentProperties> {
        private final RollingStorage instance;
        private final Iterator<SegmentProperties> results;

        RollingStorageSegmentIterator(RollingStorage instance, Iterator<SegmentProperties> results, Predicate<SegmentProperties> patternMatchPredicate) {
            this.instance = instance;
            this.results = StreamSupport.stream(Spliterators.spliteratorUnknownSize(results, 0), false).filter(patternMatchPredicate).map(this::toSegmentProperties).iterator();
        }

        public SegmentProperties toSegmentProperties(SegmentProperties segmentProperties) {
            try {
                String segmentName = NameUtils.getSegmentNameFromHeader((String)segmentProperties.getName());
                RollingSegmentHandle handle = this.instance.openHandle(segmentName, true);
                return StreamSegmentInformation.builder().name(segmentName).length(handle.length()).sealed(handle.isSealed()).build();
            }
            catch (StreamSegmentException e) {
                log.error("Exception occurred while transforming the object into SegmentProperties.");
                return null;
            }
        }

        @Override
        public boolean hasNext() {
            return this.results.hasNext();
        }

        @Override
        public SegmentProperties next() throws NoSuchElementException {
            return this.results.next();
        }
    }
}

