/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.controller.server.bucket;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.client.ClientConfig;
import io.pravega.client.SynchronizerClientFactory;
import io.pravega.client.segment.impl.NoSuchSegmentException;
import io.pravega.client.state.Revision;
import io.pravega.client.state.RevisionedStreamClient;
import io.pravega.client.state.SynchronizerConfig;
import io.pravega.client.stream.Serializer;
import io.pravega.client.stream.Stream;
import io.pravega.client.stream.StreamConfiguration;
import io.pravega.client.watermark.WatermarkSerializer;
import io.pravega.common.Exceptions;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.hash.RandomFactory;
import io.pravega.common.tracing.RequestTracker;
import io.pravega.common.tracing.TagLogger;
import io.pravega.controller.store.stream.BucketStore;
import io.pravega.controller.store.stream.OperationContext;
import io.pravega.controller.store.stream.StoreException;
import io.pravega.controller.store.stream.StreamMetadataStore;
import io.pravega.controller.store.stream.records.EpochRecord;
import io.pravega.controller.store.stream.records.StreamSegmentRecord;
import io.pravega.controller.store.stream.records.WriterMark;
import io.pravega.shared.NameUtils;
import io.pravega.shared.watermarks.SegmentWithRange;
import io.pravega.shared.watermarks.Watermark;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.ParametersAreNonnullByDefault;
import lombok.Generated;
import org.slf4j.LoggerFactory;

public class PeriodicWatermarking
implements AutoCloseable {
    private static final TagLogger log = new TagLogger(LoggerFactory.getLogger(PeriodicWatermarking.class));
    private static final int MAX_CACHE_SIZE = 500;
    private final StreamMetadataStore streamMetadataStore;
    private final BucketStore bucketStore;
    private final ScheduledExecutorService executor;
    private final LoadingCache<Stream, WatermarkClient> watermarkClientCache;
    private final LoadingCache<String, SynchronizerClientFactory> syncFactoryCache;
    private final RequestTracker requestTracker;
    private final Supplier<Long> requestIdGenerator = RandomFactory.create()::nextLong;

    public PeriodicWatermarking(StreamMetadataStore streamMetadataStore, BucketStore bucketStore, ClientConfig clientConfig, ScheduledExecutorService executor, RequestTracker requestTracker) {
        this(streamMetadataStore, bucketStore, (String s) -> SynchronizerClientFactory.withScope((String)s, (ClientConfig)clientConfig), executor, requestTracker);
    }

    @VisibleForTesting
    PeriodicWatermarking(StreamMetadataStore streamMetadataStore, BucketStore bucketStore, final Function<String, SynchronizerClientFactory> synchronizerClientFactoryFactory, ScheduledExecutorService executor, RequestTracker requestTracker) {
        this.streamMetadataStore = streamMetadataStore;
        this.bucketStore = bucketStore;
        this.executor = executor;
        this.syncFactoryCache = CacheBuilder.newBuilder().maximumSize(500L).expireAfterAccess(10L, TimeUnit.MINUTES).removalListener(notification -> ((SynchronizerClientFactory)notification.getValue()).close()).build((CacheLoader)new CacheLoader<String, SynchronizerClientFactory>(){

            @ParametersAreNonnullByDefault
            public SynchronizerClientFactory load(String scope) {
                return (SynchronizerClientFactory)synchronizerClientFactoryFactory.apply(scope);
            }
        });
        this.watermarkClientCache = CacheBuilder.newBuilder().maximumSize(500L).expireAfterAccess(10L, TimeUnit.MINUTES).removalListener(notification -> ((WatermarkClient)notification.getValue()).close()).build((CacheLoader)new CacheLoader<Stream, WatermarkClient>(){

            @ParametersAreNonnullByDefault
            public WatermarkClient load(Stream stream) {
                return new WatermarkClient(stream, (SynchronizerClientFactory)PeriodicWatermarking.this.syncFactoryCache.getUnchecked((Object)stream.getScope()));
            }
        });
        this.requestTracker = requestTracker;
    }

    @Override
    public void close() {
        this.syncFactoryCache.invalidateAll();
        this.watermarkClientCache.invalidateAll();
    }

    public CompletableFuture<Void> watermark(Stream stream) {
        String scope = stream.getScope();
        String streamName = stream.getStreamName();
        long requestId = this.requestIdGenerator.get();
        String requestDescriptor = RequestTracker.buildRequestDescriptor((String[])new String[]{"watermark", stream.getScope(), stream.getStreamName()});
        this.requestTracker.trackRequest(requestDescriptor, requestId);
        OperationContext context = this.streamMetadataStore.createStreamContext(scope, streamName, requestId);
        if (scope.equals("_system")) {
            return CompletableFuture.completedFuture(null);
        }
        log.debug(requestId, "Periodic background processing for watermarking called for stream {}/{}", new Object[]{scope, streamName});
        CompletableFuture allWriterMarks = Futures.exceptionallyExpecting(this.streamMetadataStore.getAllWriterMarks(scope, streamName, context, this.executor), e -> Exceptions.unwrap((Throwable)e) instanceof StoreException.DataNotFoundException, Collections.emptyMap());
        return ((CompletableFuture)allWriterMarks.thenCompose(writers -> {
            WatermarkClient watermarkClient = (WatermarkClient)this.watermarkClientCache.getUnchecked((Object)stream);
            try {
                watermarkClient.reinitialize();
            }
            catch (Exception e) {
                log.warn(requestId, "Watermarking client for stream {} threw exception {} during reinitialize.", new Object[]{stream, Exceptions.unwrap((Throwable)e).getClass()});
                if (Exceptions.unwrap((Throwable)e) instanceof NoSuchSegmentException) {
                    log.info(requestId, "Invalidating the watermark client in cache for stream {}.", new Object[]{stream});
                    this.watermarkClientCache.invalidate((Object)stream);
                }
                throw e;
            }
            return this.streamMetadataStore.getConfiguration(scope, streamName, context, this.executor).thenCompose(config -> this.filterWritersAndComputeWatermark(scope, streamName, context, watermarkClient, (Map<String, WriterMark>)writers, (StreamConfiguration)config));
        })).exceptionally(e -> {
            log.warn(requestId, "Exception thrown while trying to perform periodic watermark computation. Logging and ignoring.", new Object[]{e});
            return null;
        });
    }

    private CompletionStage<Void> filterWritersAndComputeWatermark(String scope, String streamName, OperationContext context, WatermarkClient watermarkClient, Map<String, WriterMark> writers, StreamConfiguration config) {
        ArrayList<Map.Entry<String, WriterMark>> activeWriters = new ArrayList<Map.Entry<String, WriterMark>>();
        ArrayList inactiveWriters = new ArrayList();
        AtomicBoolean allActiveAreParticipating = new AtomicBoolean(true);
        writers.entrySet().forEach(x -> {
            if (watermarkClient.isWriterActive((Map.Entry<String, WriterMark>)x, config.getTimestampAggregationTimeout())) {
                activeWriters.add((Map.Entry<String, WriterMark>)x);
                if (!watermarkClient.isWriterParticipating(((WriterMark)x.getValue()).getTimestamp())) {
                    allActiveAreParticipating.set(false);
                }
            } else {
                inactiveWriters.add(x);
            }
        });
        CompletableFuture removeInactiveWriters = Futures.allOfWithResults(inactiveWriters.stream().map(x -> Futures.exceptionallyExpecting((CompletableFuture)this.streamMetadataStore.removeWriter(scope, streamName, (String)x.getKey(), (WriterMark)x.getValue(), context, this.executor).thenAccept(v -> watermarkClient.untrackWriterInactivity((String)x.getKey())), e -> Exceptions.unwrap((Throwable)e) instanceof StoreException.WriteConflictException, null)).collect(Collectors.toList()));
        if (activeWriters.isEmpty()) {
            return removeInactiveWriters.thenCompose(v -> this.bucketStore.removeStreamFromBucketStore(BucketStore.ServiceType.WatermarkingService, scope, streamName, this.executor));
        }
        CompletableFuture<Object> watermarkFuture = !allActiveAreParticipating.get() ? CompletableFuture.completedFuture(null) : this.computeWatermark(scope, streamName, context, activeWriters, watermarkClient.getPreviousWatermark());
        CompletableFuture[] completableFutureArray = new CompletableFuture[2];
        completableFutureArray[0] = removeInactiveWriters;
        completableFutureArray[1] = watermarkFuture.thenAccept(watermarkClient::completeIteration);
        return CompletableFuture.allOf(completableFutureArray);
    }

    private CompletableFuture<Watermark> computeWatermark(String scope, String streamName, OperationContext context, List<Map.Entry<String, WriterMark>> activeWriters, Watermark previousWatermark) {
        long requestId = context.getRequestId();
        Watermark.WatermarkBuilder builder = Watermark.builder();
        ConcurrentHashMap upperBound = new ConcurrentHashMap();
        LongSummaryStatistics summarized = activeWriters.stream().collect(Collectors.summarizingLong(x -> ((WriterMark)x.getValue()).getTimestamp()));
        long lowerBoundOnTime = summarized.getMin();
        long upperBoundOnTime = summarized.getMax();
        if (lowerBoundOnTime > previousWatermark.getLowerTimeBound()) {
            CompletableFuture positionsFuture = Futures.allOfWithResults(activeWriters.stream().map(x -> Futures.keysAllOfWithResults(((WriterMark)x.getValue()).getPosition().entrySet().stream().collect(Collectors.toMap(y -> this.getSegmentWithRange(scope, streamName, context, (Long)y.getKey()), Map.Entry::getValue)))).collect(Collectors.toList()));
            log.debug(requestId, "Emitting watermark for stream {}/{} with time {}", new Object[]{scope, streamName, lowerBoundOnTime});
            return ((CompletableFuture)positionsFuture.thenAccept(listOfPositions -> listOfPositions.forEach(position -> this.addToUpperBound((Map<SegmentWithRange, Long>)position, upperBound)))).thenCompose(v -> this.computeStreamCut(scope, streamName, context, upperBound, previousWatermark).thenApply(streamCut -> builder.lowerTimeBound(lowerBoundOnTime).upperTimeBound(upperBoundOnTime).streamCut((Map)ImmutableMap.copyOf((Map)streamCut)).build()));
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<SegmentWithRange> getSegmentWithRange(String scope, String streamName, OperationContext context, long segmentId) {
        return this.streamMetadataStore.getSegment(scope, streamName, segmentId, context, this.executor).thenApply(this::transform);
    }

    private void addToUpperBound(Map<SegmentWithRange, Long> position, Map<SegmentWithRange, Long> upperBound) {
        for (Map.Entry<SegmentWithRange, Long> writerPos : position.entrySet()) {
            SegmentWithRange segment = writerPos.getKey();
            long offset = writerPos.getValue();
            if (upperBound.containsKey(segment)) {
                long newOffset = Math.max(offset, upperBound.get(segment));
                upperBound.put(segment, newOffset);
                continue;
            }
            if (this.hasSuccessors(segment, upperBound.keySet())) continue;
            Set<SegmentWithRange> included = upperBound.keySet();
            included.forEach(x -> {
                if (segment.overlaps(x) && segment.getSegmentId() > x.getSegmentId()) {
                    upperBound.remove(x);
                }
            });
            upperBound.put(segment, offset);
        }
    }

    private boolean hasSuccessors(SegmentWithRange segment, Set<SegmentWithRange> included) {
        return included.stream().anyMatch(x -> segment.overlaps(x) && segment.getSegmentId() < x.getSegmentId());
    }

    private CompletableFuture<Map<SegmentWithRange, Long>> computeStreamCut(String scope, String stream, OperationContext context, Map<SegmentWithRange, Long> upperBound, Watermark previousWatermark) {
        ConcurrentHashMap<SegmentWithRange, Long> streamCut = new ConcurrentHashMap<SegmentWithRange, Long>(upperBound);
        AtomicReference<Map<Double, Double>> missingRanges = new AtomicReference<Map<Double, Double>>(this.findMissingRanges(streamCut));
        if (previousWatermark != null && !previousWatermark.equals((Object)Watermark.EMPTY)) {
            this.addToUpperBound(previousWatermark.getStreamCut(), streamCut);
        }
        return Futures.doWhileLoop(() -> {
            int highestEpoch = streamCut.keySet().stream().mapToInt(x -> NameUtils.getEpoch((long)x.getSegmentId())).max().orElse(-1);
            assert (highestEpoch >= 0);
            return this.streamMetadataStore.getEpoch(scope, stream, highestEpoch, context, this.executor).thenApply(epochRecord -> {
                ((Map)missingRanges.get()).entrySet().forEach(missingRange -> {
                    List<SegmentWithRange> replacement = this.findSegmentsForMissingRange((EpochRecord)epochRecord, (Map.Entry<Double, Double>)missingRange);
                    Map<SegmentWithRange, Long> replacementSegmentOffsetMap = replacement.stream().collect(Collectors.toMap(x -> x, x -> 0L));
                    this.addToUpperBound(replacementSegmentOffsetMap, streamCut);
                });
                return (Map)missingRanges.updateAndGet(x -> this.findMissingRanges(streamCut));
            });
        }, map -> !map.isEmpty(), (Executor)this.executor).thenApply(v -> streamCut);
    }

    private SegmentWithRange transform(StreamSegmentRecord segment) {
        return SegmentWithRange.builder().segmentId(segment.segmentId()).rangeLow(segment.getKeyStart()).rangeHigh(segment.getKeyEnd()).build();
    }

    private List<SegmentWithRange> findSegmentsForMissingRange(EpochRecord epochRecord, Map.Entry<Double, Double> missingRange) {
        return epochRecord.getSegments().stream().filter(x -> x.overlaps((Double)missingRange.getKey(), (Double)missingRange.getValue())).map(this::transform).collect(Collectors.toList());
    }

    private Map<Double, Double> findMissingRanges(Map<SegmentWithRange, Long> streamCut) {
        HashMap<Double, Double> missingRanges = new HashMap<Double, Double>();
        List sorted = streamCut.entrySet().stream().sorted(Comparator.comparingDouble(x -> ((SegmentWithRange)x.getKey()).getRangeLow())).collect(Collectors.toList());
        Map.Entry previous = (Map.Entry)sorted.get(0);
        if (((SegmentWithRange)previous.getKey()).getRangeLow() > 0.0) {
            missingRanges.put(0.0, ((SegmentWithRange)previous.getKey()).getRangeLow());
        }
        for (int i = 1; i < sorted.size(); ++i) {
            Map.Entry next = (Map.Entry)sorted.get(i);
            if (((SegmentWithRange)previous.getKey()).getRangeHigh() != ((SegmentWithRange)next.getKey()).getRangeLow()) {
                missingRanges.put(((SegmentWithRange)previous.getKey()).getRangeHigh(), ((SegmentWithRange)next.getKey()).getRangeLow());
            }
            previous = next;
        }
        if (((SegmentWithRange)previous.getKey()).getRangeHigh() < 1.0) {
            missingRanges.put(((SegmentWithRange)previous.getKey()).getRangeHigh(), 1.0);
        }
        return missingRanges;
    }

    @VisibleForTesting
    boolean checkExistsInCache(Stream stream) {
        return this.watermarkClientCache.asMap().containsKey(stream);
    }

    @VisibleForTesting
    boolean checkExistsInCache(String scope) {
        return this.syncFactoryCache.asMap().containsKey(scope);
    }

    @VisibleForTesting
    void evictFromCache(String scope) {
        this.syncFactoryCache.invalidate((Object)scope);
    }

    static class WatermarkClient
    implements Closeable {
        @SuppressFBWarnings(justification="generated code")
        @Generated
        private final Object $lock = new Object[0];
        private final RevisionedStreamClient<Watermark> client;
        private final AtomicReference<Map.Entry<Revision, Watermark>> previousWatermark = new AtomicReference();
        private final AtomicReference<Revision> markRevision = new AtomicReference();
        private final ConcurrentHashMap<String, Long> inactiveWriters;

        @VisibleForTesting
        WatermarkClient(Stream stream, SynchronizerClientFactory clientFactory) {
            this.client = clientFactory.createRevisionedStreamClient(NameUtils.getMarkStreamForStream((String)stream.getStreamName()), (Serializer)new WatermarkSerializer(), SynchronizerConfig.builder().build());
            this.inactiveWriters = new ConcurrentHashMap();
        }

        Watermark getPreviousWatermark() {
            Map.Entry<Revision, Watermark> watermark = this.previousWatermark.get();
            return watermark == null ? Watermark.EMPTY : watermark.getValue();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void reinitialize() {
            Object object = this.$lock;
            synchronized (object) {
                Revision revision = this.client.getMark();
                if (revision == null) {
                    this.markRevision.set(this.client.fetchOldestRevision());
                    this.client.compareAndSetMark(null, this.markRevision.get());
                } else {
                    this.markRevision.set(revision);
                }
                ArrayList entries = Lists.newArrayList((Iterator)this.client.readFrom(this.markRevision.get()));
                if (!entries.isEmpty()) {
                    this.previousWatermark.set((Map.Entry)entries.get(entries.size() - 1));
                } else {
                    this.previousWatermark.set(null);
                }
            }
        }

        void completeIteration(Watermark newWatermark) {
            Map.Entry<Revision, Watermark> previous = this.previousWatermark.get();
            if (newWatermark != null) {
                Revision revision = previous == null ? this.markRevision.get() : previous.getKey();
                Revision newRevision = this.client.writeConditionally(revision, (Object)newWatermark);
                if (newRevision == null) {
                    return;
                }
                if (previous != null) {
                    this.client.compareAndSetMark(this.markRevision.get(), previous.getKey());
                }
            }
        }

        boolean isWriterActive(Map.Entry<String, WriterMark> writerMark, long timeout) {
            if (!this.isWriterParticipating(writerMark.getValue().getTimestamp())) {
                long currentTime = System.currentTimeMillis();
                this.inactiveWriters.putIfAbsent(writerMark.getKey(), currentTime);
                Long time = this.inactiveWriters.getOrDefault(writerMark.getKey(), currentTime);
                boolean timedOut = currentTime - time >= timeout;
                return writerMark.getValue().isAlive() && !timedOut;
            }
            this.inactiveWriters.remove(writerMark.getKey());
            return true;
        }

        boolean isWriterParticipating(long time) {
            Map.Entry<Revision, Watermark> latest = this.previousWatermark.get();
            if (latest == null) {
                return true;
            }
            return time > latest.getValue().getLowerTimeBound();
        }

        private void untrackWriterInactivity(String writerId) {
            this.inactiveWriters.remove(writerId);
        }

        @VisibleForTesting
        boolean isWriterTracked(String writerId) {
            return this.inactiveWriters.containsKey(writerId);
        }

        @Override
        public void close() {
            this.client.close();
        }
    }
}

