/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.controller.store;

import com.google.common.annotations.VisibleForTesting;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.util.ReferenceCountUtil;
import io.pravega.client.tables.impl.HashTableIteratorItem;
import io.pravega.client.tables.impl.TableSegmentEntry;
import io.pravega.client.tables.impl.TableSegmentKey;
import io.pravega.client.tables.impl.TableSegmentKeyVersion;
import io.pravega.common.Exceptions;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.tracing.TagLogger;
import io.pravega.common.util.AsyncIterator;
import io.pravega.common.util.BitConverter;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.ContinuationTokenAsyncIterator;
import io.pravega.common.util.RetriesExhaustedException;
import io.pravega.common.util.StructuredWritableBuffer;
import io.pravega.controller.server.SegmentHelper;
import io.pravega.controller.server.WireCommandFailedException;
import io.pravega.controller.server.security.auth.GrpcAuthHelper;
import io.pravega.controller.store.Version;
import io.pravega.controller.store.VersionedMetadata;
import io.pravega.controller.store.host.HostStoreException;
import io.pravega.controller.store.stream.Cache;
import io.pravega.controller.store.stream.OperationContext;
import io.pravega.controller.store.stream.StoreException;
import io.pravega.controller.util.RetryHelper;
import java.beans.ConstructorProperties;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import lombok.Generated;
import org.apache.curator.shaded.com.google.common.base.Charsets;
import org.slf4j.LoggerFactory;

public class PravegaTablesStoreHelper {
    public static final Function<Integer, byte[]> INTEGER_TO_BYTES_FUNCTION = x -> {
        byte[] bytes = new byte[4];
        BitConverter.writeInt((byte[])bytes, (int)0, (int)x);
        return bytes;
    };
    public static final Function<Long, byte[]> LONG_TO_BYTES_FUNCTION = x -> {
        byte[] bytes = new byte[8];
        BitConverter.writeLong((byte[])bytes, (int)0, (long)x);
        return bytes;
    };
    public static final Function<UUID, byte[]> UUID_TO_BYTES_FUNCTION = x -> {
        byte[] b = new byte[16];
        BitConverter.writeUUID((StructuredWritableBuffer)new ByteArraySegment(b), (UUID)x);
        return b;
    };
    public static final Function<byte[], Long> BYTES_TO_LONG_FUNCTION = data -> BitConverter.readLong((byte[])data, (int)0);
    public static final Function<byte[], Integer> BYTES_TO_INTEGER_FUNCTION = x -> BitConverter.readInt((byte[])x, (int)0);
    public static final Function<byte[], UUID> BYTES_TO_UUID_FUNCTION = x -> BitConverter.readUUID((byte[])x, (int)0);
    private static final TagLogger log = new TagLogger(LoggerFactory.getLogger(PravegaTablesStoreHelper.class));
    private static final int NUM_OF_RETRIES = 15;
    private final SegmentHelper segmentHelper;
    private final ScheduledExecutorService executor;
    private final Cache cache;
    private final AtomicReference<String> authToken;
    private final GrpcAuthHelper authHelper;
    private final int numOfRetries;

    public PravegaTablesStoreHelper(SegmentHelper segmentHelper, GrpcAuthHelper authHelper, ScheduledExecutorService executor) {
        this(segmentHelper, authHelper, executor, 15);
    }

    @VisibleForTesting
    PravegaTablesStoreHelper(SegmentHelper segmentHelper, GrpcAuthHelper authHelper, ScheduledExecutorService executor, int numOfRetries) {
        this.segmentHelper = segmentHelper;
        this.executor = executor;
        this.cache = new Cache();
        this.authHelper = authHelper;
        this.authToken = new AtomicReference<String>(authHelper.retrieveMasterToken());
        this.numOfRetries = numOfRetries;
    }

    private <T> void putInCache(String table, String key, VersionedMetadata<T> value, long time) {
        TableCacheKey cacheKey = new TableCacheKey(table, key);
        this.cache.put(cacheKey, value, time);
    }

    private <T> VersionedMetadata<T> getCachedData(String table, String key, long time, long requestId) {
        TableCacheKey cacheKey = new TableCacheKey(table, key);
        VersionedMetadata<?> cachedData = this.cache.getCachedData(cacheKey, time);
        if (cachedData == null) {
            return null;
        }
        log.trace(requestId, "found entry for key {} in table {} in cache", new Object[]{key, table});
        return this.getVersionedMetadata(cachedData);
    }

    public void invalidateCache(String table, String key) {
        this.cache.invalidateCache(new TableCacheKey(table, key));
    }

    private <T> VersionedMetadata<T> getVersionedMetadata(VersionedMetadata<?> v) {
        return new VersionedMetadata(v.getObject(), v.getVersion());
    }

    public CompletableFuture<Void> createTable(String tableName, long requestId) {
        return this.createTable(tableName, requestId, 0L);
    }

    public CompletableFuture<Void> createTable(String tableName, long requestId, long rolloverSizeBytes) {
        log.debug(requestId, "create table called for table: {}", new Object[]{tableName});
        return Futures.toVoid(this.withRetries(() -> this.segmentHelper.createTableSegment(tableName, this.authToken.get(), requestId, false, 0, rolloverSizeBytes), () -> String.format("create table: %s", tableName), requestId)).whenCompleteAsync((r, e) -> {
            if (e != null) {
                log.warn(requestId, "create table {} threw exception", new Object[]{tableName, e});
            } else {
                log.debug(requestId, "table {} created successfully", new Object[]{tableName});
            }
        }, (Executor)this.executor);
    }

    public CompletableFuture<Void> deleteTable(String tableName, boolean mustBeEmpty, long requestId) {
        log.debug(requestId, "delete table called for table: {}", new Object[]{tableName});
        return this.expectingDataNotFound(this.withRetries(() -> this.segmentHelper.deleteTableSegment(tableName, mustBeEmpty, this.authToken.get(), requestId), () -> String.format("delete table: %s", tableName), requestId), null).thenAcceptAsync(v -> log.debug(requestId, "table {} deleted successfully", new Object[]{tableName}), (Executor)this.executor);
    }

    public <T> CompletableFuture<Version> addNewEntry(String tableName, String key, T val, Function<T, byte[]> toBytes, long requestId) {
        log.trace(requestId, "addNewEntry called for : {} key : {}", new Object[]{tableName, key});
        byte[] value = toBytes.apply(val);
        List<TableSegmentEntry> entries = Collections.singletonList(TableSegmentEntry.notExists((byte[])key.getBytes(Charsets.UTF_8), (byte[])value));
        Supplier<String> errorMessage = () -> String.format("addNewEntry: key: %s table: %s", key, tableName);
        long time = System.currentTimeMillis();
        return ((CompletableFuture)((CompletableFuture)this.withRetries(() -> this.segmentHelper.updateTableEntries(tableName, entries, this.authToken.get(), requestId), errorMessage, true, requestId).exceptionally(e -> {
            Throwable unwrap = Exceptions.unwrap((Throwable)e);
            this.invalidateCache(tableName, key);
            if (unwrap instanceof StoreException.WriteConflictException) {
                throw StoreException.create(StoreException.Type.DATA_EXISTS, (String)errorMessage.get());
            }
            log.debug(requestId, "add new entry {} to {} threw exception {} {}", new Object[]{key, tableName, unwrap.getClass(), unwrap.getMessage()});
            throw new CompletionException((Throwable)e);
        })).thenApplyAsync(x -> {
            TableSegmentKeyVersion first = (TableSegmentKeyVersion)x.get(0);
            log.debug(requestId, "entry for key {} added to table {} with version {}", new Object[]{key, tableName, first.getSegmentVersion()});
            Version.LongVersion version = new Version.LongVersion(first.getSegmentVersion());
            this.putInCache(tableName, key, new VersionedMetadata<Object>(val, version), time);
            return version;
        }, (Executor)this.executor)).whenComplete((r, ex) -> this.releaseEntries(entries));
    }

    private CompletableFuture<Void> conditionalDeleteOfKey(String tableName, long requestId, String key, TableSegmentKeyVersion keyVersion) {
        return this.expectingWriteConflict(this.removeEntry(tableName, key, new Version.LongVersion(keyVersion.getSegmentVersion()), requestId), null);
    }

    public CompletableFuture<Long> getEntryCount(String tableName, long requestId) {
        log.debug(requestId, "create table called for table: {}", new Object[]{tableName});
        return this.withRetries(() -> this.segmentHelper.getTableSegmentEntryCount(tableName, this.authToken.get(), requestId), () -> String.format("GetInfo table: %s", tableName), requestId).whenCompleteAsync((r, e) -> {
            if (e != null) {
                log.warn(requestId, "Get Table Segment info for table {} threw exception", new Object[]{tableName, e});
            } else {
                log.debug(requestId, "Get Table Segment info for table {} completed successfully", new Object[]{tableName});
            }
        }, (Executor)this.executor);
    }

    public CompletableFuture<Void> getAndUpdateEntry(String tableName, String tableKey, Function<TableSegmentEntry, TableSegmentEntry> updateFunction, Predicate<TableSegmentEntry> attemptCleanup, long requestId) {
        Supplier<String> errorMessage = () -> String.format("get and update values: on table: %s for key %s", tableName, tableKey);
        List<TableSegmentKey> keys = Collections.singletonList(TableSegmentKey.unversioned((byte[])tableKey.getBytes(StandardCharsets.UTF_8)));
        return this.withRetries(() -> ((CompletableFuture)this.segmentHelper.readTable(tableName, keys, this.authToken.get(), requestId).whenComplete((v, ex) -> this.releaseKeys(keys))).thenCompose(entries -> {
            TableSegmentEntry updatedEntry = (TableSegmentEntry)updateFunction.apply((TableSegmentEntry)entries.get(0));
            boolean shouldAttemptCleanup = attemptCleanup.test(updatedEntry);
            return ((CompletableFuture)this.segmentHelper.updateTableEntries(tableName, Collections.singletonList(updatedEntry), this.authToken.get(), requestId).thenCompose(keyVersions -> {
                if (shouldAttemptCleanup) {
                    log.debug(requestId, "Delete of table key {} on table {}", new Object[]{tableKey, tableName});
                    return this.conditionalDeleteOfKey(tableName, requestId, tableKey, (TableSegmentKeyVersion)keyVersions.get(0));
                }
                return CompletableFuture.completedFuture(null);
            })).whenComplete((v, ex) -> this.releaseEntries(Collections.singletonList(updatedEntry)));
        }), errorMessage, true, requestId);
    }

    public <T> CompletableFuture<Version> addNewEntryIfAbsent(String tableName, String key, T val, Function<T, byte[]> toBytes, long requestId) {
        return this.expectingDataExists(this.addNewEntry(tableName, key, val, toBytes, requestId), null);
    }

    public <T> CompletableFuture<Void> addNewEntriesIfAbsent(String tableName, List<Map.Entry<String, T>> toAdd, Function<T, byte[]> toBytes, long requestId) {
        List entries = toAdd.stream().map(x -> TableSegmentEntry.notExists((byte[])((String)x.getKey()).getBytes(Charsets.UTF_8), (byte[])((byte[])toBytes.apply(x.getValue())))).collect(Collectors.toList());
        Supplier<String> errorMessage = () -> String.format("addNewEntriesIfAbsent: table: %s", tableName);
        long time = System.currentTimeMillis();
        return this.expectingDataExists((CompletableFuture<T>)this.withRetries(() -> this.segmentHelper.updateTableEntries(tableName, entries, this.authToken.get(), requestId), errorMessage, requestId).handle((r, e) -> {
            this.releaseEntries(entries);
            if (e != null) {
                Throwable unwrap = Exceptions.unwrap((Throwable)e);
                toAdd.forEach(entry -> this.invalidateCache(tableName, (String)entry.getKey()));
                if (unwrap instanceof StoreException.WriteConflictException) {
                    throw StoreException.create(StoreException.Type.DATA_EXISTS, (String)errorMessage.get());
                }
                log.debug(requestId, "add new entries to {} threw exception {} {}", new Object[]{tableName, unwrap.getClass(), unwrap.getMessage()});
                throw new CompletionException((Throwable)e);
            }
            log.debug(requestId, "entries added {} to table {}", new Object[]{toAdd, tableName});
            for (int i = 0; i < r.size(); ++i) {
                this.putInCache(tableName, (String)((Map.Entry)toAdd.get(i)).getKey(), new VersionedMetadata(((Map.Entry)toAdd.get(i)).getValue(), new Version.LongVersion(((TableSegmentKeyVersion)r.get(i)).getSegmentVersion())), time);
            }
            return null;
        }), null);
    }

    public <T> CompletableFuture<Version> updateEntry(String tableName, String key, T val, Function<T, byte[]> toBytes, Version ver, long requestId) {
        long version = ver.asLongVersion().getLongValue();
        log.trace(requestId, "updateEntry entry called for : {} key : {} version {}", new Object[]{tableName, key, version});
        byte[] value = toBytes.apply(val);
        long time = System.currentTimeMillis();
        List<TableSegmentEntry> entries = Collections.singletonList(TableSegmentEntry.versioned((byte[])key.getBytes(Charsets.UTF_8), (byte[])value, (long)version));
        return ((CompletableFuture)((CompletableFuture)this.withRetries(() -> this.segmentHelper.updateTableEntries(tableName, entries, this.authToken.get(), requestId), () -> String.format("updateEntry: key: %s table: %s", key, tableName), true, requestId).thenApplyAsync(x -> {
            TableSegmentKeyVersion first = (TableSegmentKeyVersion)x.get(0);
            log.debug(requestId, "entry for key {} updated to table {} with new version {}", new Object[]{key, tableName, first.getSegmentVersion()});
            Version.LongVersion newVersion = new Version.LongVersion(first.getSegmentVersion());
            this.putInCache(tableName, key, new VersionedMetadata<Object>(val, newVersion), time);
            return newVersion;
        }, (Executor)this.executor)).exceptionally(e -> {
            this.invalidateCache(tableName, key);
            throw new CompletionException((Throwable)e);
        })).whenComplete((r, ex) -> this.releaseEntries(entries));
    }

    public <T> CompletableFuture<VersionedMetadata<T>> getEntry(String tableName, String key, Function<byte[], T> fromBytes, long requestId) {
        log.trace(requestId, "get entry called for : {} key : {}", new Object[]{tableName, key});
        List<TableSegmentKey> keys = Collections.singletonList(TableSegmentKey.unversioned((byte[])key.getBytes(Charsets.UTF_8)));
        CompletableFuture result = new CompletableFuture();
        String message = "get entry: key: %s table: %s";
        ((CompletableFuture)this.withRetries(() -> this.segmentHelper.readTable(tableName, keys, this.authToken.get(), requestId), () -> String.format(message, key, tableName), requestId).thenApplyAsync(x -> {
            try {
                TableSegmentEntry first = (TableSegmentEntry)x.get(0);
                if (first.getKey().getVersion().equals((Object)TableSegmentKeyVersion.NOT_EXISTS)) {
                    throw StoreException.create(StoreException.Type.DATA_NOT_FOUND, String.format(message, key, tableName));
                }
                log.trace(requestId, "returning entry for : {} key : {} with version {}", new Object[]{tableName, key, first.getKey().getVersion().getSegmentVersion()});
                Object deserialized = fromBytes.apply(this.getArray(first.getValue()));
                VersionedMetadata versionedMetadata = new VersionedMetadata(deserialized, new Version.LongVersion(first.getKey().getVersion().getSegmentVersion()));
                return versionedMetadata;
            }
            finally {
                this.releaseEntries((Collection<TableSegmentEntry>)x);
            }
        }, (Executor)this.executor)).whenCompleteAsync((r, e) -> {
            this.releaseKeys(keys);
            if (e != null) {
                result.completeExceptionally((Throwable)e);
            } else {
                result.complete((VersionedMetadata)r);
            }
        }, (Executor)this.executor);
        return result;
    }

    public <T> CompletableFuture<VersionedMetadata<T>> getCachedOrLoad(String tableName, String key, Function<byte[], T> fromBytes, long afterTime, long requestId) {
        log.trace(requestId, "get entry called for : {} key : {}", new Object[]{tableName, key});
        VersionedMetadata<T> cached = this.getCachedData(tableName, key, afterTime, requestId);
        if (cached != null) {
            return CompletableFuture.completedFuture(this.getVersionedMetadata(cached));
        }
        long time = System.currentTimeMillis();
        return this.getEntry(tableName, key, fromBytes, requestId).thenApply(r -> {
            this.putInCache(tableName, key, (VersionedMetadata)r, time);
            return r;
        });
    }

    public <T> CompletableFuture<List<VersionedMetadata<T>>> getEntries(String tableName, List<String> keys, Function<byte[], T> fromBytes, VersionedMetadata<T> nonExistent, long requestId) {
        log.trace(requestId, "get entries called for : {} keys : {}", new Object[]{tableName, keys});
        List tableKeys = keys.stream().map(key -> TableSegmentKey.unversioned((byte[])key.getBytes(Charsets.UTF_8))).collect(Collectors.toList());
        CompletableFuture result = new CompletableFuture();
        String message = "get entry: key: %s table: %s";
        long time = System.currentTimeMillis();
        ((CompletableFuture)this.withRetries(() -> this.segmentHelper.readTable(tableName, tableKeys, this.authToken.get(), requestId), () -> String.format(message, keys, tableName), requestId).thenApplyAsync(entries -> {
            try {
                ArrayList list = new ArrayList(keys.size());
                for (int i = 0; i < keys.size(); ++i) {
                    TableSegmentEntry entry = (TableSegmentEntry)entries.get(i);
                    if (entry.getKey().getVersion().equals((Object)TableSegmentKeyVersion.NOT_EXISTS)) {
                        list.add(nonExistent);
                        continue;
                    }
                    VersionedMetadata tVersionedMetadata = new VersionedMetadata(fromBytes.apply(this.getArray(entry.getValue())), new Version.LongVersion(entry.getKey().getVersion().getSegmentVersion()));
                    this.putInCache(tableName, (String)keys.get(i), tVersionedMetadata, time);
                    list.add(tVersionedMetadata);
                }
                ArrayList arrayList = list;
                return arrayList;
            }
            finally {
                this.releaseEntries((Collection<TableSegmentEntry>)entries);
            }
        }, (Executor)this.executor)).whenCompleteAsync((r, e) -> {
            this.releaseKeys(tableKeys);
            if (e != null) {
                result.completeExceptionally((Throwable)e);
            } else {
                result.complete((List)r);
            }
        }, (Executor)this.executor);
        return result;
    }

    public CompletableFuture<Void> removeEntry(String tableName, String key, long requestId) {
        return this.removeEntry(tableName, key, null, requestId);
    }

    public CompletableFuture<Void> removeEntry(String tableName, String key, Version ver, long requestId) {
        log.trace(requestId, "remove entry called for : {} key : {}", new Object[]{tableName, key});
        TableSegmentKey tableKey = ver == null ? TableSegmentKey.unversioned((byte[])key.getBytes(Charsets.UTF_8)) : TableSegmentKey.versioned((byte[])key.getBytes(Charsets.UTF_8), (long)ver.asLongVersion().getLongValue());
        return ((CompletableFuture)this.expectingDataNotFound(this.withRetries(() -> this.segmentHelper.removeTableKeys(tableName, Collections.singletonList(tableKey), this.authToken.get(), requestId), () -> String.format("remove entry: key: %s table: %s", key, tableName), requestId), null).thenAcceptAsync(v -> {
            this.invalidateCache(tableName, key);
            log.trace(requestId, "entry for key {} removed from table {}", new Object[]{key, tableName});
        }, (Executor)this.executor)).whenComplete((r, ex) -> this.releaseKeys(Collections.singleton(tableKey)));
    }

    public CompletableFuture<Void> removeEntries(String tableName, Collection<String> keys, long requestId) {
        log.trace(requestId, "remove entry called for : {} keys : {}", new Object[]{tableName, keys});
        List listOfKeys = keys.stream().map(x -> TableSegmentKey.unversioned((byte[])x.getBytes(Charsets.UTF_8))).collect(Collectors.toList());
        return ((CompletableFuture)this.expectingDataNotFound(this.withRetries(() -> this.segmentHelper.removeTableKeys(tableName, listOfKeys, this.authToken.get(), requestId), () -> String.format("remove entries: keys: %s table: %s", keys.toString(), tableName), requestId), null).thenAcceptAsync(v -> {
            keys.forEach(key -> this.invalidateCache(tableName, (String)key));
            log.trace(requestId, "entry for keys {} removed from table {}", new Object[]{keys, tableName});
        }, (Executor)this.executor)).whenComplete((r, ex) -> this.releaseKeys(listOfKeys));
    }

    public CompletableFuture<Map.Entry<ByteBuf, List<String>>> getKeysPaginated(String tableName, ByteBuf continuationToken, int limit, long requestId) {
        log.trace(requestId, "get keys paginated called for : {}", new Object[]{tableName});
        return this.withRetries(() -> this.segmentHelper.readTableKeys(tableName, limit, HashTableIteratorItem.State.fromBytes((ByteBuf)continuationToken), this.authToken.get(), requestId), () -> String.format("get keys paginated for table: %s", tableName), requestId).thenApplyAsync(result -> {
            try {
                List items = result.getItems().stream().map(x -> new String(this.getArray(x.getKey()), Charsets.UTF_8)).collect(Collectors.toList());
                log.trace(requestId, "get keys paginated on table {} returned items {}", new Object[]{tableName, items});
                AbstractMap.SimpleEntry simpleEntry = new AbstractMap.SimpleEntry(this.getNextToken(continuationToken, (HashTableIteratorItem<?>)result), items);
                return simpleEntry;
            }
            finally {
                this.releaseKeys(result.getItems());
            }
        }, (Executor)this.executor);
    }

    public <T> CompletableFuture<Map.Entry<ByteBuf, List<Map.Entry<String, VersionedMetadata<T>>>>> getEntriesPaginated(String tableName, ByteBuf continuationToken, int limit, Function<byte[], T> fromBytes, long requestId) {
        log.trace(requestId, "get entries paginated called for : {}", new Object[]{tableName});
        long time = System.currentTimeMillis();
        return this.withRetries(() -> this.segmentHelper.readTableEntries(tableName, limit, HashTableIteratorItem.State.fromBytes((ByteBuf)continuationToken), this.authToken.get(), requestId), () -> String.format("get entries paginated for table: %s", tableName), requestId).thenApplyAsync(result -> {
            try {
                List items = result.getItems().stream().map(x -> {
                    String key = new String(this.getArray(x.getKey().getKey()), Charsets.UTF_8);
                    Object deserialized = fromBytes.apply(this.getArray(x.getValue()));
                    VersionedMetadata value = new VersionedMetadata(deserialized, new Version.LongVersion(x.getKey().getVersion().getSegmentVersion()));
                    this.putInCache(tableName, key, value, time);
                    return new AbstractMap.SimpleEntry(key, value);
                }).collect(Collectors.toList());
                log.trace(requestId, "get keys paginated on table {} returned number of items {}", new Object[]{tableName, items.size()});
                AbstractMap.SimpleEntry simpleEntry = new AbstractMap.SimpleEntry(this.getNextToken(continuationToken, (HashTableIteratorItem<?>)result), items);
                return simpleEntry;
            }
            finally {
                this.releaseEntries(result.getItems());
            }
        }, (Executor)this.executor);
    }

    private ByteBuf getNextToken(ByteBuf continuationToken, HashTableIteratorItem<?> result) {
        return result.getItems().isEmpty() && result.getState().isEmpty() ? continuationToken : Unpooled.wrappedBuffer((ByteBuffer)result.getState().toBytes());
    }

    public AsyncIterator<String> getAllKeys(String tableName, long requestId) {
        return new ContinuationTokenAsyncIterator(token -> this.getKeysPaginated(tableName, (ByteBuf)token, 1000, requestId).thenApplyAsync(result -> {
            token.release();
            return new AbstractMap.SimpleEntry<ByteBuf, Collection>((ByteBuf)result.getKey(), (Collection)result.getValue());
        }, (Executor)this.executor), (Object)HashTableIteratorItem.State.EMPTY.getToken());
    }

    public <T> AsyncIterator<Map.Entry<String, VersionedMetadata<T>>> getAllEntries(String tableName, Function<byte[], T> fromBytes, long requestId) {
        return new ContinuationTokenAsyncIterator(token -> this.getEntriesPaginated(tableName, (ByteBuf)token, 1000, fromBytes, requestId).thenApplyAsync(result -> {
            token.release();
            return new AbstractMap.SimpleEntry<ByteBuf, Collection>((ByteBuf)result.getKey(), (Collection)result.getValue());
        }, (Executor)this.executor), (Object)HashTableIteratorItem.State.EMPTY.getToken());
    }

    public <T> CompletableFuture<T> expectingDataNotFound(CompletableFuture<T> future, T toReturn) {
        return Futures.exceptionallyExpecting(future, e -> Exceptions.unwrap((Throwable)e) instanceof StoreException.DataNotFoundException, toReturn);
    }

    public <T> CompletableFuture<T> expectingWriteConflict(CompletableFuture<T> future, T toReturn) {
        return Futures.exceptionallyExpecting(future, e -> Exceptions.unwrap((Throwable)e) instanceof StoreException.WriteConflictException, toReturn);
    }

    <T> CompletableFuture<T> expectingDataExists(CompletableFuture<T> future, T toReturn) {
        return Futures.exceptionallyExpecting(future, e -> Exceptions.unwrap((Throwable)e) instanceof StoreException.DataExistsException, toReturn);
    }

    private <T> Supplier<CompletableFuture<T>> exceptionalCallback(Supplier<CompletableFuture<T>> future, Supplier<String> errorMessageSupplier, boolean throwOriginalOnCFE, long requestId) {
        return () -> ((CompletableFuture)CompletableFuture.completedFuture(null).thenComposeAsync(arg_0 -> PravegaTablesStoreHelper.lambda$exceptionalCallback$71((Supplier)future, arg_0), (Executor)this.executor)).exceptionally(arg_0 -> this.lambda$exceptionalCallback$72((Supplier)errorMessageSupplier, throwOriginalOnCFE, requestId, arg_0));
    }

    private <T> CompletableFuture<T> withRetries(Supplier<CompletableFuture<T>> futureSupplier, Supplier<String> errorMessage, long requestId) {
        return this.withRetries(futureSupplier, errorMessage, false, requestId);
    }

    private <T> CompletableFuture<T> withRetries(Supplier<CompletableFuture<T>> futureSupplier, Supplier<String> errorMessage, boolean throwOriginalOnCfe, long requestId) {
        return RetryHelper.withRetriesAsync(this.exceptionalCallback(futureSupplier, errorMessage, throwOriginalOnCfe, requestId), e -> Exceptions.unwrap((Throwable)e) instanceof StoreException.StoreConnectionException, this.numOfRetries, this.executor).exceptionally(e -> {
            Throwable t = Exceptions.unwrap((Throwable)e);
            if (t instanceof RetriesExhaustedException) {
                throw new CompletionException(t.getCause());
            }
            Throwable unwrap = Exceptions.unwrap((Throwable)e);
            if (unwrap instanceof WireCommandFailedException && (((WireCommandFailedException)unwrap).getReason().equals((Object)WireCommandFailedException.Reason.ConnectionDropped) || ((WireCommandFailedException)unwrap).getReason().equals((Object)WireCommandFailedException.Reason.ConnectionFailed))) {
                throw new CompletionException(StoreException.create(StoreException.Type.CONNECTION_ERROR, (String)errorMessage.get()));
            }
            throw new CompletionException(unwrap);
        });
    }

    byte[] getArray(ByteBuf buf) {
        byte[] bytes = new byte[buf.readableBytes()];
        int readerIndex = buf.readerIndex();
        buf.getBytes(readerIndex, bytes);
        return bytes;
    }

    private void releaseKeys(Collection<TableSegmentKey> keys) {
        for (TableSegmentKey k : keys) {
            ReferenceCountUtil.safeRelease((Object)k.getKey());
        }
    }

    private void releaseEntries(Collection<TableSegmentEntry> entries) {
        for (TableSegmentEntry e : entries) {
            ReferenceCountUtil.safeRelease((Object)e.getKey().getKey());
            ReferenceCountUtil.safeRelease((Object)e.getValue());
        }
    }

    public <T> CompletableFuture<VersionedMetadata<T>> loadFromTableHandleStaleTableName(BiFunction<Boolean, OperationContext, CompletableFuture<String>> tableNameSupplier, String key, Function<byte[], T> valueFunction, OperationContext context) {
        return Futures.exceptionallyComposeExpecting((CompletableFuture)tableNameSupplier.apply(false, context).thenCompose(tableName -> this.getCachedOrLoad((String)tableName, key, valueFunction, context.getOperationStartTime(), context.getRequestId())), e -> Exceptions.unwrap((Throwable)e) instanceof StoreException.DataContainerNotFoundException, () -> ((CompletableFuture)tableNameSupplier.apply(true, context)).thenCompose(tableName -> this.getCachedOrLoad((String)tableName, key, valueFunction, context.getOperationStartTime(), context.getRequestId())));
    }

    private /* synthetic */ Object lambda$exceptionalCallback$72(Supplier errorMessageSupplier, boolean throwOriginalOnCFE, long requestId, Throwable t) {
        RuntimeException toThrow;
        String errorMessage = (String)errorMessageSupplier.get();
        Throwable cause = Exceptions.unwrap((Throwable)t);
        if (cause instanceof WireCommandFailedException) {
            WireCommandFailedException wcfe = (WireCommandFailedException)cause;
            switch (wcfe.getReason()) {
                case ConnectionDropped: 
                case ConnectionFailed: {
                    toThrow = throwOriginalOnCFE ? wcfe : StoreException.create(StoreException.Type.CONNECTION_ERROR, wcfe, errorMessage);
                    break;
                }
                case UnknownHost: {
                    toThrow = StoreException.create(StoreException.Type.CONNECTION_ERROR, wcfe, errorMessage);
                    break;
                }
                case PreconditionFailed: {
                    toThrow = StoreException.create(StoreException.Type.ILLEGAL_STATE, wcfe, errorMessage);
                    break;
                }
                case AuthFailed: {
                    this.authToken.set(this.authHelper.retrieveMasterToken());
                    toThrow = StoreException.create(StoreException.Type.CONNECTION_ERROR, wcfe, errorMessage);
                    break;
                }
                case SegmentDoesNotExist: {
                    toThrow = StoreException.create(StoreException.Type.DATA_CONTAINER_NOT_FOUND, wcfe, errorMessage);
                    break;
                }
                case TableSegmentNotEmpty: {
                    toThrow = StoreException.create(StoreException.Type.DATA_CONTAINS_ELEMENTS, wcfe, errorMessage);
                    break;
                }
                case TableKeyDoesNotExist: {
                    toThrow = StoreException.create(StoreException.Type.DATA_NOT_FOUND, wcfe, errorMessage);
                    break;
                }
                case TableKeyBadVersion: {
                    toThrow = StoreException.create(StoreException.Type.WRITE_CONFLICT, wcfe, errorMessage);
                    break;
                }
                default: {
                    toThrow = StoreException.create(StoreException.Type.UNKNOWN, wcfe, errorMessage);
                    break;
                }
            }
        } else if (cause instanceof HostStoreException) {
            log.warn(requestId, "Host Store exception {}", new Object[]{cause.getMessage()});
            toThrow = StoreException.create(StoreException.Type.CONNECTION_ERROR, cause, errorMessage);
        } else {
            log.warn(requestId, "exception of unknown type thrown {} ", new Object[]{errorMessage, cause});
            toThrow = StoreException.create(StoreException.Type.UNKNOWN, cause, errorMessage);
        }
        throw new CompletionException(toThrow);
    }

    private static /* synthetic */ CompletionStage lambda$exceptionalCallback$71(Supplier future, Object v) {
        return (CompletionStage)future.get();
    }

    private static class TableCacheKey
    implements Cache.CacheKey {
        private final String table;
        private final String key;

        @ConstructorProperties(value={"table", "key"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public TableCacheKey(String table, String key) {
            this.table = table;
            this.key = key;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String getTable() {
            return this.table;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String getKey() {
            return this.key;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof TableCacheKey)) {
                return false;
            }
            TableCacheKey other = (TableCacheKey)o;
            if (!other.canEqual(this)) {
                return false;
            }
            String this$table = this.getTable();
            String other$table = other.getTable();
            if (this$table == null ? other$table != null : !this$table.equals(other$table)) {
                return false;
            }
            String this$key = this.getKey();
            String other$key = other.getKey();
            return !(this$key == null ? other$key != null : !this$key.equals(other$key));
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        protected boolean canEqual(Object other) {
            return other instanceof TableCacheKey;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            String $table = this.getTable();
            result = result * 59 + ($table == null ? 43 : $table.hashCode());
            String $key = this.getKey();
            result = result * 59 + ($key == null ? 43 : $key.hashCode());
            return result;
        }

        @SuppressFBWarnings(justification="generated code")
        @Generated
        public String toString() {
            return "PravegaTablesStoreHelper.TableCacheKey(table=" + this.getTable() + ", key=" + this.getKey() + ")";
        }
    }
}

