/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package io.prestosql.hadoop.$internal.org.apache.http.impl;

import java.io.IOException;
import java.net.Socket;

import io.prestosql.hadoop.$internal.org.apache.http.annotation.NotThreadSafe;
import io.prestosql.hadoop.$internal.org.apache.http.params.HttpConnectionParams;
import io.prestosql.hadoop.$internal.org.apache.http.params.HttpParams;

/**
 * Default implementation of a server-side HTTP connection.
 * <p>
 * The following parameters can be used to customize the behavior of this
 * class:
 * <ul>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreProtocolPNames#HTTP_ELEMENT_CHARSET}</li>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreConnectionPNames#TCP_NODELAY}</li>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreConnectionPNames#SO_TIMEOUT}</li>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreConnectionPNames#SO_LINGER}</li>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreConnectionPNames#SO_KEEPALIVE}</li>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreConnectionPNames#SOCKET_BUFFER_SIZE}</li>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreConnectionPNames#MAX_LINE_LENGTH}</li>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreConnectionPNames#MAX_HEADER_COUNT}</li>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreConnectionPNames#MIN_CHUNK_LIMIT}</li>
 * </ul>
 *
 * @since 4.0
 */
@NotThreadSafe
public class DefaultHttpServerConnection extends SocketHttpServerConnection {

    public DefaultHttpServerConnection() {
        super();
    }

    @Override
    public void bind(final Socket socket, final HttpParams params) throws IOException {
        if (socket == null) {
            throw new IllegalArgumentException("Socket may not be null");
        }
        if (params == null) {
            throw new IllegalArgumentException("HTTP parameters may not be null");
        }
        assertNotOpen();
        socket.setTcpNoDelay(HttpConnectionParams.getTcpNoDelay(params));
        socket.setSoTimeout(HttpConnectionParams.getSoTimeout(params));
        socket.setKeepAlive(HttpConnectionParams.getSoKeepalive(params));

        int linger = HttpConnectionParams.getLinger(params);
        if (linger >= 0) {
            socket.setSoLinger(linger > 0, linger);
        }
        super.bind(socket, params);
    }

}
