/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package io.prestosql.hadoop.$internal.org.apache.http.impl.client;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Queue;

import io.prestosql.hadoop.$internal.org.apache.commons.logging.Log;
import io.prestosql.hadoop.$internal.org.apache.commons.logging.LogFactory;
import io.prestosql.hadoop.$internal.org.apache.http.FormattedHeader;
import io.prestosql.hadoop.$internal.org.apache.http.Header;
import io.prestosql.hadoop.$internal.org.apache.http.HttpHost;
import io.prestosql.hadoop.$internal.org.apache.http.HttpResponse;
import io.prestosql.hadoop.$internal.org.apache.http.annotation.Immutable;
import io.prestosql.hadoop.$internal.org.apache.http.auth.AuthOption;
import io.prestosql.hadoop.$internal.org.apache.http.auth.AuthScheme;
import io.prestosql.hadoop.$internal.org.apache.http.auth.AuthSchemeRegistry;
import io.prestosql.hadoop.$internal.org.apache.http.auth.AuthScope;
import io.prestosql.hadoop.$internal.org.apache.http.auth.Credentials;
import io.prestosql.hadoop.$internal.org.apache.http.auth.MalformedChallengeException;
import io.prestosql.hadoop.$internal.org.apache.http.client.AuthCache;
import io.prestosql.hadoop.$internal.org.apache.http.client.AuthenticationStrategy;
import io.prestosql.hadoop.$internal.org.apache.http.client.CredentialsProvider;
import io.prestosql.hadoop.$internal.org.apache.http.client.params.AuthPolicy;
import io.prestosql.hadoop.$internal.org.apache.http.client.protocol.ClientContext;
import io.prestosql.hadoop.$internal.org.apache.http.protocol.HTTP;
import io.prestosql.hadoop.$internal.org.apache.http.protocol.HttpContext;
import io.prestosql.hadoop.$internal.org.apache.http.util.CharArrayBuffer;

@Immutable
class AuthenticationStrategyImpl implements AuthenticationStrategy {

    private final Log log = LogFactory.getLog(getClass());

    private static final List<String> DEFAULT_SCHEME_PRIORITY =
        Collections.unmodifiableList(Arrays.asList(new String[] {
                AuthPolicy.SPNEGO,
                AuthPolicy.KERBEROS,
                AuthPolicy.NTLM,
                AuthPolicy.DIGEST,
                AuthPolicy.BASIC
    }));

    private final int challengeCode;
    private final String headerName;
    private final String prefParamName;

    AuthenticationStrategyImpl(int challengeCode, final String headerName, final String prefParamName) {
        super();
        this.challengeCode = challengeCode;
        this.headerName = headerName;
        this.prefParamName = prefParamName;
    }

    public boolean isAuthenticationRequested(
            final HttpHost authhost,
            final HttpResponse response,
            final HttpContext context) {
        if (response == null) {
            throw new IllegalArgumentException("HTTP response may not be null");
        }
        int status = response.getStatusLine().getStatusCode();
        return status == this.challengeCode;
    }

    public Map<String, Header> getChallenges(
            final HttpHost authhost,
            final HttpResponse response,
            final HttpContext context) throws MalformedChallengeException {
        if (response == null) {
            throw new IllegalArgumentException("HTTP response may not be null");
        }
        Header[] headers = response.getHeaders(this.headerName);
        Map<String, Header> map = new HashMap<String, Header>(headers.length);
        for (Header header : headers) {
            CharArrayBuffer buffer;
            int pos;
            if (header instanceof FormattedHeader) {
                buffer = ((FormattedHeader) header).getBuffer();
                pos = ((FormattedHeader) header).getValuePos();
            } else {
                String s = header.getValue();
                if (s == null) {
                    throw new MalformedChallengeException("Header value is null");
                }
                buffer = new CharArrayBuffer(s.length());
                buffer.append(s);
                pos = 0;
            }
            while (pos < buffer.length() && HTTP.isWhitespace(buffer.charAt(pos))) {
                pos++;
            }
            int beginIndex = pos;
            while (pos < buffer.length() && !HTTP.isWhitespace(buffer.charAt(pos))) {
                pos++;
            }
            int endIndex = pos;
            String s = buffer.substring(beginIndex, endIndex);
            map.put(s.toLowerCase(Locale.US), header);
        }
        return map;
    }

    public Queue<AuthOption> select(
            final Map<String, Header> challenges,
            final HttpHost authhost,
            final HttpResponse response,
            final HttpContext context) throws MalformedChallengeException {
        if (challenges == null) {
            throw new IllegalArgumentException("Map of auth challenges may not be null");
        }
        if (authhost == null) {
            throw new IllegalArgumentException("Host may not be null");
        }
        if (response == null) {
            throw new IllegalArgumentException("HTTP response may not be null");
        }
        if (context == null) {
            throw new IllegalArgumentException("HTTP context may not be null");
        }

        Queue<AuthOption> options = new LinkedList<AuthOption>();
        AuthSchemeRegistry registry = (AuthSchemeRegistry) context.getAttribute(
                ClientContext.AUTHSCHEME_REGISTRY);
        if (registry == null) {
            this.log.debug("Auth scheme registry not set in the context");
            return options;
        }
        CredentialsProvider credsProvider = (CredentialsProvider) context.getAttribute(
                ClientContext.CREDS_PROVIDER);
        if (credsProvider == null) {
            this.log.debug("Credentials provider not set in the context");
            return options;
        }

        @SuppressWarnings("unchecked")
        List<String> authPrefs = (List<String>) response.getParams().getParameter(this.prefParamName);
        if (authPrefs == null) {
            authPrefs = DEFAULT_SCHEME_PRIORITY;
        }
        if (this.log.isDebugEnabled()) {
            this.log.debug("Authentication schemes in the order of preference: " + authPrefs);
        }

        for (String id: authPrefs) {
            Header challenge = challenges.get(id.toLowerCase(Locale.US));
            if (challenge != null) {
                try {
                    AuthScheme authScheme = registry.getAuthScheme(id, response.getParams());
                    authScheme.processChallenge(challenge);

                    AuthScope authScope = new AuthScope(
                            authhost.getHostName(),
                            authhost.getPort(),
                            authScheme.getRealm(),
                            authScheme.getSchemeName());

                    Credentials credentials = credsProvider.getCredentials(authScope);
                    if (credentials != null) {
                        options.add(new AuthOption(authScheme, credentials));
                    }
                } catch (IllegalStateException e) {
                    if (this.log.isWarnEnabled()) {
                        this.log.warn("Authentication scheme " + id + " not supported");
                        // Try again
                    }
                }
            } else {
                if (this.log.isDebugEnabled()) {
                    this.log.debug("Challenge for " + id + " authentication scheme not available");
                    // Try again
                }
            }
        }
        return options;
    }

    public void authSucceeded(
            final HttpHost authhost, final AuthScheme authScheme, final HttpContext context) {
        if (authhost == null) {
            throw new IllegalArgumentException("Host may not be null");
        }
        if (authScheme == null) {
            throw new IllegalArgumentException("Auth scheme may not be null");
        }
        if (context == null) {
            throw new IllegalArgumentException("HTTP context may not be null");
        }
        if (isCachable(authScheme)) {
            AuthCache authCache = (AuthCache) context.getAttribute(ClientContext.AUTH_CACHE);
            if (authCache == null) {
                authCache = new BasicAuthCache();
                context.setAttribute(ClientContext.AUTH_CACHE, authCache);
            }
            if (this.log.isDebugEnabled()) {
                this.log.debug("Caching '" + authScheme.getSchemeName() +
                        "' auth scheme for " + authhost);
            }
            authCache.put(authhost, authScheme);
        }
    }

    protected boolean isCachable(final AuthScheme authScheme) {
        if (authScheme == null || !authScheme.isComplete()) {
            return false;
        }
        String schemeName = authScheme.getSchemeName();
        return schemeName.equalsIgnoreCase(AuthPolicy.BASIC) ||
                schemeName.equalsIgnoreCase(AuthPolicy.DIGEST);
    }

    public void authFailed(
            final HttpHost authhost, final AuthScheme authScheme, final HttpContext context) {
        if (authhost == null) {
            throw new IllegalArgumentException("Host may not be null");
        }
        if (context == null) {
            throw new IllegalArgumentException("HTTP context may not be null");
        }
        AuthCache authCache = (AuthCache) context.getAttribute(ClientContext.AUTH_CACHE);
        if (authCache != null) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("Clearing cached auth scheme for " + authhost);
            }
            authCache.remove(authhost);
        }
    }

}
