/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package io.prestosql.hadoop.$internal.org.apache.http.protocol;

import java.io.IOException;

import io.prestosql.hadoop.$internal.org.apache.http.HttpException;
import io.prestosql.hadoop.$internal.org.apache.http.HttpResponse;
import io.prestosql.hadoop.$internal.org.apache.http.HttpResponseInterceptor;
import io.prestosql.hadoop.$internal.org.apache.http.annotation.Immutable;
import io.prestosql.hadoop.$internal.org.apache.http.params.CoreProtocolPNames;

/**
 * ResponseServer is responsible for adding <code>Server</code> header. This
 * interceptor is recommended for server side protocol processors.
 * <p>
 * The following parameters can be used to customize the behavior of this
 * class:
 * <ul>
 *  <li>{@link io.prestosql.hadoop.$internal.org.apache.http.params.CoreProtocolPNames#ORIGIN_SERVER}</li>
 * </ul>
 *
 * @since 4.0
 */
@Immutable
public class ResponseServer implements HttpResponseInterceptor {

    public ResponseServer() {
        super();
    }

    public void process(final HttpResponse response, final HttpContext context)
            throws HttpException, IOException {
        if (response == null) {
            throw new IllegalArgumentException("HTTP request may not be null");
        }
        if (!response.containsHeader(HTTP.SERVER_HEADER)) {
            String s = (String) response.getParams().getParameter(
                    CoreProtocolPNames.ORIGIN_SERVER);
            if (s != null) {
                response.addHeader(HTTP.SERVER_HEADER, s);
            }
        }
    }

}
