/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode.ha;

import io.prestosql.hadoop.$internal.com.google.common.annotations.VisibleForTesting;
import io.prestosql.hadoop.$internal.com.google.common.base.Preconditions;
import io.prestosql.hadoop.$internal.com.google.common.collect.Iterators;
import io.prestosql.hadoop.$internal.com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.prestosql.hadoop.$internal.org.slf4j.Logger;
import io.prestosql.hadoop.$internal.org.slf4j.LoggerFactory;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.PrivilegedAction;
import java.security.PrivilegedExceptionAction;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocolPB.NamenodeProtocolPB;
import org.apache.hadoop.hdfs.protocolPB.NamenodeProtocolTranslatorPB;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputException;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSImage;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.ha.RemoteNameNodeInfo;
import org.apache.hadoop.hdfs.server.protocol.NamenodeProtocol;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class EditLogTailer {
    public static final Logger LOG = LoggerFactory.getLogger(EditLogTailer.class);
    public static final String DFS_HA_TAILEDITS_MAX_TXNS_PER_LOCK_KEY = "dfs.ha.tail-edits.max-txns-per-lock";
    public static final long DFS_HA_TAILEDITS_MAX_TXNS_PER_LOCK_DEFAULT = Long.MAX_VALUE;
    private final EditLogTailerThread tailerThread = new EditLogTailerThread();
    private final Configuration conf;
    private final FSNamesystem namesystem;
    private final Iterator<RemoteNameNodeInfo> nnLookup;
    private FSEditLog editLog;
    private RemoteNameNodeInfo currentNN;
    private long lastRollTriggerTxId = -12345L;
    private long lastLoadedTxnId = -12345L;
    private long lastLoadTimeMs;
    private final long logRollPeriodMs;
    private final long rollEditsTimeoutMs;
    private final ExecutorService rollEditsRpcExecutor;
    private final long sleepTimeMs;
    private final int nnCount;
    private NamenodeProtocol cachedActiveProxy = null;
    private int nnLoopCount = 0;
    private int maxRetries;
    private final boolean inProgressOk;
    private final long maxTxnsPerLock;

    public EditLogTailer(FSNamesystem namesystem, Configuration conf) {
        this.conf = conf;
        this.namesystem = namesystem;
        this.editLog = namesystem.getEditLog();
        this.lastLoadTimeMs = Time.monotonicNow();
        this.logRollPeriodMs = conf.getTimeDuration("dfs.ha.log-roll.period", 120L, TimeUnit.SECONDS) * 1000L;
        List<Object> nns = Collections.emptyList();
        if (this.logRollPeriodMs >= 0L) {
            try {
                nns = RemoteNameNodeInfo.getRemoteNameNodes(conf);
            }
            catch (IOException e) {
                throw new IllegalArgumentException("Remote NameNodes not correctly configured!", e);
            }
            for (RemoteNameNodeInfo remoteNameNodeInfo : nns) {
                InetSocketAddress ipc = NameNode.getServiceAddress(remoteNameNodeInfo.getConfiguration(), true);
                Preconditions.checkArgument(ipc.getPort() > 0, "Active NameNode must have an IPC port configured. Got address '%s'", ipc);
                remoteNameNodeInfo.setIpcAddress(ipc);
            }
            LOG.info("Will roll logs on active node every " + this.logRollPeriodMs / 1000L + " seconds.");
        } else {
            LOG.info("Not going to trigger log rolls on active node because dfs.ha.log-roll.period is negative.");
        }
        this.sleepTimeMs = conf.getTimeDuration("dfs.ha.tail-edits.period", 60L, TimeUnit.SECONDS) * 1000L;
        this.rollEditsTimeoutMs = conf.getInt("dfs.ha.tail-edits.rolledits.timeout", 60) * 1000;
        this.rollEditsRpcExecutor = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setDaemon(true).build());
        this.maxRetries = conf.getInt("dfs.ha.tail-edits.namenode-retries", 3);
        if (this.maxRetries <= 0) {
            LOG.error("Specified a non-positive number of retries for the number of retries for the namenode connection when manipulating the edit log (dfs.ha.tail-edits.namenode-retries), setting to default: 3");
            this.maxRetries = 3;
        }
        this.inProgressOk = conf.getBoolean("dfs.ha.tail-edits.in-progress", false);
        this.maxTxnsPerLock = conf.getLong(DFS_HA_TAILEDITS_MAX_TXNS_PER_LOCK_KEY, Long.MAX_VALUE);
        this.nnCount = nns.size();
        this.nnLookup = Iterators.cycle(nns);
        LOG.debug("logRollPeriodMs=" + this.logRollPeriodMs + " sleepTime=" + this.sleepTimeMs);
    }

    public void start() {
        this.tailerThread.start();
    }

    public void stop() throws IOException {
        this.tailerThread.setShouldRun(false);
        this.tailerThread.interrupt();
        try {
            this.tailerThread.join();
        }
        catch (InterruptedException e) {
            LOG.warn("Edit log tailer thread exited with an exception");
            throw new IOException(e);
        }
        finally {
            this.rollEditsRpcExecutor.shutdown();
        }
    }

    @VisibleForTesting
    FSEditLog getEditLog() {
        return this.editLog;
    }

    @VisibleForTesting
    public void setEditLog(FSEditLog editLog) {
        this.editLog = editLog;
    }

    public void catchupDuringFailover() throws IOException {
        Preconditions.checkState(this.tailerThread == null || !this.tailerThread.isAlive(), "Tailer thread should not be running once failover starts");
        SecurityUtil.doAsLoginUser(new PrivilegedExceptionAction<Void>(){

            @Override
            public Void run() throws Exception {
                try {
                    EditLogTailer.this.doTailEdits();
                }
                catch (InterruptedException e) {
                    throw new IOException(e);
                }
                return null;
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @VisibleForTesting
    void doTailEdits() throws IOException, InterruptedException {
        long editsLoaded;
        FSImage image;
        block18: {
            Collection<EditLogInputStream> streams;
            this.namesystem.writeLockInterruptibly();
            image = this.namesystem.getFSImage();
            long lastTxnId = image.getLastAppliedTxId();
            if (LOG.isDebugEnabled()) {
                LOG.debug("lastTxnId: " + lastTxnId);
            }
            long startTime = Time.monotonicNow();
            try {
                streams = this.editLog.selectInputStreams(lastTxnId + 1L, 0L, null, this.inProgressOk, true);
            }
            catch (IOException ioe) {
                LOG.warn("Edits tailer failed to find any streams. Will try again later.", ioe);
                this.namesystem.writeUnlock();
                return;
            }
            finally {
                NameNode.getNameNodeMetrics().addEditLogFetchTime(Time.monotonicNow() - startTime);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("edit streams to load from: " + streams.size());
            }
            editsLoaded = 0L;
            try {
                editsLoaded = image.loadEdits(streams, this.namesystem, this.maxTxnsPerLock, null, null);
                if (editsLoaded <= 0L && !LOG.isDebugEnabled()) break block18;
            }
            catch (EditLogInputException elie) {
                try {
                    editsLoaded = elie.getNumEditsLoaded();
                    throw elie;
                }
                catch (Throwable throwable) {
                    if (editsLoaded > 0L || LOG.isDebugEnabled()) {
                        LOG.debug(String.format("Loaded %d edits starting from txid %d ", editsLoaded, lastTxnId));
                    }
                    NameNode.getNameNodeMetrics().addNumEditLogLoaded(editsLoaded);
                    throw throwable;
                }
            }
            LOG.debug(String.format("Loaded %d edits starting from txid %d ", editsLoaded, lastTxnId));
        }
        NameNode.getNameNodeMetrics().addNumEditLogLoaded(editsLoaded);
        if (editsLoaded > 0L) {
            this.lastLoadTimeMs = Time.monotonicNow();
        }
        this.lastLoadedTxnId = image.getLastAppliedTxId();
        return;
        finally {
            this.namesystem.writeUnlock();
        }
    }

    public long getLastLoadTimeMs() {
        return this.lastLoadTimeMs;
    }

    private boolean tooLongSinceLastLoad() {
        return this.logRollPeriodMs >= 0L && Time.monotonicNow() - this.lastLoadTimeMs > this.logRollPeriodMs;
    }

    @VisibleForTesting
    Callable<Void> getNameNodeProxy() {
        return new MultipleNameNodeProxy<Void>(){

            @Override
            protected Void doWork() throws IOException {
                EditLogTailer.this.cachedActiveProxy.rollEditLog();
                return null;
            }
        };
    }

    @VisibleForTesting
    void triggerActiveLogRoll() {
        LOG.info("Triggering log roll on remote NameNode");
        Future<Void> future = null;
        try {
            future = this.rollEditsRpcExecutor.submit(this.getNameNodeProxy());
            future.get(this.rollEditsTimeoutMs, TimeUnit.MILLISECONDS);
            this.lastRollTriggerTxId = this.lastLoadedTxnId;
        }
        catch (ExecutionException e) {
            LOG.warn("Unable to trigger a roll of the active NN", e);
        }
        catch (TimeoutException e) {
            if (future != null) {
                future.cancel(true);
            }
            LOG.warn(String.format("Unable to finish rolling edits in %d ms", this.rollEditsTimeoutMs));
        }
        catch (InterruptedException e) {
            LOG.warn("Unable to trigger a roll of the active NN", e);
        }
    }

    @VisibleForTesting
    abstract class MultipleNameNodeProxy<T>
    implements Callable<T> {
        MultipleNameNodeProxy() {
        }

        protected abstract T doWork() throws IOException;

        @Override
        public T call() throws IOException {
            EditLogTailer.this.nnLoopCount = 0;
            while ((EditLogTailer.this.cachedActiveProxy = this.getActiveNodeProxy()) != null) {
                try {
                    T ret = this.doWork();
                    return ret;
                }
                catch (IOException e) {
                    LOG.warn("Exception from remote name node " + EditLogTailer.this.currentNN + ", try next.", e);
                    EditLogTailer.this.cachedActiveProxy = null;
                    EditLogTailer.this.nnLoopCount++;
                }
            }
            throw new IOException("Cannot find any valid remote NN to service request!");
        }

        private NamenodeProtocol getActiveNodeProxy() throws IOException {
            if (EditLogTailer.this.cachedActiveProxy == null) {
                while (true) {
                    if (EditLogTailer.this.nnLoopCount / EditLogTailer.this.nnCount >= EditLogTailer.this.maxRetries) {
                        return null;
                    }
                    EditLogTailer.this.currentNN = (RemoteNameNodeInfo)EditLogTailer.this.nnLookup.next();
                    try {
                        int rpcTimeout = EditLogTailer.this.conf.getInt("dfs.ha.log-roll.rpc.timeout", 20000);
                        NamenodeProtocolPB proxy = RPC.waitForProxy(NamenodeProtocolPB.class, RPC.getProtocolVersion(NamenodeProtocolPB.class), EditLogTailer.this.currentNN.getIpcAddress(), EditLogTailer.this.conf, rpcTimeout, Long.MAX_VALUE);
                        EditLogTailer.this.cachedActiveProxy = new NamenodeProtocolTranslatorPB(proxy);
                    }
                    catch (IOException e) {
                        LOG.info("Failed to reach " + EditLogTailer.this.currentNN, e);
                        EditLogTailer.this.nnLoopCount++;
                        continue;
                    }
                    break;
                }
            }
            assert (EditLogTailer.this.cachedActiveProxy != null);
            return EditLogTailer.this.cachedActiveProxy;
        }
    }

    private class EditLogTailerThread
    extends Thread {
        private volatile boolean shouldRun;

        private EditLogTailerThread() {
            super("Edit log tailer");
            this.shouldRun = true;
        }

        private void setShouldRun(boolean shouldRun) {
            this.shouldRun = shouldRun;
        }

        @Override
        public void run() {
            SecurityUtil.doAsLoginUserOrFatal(new PrivilegedAction<Object>(){

                @Override
                public Object run() {
                    EditLogTailerThread.this.doWork();
                    return null;
                }
            });
        }

        private void doWork() {
            while (this.shouldRun) {
                try {
                    if (EditLogTailer.this.tooLongSinceLastLoad() && EditLogTailer.this.lastRollTriggerTxId < EditLogTailer.this.lastLoadedTxnId) {
                        EditLogTailer.this.triggerActiveLogRoll();
                    }
                    if (!this.shouldRun) break;
                    EditLogTailer.this.namesystem.cpLockInterruptibly();
                    long startTime = Time.monotonicNow();
                    try {
                        NameNode.getNameNodeMetrics().addEditLogTailInterval(startTime - EditLogTailer.this.lastLoadTimeMs);
                        EditLogTailer.this.doTailEdits();
                    }
                    finally {
                        EditLogTailer.this.namesystem.cpUnlock();
                        NameNode.getNameNodeMetrics().addEditLogTailTime(Time.monotonicNow() - startTime);
                    }
                    EditLogTailer.this.namesystem.getFSImage().getStorage().updateNameDirSize();
                }
                catch (EditLogInputException elie) {
                    LOG.warn("Error while reading edits from disk. Will try again.", elie);
                }
                catch (InterruptedException ie) {
                    continue;
                }
                catch (Throwable t) {
                    LOG.error("Unknown error encountered while tailing edits. Shutting down standby NN.", t);
                    ExitUtil.terminate(1, t);
                }
                try {
                    Thread.sleep(EditLogTailer.this.sleepTimeMs);
                }
                catch (InterruptedException e) {
                    LOG.warn("Edit log tailer interrupted", e);
                }
            }
        }
    }
}

