/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.cache.results;

import io.prestosql.hive.$internal.com.google.common.annotations.VisibleForTesting;
import io.prestosql.hive.$internal.com.google.common.base.Preconditions;
import io.prestosql.hive.$internal.com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.prestosql.hive.$internal.org.slf4j.Logger;
import io.prestosql.hive.$internal.org.slf4j.LoggerFactory;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Supplier;
import java.util.stream.Stream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hive.common.ValidTxnWriteIdList;
import org.apache.hadoop.hive.common.ValidWriteIdList;
import org.apache.hadoop.hive.common.metrics.common.Metrics;
import org.apache.hadoop.hive.common.metrics.common.MetricsFactory;
import org.apache.hadoop.hive.common.metrics.common.MetricsVariable;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.hadoop.hive.metastore.api.NotificationEvent;
import org.apache.hadoop.hive.ql.exec.Utilities;
import org.apache.hadoop.hive.ql.hooks.Entity;
import org.apache.hadoop.hive.ql.hooks.ReadEntity;
import org.apache.hadoop.hive.ql.io.AcidUtils;
import org.apache.hadoop.hive.ql.metadata.SessionHiveMetaStoreClient;
import org.apache.hadoop.hive.ql.metadata.Table;
import org.apache.hadoop.hive.ql.metadata.events.EventConsumer;
import org.apache.hadoop.hive.ql.parse.ColumnAccessInfo;
import org.apache.hadoop.hive.ql.parse.TableAccessInfo;
import org.apache.hadoop.hive.ql.plan.FetchWork;
import org.apache.hadoop.hive.ql.plan.HiveOperation;
import org.apache.hive.common.util.TxnIdUtils;

public final class QueryResultsCache {
    private static final Logger LOG = LoggerFactory.getLogger(QueryResultsCache.class);
    private final Map<String, Set<CacheEntry>> queryMap = new HashMap<String, Set<CacheEntry>>();
    private final Map<CacheEntry, CacheEntry> lru = Collections.synchronizedMap(new LinkedHashMap(16, 0.75f, true));
    private final Map<String, Set<CacheEntry>> tableToEntryMap = new HashMap<String, Set<CacheEntry>>();
    private final HiveConf conf;
    private Path cacheDirPath;
    private Path zeroRowsPath;
    private long cacheSize = 0L;
    private long maxCacheSize;
    private long maxEntrySize;
    private long maxEntryLifetime;
    private ReadWriteLock rwLock = new ReentrantReadWriteLock();
    private ScheduledFuture<?> invalidationPollFuture;
    private static final AtomicBoolean inited = new AtomicBoolean(false);
    private static QueryResultsCache instance;
    private static final int INITIAL_LRU_SIZE = 16;
    private static final float LRU_LOAD_FACTOR = 0.75f;
    private static final CacheEntry[] EMPTY_CACHEENTRY_ARRAY;
    private static ScheduledExecutorService invalidationExecutor;
    private static ExecutorService deletionExecutor;

    private QueryResultsCache(HiveConf configuration) throws IOException {
        this.conf = configuration;
        Path rootCacheDir = new Path(this.conf.getVar(HiveConf.ConfVars.HIVE_QUERY_RESULTS_CACHE_DIRECTORY));
        LOG.info("Initializing query results cache at {}", (Object)rootCacheDir);
        Utilities.ensurePathIsWritable(rootCacheDir, this.conf);
        String currentCacheDirName = "results-" + UUID.randomUUID().toString();
        this.cacheDirPath = new Path(rootCacheDir, currentCacheDirName);
        FileSystem fs = this.cacheDirPath.getFileSystem((Configuration)this.conf);
        FsPermission fsPermission = new FsPermission("700");
        fs.mkdirs(this.cacheDirPath, fsPermission);
        this.zeroRowsPath = new Path(this.cacheDirPath, "dummy_zero_rows");
        fs.deleteOnExit(this.cacheDirPath);
        this.maxCacheSize = this.conf.getLongVar(HiveConf.ConfVars.HIVE_QUERY_RESULTS_CACHE_MAX_SIZE);
        this.maxEntrySize = this.conf.getLongVar(HiveConf.ConfVars.HIVE_QUERY_RESULTS_CACHE_MAX_ENTRY_SIZE);
        this.maxEntryLifetime = this.conf.getTimeVar(HiveConf.ConfVars.HIVE_QUERY_RESULTS_CACHE_MAX_ENTRY_LIFETIME, TimeUnit.MILLISECONDS);
        LOG.info("Query results cache: cacheDirectory {}, maxCacheSize {}, maxEntrySize {}, maxEntryLifetime {}", this.cacheDirPath, this.maxCacheSize, this.maxEntrySize, this.maxEntryLifetime);
    }

    public static void initialize(HiveConf conf) throws IOException {
        if (!inited.getAndSet(true)) {
            try {
                instance = new QueryResultsCache(conf);
                Metrics metrics = MetricsFactory.getInstance();
                if (metrics != null) {
                    QueryResultsCache.registerMetrics(metrics, instance);
                }
            }
            catch (IOException err) {
                inited.set(false);
                throw err;
            }
        }
    }

    public static QueryResultsCache getInstance() {
        return instance;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CacheEntry lookup(LookupInfo request) {
        CacheEntry result = null;
        LOG.debug("QueryResultsCache lookup for query: {}", (Object)request.queryText);
        boolean foundPending = false;
        HashSet<CacheEntry> entriesToRemove = new HashSet<CacheEntry>();
        Lock readLock = this.rwLock.readLock();
        try {
            readLock.lock();
            Set<CacheEntry> candidates = this.queryMap.get(request.queryText);
            if (candidates != null) {
                CacheEntry pendingResult = null;
                for (CacheEntry candidate : candidates) {
                    if (!this.entryMatches(request, candidate, entriesToRemove)) continue;
                    CacheEntryStatus entryStatus = candidate.status;
                    if (entryStatus == CacheEntryStatus.VALID) {
                        result = candidate;
                        break;
                    }
                    if (entryStatus != CacheEntryStatus.PENDING || pendingResult != null) continue;
                    pendingResult = candidate;
                }
                if (result == null && pendingResult != null) {
                    result = pendingResult;
                    foundPending = true;
                }
                if (result != null) {
                    this.lru.get(result);
                }
            }
        }
        finally {
            readLock.unlock();
        }
        for (CacheEntry invalidEntry : entriesToRemove) {
            this.removeEntry(invalidEntry);
        }
        LOG.debug("QueryResultsCache lookup result: {}", (Object)result);
        QueryResultsCache.incrementMetric("qc_lookups");
        if (result != null) {
            if (foundPending) {
                QueryResultsCache.incrementMetric("qc_pending_hits");
            } else {
                QueryResultsCache.incrementMetric("qc_valid_hits");
            }
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CacheEntry addToCache(QueryInfo queryInfo, ValidTxnWriteIdList txnWriteIdList) {
        String queryText = queryInfo.getLookupInfo().getQueryText();
        CacheEntry addedEntry = new CacheEntry();
        addedEntry.queryInfo = queryInfo;
        addedEntry.txnWriteIdList = txnWriteIdList;
        Lock writeLock = this.rwLock.writeLock();
        try {
            writeLock.lock();
            LOG.info("Adding placeholder cache entry for query '{}'", (Object)queryText);
            QueryResultsCache.addToEntryMap(this.queryMap, queryText, addedEntry);
            this.lru.put(addedEntry, addedEntry);
            addedEntry.getTableNames().forEach(tableName -> QueryResultsCache.addToEntryMap(this.tableToEntryMap, tableName, addedEntry));
        }
        finally {
            writeLock.unlock();
        }
        return addedEntry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean setEntryValid(CacheEntry cacheEntry, FetchWork fetchWork) {
        String queryText = cacheEntry.getQueryText();
        boolean dataDirMoved = false;
        Path queryResultsPath = null;
        Path cachedResultsPath = null;
        try {
            long resultSize;
            boolean requiresMove = true;
            queryResultsPath = fetchWork.getTblDir();
            FileSystem resultsFs = queryResultsPath.getFileSystem((Configuration)this.conf);
            if (resultsFs.exists(queryResultsPath)) {
                ContentSummary cs = resultsFs.getContentSummary(queryResultsPath);
                resultSize = cs.getLength();
            } else {
                cachedResultsPath = this.zeroRowsPath;
                FileSystem cacheFs = cachedResultsPath.getFileSystem((Configuration)this.conf);
                boolean fakePathExists = cacheFs.exists(this.zeroRowsPath);
                resultSize = 0L;
                requiresMove = false;
            }
            if (!this.shouldEntryBeAdded(cacheEntry, resultSize)) {
                return false;
            }
            CacheEntry cacheEntry2 = cacheEntry;
            synchronized (cacheEntry2) {
                if (cacheEntry.getStatus() == CacheEntryStatus.INVALID) {
                    return false;
                }
                if (requiresMove) {
                    cachedResultsPath = this.moveResultsToCacheDirectory(queryResultsPath);
                    dataDirMoved = true;
                }
                LOG.info("Moved query results from {} to {} (size {}) for query '{}'", queryResultsPath, cachedResultsPath, resultSize, queryText);
                FetchWork fetchWorkForCache = new FetchWork(cachedResultsPath, fetchWork.getTblDesc(), fetchWork.getLimit());
                fetchWorkForCache.setCachedResult(true);
                cacheEntry.fetchWork = fetchWorkForCache;
                cacheEntry.cachedResultsPath = cachedResultsPath;
                cacheEntry.size = resultSize;
                this.cacheSize += resultSize;
                cacheEntry.setStatus(CacheEntryStatus.VALID);
                cacheEntry.addReader();
                this.scheduleEntryInvalidation(cacheEntry);
                cacheEntry.notifyAll();
            }
            QueryResultsCache.incrementMetric("qc_valid_entries");
            QueryResultsCache.incrementMetric("qc_total_entries_added");
        }
        catch (Exception err) {
            LOG.error("Failed to create cache entry for query results for query: " + queryText, err);
            if (dataDirMoved) {
                LOG.info("Restoring query results from {} back to {}", (Object)cachedResultsPath, (Object)queryResultsPath);
                try {
                    FileSystem fs = cachedResultsPath.getFileSystem((Configuration)this.conf);
                    fs.rename(cachedResultsPath, queryResultsPath);
                    cacheEntry.size = 0L;
                    cacheEntry.cachedResultsPath = null;
                }
                catch (Exception err2) {
                    String errMsg = "Failed cleanup during failed attempt to cache query: " + queryText;
                    LOG.error(errMsg);
                    throw new RuntimeException(errMsg);
                }
            }
            cacheEntry.invalidate();
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clear() {
        Lock writeLock = this.rwLock.writeLock();
        try {
            writeLock.lock();
            LOG.info("Clearing the results cache");
            CacheEntry[] allEntries = null;
            CacheEntry[] cacheEntryArray = this.lru;
            synchronized (this.lru) {
                allEntries = this.lru.keySet().toArray(EMPTY_CACHEENTRY_ARRAY);
                // ** MonitorExit[var3_3] (shouldn't be in output)
                for (CacheEntry entry : allEntries) {
                    try {
                        this.removeEntry(entry);
                    }
                    catch (Exception err) {
                        LOG.error("Error removing cache entry " + entry, err);
                    }
                }
            }
        }
        finally {
            writeLock.unlock();
        }
        {
            return;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getSize() {
        Lock readLock = this.rwLock.readLock();
        try {
            readLock.lock();
            long l = this.cacheSize;
            return l;
        }
        finally {
            readLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void notifyTableChanged(String dbName, String tableName, long updateTime) {
        LOG.debug("Table changed: {}.{}, at {}", dbName, tableName, updateTime);
        ArrayList<CacheEntry> entriesToInvalidate = null;
        this.rwLock.writeLock().lock();
        try {
            String key = dbName.toLowerCase() + "." + tableName.toLowerCase();
            Set<CacheEntry> entriesForTable = this.tableToEntryMap.get(key);
            if (entriesForTable != null) {
                entriesToInvalidate = new ArrayList<CacheEntry>(entriesForTable);
            }
            if (entriesToInvalidate != null) {
                for (CacheEntry entry : entriesToInvalidate) {
                    if (entry.getQueryInfo().getQueryTime() > updateTime) continue;
                    this.removeEntry(entry);
                }
            }
        }
        finally {
            this.rwLock.writeLock().unlock();
        }
    }

    private boolean entryMatches(LookupInfo lookupInfo, CacheEntry entry, Set<CacheEntry> entriesToRemove) {
        QueryInfo queryInfo = entry.getQueryInfo();
        for (ReadEntity readEntity : queryInfo.getInputs()) {
            if (readEntity.getType() != Entity.Type.TABLE) continue;
            Table tableUsed = readEntity.getTable();
            Map<String, Table> tempTables = SessionHiveMetaStoreClient.getTempTablesForDatabase(tableUsed.getDbName(), tableUsed.getTableName());
            if (tempTables != null && tempTables.containsKey(tableUsed.getTableName())) {
                LOG.info("{} resolves to a temporary table in the current session. This query cannot use the cache.", (Object)tableUsed.getTableName());
                return false;
            }
            if (!AcidUtils.isTransactionalTable(tableUsed)) continue;
            boolean writeIdCheckPassed = false;
            String tableName = tableUsed.getFullyQualifiedName();
            ValidTxnWriteIdList currentTxnWriteIdList = (ValidTxnWriteIdList)lookupInfo.txnWriteIdListProvider.get();
            if (currentTxnWriteIdList == null) {
                LOG.warn("Current query's txnWriteIdList is null!");
                return false;
            }
            if (entry.txnWriteIdList == null) {
                LOG.warn("Cache entry's txnWriteIdList is null!");
                return false;
            }
            ValidWriteIdList currentWriteIdForTable = currentTxnWriteIdList.getTableValidWriteIdList(tableName);
            ValidWriteIdList cachedWriteIdForTable = entry.txnWriteIdList.getTableValidWriteIdList(tableName);
            LOG.debug("Checking writeIds for table {}: currentWriteIdForTable {}, cachedWriteIdForTable {}", tableName, currentWriteIdForTable, cachedWriteIdForTable);
            if (currentWriteIdForTable != null && cachedWriteIdForTable != null && TxnIdUtils.checkEquivalentWriteIds(currentWriteIdForTable, cachedWriteIdForTable)) {
                writeIdCheckPassed = true;
            }
            if (writeIdCheckPassed) continue;
            LOG.debug("Cached query no longer valid due to table {}", (Object)tableUsed.getFullyQualifiedName());
            entriesToRemove.add(entry);
            entry.invalidate();
            return false;
        }
        return true;
    }

    public void removeEntry(CacheEntry entry) {
        entry.invalidate();
        this.rwLock.writeLock().lock();
        try {
            this.removeFromLookup(entry);
            this.lru.remove(entry);
            this.cacheSize -= entry.size;
        }
        finally {
            this.rwLock.writeLock().unlock();
        }
    }

    private void removeFromLookup(CacheEntry entry) {
        String queryString = entry.getQueryText();
        if (!QueryResultsCache.removeFromEntryMap(this.queryMap, queryString, entry)) {
            LOG.warn("Attempted to remove entry but it was not in the cache: {}", (Object)entry);
        }
        entry.getTableNames().forEach(tableName -> QueryResultsCache.removeFromEntryMap(this.tableToEntryMap, tableName, entry));
    }

    private void calculateEntrySize(CacheEntry entry, FetchWork fetchWork) throws IOException {
        Path queryResultsPath = fetchWork.getTblDir();
        FileSystem resultsFs = queryResultsPath.getFileSystem((Configuration)this.conf);
        ContentSummary cs = resultsFs.getContentSummary(queryResultsPath);
        entry.size = cs.getLength();
    }

    private boolean shouldEntryBeAdded(CacheEntry entry, long size) {
        if (this.maxEntrySize >= 0L && size > this.maxEntrySize) {
            LOG.debug("Cache entry size {} larger than max entry size ({})", (Object)size, (Object)this.maxEntrySize);
            QueryResultsCache.incrementMetric("qc_rejected_too_large");
            return false;
        }
        return this.clearSpaceForCacheEntry(entry, size);
    }

    private Path moveResultsToCacheDirectory(Path queryResultsPath) throws IOException {
        String dirName = UUID.randomUUID().toString();
        Path cachedResultsPath = new Path(this.cacheDirPath, dirName);
        FileSystem fs = cachedResultsPath.getFileSystem((Configuration)this.conf);
        fs.rename(queryResultsPath, cachedResultsPath);
        return cachedResultsPath;
    }

    private boolean hasSpaceForCacheEntry(CacheEntry entry, long size) {
        if (this.maxCacheSize >= 0L) {
            return this.cacheSize + size <= this.maxCacheSize;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CacheEntry findEntryToRemove() {
        Set<CacheEntry> entries = this.lru.keySet();
        Map<CacheEntry, CacheEntry> map = this.lru;
        synchronized (map) {
            for (CacheEntry removalCandidate : entries) {
                if (removalCandidate.getStatus() != CacheEntryStatus.VALID) continue;
                return removalCandidate;
            }
        }
        return null;
    }

    private boolean clearSpaceForCacheEntry(CacheEntry entry, long size) {
        CacheEntry removalCandidate;
        if (this.hasSpaceForCacheEntry(entry, size)) {
            return true;
        }
        LOG.info("Clearing space for cache entry for query: [{}] with size {}", (Object)entry.getQueryText(), (Object)size);
        while ((removalCandidate = this.findEntryToRemove()) != null) {
            LOG.info("Removing entry: {}", (Object)removalCandidate);
            this.removeEntry(removalCandidate);
            if (!this.hasSpaceForCacheEntry(entry, size)) continue;
            return true;
        }
        LOG.info("Could not free enough space for cache entry for query: [{}] withe size {}", (Object)entry.getQueryText(), (Object)size);
        return false;
    }

    private static void addToEntryMap(Map<String, Set<CacheEntry>> entryMap, String key, CacheEntry entry) {
        Set<CacheEntry> entriesForKey = entryMap.get(key);
        if (entriesForKey == null) {
            entriesForKey = new HashSet<CacheEntry>();
            entryMap.put(key, entriesForKey);
        }
        entriesForKey.add(entry);
    }

    private static boolean removeFromEntryMap(Map<String, Set<CacheEntry>> entryMap, String key, CacheEntry entry) {
        Set<CacheEntry> entries = entryMap.get(key);
        if (entries == null) {
            return false;
        }
        boolean deleted = entries.remove(entry);
        if (!deleted) {
            return false;
        }
        if (entries.isEmpty()) {
            entryMap.remove(key);
        }
        return true;
    }

    @VisibleForTesting
    public static void cleanupInstance() {
        if (inited.get()) {
            if (QueryResultsCache.instance.invalidationPollFuture != null) {
                QueryResultsCache.instance.invalidationPollFuture.cancel(true);
                QueryResultsCache.instance.invalidationPollFuture = null;
            }
            instance.clear();
            instance = null;
            inited.set(false);
        }
    }

    private void scheduleEntryInvalidation(final CacheEntry entry) {
        if (this.maxEntryLifetime >= 0L) {
            ScheduledFuture<?> future = invalidationExecutor.schedule(new Runnable(){

                @Override
                public void run() {
                    QueryResultsCache.this.removeEntry(entry);
                }
            }, this.maxEntryLifetime, TimeUnit.MILLISECONDS);
            entry.invalidationFuture = future;
        }
    }

    private static void cleanupEntry(final CacheEntry entry) {
        Preconditions.checkState(entry.getStatus() == CacheEntryStatus.INVALID);
        HiveConf conf = QueryResultsCache.getInstance().conf;
        if (entry.cachedResultsPath != null && !QueryResultsCache.getInstance().zeroRowsPath.equals((Object)entry.cachedResultsPath)) {
            deletionExecutor.execute(new Runnable(){

                @Override
                public void run() {
                    Path path = entry.cachedResultsPath;
                    LOG.info("Cache directory cleanup: deleting {}", (Object)path);
                    try {
                        FileSystem fs = entry.cachedResultsPath.getFileSystem((Configuration)QueryResultsCache.getInstance().conf);
                        fs.delete(entry.cachedResultsPath, true);
                    }
                    catch (Exception err) {
                        LOG.error("Error while trying to delete " + path, err);
                    }
                }
            });
        }
    }

    public static void incrementMetric(String name, long count) {
        Metrics metrics = MetricsFactory.getInstance();
        if (metrics != null) {
            metrics.incrementCounter(name, count);
        }
    }

    public static void decrementMetric(String name, long count) {
        Metrics metrics = MetricsFactory.getInstance();
        if (metrics != null) {
            metrics.decrementCounter(name, count);
        }
    }

    public static void incrementMetric(String name) {
        QueryResultsCache.incrementMetric(name, 1L);
    }

    public static void decrementMetric(String name) {
        QueryResultsCache.decrementMetric(name, 1L);
    }

    private static void registerMetrics(Metrics metrics, final QueryResultsCache cache) {
        MetricsVariable<Long> maxCacheSize = new MetricsVariable<Long>(){

            @Override
            public Long getValue() {
                return cache.maxCacheSize;
            }
        };
        MetricsVariable<Long> curCacheSize = new MetricsVariable<Long>(){

            @Override
            public Long getValue() {
                return cache.cacheSize;
            }
        };
        metrics.addGauge("qc_max_size", maxCacheSize);
        metrics.addGauge("qc_current_size", curCacheSize);
    }

    static {
        EMPTY_CACHEENTRY_ARRAY = new CacheEntry[0];
        invalidationExecutor = null;
        deletionExecutor = null;
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("QueryResultsCache %d").build();
        invalidationExecutor = Executors.newSingleThreadScheduledExecutor(threadFactory);
        deletionExecutor = Executors.newSingleThreadExecutor(threadFactory);
    }

    public static class InvalidationEventConsumer
    implements EventConsumer {
        Configuration conf;

        public Configuration getConf() {
            return this.conf;
        }

        public void setConf(Configuration conf) {
            this.conf = conf;
        }

        @Override
        public void accept(NotificationEvent event) {
            String tableName;
            String dbName;
            switch (event.getEventType()) {
                case "ADD_PARTITION": 
                case "ALTER_PARTITION": 
                case "DROP_PARTITION": 
                case "ALTER_TABLE": 
                case "DROP_TABLE": 
                case "INSERT": {
                    dbName = event.getDbName();
                    tableName = event.getTableName();
                    break;
                }
                default: {
                    return;
                }
            }
            if (dbName == null || tableName == null) {
                LOG.info("Possibly malformed notification event, missing db or table name: {}", (Object)event);
                return;
            }
            LOG.debug("Handling event {} on table {}.{}", event.getEventType(), dbName, tableName);
            QueryResultsCache cache = QueryResultsCache.getInstance();
            if (cache != null) {
                long eventTime = (long)event.getEventTime() * 1000L;
                cache.notifyTableChanged(dbName, tableName, eventTime);
            } else {
                LOG.debug("Cache not instantiated, skipping event on {}.{}", (Object)dbName, (Object)tableName);
            }
        }
    }

    public static class CacheEntry {
        private QueryInfo queryInfo;
        private FetchWork fetchWork;
        private Path cachedResultsPath;
        private long size;
        private AtomicInteger readers = new AtomicInteger(0);
        private ScheduledFuture<?> invalidationFuture = null;
        private volatile CacheEntryStatus status = CacheEntryStatus.PENDING;
        private ValidTxnWriteIdList txnWriteIdList;

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void releaseReader() {
            int readerCount = 0;
            CacheEntry cacheEntry = this;
            synchronized (cacheEntry) {
                readerCount = this.readers.decrementAndGet();
            }
            LOG.debug("releaseReader: entry: {}, readerCount: {}", (Object)this, (Object)readerCount);
            this.cleanupIfNeeded();
        }

        public String toString() {
            return "CacheEntry query: [" + this.getQueryInfo().getLookupInfo().getQueryText() + "], status: " + (Object)((Object)this.status) + ", location: " + this.cachedResultsPath + ", size: " + this.size;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public boolean addReader() {
            boolean added = false;
            int readerCount = 0;
            CacheEntry cacheEntry = this;
            synchronized (cacheEntry) {
                if (this.status == CacheEntryStatus.VALID) {
                    readerCount = this.readers.incrementAndGet();
                    added = true;
                }
            }
            LOG.debug("addReader: entry: {}, readerCount: {}, added: {}", this, readerCount, added);
            return added;
        }

        private int numReaders() {
            return this.readers.get();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void invalidate() {
            LOG.info("Invalidating cache entry: {}", (Object)this);
            CacheEntryStatus prevStatus = this.setStatus(CacheEntryStatus.INVALID);
            if (prevStatus == CacheEntryStatus.VALID) {
                if (this.invalidationFuture != null) {
                    this.invalidationFuture.cancel(false);
                }
                this.cleanupIfNeeded();
                QueryResultsCache.decrementMetric("qc_valid_entries");
            } else if (prevStatus == CacheEntryStatus.PENDING) {
                CacheEntry cacheEntry = this;
                synchronized (cacheEntry) {
                    this.notifyAll();
                }
                QueryResultsCache.decrementMetric("qc_pending_fails");
            }
        }

        public CacheEntryStatus getStatus() {
            return this.status;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private CacheEntryStatus setStatus(CacheEntryStatus newStatus) {
            CacheEntry cacheEntry = this;
            synchronized (cacheEntry) {
                CacheEntryStatus oldStatus = this.status;
                this.status = newStatus;
                return oldStatus;
            }
        }

        private void cleanupIfNeeded() {
            if (this.status == CacheEntryStatus.INVALID && this.readers.get() <= 0) {
                QueryResultsCache.cleanupEntry(this);
            }
        }

        private String getQueryText() {
            return this.getQueryInfo().getLookupInfo().getQueryText();
        }

        public FetchWork getFetchWork() {
            FetchWork fetch = new FetchWork(this.cachedResultsPath, this.fetchWork.getTblDesc(), this.fetchWork.getLimit());
            fetch.setCachedResult(true);
            return fetch;
        }

        public QueryInfo getQueryInfo() {
            return this.queryInfo;
        }

        public Path getCachedResultsPath() {
            return this.cachedResultsPath;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        public boolean waitForValidStatus() {
            LOG.info("Waiting on pending cacheEntry");
            long timeout = 1000L;
            long startTime = System.nanoTime();
            try {
                while (true) {
                    switch (this.status) {
                        case VALID: {
                            long endTime = System.nanoTime();
                            QueryResultsCache.incrementMetric("qc_pending_success_wait_time", TimeUnit.MILLISECONDS.convert(endTime - startTime, TimeUnit.NANOSECONDS));
                            return true;
                        }
                        case INVALID: {
                            long endTime = System.nanoTime();
                            QueryResultsCache.incrementMetric("qc_pending_fails_wait_time", TimeUnit.MILLISECONDS.convert(endTime - startTime, TimeUnit.NANOSECONDS));
                            return false;
                        }
                    }
                    CacheEntry cacheEntry = this;
                    synchronized (cacheEntry) {
                        this.wait(timeout);
                    }
                }
            }
            catch (InterruptedException err) {
                Thread.currentThread().interrupt();
                return false;
            }
        }

        public Stream<String> getTableNames() {
            return this.queryInfo.getInputs().stream().filter(readEntity -> readEntity.getType() == Entity.Type.TABLE).map(readEntity -> readEntity.getTable().getFullyQualifiedName());
        }
    }

    public static enum CacheEntryStatus {
        VALID,
        INVALID,
        PENDING;

    }

    public static class QueryInfo {
        private long queryTime;
        private LookupInfo lookupInfo;
        private HiveOperation hiveOperation;
        private List<FieldSchema> resultSchema;
        private TableAccessInfo tableAccessInfo;
        private ColumnAccessInfo columnAccessInfo;
        private Set<ReadEntity> inputs;

        public QueryInfo(long queryTime, LookupInfo lookupInfo, HiveOperation hiveOperation, List<FieldSchema> resultSchema, TableAccessInfo tableAccessInfo, ColumnAccessInfo columnAccessInfo, Set<ReadEntity> inputs) {
            this.queryTime = queryTime;
            this.lookupInfo = lookupInfo;
            this.hiveOperation = hiveOperation;
            this.resultSchema = resultSchema;
            this.tableAccessInfo = tableAccessInfo;
            this.columnAccessInfo = columnAccessInfo;
            this.inputs = inputs;
        }

        public LookupInfo getLookupInfo() {
            return this.lookupInfo;
        }

        public void setLookupInfo(LookupInfo lookupInfo) {
            this.lookupInfo = lookupInfo;
        }

        public HiveOperation getHiveOperation() {
            return this.hiveOperation;
        }

        public void setHiveOperation(HiveOperation hiveOperation) {
            this.hiveOperation = hiveOperation;
        }

        public List<FieldSchema> getResultSchema() {
            return this.resultSchema;
        }

        public void setResultSchema(List<FieldSchema> resultSchema) {
            this.resultSchema = resultSchema;
        }

        public TableAccessInfo getTableAccessInfo() {
            return this.tableAccessInfo;
        }

        public void setTableAccessInfo(TableAccessInfo tableAccessInfo) {
            this.tableAccessInfo = tableAccessInfo;
        }

        public ColumnAccessInfo getColumnAccessInfo() {
            return this.columnAccessInfo;
        }

        public void setColumnAccessInfo(ColumnAccessInfo columnAccessInfo) {
            this.columnAccessInfo = columnAccessInfo;
        }

        public Set<ReadEntity> getInputs() {
            return this.inputs;
        }

        public void setInputs(Set<ReadEntity> inputs) {
            this.inputs = inputs;
        }

        public long getQueryTime() {
            return this.queryTime;
        }

        public void setQueryTime(long queryTime) {
            this.queryTime = queryTime;
        }
    }

    public static class LookupInfo {
        private String queryText;
        private Supplier<ValidTxnWriteIdList> txnWriteIdListProvider;

        public LookupInfo(String queryText, Supplier<ValidTxnWriteIdList> txnWriteIdListProvider) {
            this.queryText = queryText;
            this.txnWriteIdListProvider = txnWriteIdListProvider;
        }

        public String getQueryText() {
            return this.queryText;
        }
    }
}

