/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.plugin.jdbc;

import com.google.common.cache.CacheStats;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.airlift.units.Duration;
import io.prestosql.plugin.jdbc.CachingJdbcClient;
import io.prestosql.plugin.jdbc.JdbcClient;
import io.prestosql.plugin.jdbc.JdbcColumnHandle;
import io.prestosql.plugin.jdbc.JdbcTableHandle;
import io.prestosql.plugin.jdbc.SessionPropertiesProvider;
import io.prestosql.plugin.jdbc.TestingDatabase;
import io.prestosql.spi.connector.ColumnMetadata;
import io.prestosql.spi.connector.ConnectorSession;
import io.prestosql.spi.connector.ConnectorTableMetadata;
import io.prestosql.spi.connector.SchemaTableName;
import io.prestosql.spi.connector.TableNotFoundException;
import io.prestosql.spi.session.PropertyMetadata;
import io.prestosql.spi.testing.InterfaceTestUtils;
import io.prestosql.spi.type.IntegerType;
import io.prestosql.spi.type.Type;
import io.prestosql.testing.TestingConnectorSession;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

@Test(singleThreaded=true)
public class TestCachingJdbcClient {
    private static final Duration FOREVER = Duration.succinctDuration((double)1.0, (TimeUnit)TimeUnit.DAYS);
    private static final Duration ZERO = Duration.succinctDuration((double)0.0, (TimeUnit)TimeUnit.MILLISECONDS);
    private static final ImmutableList<PropertyMetadata<?>> PROPERTY_METADATA = ImmutableList.of((Object)PropertyMetadata.stringProperty((String)"session_name", (String)"Session name", null, (boolean)false));
    private static final ConnectorSession SESSION = TestingConnectorSession.builder().setPropertyMetadata(PROPERTY_METADATA).build();
    private TestingDatabase database;
    private CachingJdbcClient cachingJdbcClient;
    private JdbcClient jdbcClient;
    private String schema;

    @BeforeMethod
    public void setUp() throws Exception {
        this.database = new TestingDatabase();
        this.cachingJdbcClient = this.createCachingJdbcClient(true);
        this.jdbcClient = this.database.getJdbcClient();
        this.schema = (String)this.jdbcClient.getSchemaNames(SESSION).iterator().next();
    }

    private CachingJdbcClient createCachingJdbcClient(Duration cacheTtl, boolean cacheMissing) {
        return new CachingJdbcClient(this.database.getJdbcClient(), Set.of(TestCachingJdbcClient.getTestSessionPropertiesProvider()), cacheTtl, cacheMissing);
    }

    private CachingJdbcClient createCachingJdbcClient(boolean cacheMissing) {
        return this.createCachingJdbcClient(FOREVER, cacheMissing);
    }

    @AfterMethod(alwaysRun=true)
    public void tearDown() throws Exception {
        this.database.close();
    }

    @Test
    public void testSchemaNamesCached() {
        String phantomSchema = "phantom_schema";
        this.jdbcClient.createSchema(SESSION, phantomSchema);
        Assertions.assertThat((Iterable)this.cachingJdbcClient.getSchemaNames(SESSION)).contains((Object[])new String[]{phantomSchema});
        this.jdbcClient.dropSchema(SESSION, phantomSchema);
        Assertions.assertThat((Iterable)this.jdbcClient.getSchemaNames(SESSION)).doesNotContain((Object[])new String[]{phantomSchema});
        Assertions.assertThat((Iterable)this.cachingJdbcClient.getSchemaNames(SESSION)).contains((Object[])new String[]{phantomSchema});
    }

    @Test
    public void testTableNamesCached() {
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        Assertions.assertThat((List)this.cachingJdbcClient.getTableNames(SESSION, Optional.of(this.schema))).contains((Object[])new SchemaTableName[]{phantomTable});
        this.dropTable(phantomTable);
        Assertions.assertThat((List)this.jdbcClient.getTableNames(SESSION, Optional.of(this.schema))).doesNotContain((Object[])new SchemaTableName[]{phantomTable});
        Assertions.assertThat((List)this.cachingJdbcClient.getTableNames(SESSION, Optional.of(this.schema))).contains((Object[])new SchemaTableName[]{phantomTable});
    }

    @Test
    public void testTableHandleCached() {
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        Optional cachedTable = this.cachingJdbcClient.getTableHandle(SESSION, phantomTable);
        this.dropTable(phantomTable);
        Assertions.assertThat((Optional)this.jdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        Assertions.assertThat((Optional)this.cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEqualTo((Object)cachedTable);
    }

    @Test
    public void testEmptyTableHandleIsCachedWhenCacheMissingIsTrue() {
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        Assertions.assertThat((Optional)this.cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        this.createTable(phantomTable);
        Assertions.assertThat((Optional)this.cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        this.dropTable(phantomTable);
    }

    @Test
    public void testEmptyTableHandleNotCachedWhenCacheMissingIsFalse() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient(false);
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        this.createTable(phantomTable);
        Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isPresent();
        this.dropTable(phantomTable);
    }

    private JdbcTableHandle createTable(SchemaTableName phantomTable) {
        this.jdbcClient.createTable(SESSION, new ConnectorTableMetadata(phantomTable, Collections.emptyList()));
        return (JdbcTableHandle)this.jdbcClient.getTableHandle(SESSION, phantomTable).orElseThrow();
    }

    private void dropTable(SchemaTableName phantomTable) {
        JdbcTableHandle tableHandle = (JdbcTableHandle)this.jdbcClient.getTableHandle(SESSION, phantomTable).orElseThrow();
        this.jdbcClient.dropTable(SESSION, tableHandle);
    }

    @Test
    public void testColumnsCached() {
        JdbcTableHandle table = this.getAnyTable(this.schema);
        JdbcColumnHandle phantomColumn = this.addColumn(table);
        int expectedLoad = 0;
        int expectedHit = 0;
        int expectedMiss = 0;
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), ++expectedLoad, expectedHit, ++expectedMiss);
        this.jdbcClient.dropColumn(SESSION, table, phantomColumn);
        Assertions.assertThat((List)this.jdbcClient.getColumns(SESSION, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), expectedLoad, ++expectedHit, expectedMiss);
    }

    @Test
    public void testColumnsCachedPerSession() {
        ConnectorSession firstSession = TestCachingJdbcClient.createSession("first");
        ConnectorSession secondSession = TestCachingJdbcClient.createSession("second");
        JdbcTableHandle table = this.getAnyTable(this.schema);
        JdbcColumnHandle phantomColumn = this.addColumn(table);
        int expectedLoad = 0;
        int expectedHit = 0;
        int expectedMiss = 0;
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), ++expectedLoad, expectedHit, ++expectedMiss);
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), ++expectedLoad, expectedHit, ++expectedMiss);
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), expectedLoad, ++expectedHit, expectedMiss);
        this.cachingJdbcClient.dropColumn(firstSession, table, phantomColumn);
        Assertions.assertThat((List)this.jdbcClient.getColumns(firstSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), expectedLoad += 2, expectedHit, expectedMiss += 2);
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), expectedLoad, expectedHit + 2, expectedMiss);
    }

    @Test
    public void testColumnsCacheInvalidationOnTableDrop() {
        ConnectorSession firstSession = TestCachingJdbcClient.createSession("first");
        ConnectorSession secondSession = TestCachingJdbcClient.createSession("second");
        JdbcTableHandle firstTable = this.createTable(new SchemaTableName(this.schema, "first_table"));
        JdbcTableHandle secondTable = this.createTable(new SchemaTableName(this.schema, "second_table"));
        JdbcColumnHandle firstColumn = this.addColumn(firstTable, "first_column");
        JdbcColumnHandle secondColumn = this.addColumn(secondTable, "second_column");
        int expectedLoad = 0;
        int expectedHit = 0;
        int expectedMiss = 0;
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, firstTable)).contains((Object[])new JdbcColumnHandle[]{firstColumn});
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, firstTable)).contains((Object[])new JdbcColumnHandle[]{firstColumn});
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), expectedLoad += 4, expectedHit, expectedMiss += 4);
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, firstTable)).contains((Object[])new JdbcColumnHandle[]{firstColumn});
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), expectedLoad, expectedHit += 2, expectedMiss);
        this.cachingJdbcClient.renameColumn(firstSession, firstTable, firstColumn, "another_column");
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, firstTable)).doesNotContain((Object[])new JdbcColumnHandle[]{firstColumn}).containsAll((Iterable)this.jdbcClient.getColumns(SESSION, firstTable));
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), ++expectedLoad, expectedHit, ++expectedMiss);
        this.cachingJdbcClient.dropTable(secondSession, firstTable);
        Assertions.assertThatThrownBy(() -> this.cachingJdbcClient.getColumns(firstSession, firstTable)).isInstanceOf(TableNotFoundException.class);
        Assertions.assertThatThrownBy(() -> this.cachingJdbcClient.getColumns(secondSession, firstTable)).isInstanceOf(TableNotFoundException.class);
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), expectedLoad += 2, expectedHit, expectedMiss += 2);
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        this.assertCacheLoadsHitsAndMisses(this.cachingJdbcClient.getColumnsCacheStats(), expectedLoad, expectedHit += 2, expectedMiss);
        this.cachingJdbcClient.dropTable(secondSession, secondTable);
    }

    @Test
    public void testColumnsNotCachedWhenCacheDisabled() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient(ZERO, true);
        ConnectorSession firstSession = TestCachingJdbcClient.createSession("first");
        ConnectorSession secondSession = TestCachingJdbcClient.createSession("second");
        JdbcTableHandle firstTable = this.createTable(new SchemaTableName(this.schema, "first_table"));
        JdbcTableHandle secondTable = this.createTable(new SchemaTableName(this.schema, "second_table"));
        JdbcColumnHandle firstColumn = this.addColumn(firstTable, "first_column");
        JdbcColumnHandle secondColumn = this.addColumn(secondTable, "second_column");
        int expectedLoad = 0;
        int expectedHit = 0;
        int expectedMiss = 0;
        Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
        Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
        Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
        Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
        this.assertCacheLoadsHitsAndMisses(cachingJdbcClient.getColumnsCacheStats(), expectedLoad += 4, expectedHit, expectedMiss += 4);
        Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
        Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
        Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
        Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
        this.assertCacheLoadsHitsAndMisses(cachingJdbcClient.getColumnsCacheStats(), expectedLoad += 4, expectedHit, expectedMiss += 4);
        this.jdbcClient.dropTable(SESSION, firstTable);
        this.jdbcClient.dropTable(SESSION, secondTable);
        Assertions.assertThatThrownBy(() -> cachingJdbcClient.getColumns(firstSession, firstTable)).isInstanceOf(TableNotFoundException.class);
        Assertions.assertThatThrownBy(() -> cachingJdbcClient.getColumns(firstSession, secondTable)).isInstanceOf(TableNotFoundException.class);
        this.assertCacheLoadsHitsAndMisses(cachingJdbcClient.getColumnsCacheStats(), expectedLoad += 2, expectedHit, expectedMiss += 2);
    }

    private void assertCacheLoadsHitsAndMisses(CacheStats stats, int expectedLoad, int expectedHit, int expectedMiss) {
        ((AbstractLongAssert)Assertions.assertThat((long)stats.loadCount()).withFailMessage("Expected load count is %d but actual is %d", new Object[]{expectedLoad, stats.loadCount()})).isEqualTo((long)expectedLoad);
        ((AbstractLongAssert)Assertions.assertThat((long)stats.hitCount()).withFailMessage("Expected hit count is %d but actual is %d", new Object[]{expectedHit, stats.hitCount()})).isEqualTo((long)expectedHit);
        ((AbstractLongAssert)Assertions.assertThat((long)stats.missCount()).withFailMessage("Expected miss count is %d but actual is %d", new Object[]{expectedMiss, stats.missCount()})).isEqualTo((long)expectedMiss);
    }

    private JdbcTableHandle getAnyTable(String schema) {
        SchemaTableName tableName = this.jdbcClient.getTableNames(SESSION, Optional.of(schema)).stream().filter(schemaTableName -> !"public".equals(schemaTableName.getTableName())).findAny().orElseThrow();
        return (JdbcTableHandle)this.jdbcClient.getTableHandle(SESSION, tableName).orElseThrow();
    }

    private JdbcColumnHandle addColumn(JdbcTableHandle tableHandle) {
        return this.addColumn(tableHandle, "phantom_column");
    }

    private JdbcColumnHandle addColumn(JdbcTableHandle tableHandle, String columnName) {
        ColumnMetadata columnMetadata = new ColumnMetadata(columnName, (Type)IntegerType.INTEGER);
        this.jdbcClient.addColumn(SESSION, tableHandle, columnMetadata);
        return this.jdbcClient.getColumns(SESSION, tableHandle).stream().filter(jdbcColumnHandle -> jdbcColumnHandle.getColumnMetadata().equals((Object)columnMetadata)).findAny().orElseThrow();
    }

    private static SessionPropertiesProvider getTestSessionPropertiesProvider() {
        return new SessionPropertiesProvider(){

            public List<PropertyMetadata<?>> getSessionProperties() {
                return PROPERTY_METADATA;
            }
        };
    }

    private static ConnectorSession createSession(String sessionName) {
        return TestingConnectorSession.builder().setPropertyMetadata(PROPERTY_METADATA).setPropertyValues((Map)ImmutableMap.of((Object)"session_name", (Object)sessionName)).build();
    }

    @Test
    public void testEverythingImplemented() {
        InterfaceTestUtils.assertAllMethodsOverridden(JdbcClient.class, CachingJdbcClient.class, TestCachingJdbcClient.nonOverridenMethods());
    }

    private static Set<Method> nonOverridenMethods() {
        try {
            return ImmutableSet.builder().add((Object)JdbcClient.class.getMethod("schemaExists", ConnectorSession.class, String.class)).build();
        }
        catch (NoSuchMethodException e) {
            throw new AssertionError((Object)e);
        }
    }
}

