/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.plugin.hive.parquet;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.airlift.units.DataSize;
import io.prestosql.parquet.ChunkReader;
import io.prestosql.parquet.DiskRange;
import io.prestosql.parquet.ParquetDataSource;
import io.prestosql.parquet.ParquetDataSourceId;
import io.prestosql.parquet.ParquetReaderOptions;
import io.prestosql.plugin.hive.FileFormatDataSourceStats;
import io.prestosql.plugin.hive.HiveErrorCode;
import io.prestosql.spi.ErrorCodeSupplier;
import io.prestosql.spi.PrestoException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.Path;

public class HdfsParquetDataSource
implements ParquetDataSource {
    private final ParquetDataSourceId id;
    private final long size;
    private final FSDataInputStream inputStream;
    private long readTimeNanos;
    private long readBytes;
    private final FileFormatDataSourceStats stats;
    private final ParquetReaderOptions options;

    public HdfsParquetDataSource(ParquetDataSourceId id, long size, FSDataInputStream inputStream, FileFormatDataSourceStats stats, ParquetReaderOptions options) {
        this.id = Objects.requireNonNull(id, "id is null");
        this.size = size;
        this.inputStream = inputStream;
        this.stats = stats;
        this.options = Objects.requireNonNull(options, "options is null");
    }

    public ParquetDataSourceId getId() {
        return this.id;
    }

    public final long getReadBytes() {
        return this.readBytes;
    }

    public long getReadTimeNanos() {
        return this.readTimeNanos;
    }

    public final long getSize() {
        return this.size;
    }

    public void close() throws IOException {
        this.inputStream.close();
    }

    public final void readFully(long position, byte[] buffer) {
        this.readFully(position, buffer, 0, buffer.length);
    }

    public final void readFully(long position, byte[] buffer, int bufferOffset, int bufferLength) {
        this.readBytes += (long)bufferLength;
        long start = System.nanoTime();
        try {
            this.inputStream.readFully(position, buffer, bufferOffset, bufferLength);
        }
        catch (PrestoException e) {
            throw e;
        }
        catch (Exception e) {
            throw new PrestoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, String.format("Error reading from %s at position %s", this.id, position), (Throwable)e);
        }
        long currentReadTimeNanos = System.nanoTime() - start;
        this.readTimeNanos += currentReadTimeNanos;
        this.stats.readDataBytesPerSecond(bufferLength, currentReadTimeNanos);
    }

    public final <K> Map<K, ChunkReader> planRead(Map<K, DiskRange> diskRanges) {
        Objects.requireNonNull(diskRanges, "diskRanges is null");
        if (diskRanges.isEmpty()) {
            return ImmutableMap.of();
        }
        ImmutableMap.Builder smallRangesBuilder = ImmutableMap.builder();
        ImmutableMap.Builder largeRangesBuilder = ImmutableMap.builder();
        for (Map.Entry<K, DiskRange> entry : diskRanges.entrySet()) {
            if ((long)entry.getValue().getLength() <= this.options.getMaxBufferSize().toBytes()) {
                smallRangesBuilder.put(entry);
                continue;
            }
            largeRangesBuilder.put(entry);
        }
        ImmutableMap smallRanges = smallRangesBuilder.build();
        ImmutableMap largeRanges = largeRangesBuilder.build();
        ImmutableMap.Builder slices = ImmutableMap.builder();
        slices.putAll(this.readSmallDiskRanges((Map<K, DiskRange>)smallRanges));
        slices.putAll(this.readLargeDiskRanges((Map<K, DiskRange>)largeRanges));
        return slices.build();
    }

    private <K> Map<K, ChunkReader> readSmallDiskRanges(Map<K, DiskRange> diskRanges) {
        if (diskRanges.isEmpty()) {
            return ImmutableMap.of();
        }
        List<DiskRange> mergedRanges = HdfsParquetDataSource.mergeAdjacentDiskRanges(diskRanges.values(), this.options.getMaxMergeDistance(), this.options.getMaxBufferSize());
        ImmutableMap.Builder slices = ImmutableMap.builder();
        for (final DiskRange mergedRange : mergedRanges) {
            final ReferenceCountedReader mergedRangeLoader = new ReferenceCountedReader(mergedRange);
            for (Map.Entry<K, DiskRange> diskRangeEntry : diskRanges.entrySet()) {
                final DiskRange diskRange = diskRangeEntry.getValue();
                if (!mergedRange.contains(diskRange)) continue;
                mergedRangeLoader.addReference();
                slices.put(diskRangeEntry.getKey(), (Object)new ChunkReader(){

                    public Slice read() {
                        int offset = Math.toIntExact(diskRange.getOffset() - mergedRange.getOffset());
                        return mergedRangeLoader.read().slice(offset, diskRange.getLength());
                    }

                    public void free() {
                        mergedRangeLoader.free();
                    }
                });
            }
            mergedRangeLoader.free();
        }
        ImmutableMap sliceStreams = slices.build();
        Verify.verify((boolean)sliceStreams.keySet().equals(diskRanges.keySet()));
        return sliceStreams;
    }

    private <K> Map<K, ChunkReader> readLargeDiskRanges(Map<K, DiskRange> diskRanges) {
        if (diskRanges.isEmpty()) {
            return ImmutableMap.of();
        }
        ImmutableMap.Builder slices = ImmutableMap.builder();
        for (Map.Entry<K, DiskRange> entry : diskRanges.entrySet()) {
            slices.put(entry.getKey(), (Object)new ReferenceCountedReader(entry.getValue()));
        }
        return slices.build();
    }

    public static HdfsParquetDataSource buildHdfsParquetDataSource(FSDataInputStream inputStream, Path path, long fileSize, FileFormatDataSourceStats stats, ParquetReaderOptions options) {
        return new HdfsParquetDataSource(new ParquetDataSourceId(path.toString()), fileSize, inputStream, stats, options);
    }

    public static List<DiskRange> mergeAdjacentDiskRanges(Collection<DiskRange> diskRanges, DataSize maxMergeDistance, DataSize maxReadSize) {
        ArrayList<DiskRange> ranges = new ArrayList<DiskRange>(diskRanges);
        ranges.sort(Comparator.comparingLong(DiskRange::getOffset));
        long maxReadSizeBytes = maxReadSize.toBytes();
        long maxMergeDistanceBytes = maxMergeDistance.toBytes();
        ImmutableList.Builder result = ImmutableList.builder();
        DiskRange last = (DiskRange)ranges.get(0);
        for (int i = 1; i < ranges.size(); ++i) {
            DiskRange current = (DiskRange)ranges.get(i);
            DiskRange merged = null;
            boolean blockTooLong = false;
            try {
                merged = last.span(current);
            }
            catch (ArithmeticException e) {
                blockTooLong = true;
            }
            if (!blockTooLong && (long)merged.getLength() <= maxReadSizeBytes && last.getEnd() + maxMergeDistanceBytes >= current.getOffset()) {
                last = merged;
                continue;
            }
            result.add((Object)last);
            last = current;
        }
        result.add((Object)last);
        return result.build();
    }

    private class ReferenceCountedReader
    implements ChunkReader {
        private final DiskRange range;
        private Slice data;
        private int referenceCount = 1;

        public ReferenceCountedReader(DiskRange range) {
            this.range = range;
        }

        public void addReference() {
            Preconditions.checkState((this.referenceCount > 0 ? 1 : 0) != 0, (Object)"Chunk reader is already closed");
            ++this.referenceCount;
        }

        public Slice read() {
            Preconditions.checkState((this.referenceCount > 0 ? 1 : 0) != 0, (Object)"Chunk reader is already closed");
            if (this.data == null) {
                byte[] buffer = new byte[this.range.getLength()];
                HdfsParquetDataSource.this.readFully(this.range.getOffset(), buffer, 0, buffer.length);
                this.data = Slices.wrappedBuffer((byte[])buffer);
            }
            return this.data;
        }

        public void free() {
            Preconditions.checkState((this.referenceCount > 0 ? 1 : 0) != 0, (Object)"Reference count is already 0");
            --this.referenceCount;
            if (this.referenceCount == 0) {
                this.data = null;
            }
        }
    }
}

