/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.plugin.iceberg;

import io.airlift.slice.Murmur3Hash32;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceUtf8;
import io.airlift.slice.Slices;
import io.prestosql.spi.block.Block;
import io.prestosql.spi.block.BlockBuilder;
import io.prestosql.spi.type.BigintType;
import io.prestosql.spi.type.DateTimeEncoding;
import io.prestosql.spi.type.DateType;
import io.prestosql.spi.type.DecimalType;
import io.prestosql.spi.type.Decimals;
import io.prestosql.spi.type.IntegerType;
import io.prestosql.spi.type.TimeType;
import io.prestosql.spi.type.TimestampType;
import io.prestosql.spi.type.TimestampWithTimeZoneType;
import io.prestosql.spi.type.Type;
import io.prestosql.spi.type.VarbinaryType;
import io.prestosql.spi.type.VarcharType;
import io.prestosql.spi.type.Varchars;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.IntUnaryOperator;
import java.util.function.LongUnaryOperator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.iceberg.PartitionField;
import org.joda.time.DurationField;
import org.joda.time.chrono.ISOChronology;

public final class PartitionTransforms {
    private static final Pattern BUCKET_PATTERN = Pattern.compile("bucket\\[(\\d+)]");
    private static final Pattern TRUNCATE_PATTERN = Pattern.compile("truncate\\[(\\d+)]");
    private static final ISOChronology UTC_CHRONOLOGY = ISOChronology.getInstanceUTC();
    private static final DurationField YEARS_DURATION = UTC_CHRONOLOGY.years();
    private static final DurationField MONTHS_DURATION = UTC_CHRONOLOGY.months();
    private static final DurationField DAYS_DURATION = UTC_CHRONOLOGY.days();
    private static final DurationField HOURS_DURATION = UTC_CHRONOLOGY.hours();

    private PartitionTransforms() {
    }

    public static ColumnTransform getColumnTransform(PartitionField field, Type type) {
        String transform;
        switch (transform = field.transform().toString()) {
            case "identity": {
                return new ColumnTransform(type, Function.identity());
            }
            case "year": {
                if (type.equals(DateType.DATE)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::yearsFromDate);
                }
                if (type.equals(TimestampType.TIMESTAMP)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::yearsFromTimestamp);
                }
                if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_WITH_TIME_ZONE)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::yearsFromTimestampWithTimeZone);
                }
                throw new UnsupportedOperationException("Unsupported type for 'year': " + field);
            }
            case "month": {
                if (type.equals(DateType.DATE)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::monthsFromDate);
                }
                if (type.equals(TimestampType.TIMESTAMP)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::monthsFromTimestamp);
                }
                if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_WITH_TIME_ZONE)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::monthsFromTimestampWithTimeZone);
                }
                throw new UnsupportedOperationException("Unsupported type for 'month': " + field);
            }
            case "day": {
                if (type.equals(DateType.DATE)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::daysFromDate);
                }
                if (type.equals(TimestampType.TIMESTAMP)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::daysFromTimestamp);
                }
                if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_WITH_TIME_ZONE)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::daysFromTimestampWithTimeZone);
                }
                throw new UnsupportedOperationException("Unsupported type for 'day': " + field);
            }
            case "hour": {
                if (type.equals(TimestampType.TIMESTAMP)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::hoursFromTimestamp);
                }
                if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_WITH_TIME_ZONE)) {
                    return new ColumnTransform((Type)IntegerType.INTEGER, PartitionTransforms::hoursFromTimestampWithTimeZone);
                }
                throw new UnsupportedOperationException("Unsupported type for 'hour': " + field);
            }
        }
        Matcher matcher = BUCKET_PATTERN.matcher(transform);
        if (matcher.matches()) {
            int count = Integer.parseInt(matcher.group(1));
            if (type.equals(IntegerType.INTEGER)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketInteger(block, count));
            }
            if (type.equals(BigintType.BIGINT)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketBigint(block, count));
            }
            if (Decimals.isShortDecimal((Type)type)) {
                DecimalType decimal = (DecimalType)type;
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketShortDecimal(decimal, block, count));
            }
            if (Decimals.isLongDecimal((Type)type)) {
                DecimalType decimal = (DecimalType)type;
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketLongDecimal(decimal, block, count));
            }
            if (type.equals(DateType.DATE)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketDate(block, count));
            }
            if (type.equals(TimeType.TIME)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketTime(block, count));
            }
            if (type.equals(TimestampType.TIMESTAMP)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketTimestamp(block, count));
            }
            if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_WITH_TIME_ZONE)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketTimestampWithTimeZone(block, count));
            }
            if (Varchars.isVarcharType((Type)type)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketVarchar(block, count));
            }
            if (type.equals(VarbinaryType.VARBINARY)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketVarbinary(block, count));
            }
            if (type.getTypeSignature().getBase().equals("uuid")) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketUuid(block, count));
            }
            throw new UnsupportedOperationException("Unsupported type for 'bucket': " + field);
        }
        matcher = TRUNCATE_PATTERN.matcher(transform);
        if (matcher.matches()) {
            int width = Integer.parseInt(matcher.group(1));
            if (type.equals(IntegerType.INTEGER)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.truncateInteger(block, width));
            }
            if (type.equals(BigintType.BIGINT)) {
                return new ColumnTransform((Type)BigintType.BIGINT, block -> PartitionTransforms.truncateBigint(block, width));
            }
            if (Decimals.isShortDecimal((Type)type)) {
                DecimalType decimal = (DecimalType)type;
                return new ColumnTransform(type, block -> PartitionTransforms.truncateShortDecimal(decimal, block, width));
            }
            if (Decimals.isLongDecimal((Type)type)) {
                DecimalType decimal = (DecimalType)type;
                return new ColumnTransform(type, block -> PartitionTransforms.truncateLongDecimal(decimal, block, width));
            }
            if (Varchars.isVarcharType((Type)type)) {
                return new ColumnTransform((Type)VarcharType.VARCHAR, block -> PartitionTransforms.truncateVarchar(block, width));
            }
            if (type.equals(VarbinaryType.VARBINARY)) {
                return new ColumnTransform((Type)VarbinaryType.VARBINARY, block -> PartitionTransforms.truncateVarbinary(block, width));
            }
            throw new UnsupportedOperationException("Unsupported type for 'truncate': " + field);
        }
        throw new UnsupportedOperationException("Unsupported partition transform: " + field);
    }

    private static Block yearsFromDate(Block block) {
        return PartitionTransforms.extractDate(block, value -> YEARS_DURATION.getValueAsLong(TimeUnit.DAYS.toMillis(value)));
    }

    private static Block monthsFromDate(Block block) {
        return PartitionTransforms.extractDate(block, value -> MONTHS_DURATION.getValueAsLong(TimeUnit.DAYS.toMillis(value)));
    }

    private static Block daysFromDate(Block block) {
        return PartitionTransforms.extractDate(block, LongUnaryOperator.identity());
    }

    private static Block extractDate(Block block, LongUnaryOperator function) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            long value = DateType.DATE.getLong(block, position);
            value = function.applyAsLong(value);
            IntegerType.INTEGER.writeLong(builder, value);
        }
        return builder.build();
    }

    private static Block yearsFromTimestamp(Block block) {
        return PartitionTransforms.extractTimestamp(block, arg_0 -> ((DurationField)YEARS_DURATION).getValueAsLong(arg_0));
    }

    private static Block monthsFromTimestamp(Block block) {
        return PartitionTransforms.extractTimestamp(block, arg_0 -> ((DurationField)MONTHS_DURATION).getValueAsLong(arg_0));
    }

    private static Block daysFromTimestamp(Block block) {
        return PartitionTransforms.extractTimestamp(block, arg_0 -> ((DurationField)DAYS_DURATION).getValueAsLong(arg_0));
    }

    private static Block hoursFromTimestamp(Block block) {
        return PartitionTransforms.extractTimestamp(block, arg_0 -> ((DurationField)HOURS_DURATION).getValueAsLong(arg_0));
    }

    private static Block extractTimestamp(Block block, LongUnaryOperator function) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            long value = TimestampType.TIMESTAMP.getLong(block, position);
            value = function.applyAsLong(value);
            IntegerType.INTEGER.writeLong(builder, value);
        }
        return builder.build();
    }

    private static Block yearsFromTimestampWithTimeZone(Block block) {
        return PartitionTransforms.extractTimestampWithTimeZone(block, arg_0 -> ((DurationField)YEARS_DURATION).getValueAsLong(arg_0));
    }

    private static Block monthsFromTimestampWithTimeZone(Block block) {
        return PartitionTransforms.extractTimestampWithTimeZone(block, arg_0 -> ((DurationField)MONTHS_DURATION).getValueAsLong(arg_0));
    }

    private static Block daysFromTimestampWithTimeZone(Block block) {
        return PartitionTransforms.extractTimestampWithTimeZone(block, arg_0 -> ((DurationField)DAYS_DURATION).getValueAsLong(arg_0));
    }

    private static Block hoursFromTimestampWithTimeZone(Block block) {
        return PartitionTransforms.extractTimestampWithTimeZone(block, arg_0 -> ((DurationField)HOURS_DURATION).getValueAsLong(arg_0));
    }

    private static Block extractTimestampWithTimeZone(Block block, LongUnaryOperator function) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            long value = DateTimeEncoding.unpackMillisUtc((long)TimestampWithTimeZoneType.TIMESTAMP_WITH_TIME_ZONE.getLong(block, position));
            value = function.applyAsLong(value);
            IntegerType.INTEGER.writeLong(builder, value);
        }
        return builder.build();
    }

    private static Block bucketInteger(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(IntegerType.INTEGER.getLong(block, position)));
    }

    private static Block bucketBigint(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(BigintType.BIGINT.getLong(block, position)));
    }

    private static Block bucketShortDecimal(DecimalType decimal, Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> {
            BigDecimal value = Decimals.readBigDecimal((DecimalType)decimal, (Block)block, (int)position);
            return PartitionTransforms.bucketHash(Slices.wrappedBuffer((byte[])value.unscaledValue().toByteArray()));
        });
    }

    private static Block bucketLongDecimal(DecimalType decimal, Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> {
            BigDecimal value = Decimals.readBigDecimal((DecimalType)decimal, (Block)block, (int)position);
            return PartitionTransforms.bucketHash(Slices.wrappedBuffer((byte[])value.unscaledValue().toByteArray()));
        });
    }

    private static Block bucketDate(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(DateType.DATE.getLong(block, position)));
    }

    private static Block bucketTime(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> {
            long value = TimeType.TIME.getLong(block, position);
            return PartitionTransforms.bucketHash(TimeUnit.MILLISECONDS.toMicros(value));
        });
    }

    private static Block bucketTimestamp(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> {
            long value = TimestampType.TIMESTAMP.getLong(block, position);
            return PartitionTransforms.bucketHash(TimeUnit.MILLISECONDS.toMicros(value));
        });
    }

    private static Block bucketTimestampWithTimeZone(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> {
            long value = DateTimeEncoding.unpackMillisUtc((long)TimestampWithTimeZoneType.TIMESTAMP_WITH_TIME_ZONE.getLong(block, position));
            return PartitionTransforms.bucketHash(TimeUnit.MILLISECONDS.toMicros(value));
        });
    }

    private static Block bucketVarchar(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(VarcharType.VARCHAR.getSlice(block, position)));
    }

    private static Block bucketVarbinary(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(VarcharType.VARCHAR.getSlice(block, position)));
    }

    private static Block bucketUuid(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(VarcharType.VARCHAR.getSlice(block, position)));
    }

    private static Block bucketBlock(Block block, int count, IntUnaryOperator hasher) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            int hash = hasher.applyAsInt(position);
            int bucket = (hash & Integer.MAX_VALUE) % count;
            IntegerType.INTEGER.writeLong(builder, (long)bucket);
        }
        return builder.build();
    }

    private static int bucketHash(long value) {
        return Murmur3Hash32.hash((long)value);
    }

    private static int bucketHash(Slice value) {
        return Murmur3Hash32.hash((Slice)value);
    }

    private static Block truncateInteger(Block block, int width) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            long value = IntegerType.INTEGER.getLong(block, position);
            value -= (value % (long)width + (long)width) % (long)width;
            IntegerType.INTEGER.writeLong(builder, value);
        }
        return builder.build();
    }

    private static Block truncateBigint(Block block, int width) {
        BlockBuilder builder = BigintType.BIGINT.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            long value = BigintType.BIGINT.getLong(block, position);
            value -= (value % (long)width + (long)width) % (long)width;
            BigintType.BIGINT.writeLong(builder, value);
        }
        return builder.build();
    }

    private static Block truncateShortDecimal(DecimalType type, Block block, int width) {
        BigInteger unscaledWidth = BigInteger.valueOf(width);
        BlockBuilder builder = type.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            BigDecimal value = Decimals.readBigDecimal((DecimalType)type, (Block)block, (int)position);
            value = PartitionTransforms.truncateDecimal(value, unscaledWidth);
            type.writeLong(builder, Decimals.encodeShortScaledValue((BigDecimal)value, (int)type.getScale()));
        }
        return builder.build();
    }

    private static Block truncateLongDecimal(DecimalType type, Block block, int width) {
        BigInteger unscaledWidth = BigInteger.valueOf(width);
        BlockBuilder builder = type.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            BigDecimal value = Decimals.readBigDecimal((DecimalType)type, (Block)block, (int)position);
            value = PartitionTransforms.truncateDecimal(value, unscaledWidth);
            type.writeSlice(builder, Decimals.encodeScaledValue((BigDecimal)value, (int)type.getScale()));
        }
        return builder.build();
    }

    private static BigDecimal truncateDecimal(BigDecimal value, BigInteger unscaledWidth) {
        BigDecimal remainder = new BigDecimal(value.unscaledValue().remainder(unscaledWidth).add(unscaledWidth).remainder(unscaledWidth), value.scale());
        return value.subtract(remainder);
    }

    private static Block truncateVarchar(Block block, int max) {
        BlockBuilder builder = VarcharType.VARCHAR.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            Slice value = VarcharType.VARCHAR.getSlice(block, position);
            value = PartitionTransforms.truncateVarchar(value, max);
            VarcharType.VARCHAR.writeSlice(builder, value);
        }
        return builder.build();
    }

    private static Slice truncateVarchar(Slice value, int max) {
        if (value.length() <= max) {
            return value;
        }
        int end = SliceUtf8.offsetOfCodePoint((Slice)value, (int)0, (int)max);
        if (end < 0) {
            return value;
        }
        return value.slice(0, end);
    }

    private static Block truncateVarbinary(Block block, int max) {
        BlockBuilder builder = VarbinaryType.VARBINARY.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            Slice value = VarbinaryType.VARBINARY.getSlice(block, position);
            if (value.length() > max) {
                value = value.slice(0, max);
            }
            VarbinaryType.VARBINARY.writeSlice(builder, value);
        }
        return builder.build();
    }

    public static class ColumnTransform {
        private final Type type;
        private final Function<Block, Block> transform;

        public ColumnTransform(Type type, Function<Block, Block> transform) {
            this.type = Objects.requireNonNull(type, "resultType is null");
            this.transform = Objects.requireNonNull(transform, "transform is null");
        }

        public Type getType() {
            return this.type;
        }

        public Function<Block, Block> getTransform() {
            return this.transform;
        }
    }
}

