/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonNode;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.iceberg.Schema;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.JsonUtil;

public class SchemaParser {
    private static final String TYPE = "type";
    private static final String STRUCT = "struct";
    private static final String LIST = "list";
    private static final String MAP = "map";
    private static final String FIELDS = "fields";
    private static final String ELEMENT = "element";
    private static final String KEY = "key";
    private static final String VALUE = "value";
    private static final String DOC = "doc";
    private static final String NAME = "name";
    private static final String ID = "id";
    private static final String ELEMENT_ID = "element-id";
    private static final String KEY_ID = "key-id";
    private static final String VALUE_ID = "value-id";
    private static final String REQUIRED = "required";
    private static final String ELEMENT_REQUIRED = "element-required";
    private static final String VALUE_REQUIRED = "value-required";
    private static final Cache<String, Schema> SCHEMA_CACHE = Caffeine.newBuilder().weakValues().build();

    private SchemaParser() {
    }

    static void toJson(Types.StructType struct, JsonGenerator generator) throws IOException {
        generator.writeStartObject();
        generator.writeStringField(TYPE, STRUCT);
        generator.writeArrayFieldStart(FIELDS);
        for (Types.NestedField field : struct.fields()) {
            generator.writeStartObject();
            generator.writeNumberField(ID, field.fieldId());
            generator.writeStringField(NAME, field.name());
            generator.writeBooleanField(REQUIRED, field.isRequired());
            generator.writeFieldName(TYPE);
            SchemaParser.toJson(field.type(), generator);
            if (field.doc() != null) {
                generator.writeStringField(DOC, field.doc());
            }
            generator.writeEndObject();
        }
        generator.writeEndArray();
        generator.writeEndObject();
    }

    static void toJson(Types.ListType list, JsonGenerator generator) throws IOException {
        generator.writeStartObject();
        generator.writeStringField(TYPE, LIST);
        generator.writeNumberField(ELEMENT_ID, list.elementId());
        generator.writeFieldName(ELEMENT);
        SchemaParser.toJson(list.elementType(), generator);
        generator.writeBooleanField(ELEMENT_REQUIRED, !list.isElementOptional());
        generator.writeEndObject();
    }

    static void toJson(Types.MapType map, JsonGenerator generator) throws IOException {
        generator.writeStartObject();
        generator.writeStringField(TYPE, MAP);
        generator.writeNumberField(KEY_ID, map.keyId());
        generator.writeFieldName(KEY);
        SchemaParser.toJson(map.keyType(), generator);
        generator.writeNumberField(VALUE_ID, map.valueId());
        generator.writeFieldName(VALUE);
        SchemaParser.toJson(map.valueType(), generator);
        generator.writeBooleanField(VALUE_REQUIRED, !map.isValueOptional());
        generator.writeEndObject();
    }

    static void toJson(Type.PrimitiveType primitive, JsonGenerator generator) throws IOException {
        generator.writeString(primitive.toString());
    }

    static void toJson(Type type, JsonGenerator generator) throws IOException {
        if (type.isPrimitiveType()) {
            SchemaParser.toJson(type.asPrimitiveType(), generator);
        } else {
            Type.NestedType nested = type.asNestedType();
            switch (type.typeId()) {
                case STRUCT: {
                    SchemaParser.toJson(nested.asStructType(), generator);
                    break;
                }
                case LIST: {
                    SchemaParser.toJson(nested.asListType(), generator);
                    break;
                }
                case MAP: {
                    SchemaParser.toJson(nested.asMapType(), generator);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Cannot write unknown type: " + type);
                }
            }
        }
    }

    public static void toJson(Schema schema, JsonGenerator generator) throws IOException {
        SchemaParser.toJson(schema.asStruct(), generator);
    }

    public static String toJson(Schema schema) {
        return SchemaParser.toJson(schema, false);
    }

    public static String toJson(Schema schema, boolean pretty) {
        try {
            StringWriter writer = new StringWriter();
            JsonGenerator generator = JsonUtil.factory().createGenerator((Writer)writer);
            if (pretty) {
                generator.useDefaultPrettyPrinter();
            }
            SchemaParser.toJson(schema.asStruct(), generator);
            generator.flush();
            return writer.toString();
        }
        catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private static Type typeFromJson(JsonNode json) {
        if (json.isTextual()) {
            return Types.fromPrimitiveString((String)json.asText());
        }
        if (json.isObject()) {
            String type = json.get(TYPE).asText();
            if (STRUCT.equals(type)) {
                return SchemaParser.structFromJson(json);
            }
            if (LIST.equals(type)) {
                return SchemaParser.listFromJson(json);
            }
            if (MAP.equals(type)) {
                return SchemaParser.mapFromJson(json);
            }
        }
        throw new IllegalArgumentException("Cannot parse type from json: " + json);
    }

    private static Types.StructType structFromJson(JsonNode json) {
        JsonNode fieldArray = json.get(FIELDS);
        Preconditions.checkArgument((boolean)fieldArray.isArray(), (String)"Cannot parse struct fields from non-array: %s", (Object)fieldArray);
        ArrayList fields = Lists.newArrayListWithExpectedSize((int)fieldArray.size());
        Iterator iterator = fieldArray.elements();
        while (iterator.hasNext()) {
            JsonNode field = (JsonNode)iterator.next();
            Preconditions.checkArgument((boolean)field.isObject(), (String)"Cannot parse struct field from non-object: %s", (Object)field);
            int id = JsonUtil.getInt(ID, field);
            String name = JsonUtil.getString(NAME, field);
            Type type = SchemaParser.typeFromJson(field.get(TYPE));
            String doc = JsonUtil.getStringOrNull(DOC, field);
            boolean isRequired = JsonUtil.getBool(REQUIRED, field);
            if (isRequired) {
                fields.add(Types.NestedField.required((int)id, (String)name, (Type)type, (String)doc));
                continue;
            }
            fields.add(Types.NestedField.optional((int)id, (String)name, (Type)type, (String)doc));
        }
        return Types.StructType.of((List)fields);
    }

    private static Types.ListType listFromJson(JsonNode json) {
        int elementId = JsonUtil.getInt(ELEMENT_ID, json);
        Type elementType = SchemaParser.typeFromJson(json.get(ELEMENT));
        boolean isRequired = JsonUtil.getBool(ELEMENT_REQUIRED, json);
        if (isRequired) {
            return Types.ListType.ofRequired((int)elementId, (Type)elementType);
        }
        return Types.ListType.ofOptional((int)elementId, (Type)elementType);
    }

    private static Types.MapType mapFromJson(JsonNode json) {
        int keyId = JsonUtil.getInt(KEY_ID, json);
        Type keyType = SchemaParser.typeFromJson(json.get(KEY));
        int valueId = JsonUtil.getInt(VALUE_ID, json);
        Type valueType = SchemaParser.typeFromJson(json.get(VALUE));
        boolean isRequired = JsonUtil.getBool(VALUE_REQUIRED, json);
        if (isRequired) {
            return Types.MapType.ofRequired((int)keyId, (int)valueId, (Type)keyType, (Type)valueType);
        }
        return Types.MapType.ofOptional((int)keyId, (int)valueId, (Type)keyType, (Type)valueType);
    }

    public static Schema fromJson(JsonNode json) {
        Type type = SchemaParser.typeFromJson(json);
        Preconditions.checkArgument((type.isNestedType() && type.asNestedType().isStructType() ? 1 : 0) != 0, (String)"Cannot create schema, not a struct type: %s", (Object)type);
        return new Schema(type.asNestedType().asStructType().fields());
    }

    public static Schema fromJson(String json) {
        return (Schema)SCHEMA_CACHE.get((Object)json, jsonKey -> {
            try {
                return SchemaParser.fromJson((JsonNode)JsonUtil.mapper().readValue(jsonKey, JsonNode.class));
            }
            catch (IOException e) {
                throw new RuntimeIOException(e);
            }
        });
    }
}

