/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.parquet;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.InputStream;
import org.apache.iceberg.parquet.TripleIterator;
import org.apache.parquet.CorruptDeltaByteArrays;
import org.apache.parquet.bytes.ByteBufferInputStream;
import org.apache.parquet.bytes.BytesInput;
import org.apache.parquet.bytes.BytesUtils;
import org.apache.parquet.column.ColumnDescriptor;
import org.apache.parquet.column.Dictionary;
import org.apache.parquet.column.Encoding;
import org.apache.parquet.column.ValuesType;
import org.apache.parquet.column.page.DataPage;
import org.apache.parquet.column.page.DataPageV1;
import org.apache.parquet.column.page.DataPageV2;
import org.apache.parquet.column.values.RequiresPreviousReader;
import org.apache.parquet.column.values.ValuesReader;
import org.apache.parquet.column.values.rle.RunLengthBitPackingHybridDecoder;
import org.apache.parquet.io.ParquetDecodingException;
import org.apache.parquet.io.api.Binary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class PageIterator<T>
implements TripleIterator<T> {
    private static final Logger LOG = LoggerFactory.getLogger(PageIterator.class);
    private final ColumnDescriptor desc;
    private final String writerVersion;
    private boolean hasNext = false;
    private int triplesRead = 0;
    private int currentDL = 0;
    private int currentRL = 0;
    private Dictionary dict = null;
    private DataPage page = null;
    private int triplesCount = 0;
    private Encoding valueEncoding = null;
    private IntIterator definitionLevels = null;
    private IntIterator repetitionLevels = null;
    private ValuesReader values = null;

    static <T> PageIterator<T> newIterator(ColumnDescriptor desc, String writerVersion) {
        switch (desc.getPrimitiveType().getPrimitiveTypeName()) {
            case BOOLEAN: {
                return new PageIterator<Boolean>(desc, writerVersion){

                    @Override
                    public Boolean next() {
                        return this.nextBoolean();
                    }
                };
            }
            case INT32: {
                return new PageIterator<Integer>(desc, writerVersion){

                    @Override
                    public Integer next() {
                        return this.nextInteger();
                    }
                };
            }
            case INT64: {
                return new PageIterator<Long>(desc, writerVersion){

                    @Override
                    public Long next() {
                        return this.nextLong();
                    }
                };
            }
            case FLOAT: {
                return new PageIterator<Float>(desc, writerVersion){

                    @Override
                    public Float next() {
                        return Float.valueOf(this.nextFloat());
                    }
                };
            }
            case DOUBLE: {
                return new PageIterator<Double>(desc, writerVersion){

                    @Override
                    public Double next() {
                        return this.nextDouble();
                    }
                };
            }
            case FIXED_LEN_BYTE_ARRAY: 
            case BINARY: {
                return new PageIterator<Binary>(desc, writerVersion){

                    @Override
                    public Binary next() {
                        return this.nextBinary();
                    }
                };
            }
        }
        throw new UnsupportedOperationException("Unsupported primitive type: " + desc.getPrimitiveType().getPrimitiveTypeName());
    }

    private PageIterator(ColumnDescriptor desc, String writerVersion) {
        this.desc = desc;
        this.writerVersion = writerVersion;
    }

    public void setPage(DataPage page) {
        Preconditions.checkNotNull((Object)page, (Object)"Cannot read from null page");
        this.page = page;
        this.page.accept((DataPage.Visitor)new DataPage.Visitor<ValuesReader>(){

            public ValuesReader visit(DataPageV1 dataPageV1) {
                PageIterator.this.initFromPage(dataPageV1);
                return null;
            }

            public ValuesReader visit(DataPageV2 dataPageV2) {
                PageIterator.this.initFromPage(dataPageV2);
                return null;
            }
        });
        this.triplesRead = 0;
        this.advance();
    }

    public void setDictionary(Dictionary dict) {
        this.dict = dict;
    }

    public void reset() {
        this.page = null;
        this.triplesCount = 0;
        this.triplesRead = 0;
        this.definitionLevels = null;
        this.repetitionLevels = null;
        this.values = null;
        this.hasNext = false;
    }

    public int currentPageCount() {
        return this.triplesCount;
    }

    @Override
    public boolean hasNext() {
        return this.hasNext;
    }

    @Override
    public int currentDefinitionLevel() {
        Preconditions.checkArgument((this.currentDL >= 0 ? 1 : 0) != 0, (Object)"Should not read definition, past page end");
        return this.currentDL;
    }

    @Override
    public int currentRepetitionLevel() {
        return this.currentRL;
    }

    @Override
    public boolean nextBoolean() {
        this.advance();
        try {
            return this.values.readBoolean();
        }
        catch (RuntimeException e) {
            throw this.handleRuntimeException(e);
        }
    }

    @Override
    public int nextInteger() {
        this.advance();
        try {
            return this.values.readInteger();
        }
        catch (RuntimeException e) {
            throw this.handleRuntimeException(e);
        }
    }

    @Override
    public long nextLong() {
        this.advance();
        try {
            return this.values.readLong();
        }
        catch (RuntimeException e) {
            throw this.handleRuntimeException(e);
        }
    }

    @Override
    public float nextFloat() {
        this.advance();
        try {
            return this.values.readFloat();
        }
        catch (RuntimeException e) {
            throw this.handleRuntimeException(e);
        }
    }

    @Override
    public double nextDouble() {
        this.advance();
        try {
            return this.values.readDouble();
        }
        catch (RuntimeException e) {
            throw this.handleRuntimeException(e);
        }
    }

    @Override
    public Binary nextBinary() {
        this.advance();
        try {
            return this.values.readBytes();
        }
        catch (RuntimeException e) {
            throw this.handleRuntimeException(e);
        }
    }

    @Override
    public <V> V nextNull() {
        this.advance();
        return null;
    }

    private void advance() {
        if (this.triplesRead < this.triplesCount) {
            this.currentDL = this.definitionLevels.nextInt();
            this.currentRL = this.repetitionLevels.nextInt();
            ++this.triplesRead;
            this.hasNext = true;
        } else {
            this.currentDL = -1;
            this.currentRL = -1;
            this.hasNext = false;
        }
    }

    RuntimeException handleRuntimeException(RuntimeException e) {
        if (CorruptDeltaByteArrays.requiresSequentialReads((String)this.writerVersion, (Encoding)this.valueEncoding) && e instanceof ArrayIndexOutOfBoundsException) {
            throw new ParquetDecodingException("Read failure possibly due to PARQUET-246: try setting parquet.split.files to false", (Throwable)new ParquetDecodingException(String.format("Can't read value in column %s at value %d out of %d in current page. repetition level: %d, definition level: %d", this.desc, this.triplesRead, this.triplesCount, this.currentRL, this.currentDL), (Throwable)e));
        }
        throw new ParquetDecodingException(String.format("Can't read value in column %s at value %d out of %d in current page. repetition level: %d, definition level: %d", this.desc, this.triplesRead, this.triplesCount, this.currentRL, this.currentDL), (Throwable)e);
    }

    private void initDataReader(Encoding dataEncoding, ByteBufferInputStream in, int valueCount) {
        ValuesReader previousReader = this.values;
        this.valueEncoding = dataEncoding;
        if (dataEncoding.usesDictionary()) {
            if (this.dict == null) {
                throw new ParquetDecodingException("could not read page in col " + this.desc + " as the dictionary was missing for encoding " + dataEncoding);
            }
            this.values = dataEncoding.getDictionaryBasedValuesReader(this.desc, ValuesType.VALUES, this.dict);
        } else {
            this.values = dataEncoding.getValuesReader(this.desc, ValuesType.VALUES);
        }
        try {
            this.values.initFromPage(valueCount, in);
        }
        catch (IOException e) {
            throw new ParquetDecodingException("could not read page in col " + this.desc, (Throwable)e);
        }
        if (CorruptDeltaByteArrays.requiresSequentialReads((String)this.writerVersion, (Encoding)dataEncoding) && previousReader != null && previousReader instanceof RequiresPreviousReader) {
            ((RequiresPreviousReader)this.values).setPreviousReader(previousReader);
        }
    }

    private void initFromPage(DataPageV1 page) {
        this.triplesCount = page.getValueCount();
        ValuesReader rlReader = page.getRlEncoding().getValuesReader(this.desc, ValuesType.REPETITION_LEVEL);
        ValuesReader dlReader = page.getDlEncoding().getValuesReader(this.desc, ValuesType.DEFINITION_LEVEL);
        this.repetitionLevels = new ValuesReaderIntIterator(rlReader);
        this.definitionLevels = new ValuesReaderIntIterator(dlReader);
        try {
            BytesInput bytes = page.getBytes();
            LOG.debug("page size {} bytes and {} records", (Object)bytes.size(), (Object)this.triplesCount);
            LOG.debug("reading repetition levels at 0");
            ByteBufferInputStream in = bytes.toInputStream();
            rlReader.initFromPage(this.triplesCount, in);
            LOG.debug("reading definition levels at {}", (Object)in.position());
            dlReader.initFromPage(this.triplesCount, in);
            LOG.debug("reading data at {}", (Object)in.position());
            this.initDataReader(page.getValueEncoding(), in, page.getValueCount());
        }
        catch (IOException e) {
            throw new ParquetDecodingException("could not read page " + page + " in col " + this.desc, (Throwable)e);
        }
    }

    private void initFromPage(DataPageV2 page) {
        this.triplesCount = page.getValueCount();
        this.repetitionLevels = this.newRLEIterator(this.desc.getMaxRepetitionLevel(), page.getRepetitionLevels());
        this.definitionLevels = this.newRLEIterator(this.desc.getMaxDefinitionLevel(), page.getDefinitionLevels());
        LOG.debug("page data size {} bytes and {} records", (Object)page.getData().size(), (Object)this.triplesCount);
        try {
            this.initDataReader(page.getDataEncoding(), page.getData().toInputStream(), this.triplesCount);
        }
        catch (IOException e) {
            throw new ParquetDecodingException("could not read page " + page + " in col " + this.desc, (Throwable)e);
        }
    }

    private IntIterator newRLEIterator(int maxLevel, BytesInput bytes) {
        try {
            if (maxLevel == 0) {
                return new NullIntIterator();
            }
            return new RLEIntIterator(new RunLengthBitPackingHybridDecoder(BytesUtils.getWidthFromMaxInt((int)maxLevel), (InputStream)bytes.toInputStream()));
        }
        catch (IOException e) {
            throw new ParquetDecodingException("could not read levels in page for col " + this.desc, (Throwable)e);
        }
    }

    private static final class NullIntIterator
    extends IntIterator {
        private NullIntIterator() {
        }

        @Override
        int nextInt() {
            return 0;
        }
    }

    static class RLEIntIterator
    extends IntIterator {
        RunLengthBitPackingHybridDecoder delegate;

        RLEIntIterator(RunLengthBitPackingHybridDecoder delegate) {
            this.delegate = delegate;
        }

        @Override
        int nextInt() {
            try {
                return this.delegate.readInt();
            }
            catch (IOException e) {
                throw new ParquetDecodingException((Throwable)e);
            }
        }
    }

    static class ValuesReaderIntIterator
    extends IntIterator {
        ValuesReader delegate;

        ValuesReaderIntIterator(ValuesReader delegate) {
            this.delegate = delegate;
        }

        @Override
        int nextInt() {
            return this.delegate.readInteger();
        }
    }

    static abstract class IntIterator {
        IntIterator() {
        }

        abstract int nextInt();
    }
}

