/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TreeSet;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.iceberg.HistoryEntry;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.PartitionSpecParser;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SchemaParser;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.SnapshotParser;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.io.OutputFile;
import org.apache.iceberg.io.PositionOutputStream;
import org.apache.iceberg.util.JsonUtil;

public class TableMetadataParser {
    static final String FORMAT_VERSION = "format-version";
    static final String TABLE_UUID = "table-uuid";
    static final String LOCATION = "location";
    static final String LAST_UPDATED_MILLIS = "last-updated-ms";
    static final String LAST_COLUMN_ID = "last-column-id";
    static final String SCHEMA = "schema";
    static final String PARTITION_SPEC = "partition-spec";
    static final String PARTITION_SPECS = "partition-specs";
    static final String DEFAULT_SPEC_ID = "default-spec-id";
    static final String PROPERTIES = "properties";
    static final String CURRENT_SNAPSHOT_ID = "current-snapshot-id";
    static final String SNAPSHOTS = "snapshots";
    static final String SNAPSHOT_ID = "snapshot-id";
    static final String TIMESTAMP_MS = "timestamp-ms";
    static final String SNAPSHOT_LOG = "snapshot-log";

    private TableMetadataParser() {
    }

    public static void overwrite(TableMetadata metadata, OutputFile outputFile) {
        TableMetadataParser.internalWrite(metadata, outputFile, true);
    }

    public static void write(TableMetadata metadata, OutputFile outputFile) {
        TableMetadataParser.internalWrite(metadata, outputFile, false);
    }

    public static void internalWrite(TableMetadata metadata, OutputFile outputFile, boolean overwrite) {
        boolean isGzip = Codec.fromFileName(outputFile.location()) == Codec.GZIP;
        PositionOutputStream stream = overwrite ? outputFile.createOrOverwrite() : outputFile.create();
        try (OutputStreamWriter writer = new OutputStreamWriter(isGzip ? new GZIPOutputStream((OutputStream)stream) : stream);){
            JsonGenerator generator = JsonUtil.factory().createGenerator((Writer)writer);
            generator.useDefaultPrettyPrinter();
            TableMetadataParser.toJson(metadata, generator);
            generator.flush();
        }
        catch (IOException e) {
            throw new RuntimeIOException(e, "Failed to write json to file: %s", new Object[]{outputFile});
        }
    }

    public static String getFileExtension(String codecName) {
        return TableMetadataParser.getFileExtension(Codec.fromName(codecName));
    }

    public static String getFileExtension(Codec codec) {
        return codec.extension + ".metadata.json";
    }

    public static String getOldFileExtension(Codec codec) {
        return ".metadata.json" + codec.extension;
    }

    public static String toJson(TableMetadata metadata) {
        StringWriter writer = new StringWriter();
        try {
            JsonGenerator generator = JsonUtil.factory().createGenerator((Writer)writer);
            TableMetadataParser.toJson(metadata, generator);
            generator.flush();
        }
        catch (IOException e) {
            throw new RuntimeIOException(e, "Failed to write json for: %s", new Object[]{metadata});
        }
        return writer.toString();
    }

    private static void toJson(TableMetadata metadata, JsonGenerator generator) throws IOException {
        generator.writeStartObject();
        generator.writeNumberField(FORMAT_VERSION, 1);
        generator.writeStringField(TABLE_UUID, metadata.uuid());
        generator.writeStringField(LOCATION, metadata.location());
        generator.writeNumberField(LAST_UPDATED_MILLIS, metadata.lastUpdatedMillis());
        generator.writeNumberField(LAST_COLUMN_ID, metadata.lastColumnId());
        generator.writeFieldName(SCHEMA);
        SchemaParser.toJson(metadata.schema(), generator);
        generator.writeFieldName(PARTITION_SPEC);
        PartitionSpecParser.toJsonFields(metadata.spec(), generator);
        generator.writeNumberField(DEFAULT_SPEC_ID, metadata.defaultSpecId());
        generator.writeArrayFieldStart(PARTITION_SPECS);
        for (PartitionSpec partitionSpec : metadata.specs()) {
            PartitionSpecParser.toJson(partitionSpec, generator);
        }
        generator.writeEndArray();
        generator.writeObjectFieldStart(PROPERTIES);
        for (Map.Entry entry : metadata.properties().entrySet()) {
            generator.writeStringField((String)entry.getKey(), (String)entry.getValue());
        }
        generator.writeEndObject();
        generator.writeNumberField(CURRENT_SNAPSHOT_ID, metadata.currentSnapshot() != null ? metadata.currentSnapshot().snapshotId() : -1L);
        generator.writeArrayFieldStart(SNAPSHOTS);
        for (Snapshot snapshot : metadata.snapshots()) {
            SnapshotParser.toJson(snapshot, generator);
        }
        generator.writeEndArray();
        generator.writeArrayFieldStart(SNAPSHOT_LOG);
        for (HistoryEntry historyEntry : metadata.snapshotLog()) {
            generator.writeStartObject();
            generator.writeNumberField(TIMESTAMP_MS, historyEntry.timestampMillis());
            generator.writeNumberField(SNAPSHOT_ID, historyEntry.snapshotId());
            generator.writeEndObject();
        }
        generator.writeEndArray();
        generator.writeEndObject();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static TableMetadata read(TableOperations ops, InputFile file) {
        Codec codec = Codec.fromFileName(file.location());
        try (GZIPInputStream is = codec == Codec.GZIP ? new GZIPInputStream((InputStream)file.newStream()) : file.newStream();){
            TableMetadata tableMetadata = TableMetadataParser.fromJson(ops, file, (JsonNode)JsonUtil.mapper().readValue((InputStream)is, JsonNode.class));
            return tableMetadata;
        }
        catch (IOException e) {
            throw new RuntimeIOException(e, "Failed to read file: %s", new Object[]{file});
        }
    }

    static TableMetadata fromJson(TableOperations ops, InputFile file, JsonNode node) {
        ImmutableList specs;
        int defaultSpecId;
        Preconditions.checkArgument((boolean)node.isObject(), (String)"Cannot parse metadata from a non-object: %s", (Object)node);
        int formatVersion = JsonUtil.getInt(FORMAT_VERSION, node);
        Preconditions.checkArgument((formatVersion == 1 ? 1 : 0) != 0, (String)"Cannot read unsupported version %d", (int)formatVersion);
        String uuid = JsonUtil.getStringOrNull(TABLE_UUID, node);
        String location = JsonUtil.getString(LOCATION, node);
        int lastAssignedColumnId = JsonUtil.getInt(LAST_COLUMN_ID, node);
        Schema schema = SchemaParser.fromJson(node.get(SCHEMA));
        JsonNode specArray = node.get(PARTITION_SPECS);
        if (specArray != null) {
            Preconditions.checkArgument((boolean)specArray.isArray(), (String)"Cannot parse partition specs from non-array: %s", (Object)specArray);
            defaultSpecId = JsonUtil.getInt(DEFAULT_SPEC_ID, node);
            ImmutableList.Builder builder = ImmutableList.builder();
            for (JsonNode spec : specArray) {
                builder.add((Object)PartitionSpecParser.fromJson(schema, spec));
            }
            specs = builder.build();
        } else {
            defaultSpecId = 0;
            specs = ImmutableList.of((Object)PartitionSpecParser.fromJsonFields(schema, 0, node.get(PARTITION_SPEC)));
        }
        Map<String, String> properties = JsonUtil.getStringMap(PROPERTIES, node);
        long currentVersionId = JsonUtil.getLong(CURRENT_SNAPSHOT_ID, node);
        long lastUpdatedMillis = JsonUtil.getLong(LAST_UPDATED_MILLIS, node);
        JsonNode snapshotArray = node.get(SNAPSHOTS);
        Preconditions.checkArgument((boolean)snapshotArray.isArray(), (String)"Cannot parse snapshots from non-array: %s", (Object)snapshotArray);
        ArrayList snapshots = Lists.newArrayListWithExpectedSize((int)snapshotArray.size());
        Iterator iterator = snapshotArray.elements();
        while (iterator.hasNext()) {
            snapshots.add(SnapshotParser.fromJson(ops, (JsonNode)iterator.next()));
        }
        TreeSet entries = Sets.newTreeSet(Comparator.comparingLong(TableMetadata.SnapshotLogEntry::timestampMillis));
        if (node.has(SNAPSHOT_LOG)) {
            Iterator logIterator = node.get(SNAPSHOT_LOG).elements();
            while (logIterator.hasNext()) {
                JsonNode entryNode = (JsonNode)logIterator.next();
                entries.add(new TableMetadata.SnapshotLogEntry(JsonUtil.getLong(TIMESTAMP_MS, entryNode), JsonUtil.getLong(SNAPSHOT_ID, entryNode)));
            }
        }
        return new TableMetadata(ops, file, uuid, location, lastUpdatedMillis, lastAssignedColumnId, schema, defaultSpecId, (List<PartitionSpec>)specs, properties, currentVersionId, snapshots, (List<HistoryEntry>)ImmutableList.copyOf(entries.iterator()));
    }

    public static enum Codec {
        NONE(""),
        GZIP(".gz");

        private final String extension;

        private Codec(String extension) {
            this.extension = extension;
        }

        public static Codec fromName(String codecName) {
            Preconditions.checkArgument((codecName != null ? 1 : 0) != 0, (Object)"Codec name is null");
            return Codec.valueOf(codecName.toUpperCase(Locale.ENGLISH));
        }

        public static Codec fromFileName(String fileName) {
            Preconditions.checkArgument((boolean)fileName.contains(".metadata.json"), (String)"%s is not a valid metadata file", (Object)fileName);
            if (fileName.endsWith(".metadata.json.gz")) {
                return GZIP;
            }
            String fileNameWithoutSuffix = fileName.substring(0, fileName.lastIndexOf(".metadata.json"));
            if (fileNameWithoutSuffix.endsWith(Codec.GZIP.extension)) {
                return GZIP;
            }
            return NONE;
        }
    }
}

