/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.orc;

import com.google.common.base.Joiner;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.airlift.units.DataSize;
import io.prestosql.memory.context.AggregatedMemoryContext;
import io.prestosql.orc.CachingOrcDataSource;
import io.prestosql.orc.DiskRange;
import io.prestosql.orc.OrcColumn;
import io.prestosql.orc.OrcCorruptionException;
import io.prestosql.orc.OrcDataSource;
import io.prestosql.orc.OrcDataSourceId;
import io.prestosql.orc.OrcDecompressor;
import io.prestosql.orc.OrcPredicate;
import io.prestosql.orc.OrcReaderOptions;
import io.prestosql.orc.OrcRecordReader;
import io.prestosql.orc.OrcWriteValidation;
import io.prestosql.orc.metadata.ColumnMetadata;
import io.prestosql.orc.metadata.CompressionKind;
import io.prestosql.orc.metadata.ExceptionWrappingMetadataReader;
import io.prestosql.orc.metadata.Footer;
import io.prestosql.orc.metadata.Metadata;
import io.prestosql.orc.metadata.OrcColumnId;
import io.prestosql.orc.metadata.OrcMetadataReader;
import io.prestosql.orc.metadata.OrcType;
import io.prestosql.orc.metadata.PostScript;
import io.prestosql.orc.stream.OrcChunkLoader;
import io.prestosql.orc.stream.OrcInputStream;
import io.prestosql.spi.Page;
import io.prestosql.spi.type.Type;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.joda.time.DateTimeZone;

public class OrcReader {
    public static final int MAX_BATCH_SIZE = 1024;
    public static final int INITIAL_BATCH_SIZE = 1;
    public static final int BATCH_SIZE_GROWTH_FACTOR = 2;
    private static final Logger log = Logger.get(OrcReader.class);
    private static final int CURRENT_MAJOR_VERSION = 0;
    private static final int CURRENT_MINOR_VERSION = 12;
    private static final int EXPECTED_FOOTER_SIZE = 16384;
    private final OrcDataSource orcDataSource;
    private final ExceptionWrappingMetadataReader metadataReader;
    private final OrcReaderOptions options;
    private final PostScript.HiveWriterVersion hiveWriterVersion;
    private final int bufferSize;
    private final CompressionKind compressionKind;
    private final Optional<OrcDecompressor> decompressor;
    private final Footer footer;
    private final Metadata metadata;
    private final OrcColumn rootColumn;
    private final Optional<OrcWriteValidation> writeValidation;

    public OrcReader(OrcDataSource orcDataSource, OrcReaderOptions options) throws IOException {
        this(orcDataSource, options, Optional.empty());
    }

    private OrcReader(OrcDataSource orcDataSource, OrcReaderOptions options, Optional<OrcWriteValidation> writeValidation) throws IOException {
        PostScript postScript;
        this.options = Objects.requireNonNull(options, "options is null");
        this.orcDataSource = orcDataSource = OrcReader.wrapWithCacheIfTiny(orcDataSource, options.getTinyStripeThreshold());
        this.metadataReader = new ExceptionWrappingMetadataReader(orcDataSource.getId(), new OrcMetadataReader());
        this.writeValidation = Objects.requireNonNull(writeValidation, "writeValidation is null");
        long size = orcDataSource.getSize();
        if (size <= (long)PostScript.MAGIC.length()) {
            throw new OrcCorruptionException(orcDataSource.getId(), "Invalid file size %s", size);
        }
        int expectedBufferSize = Math.toIntExact(Math.min(size, 16384L));
        Slice buffer = orcDataSource.readFully(size - (long)expectedBufferSize, expectedBufferSize);
        short postScriptSize = buffer.getUnsignedByte(buffer.length() - 1);
        if (postScriptSize >= buffer.length()) {
            throw new OrcCorruptionException(orcDataSource.getId(), "Invalid postscript length %s", postScriptSize);
        }
        try {
            postScript = this.metadataReader.readPostScript((InputStream)buffer.slice(buffer.length() - 1 - postScriptSize, (int)postScriptSize).getInput());
        }
        catch (OrcCorruptionException e) {
            if (!OrcReader.isValidHeaderMagic(orcDataSource)) {
                throw new OrcCorruptionException(orcDataSource.getId(), "Not an ORC file");
            }
            throw e;
        }
        OrcReader.checkOrcVersion(orcDataSource, postScript.getVersion());
        this.validateWrite(validation -> validation.getVersion().equals(postScript.getVersion()), "Unexpected version", new Object[0]);
        this.bufferSize = Math.toIntExact(postScript.getCompressionBlockSize());
        this.compressionKind = postScript.getCompression();
        this.decompressor = OrcDecompressor.createOrcDecompressor(orcDataSource.getId(), this.compressionKind, this.bufferSize);
        this.validateWrite(validation -> validation.getCompression() == this.compressionKind, "Unexpected compression", new Object[0]);
        this.hiveWriterVersion = postScript.getHiveWriterVersion();
        int footerSize = Math.toIntExact(postScript.getFooterLength());
        int metadataSize = Math.toIntExact(postScript.getMetadataLength());
        int completeFooterSize = footerSize + metadataSize + postScriptSize + 1;
        Slice completeFooterSlice = completeFooterSize > buffer.length() ? orcDataSource.readFully(size - (long)completeFooterSize, completeFooterSize) : buffer.slice(buffer.length() - completeFooterSize, completeFooterSize);
        Slice metadataSlice = completeFooterSlice.slice(0, metadataSize);
        try (OrcInputStream metadataInputStream = new OrcInputStream(OrcChunkLoader.create(orcDataSource.getId(), metadataSlice, this.decompressor, AggregatedMemoryContext.newSimpleAggregatedMemoryContext()));){
            this.metadata = this.metadataReader.readMetadata(this.hiveWriterVersion, metadataInputStream);
        }
        Slice footerSlice = completeFooterSlice.slice(metadataSize, footerSize);
        try (OrcInputStream footerInputStream = new OrcInputStream(OrcChunkLoader.create(orcDataSource.getId(), footerSlice, this.decompressor, AggregatedMemoryContext.newSimpleAggregatedMemoryContext()));){
            this.footer = this.metadataReader.readFooter(this.hiveWriterVersion, footerInputStream);
        }
        if (this.footer.getTypes().size() == 0) {
            throw new OrcCorruptionException(orcDataSource.getId(), "File has no columns");
        }
        this.rootColumn = OrcReader.createOrcColumn("", "", new OrcColumnId(0), this.footer.getTypes(), orcDataSource.getId());
        this.validateWrite(validation -> validation.getColumnNames().equals(this.getColumnNames()), "Unexpected column names", new Object[0]);
        this.validateWrite(validation -> validation.getRowGroupMaxRowCount() == this.footer.getRowsInRowGroup().orElse(0), "Unexpected rows in group", new Object[0]);
        if (writeValidation.isPresent()) {
            writeValidation.get().validateMetadata(orcDataSource.getId(), this.footer.getUserMetadata());
            writeValidation.get().validateFileStatistics(orcDataSource.getId(), this.footer.getFileStats());
            writeValidation.get().validateStripeStatistics(orcDataSource.getId(), this.footer.getStripes(), this.metadata.getStripeStatsList());
        }
    }

    public List<String> getColumnNames() {
        return this.footer.getTypes().get(OrcColumnId.ROOT_COLUMN).getFieldNames();
    }

    public Footer getFooter() {
        return this.footer;
    }

    public Metadata getMetadata() {
        return this.metadata;
    }

    public OrcColumn getRootColumn() {
        return this.rootColumn;
    }

    public int getBufferSize() {
        return this.bufferSize;
    }

    public CompressionKind getCompressionKind() {
        return this.compressionKind;
    }

    public OrcRecordReader createRecordReader(List<OrcColumn> readColumns, List<Type> readTypes, OrcPredicate predicate, DateTimeZone hiveStorageTimeZone, AggregatedMemoryContext systemMemoryUsage, int initialBatchSize, Function<Exception, RuntimeException> exceptionTransform) throws OrcCorruptionException {
        return this.createRecordReader(readColumns, readTypes, predicate, 0L, this.orcDataSource.getSize(), hiveStorageTimeZone, systemMemoryUsage, initialBatchSize, exceptionTransform);
    }

    public OrcRecordReader createRecordReader(List<OrcColumn> readColumns, List<Type> readTypes, OrcPredicate predicate, long offset, long length, DateTimeZone hiveStorageTimeZone, AggregatedMemoryContext systemMemoryUsage, int initialBatchSize, Function<Exception, RuntimeException> exceptionTransform) throws OrcCorruptionException {
        return this.createRecordReader(readColumns, readTypes, Collections.nCopies(readColumns.size(), ProjectedLayout.fullyProjectedLayout()), predicate, offset, length, hiveStorageTimeZone, systemMemoryUsage, initialBatchSize, exceptionTransform);
    }

    public OrcRecordReader createRecordReader(List<OrcColumn> readColumns, List<Type> readTypes, List<ProjectedLayout> readLayouts, OrcPredicate predicate, long offset, long length, DateTimeZone hiveStorageTimeZone, AggregatedMemoryContext systemMemoryUsage, int initialBatchSize, Function<Exception, RuntimeException> exceptionTransform) throws OrcCorruptionException {
        return new OrcRecordReader(Objects.requireNonNull(readColumns, "readColumns is null"), Objects.requireNonNull(readTypes, "readTypes is null"), Objects.requireNonNull(readLayouts, "readLayouts is null"), Objects.requireNonNull(predicate, "predicate is null"), this.footer.getNumberOfRows(), this.footer.getStripes(), this.footer.getFileStats(), this.metadata.getStripeStatsList(), this.orcDataSource, offset, length, this.footer.getTypes(), this.decompressor, this.footer.getRowsInRowGroup(), Objects.requireNonNull(hiveStorageTimeZone, "hiveStorageTimeZone is null"), this.hiveWriterVersion, this.metadataReader, this.options, this.footer.getUserMetadata(), systemMemoryUsage, this.writeValidation, initialBatchSize, exceptionTransform);
    }

    private static OrcDataSource wrapWithCacheIfTiny(OrcDataSource dataSource, DataSize maxCacheSize) {
        if (dataSource instanceof CachingOrcDataSource) {
            return dataSource;
        }
        if (dataSource.getSize() > maxCacheSize.toBytes()) {
            return dataSource;
        }
        DiskRange diskRange = new DiskRange(0L, Math.toIntExact(dataSource.getSize()));
        return new CachingOrcDataSource(dataSource, desiredOffset -> diskRange);
    }

    private static OrcColumn createOrcColumn(String parentStreamName, String fieldName, OrcColumnId columnId, ColumnMetadata<OrcType> types, OrcDataSourceId orcDataSourceId) {
        String path = fieldName.isEmpty() ? parentStreamName : parentStreamName + "." + fieldName;
        OrcType orcType = types.get(columnId);
        Object nestedColumns = ImmutableList.of();
        if (orcType.getOrcTypeKind() == OrcType.OrcTypeKind.STRUCT) {
            nestedColumns = (List)IntStream.range(0, orcType.getFieldCount()).mapToObj(fieldId -> OrcReader.createOrcColumn(path, orcType.getFieldName(fieldId), orcType.getFieldTypeIndex(fieldId), types, orcDataSourceId)).collect(ImmutableList.toImmutableList());
        } else if (orcType.getOrcTypeKind() == OrcType.OrcTypeKind.LIST) {
            nestedColumns = ImmutableList.of((Object)OrcReader.createOrcColumn(path, "item", orcType.getFieldTypeIndex(0), types, orcDataSourceId));
        } else if (orcType.getOrcTypeKind() == OrcType.OrcTypeKind.MAP) {
            nestedColumns = ImmutableList.of((Object)OrcReader.createOrcColumn(path, "key", orcType.getFieldTypeIndex(0), types, orcDataSourceId), (Object)OrcReader.createOrcColumn(path, "value", orcType.getFieldTypeIndex(1), types, orcDataSourceId));
        }
        return new OrcColumn(path, columnId, fieldName, orcType.getOrcTypeKind(), orcDataSourceId, (List<OrcColumn>)nestedColumns, orcType.getAttributes());
    }

    private static boolean isValidHeaderMagic(OrcDataSource source) throws IOException {
        Slice headerMagic = source.readFully(0L, PostScript.MAGIC.length());
        return PostScript.MAGIC.equals((Object)headerMagic);
    }

    private static void checkOrcVersion(OrcDataSource orcDataSource, List<Integer> version) {
        if (version.size() >= 1) {
            int major = version.get(0);
            int minor = 0;
            if (version.size() > 1) {
                minor = version.get(1);
            }
            if (major > 0 || major == 0 && minor > 12) {
                log.warn("ORC file %s was written by a newer Hive version %s. This file may not be readable by this version of Hive (%s.%s).", new Object[]{orcDataSource, Joiner.on((char)'.').join(version), 0, 12});
            }
        }
    }

    private void validateWrite(Predicate<OrcWriteValidation> test, String messageFormat, Object ... args) throws OrcCorruptionException {
        if (this.writeValidation.isPresent() && !test.test(this.writeValidation.get())) {
            throw new OrcCorruptionException(this.orcDataSource.getId(), "Write validation failed: " + messageFormat, args);
        }
    }

    static void validateFile(OrcWriteValidation writeValidation, OrcDataSource input, List<Type> readTypes, DateTimeZone hiveStorageTimeZone) throws OrcCorruptionException {
        try {
            OrcReader orcReader = new OrcReader(input, new OrcReaderOptions(), Optional.of(writeValidation));
            try (OrcRecordReader orcRecordReader = orcReader.createRecordReader(orcReader.getRootColumn().getNestedColumns(), readTypes, OrcPredicate.TRUE, hiveStorageTimeZone, AggregatedMemoryContext.newSimpleAggregatedMemoryContext(), 1, exception -> {
                Throwables.throwIfUnchecked((Throwable)exception);
                return new RuntimeException((Throwable)exception);
            });){
                Page page = orcRecordReader.nextPage();
                while (page != null) {
                    page.getLoadedPage();
                    page = orcRecordReader.nextPage();
                }
            }
        }
        catch (IOException e) {
            throw new OrcCorruptionException(e, input.getId(), "Validation failed", new Object[0]);
        }
    }

    public static class ProjectedLayout {
        private final Optional<Map<String, ProjectedLayout>> fieldLayouts;

        private ProjectedLayout(Optional<Map<String, ProjectedLayout>> fieldLayouts) {
            this.fieldLayouts = Objects.requireNonNull(fieldLayouts, "fieldLayouts is null");
        }

        public ProjectedLayout getFieldLayout(String name) {
            if (this.fieldLayouts.isPresent()) {
                return this.fieldLayouts.get().get(name);
            }
            return ProjectedLayout.fullyProjectedLayout();
        }

        public static ProjectedLayout fullyProjectedLayout() {
            return new ProjectedLayout(Optional.empty());
        }

        public static ProjectedLayout createProjectedLayout(OrcColumn root, List<List<String>> dereferences) {
            if (dereferences.stream().map(List::size).anyMatch(Predicate.isEqual(0))) {
                return ProjectedLayout.fullyProjectedLayout();
            }
            Map dereferencesByField = dereferences.stream().collect(Collectors.groupingBy(sequence -> (String)sequence.get(0), Collectors.mapping(sequence -> sequence.subList(1, sequence.size()), Collectors.toList())));
            ImmutableMap.Builder fieldLayouts = ImmutableMap.builder();
            for (OrcColumn nestedColumn : root.getNestedColumns()) {
                String fieldName = nestedColumn.getColumnName().toLowerCase(Locale.ENGLISH);
                if (!dereferencesByField.containsKey(fieldName)) continue;
                fieldLayouts.put((Object)fieldName, (Object)ProjectedLayout.createProjectedLayout(nestedColumn, dereferencesByField.get(fieldName)));
            }
            return new ProjectedLayout(Optional.of(fieldLayouts.build()));
        }
    }
}

