/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.jdbc;

import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.io.Closer;
import io.airlift.log.Logger;
import io.airlift.log.Logging;
import io.prestosql.jdbc.PrestoConnection;
import io.prestosql.server.testing.TestingPrestoServer;
import java.io.Closeable;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.JDBCType;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.TimeZone;
import oracle.jdbc.OracleType;
import org.assertj.core.api.AbstractIntegerAssert;
import org.assertj.core.api.Assertions;
import org.testcontainers.containers.OracleContainer;
import org.testcontainers.containers.PostgreSQLContainer;
import org.testng.Assert;
import org.testng.annotations.AfterClass;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

@Test(singleThreaded=true)
public class TestJdbcVendorCompatibility {
    private static final String OTHER_TIMEZONE = "Asia/Kathmandu";
    private Logger log;
    private TestingPrestoServer server;
    private List<ReferenceDriver> referenceDrivers;
    private Connection connection;
    private Statement statement;

    @BeforeClass
    public void setupServer() {
        Assert.assertNotEquals((Object)OTHER_TIMEZONE, (Object)TimeZone.getDefault().getID(), (String)"We need a timezone different from the default JVM one");
        Logging.initialize();
        this.log = Logger.get(TestJdbcVendorCompatibility.class);
        this.server = TestingPrestoServer.create();
        this.referenceDrivers = ImmutableList.of((Object)new PostgresqlReferenceDriver(), (Object)new OracleReferenceDriver());
    }

    @AfterClass(alwaysRun=true)
    public void tearDownServer() throws Exception {
        try (Closer closer = Closer.create();){
            this.referenceDrivers.forEach(arg_0 -> ((Closer)closer).register(arg_0));
            closer.register((Closeable)this.server);
        }
    }

    @BeforeMethod
    public void setUp() throws Exception {
        this.connection = DriverManager.getConnection("jdbc:presto://" + this.server.getAddress(), "test", null);
        this.statement = this.connection.createStatement();
        this.referenceDrivers.forEach(ReferenceDriver::setUp);
    }

    @AfterMethod(alwaysRun=true)
    public void tearDown() throws Exception {
        this.statement.close();
        this.connection.close();
        for (ReferenceDriver driver : this.referenceDrivers) {
            try {
                driver.tearDown();
            }
            catch (Exception e) {
                this.log.warn((Throwable)e, "Failed to close reference JDBC driver %s; continuing", new Object[]{driver});
            }
        }
    }

    @DataProvider
    public Object[][] timeZoneIds() {
        return new Object[][]{{Optional.empty()}, {Optional.of("UTC")}, {Optional.of("Europe/Warsaw")}, {Optional.of("America/Denver")}, {Optional.of(ZoneId.systemDefault().getId())}};
    }

    @Test
    public void testVarbinary() throws Exception {
        this.checkRepresentation("X'12345678'", (List<String>)ImmutableList.of((Object)"bytea E'\\\\x12345678'", (Object)"hextoraw('12345678')"), JDBCType.VARBINARY, Optional.empty(), (rs, reference, column) -> {
            Assertions.assertThat((byte[])rs.getBytes(column)).isEqualTo((Object)new byte[]{18, 52, 86, 120});
            Assertions.assertThat((byte[])rs.getBytes(column)).isEqualTo((Object)reference.getBytes(column));
            Assertions.assertThat((Object)rs.getObject(column)).isEqualTo(reference.getObject(column));
            Assertions.assertThat((String)rs.getString(column).replaceFirst("^0x", "")).isEqualTo(reference.getString(column).replaceFirst("^\\\\x", ""));
        });
    }

    @Test(dataProvider="timeZoneIds")
    public void testDate(Optional<String> sessionTimezoneId) throws Exception {
        this.checkRepresentation("DATE '2018-02-13'", JDBCType.DATE, sessionTimezoneId, (rs, reference, column) -> {
            Assert.assertEquals((Object)rs.getDate(column), (Object)reference.getDate(column));
            Assert.assertEquals((Object)rs.getDate(column), (Object)Date.valueOf(LocalDate.of(2018, 2, 13)));
            Assert.assertEquals((Object)rs.getDate(column, this.getCalendar()), (Object)reference.getDate(column, this.getCalendar()));
            Assert.assertEquals((Object)rs.getDate(column, this.getCalendar()), (Object)new Date(LocalDate.of(2018, 2, 13).atStartOfDay(this.getZoneId()).toInstant().toEpochMilli()));
        });
    }

    @Test(dataProvider="timeZoneIds")
    public void testTimestamp(Optional<String> sessionTimezoneId) throws Exception {
        this.checkRepresentation("TIMESTAMP '2018-02-13 13:14:15.123'", JDBCType.TIMESTAMP, sessionTimezoneId, (rs, reference, column) -> {
            Assert.assertEquals((Object)rs.getTimestamp(column), (Object)reference.getTimestamp(column));
            Assert.assertEquals((Object)rs.getTimestamp(column), (Object)Timestamp.valueOf(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 123000000)));
            Assert.assertEquals((Object)rs.getTimestamp(column, this.getCalendar()), (Object)reference.getTimestamp(column, this.getCalendar()));
            Assert.assertEquals((Object)rs.getTimestamp(column, this.getCalendar()), (Object)new Timestamp(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 123000000).atZone(this.getZoneId()).toInstant().toEpochMilli()));
        });
    }

    @Test(dataProvider="timeZoneIds")
    public void testTimestampWithTimeZone(Optional<String> sessionTimezoneId) throws Exception {
        this.checkRepresentation("TIMESTAMP '1970-01-01 00:00:00.000 +00:00'", (List<String>)ImmutableList.of((Object)"TIMESTAMP WITH TIME ZONE '1970-01-01 00:00:00.000 +00:00'", (Object)"from_tz(TIMESTAMP '1970-01-01 00:00:00.000', '+00:00')"), JDBCType.TIMESTAMP_WITH_TIMEZONE, sessionTimezoneId, (rs, reference, column) -> {
            Timestamp timestampForPointInTime = Timestamp.from(Instant.EPOCH);
            Assert.assertEquals((long)rs.getTimestamp(column).getTime(), (long)reference.getTimestamp(column).getTime());
            Assert.assertEquals((Object)rs.getTimestamp(column), (Object)reference.getTimestamp(column));
            Assert.assertEquals((Object)rs.getTimestamp(column), (Object)timestampForPointInTime);
            Assert.assertEquals((long)rs.getTimestamp(column, this.getCalendar()).getTime(), (long)reference.getTimestamp(column, this.getCalendar()).getTime());
            Assert.assertEquals((Object)rs.getTimestamp(column, this.getCalendar()), (Object)reference.getTimestamp(column, this.getCalendar()));
            Assert.assertEquals((Object)rs.getTimestamp(column, this.getCalendar()), (Object)timestampForPointInTime);
        });
        this.checkRepresentation("TIMESTAMP '2018-02-13 13:14:15.123 +03:15'", (List<String>)ImmutableList.of((Object)"TIMESTAMP WITH TIME ZONE '2018-02-13 13:14:15.123 +03:15'", (Object)"from_tz(TIMESTAMP '2018-02-13 13:14:15.123', '+03:15')"), JDBCType.TIMESTAMP_WITH_TIMEZONE, sessionTimezoneId, (rs, reference, column) -> {
            Timestamp timestampForPointInTime = Timestamp.from(ZonedDateTime.of(2018, 2, 13, 13, 14, 15, 123000000, ZoneOffset.ofHoursMinutes(3, 15)).toInstant());
            Assert.assertEquals((long)rs.getTimestamp(column).getTime(), (long)reference.getTimestamp(column).getTime());
            Assert.assertEquals((Object)rs.getTimestamp(column), (Object)reference.getTimestamp(column));
            Assert.assertEquals((Object)rs.getTimestamp(column), (Object)timestampForPointInTime);
            Assert.assertEquals((long)rs.getTimestamp(column, this.getCalendar()).getTime(), (long)reference.getTimestamp(column, this.getCalendar()).getTime());
            Assert.assertEquals((Object)rs.getTimestamp(column, this.getCalendar()), (Object)reference.getTimestamp(column, this.getCalendar()));
            Assert.assertEquals((Object)rs.getTimestamp(column, this.getCalendar()), (Object)timestampForPointInTime);
        });
        this.checkRepresentation("TIMESTAMP '2018-02-13 13:14:15.123 Europe/Warsaw'", (List<String>)ImmutableList.of((Object)"TIMESTAMP WITH TIME ZONE '2018-02-13 13:14:15.123 Europe/Warsaw'", (Object)"from_tz(TIMESTAMP '2018-02-13 13:14:15.123', 'Europe/Warsaw')"), JDBCType.TIMESTAMP_WITH_TIMEZONE, sessionTimezoneId, (rs, reference, column) -> {
            Timestamp timestampForPointInTime = Timestamp.from(ZonedDateTime.of(2018, 2, 13, 13, 14, 15, 123000000, ZoneId.of("Europe/Warsaw")).toInstant());
            Assert.assertEquals((long)rs.getTimestamp(column).getTime(), (long)reference.getTimestamp(column).getTime());
            Assert.assertEquals((Object)rs.getTimestamp(column), (Object)reference.getTimestamp(column));
            Assert.assertEquals((Object)rs.getTimestamp(column), (Object)timestampForPointInTime);
            Assert.assertEquals((long)rs.getTimestamp(column, this.getCalendar()).getTime(), (long)reference.getTimestamp(column, this.getCalendar()).getTime());
            Assert.assertEquals((Object)rs.getTimestamp(column, this.getCalendar()), (Object)reference.getTimestamp(column, this.getCalendar()));
            Assert.assertEquals((Object)rs.getTimestamp(column, this.getCalendar()), (Object)timestampForPointInTime);
        });
    }

    @Test(dataProvider="timeZoneIds")
    public void testTime(Optional<String> sessionTimezoneId) throws Exception {
        this.checkRepresentation("TIME '09:39:05'", JDBCType.TIME, sessionTimezoneId, (rs, reference, column) -> {
            Assert.assertEquals((Object)rs.getTime(column), (Object)reference.getTime(column));
            Assert.assertEquals((Object)rs.getTime(column), (Object)Time.valueOf(LocalTime.of(9, 39, 5)));
            Assert.assertEquals((Object)rs.getTime(column, this.getCalendar()), (Object)reference.getTime(column, this.getCalendar()));
            Assert.assertEquals((Object)rs.getTime(column, this.getCalendar()), (Object)new Time(LocalDate.of(1970, 1, 1).atTime(LocalTime.of(9, 39, 5)).atZone(this.getZoneId()).toInstant().toEpochMilli()));
        });
    }

    @Test(dataProvider="timeZoneIds")
    public void testDateRoundTrip(Optional<String> sessionTimezoneId) throws SQLException {
        LocalDate date = LocalDate.of(2001, 5, 6);
        Date sqlDate = Date.valueOf(date);
        java.util.Date javaDate = new java.util.Date(sqlDate.getTime());
        LocalDateTime dateTime = LocalDateTime.of(date, LocalTime.of(12, 34, 56));
        Timestamp sqlTimestamp = Timestamp.valueOf(dateTime);
        this.assertParameter(sqlDate, sessionTimezoneId, (ps, i) -> ps.setDate(i, sqlDate));
        this.assertParameter(sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, sqlDate));
        this.assertParameter(sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlDate, 91));
        this.assertParameter(sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlTimestamp, 91));
        this.assertParameter(sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)javaDate, 91));
        this.assertParameter(sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)date, 91));
        this.assertParameter(sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)dateTime, 91));
        this.assertParameter(sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)"2001-05-06", 91));
    }

    @Test(dataProvider="timeZoneIds")
    public void testTimestampRoundTrip(Optional<String> sessionTimezoneId) throws SQLException {
        LocalDateTime dateTime = LocalDateTime.of(2001, 5, 6, 12, 34, 56);
        Date sqlDate = Date.valueOf(dateTime.toLocalDate());
        Time sqlTime = Time.valueOf(dateTime.toLocalTime());
        Timestamp sqlTimestamp = Timestamp.valueOf(dateTime);
        Timestamp sameInstantInWarsawZone = Timestamp.valueOf(dateTime.atZone(ZoneId.systemDefault()).withZoneSameInstant(ZoneId.of("Europe/Warsaw")).toLocalDateTime());
        java.util.Date javaDate = java.util.Date.from(dateTime.atZone(ZoneId.systemDefault()).toInstant());
        this.assertParameter(sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setTimestamp(i, sqlTimestamp));
        this.assertParameter(sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setTimestamp(i, sqlTimestamp, null));
        this.assertParameter(sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setTimestamp(i, sqlTimestamp, Calendar.getInstance()));
        this.assertParameter(sameInstantInWarsawZone, sessionTimezoneId, (ps, i) -> ps.setTimestamp(i, sqlTimestamp, Calendar.getInstance(TimeZone.getTimeZone(ZoneId.of("Europe/Warsaw")))));
        this.assertParameter(sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, sqlTimestamp));
        this.assertParameter(new Timestamp(sqlDate.getTime()), sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlDate, 93));
        this.assertParameter(new Timestamp(sqlTime.getTime()), sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlTime, 93));
        this.assertParameter(sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlTimestamp, 93));
        this.assertParameter(sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)javaDate, 93));
        this.assertParameter(sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)dateTime, 93));
        this.assertParameter(sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)"2001-05-06 12:34:56", 93));
    }

    private void assertParameter(Object expectedValue, Optional<String> sessionTimezoneId, Binder binder) throws SQLException {
        sessionTimezoneId.ifPresent(arg_0 -> ((PrestoConnection)this.connection.unwrap(PrestoConnection.class)).setTimeZoneId(arg_0));
        try (PreparedStatement statement = this.connection.prepareStatement("SELECT ?");){
            binder.bind(statement, 1);
            try (ResultSet rs = statement.executeQuery();){
                Assert.assertTrue((boolean)rs.next());
                Assert.assertEquals((Object)expectedValue, (Object)rs.getObject(1));
                Assert.assertFalse((boolean)rs.next());
            }
        }
    }

    private void checkRepresentation(String expression, JDBCType type, Optional<String> sessionTimezoneId, ResultAssertion assertion) throws Exception {
        List referenceDriversExpressions = (List)this.referenceDrivers.stream().map(driver -> driver.supports(type) ? expression : "").collect(ImmutableList.toImmutableList());
        this.checkRepresentation(expression, referenceDriversExpressions, type, sessionTimezoneId, assertion);
    }

    private void checkRepresentation(String prestoExpression, List<String> referenceDriversExpressions, JDBCType type, Optional<String> sessionTimezoneId, ResultAssertion assertion) throws Exception {
        Verify.verify((referenceDriversExpressions.size() == this.referenceDrivers.size() ? 1 : 0) != 0, (String)"Wrong referenceDriversExpressions list size", (Object[])new Object[0]);
        int tests = 0;
        ArrayList<AssertionError> failures = new ArrayList<AssertionError>();
        for (int i = 0; i < this.referenceDrivers.size(); ++i) {
            ReferenceDriver driver = this.referenceDrivers.get(i);
            String referenceExpression = referenceDriversExpressions.get(i);
            if (!driver.supports(type)) {
                Verify.verify((boolean)referenceExpression.isEmpty(), (String)"referenceExpression must be empty for %s so that the test code clearly indicates which cases are actually tested", (Object)driver);
                continue;
            }
            ++tests;
            this.log.info("Checking behavior against %s using expression: %s", new Object[]{driver, referenceExpression});
            try {
                Verify.verify((!referenceExpression.isEmpty() ? 1 : 0) != 0, (String)"referenceExpression is empty", (Object[])new Object[0]);
                this.checkRepresentation(prestoExpression, referenceExpression, type, sessionTimezoneId, driver, assertion);
                continue;
            }
            catch (AssertionError | RuntimeException e) {
                String message = String.format("Failure when checking behavior against %s", driver);
                this.log.error((Throwable)e, message);
                failures.add(new AssertionError(message, (Throwable)e));
            }
        }
        Verify.verify((tests > 0 ? 1 : 0) != 0, (String)"No reference driver found supporting %s", (Object)type);
        if (!failures.isEmpty()) {
            if (failures.size() == 1 && tests == 1) {
                throw (AssertionError)Iterables.getOnlyElement(failures);
            }
            AssertionError error = new AssertionError((Object)String.format("Test failed for %s reference drivers out of %s applicable", failures.size(), tests));
            failures.forEach(arg_0 -> error.addSuppressed(arg_0));
            throw error;
        }
    }

    private void checkRepresentation(String prestoExpression, String referenceExpression, JDBCType type, Optional<String> sessionTimezoneId, ReferenceDriver reference, ResultAssertion assertion) throws Exception {
        try (ResultSet prestoResultSet = this.prestoQuery(prestoExpression, sessionTimezoneId);
             ResultSet referenceResultSet = reference.query(referenceExpression, sessionTimezoneId);){
            Assert.assertTrue((boolean)prestoResultSet.next());
            Assert.assertTrue((boolean)referenceResultSet.next());
            assertion.accept(prestoResultSet, referenceResultSet, 1);
            ((AbstractIntegerAssert)Assertions.assertThat((int)prestoResultSet.getMetaData().getColumnType(1)).as("Presto declared SQL type", new Object[0])).isEqualTo((Object)type.getVendorTypeNumber());
            ((AbstractIntegerAssert)Assertions.assertThat((int)referenceResultSet.getMetaData().getColumnType(1)).as("Reference driver's declared SQL type for " + type, new Object[0])).isEqualTo(reference.expectedDeclaredJdbcType(type));
            Assert.assertFalse((boolean)prestoResultSet.next());
            Assert.assertFalse((boolean)referenceResultSet.next());
        }
    }

    private ResultSet prestoQuery(String expression, Optional<String> sessionTimezoneId) throws Exception {
        sessionTimezoneId.ifPresent(arg_0 -> ((PrestoConnection)this.connection.unwrap(PrestoConnection.class)).setTimeZoneId(arg_0));
        return this.statement.executeQuery("SELECT " + expression);
    }

    private Calendar getCalendar() {
        return Calendar.getInstance(TimeZone.getTimeZone(ZoneId.of(OTHER_TIMEZONE)));
    }

    private ZoneId getZoneId() {
        return ZoneId.of(this.getCalendar().getTimeZone().getID());
    }

    private static interface Binder {
        public void bind(PreparedStatement var1, int var2) throws SQLException;
    }

    @FunctionalInterface
    private static interface ResultAssertion {
        public void accept(ResultSet var1, ResultSet var2, int var3) throws Exception;
    }

    private static class PostgresqlReferenceDriver
    implements ReferenceDriver {
        private PostgreSQLContainer<?> postgresqlContainer;
        private Connection connection;
        private Statement statement;
        private Optional<Optional<String>> timezoneSet = Optional.empty();

        PostgresqlReferenceDriver() {
            this.postgresqlContainer = new PostgreSQLContainer("postgres:12.4");
            this.postgresqlContainer.start();
        }

        @Override
        public ResultSet query(String expression, Optional<String> timezoneId) throws Exception {
            Verify.verify((!this.timezoneSet.isPresent() || Objects.equals(this.timezoneSet.get(), timezoneId) ? 1 : 0) != 0, (String)"Cannot set time zone %s while %s set previously", timezoneId, this.timezoneSet);
            this.timezoneSet = Optional.of(timezoneId);
            if (timezoneId.isPresent()) {
                this.statement.execute(String.format("SET SESSION TIME ZONE '%s'", timezoneId.get()));
            }
            return this.statement.executeQuery(String.format("SELECT %s", expression));
        }

        @Override
        public boolean supports(JDBCType type) {
            return true;
        }

        @Override
        public int expectedDeclaredJdbcType(JDBCType type) {
            switch (type) {
                case TIMESTAMP_WITH_TIMEZONE: {
                    return 93;
                }
                case VARBINARY: {
                    return -2;
                }
            }
            return type.getVendorTypeNumber();
        }

        @Override
        public void setUp() {
            try {
                this.connection = this.postgresqlContainer.createConnection("");
                this.statement = this.connection.createStatement();
                this.timezoneSet = Optional.empty();
            }
            catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public void tearDown() throws Exception {
            this.statement.close();
            this.connection.close();
        }

        @Override
        public void close() {
            this.postgresqlContainer.stop();
        }

        public String toString() {
            return "[postgresql]";
        }
    }

    private static class OracleReferenceDriver
    implements ReferenceDriver {
        private OracleContainer oracleServer;
        private Connection connection;
        private Statement statement;
        private Optional<Optional<String>> timezoneSet = Optional.empty();

        OracleReferenceDriver() {
            this.oracleServer = new OracleContainer("wnameless/oracle-xe-11g-r2");
            this.oracleServer.start();
        }

        @Override
        public ResultSet query(String expression, Optional<String> timezoneId) throws Exception {
            Verify.verify((!this.timezoneSet.isPresent() || Objects.equals(this.timezoneSet.get(), timezoneId) ? 1 : 0) != 0, (String)"Cannot set time zone %s while %s set previously", timezoneId, this.timezoneSet);
            this.timezoneSet = Optional.of(timezoneId);
            if (timezoneId.isPresent()) {
                this.statement.execute(String.format("ALTER SESSION SET TIME_ZONE='%s'", timezoneId.get()));
            }
            return this.statement.executeQuery(String.format("SELECT %s FROM dual", expression));
        }

        @Override
        public boolean supports(JDBCType type) {
            return type != JDBCType.TIME;
        }

        @Override
        public int expectedDeclaredJdbcType(JDBCType type) {
            switch (type) {
                case DATE: {
                    return 93;
                }
                case TIMESTAMP_WITH_TIMEZONE: {
                    return OracleType.TIMESTAMP_WITH_TIME_ZONE.getVendorTypeNumber();
                }
            }
            return type.getVendorTypeNumber();
        }

        @Override
        public void setUp() {
            try {
                this.connection = DriverManager.getConnection(this.oracleServer.getJdbcUrl(), this.oracleServer.getUsername(), this.oracleServer.getPassword());
                this.statement = this.connection.createStatement();
                this.timezoneSet = Optional.empty();
            }
            catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public void tearDown() throws Exception {
            this.statement.close();
            this.connection.close();
        }

        @Override
        public void close() {
            this.oracleServer.stop();
        }

        public String toString() {
            return "[oracle]";
        }
    }

    private static interface ReferenceDriver
    extends Closeable {
        public ResultSet query(String var1, Optional<String> var2) throws Exception;

        public boolean supports(JDBCType var1);

        public int expectedDeclaredJdbcType(JDBCType var1);

        public void setUp();

        public void tearDown() throws Exception;

        @Override
        public void close();
    }
}

