/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.memory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.MoreCollectors;
import com.google.common.collect.Sets;
import com.google.common.collect.Streams;
import com.google.common.io.Closer;
import io.airlift.http.client.HttpClient;
import io.airlift.json.JsonCodec;
import io.airlift.log.Logger;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.prestosql.ExceededMemoryLimitException;
import io.prestosql.SystemSessionProperties;
import io.prestosql.execution.LocationFactory;
import io.prestosql.execution.QueryExecution;
import io.prestosql.execution.QueryIdGenerator;
import io.prestosql.execution.QueryTracker;
import io.prestosql.execution.scheduler.NodeSchedulerConfig;
import io.prestosql.memory.ClusterMemoryLeakDetector;
import io.prestosql.memory.ClusterMemoryPool;
import io.prestosql.memory.ForMemoryManager;
import io.prestosql.memory.LocalMemoryManager;
import io.prestosql.memory.LowMemoryKiller;
import io.prestosql.memory.MemoryInfo;
import io.prestosql.memory.MemoryManagerConfig;
import io.prestosql.memory.MemoryPoolAssignment;
import io.prestosql.memory.MemoryPoolAssignmentsRequest;
import io.prestosql.memory.NodeMemoryConfig;
import io.prestosql.memory.NoneLowMemoryKiller;
import io.prestosql.memory.RemoteNodeMemory;
import io.prestosql.memory.VersionedMemoryPoolId;
import io.prestosql.metadata.InternalNode;
import io.prestosql.metadata.InternalNodeManager;
import io.prestosql.metadata.NodeState;
import io.prestosql.server.BasicQueryInfo;
import io.prestosql.server.ServerConfig;
import io.prestosql.spi.ErrorCodeSupplier;
import io.prestosql.spi.PrestoException;
import io.prestosql.spi.QueryId;
import io.prestosql.spi.StandardErrorCode;
import io.prestosql.spi.memory.ClusterMemoryPoolManager;
import io.prestosql.spi.memory.MemoryPoolId;
import io.prestosql.spi.memory.MemoryPoolInfo;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Supplier;
import javax.annotation.PreDestroy;
import javax.annotation.concurrent.GuardedBy;
import javax.inject.Inject;
import org.weakref.jmx.JmxException;
import org.weakref.jmx.MBeanExporter;
import org.weakref.jmx.Managed;
import org.weakref.jmx.ObjectNames;

public class ClusterMemoryManager
implements ClusterMemoryPoolManager {
    private static final Logger log = Logger.get(ClusterMemoryManager.class);
    private final ExecutorService listenerExecutor = Executors.newSingleThreadExecutor();
    private final ClusterMemoryLeakDetector memoryLeakDetector = new ClusterMemoryLeakDetector();
    private final InternalNodeManager nodeManager;
    private final LocationFactory locationFactory;
    private final HttpClient httpClient;
    private final MBeanExporter exporter;
    private final JsonCodec<MemoryInfo> memoryInfoCodec;
    private final JsonCodec<MemoryPoolAssignmentsRequest> assignmentsRequestJsonCodec;
    private final DataSize maxQueryMemory;
    private final DataSize maxQueryTotalMemory;
    private final boolean enabled;
    private final LowMemoryKiller lowMemoryKiller;
    private final Duration killOnOutOfMemoryDelay;
    private final String coordinatorId;
    private final AtomicLong totalAvailableProcessors = new AtomicLong();
    private final AtomicLong memoryPoolAssignmentsVersion = new AtomicLong();
    private final AtomicLong clusterUserMemoryReservation = new AtomicLong();
    private final AtomicLong clusterTotalMemoryReservation = new AtomicLong();
    private final AtomicLong clusterMemoryBytes = new AtomicLong();
    private final AtomicLong queriesKilledDueToOutOfMemory = new AtomicLong();
    private final boolean isWorkScheduledOnCoordinator;
    @GuardedBy(value="this")
    private final Map<String, RemoteNodeMemory> nodes = new HashMap<String, RemoteNodeMemory>();
    @GuardedBy(value="this")
    private final Map<MemoryPoolId, List<Consumer<MemoryPoolInfo>>> changeListeners = new HashMap<MemoryPoolId, List<Consumer<MemoryPoolInfo>>>();
    @GuardedBy(value="this")
    private final Map<MemoryPoolId, ClusterMemoryPool> pools;
    @GuardedBy(value="this")
    private long lastTimeNotOutOfMemory = System.nanoTime();
    @GuardedBy(value="this")
    private QueryId lastKilledQuery;

    @Inject
    public ClusterMemoryManager(@ForMemoryManager HttpClient httpClient, InternalNodeManager nodeManager, LocationFactory locationFactory, MBeanExporter exporter, JsonCodec<MemoryInfo> memoryInfoCodec, JsonCodec<MemoryPoolAssignmentsRequest> assignmentsRequestJsonCodec, QueryIdGenerator queryIdGenerator, LowMemoryKiller lowMemoryKiller, ServerConfig serverConfig, MemoryManagerConfig config, NodeMemoryConfig nodeMemoryConfig, NodeSchedulerConfig schedulerConfig) {
        Objects.requireNonNull(config, "config is null");
        Objects.requireNonNull(nodeMemoryConfig, "nodeMemoryConfig is null");
        Objects.requireNonNull(serverConfig, "serverConfig is null");
        Objects.requireNonNull(schedulerConfig, "schedulerConfig is null");
        this.nodeManager = Objects.requireNonNull(nodeManager, "nodeManager is null");
        this.locationFactory = Objects.requireNonNull(locationFactory, "locationFactory is null");
        this.httpClient = Objects.requireNonNull(httpClient, "httpClient is null");
        this.exporter = Objects.requireNonNull(exporter, "exporter is null");
        this.memoryInfoCodec = Objects.requireNonNull(memoryInfoCodec, "memoryInfoCodec is null");
        this.assignmentsRequestJsonCodec = Objects.requireNonNull(assignmentsRequestJsonCodec, "assignmentsRequestJsonCodec is null");
        this.lowMemoryKiller = Objects.requireNonNull(lowMemoryKiller, "lowMemoryKiller is null");
        this.maxQueryMemory = config.getMaxQueryMemory();
        this.maxQueryTotalMemory = config.getMaxQueryTotalMemory();
        this.coordinatorId = queryIdGenerator.getCoordinatorId();
        this.enabled = serverConfig.isCoordinator();
        this.killOnOutOfMemoryDelay = config.getKillOnOutOfMemoryDelay();
        this.isWorkScheduledOnCoordinator = schedulerConfig.isIncludeCoordinator();
        Verify.verify((this.maxQueryMemory.toBytes() <= this.maxQueryTotalMemory.toBytes() ? 1 : 0) != 0, (String)"maxQueryMemory cannot be greater than maxQueryTotalMemory", (Object[])new Object[0]);
        this.pools = this.createClusterMemoryPools(nodeMemoryConfig.isReservedPoolEnabled());
    }

    private Map<MemoryPoolId, ClusterMemoryPool> createClusterMemoryPools(boolean reservedPoolEnabled) {
        HashSet<MemoryPoolId> memoryPools = new HashSet<MemoryPoolId>();
        memoryPools.add(LocalMemoryManager.GENERAL_POOL);
        if (reservedPoolEnabled) {
            memoryPools.add(LocalMemoryManager.RESERVED_POOL);
        }
        ImmutableMap.Builder builder = ImmutableMap.builder();
        for (MemoryPoolId poolId : memoryPools) {
            ClusterMemoryPool pool = new ClusterMemoryPool(poolId);
            builder.put((Object)poolId, (Object)pool);
            try {
                this.exporter.exportWithGeneratedName((Object)pool, ClusterMemoryPool.class, poolId.toString());
            }
            catch (JmxException e) {
                log.error((Throwable)e, "Error exporting memory pool %s", new Object[]{poolId});
            }
        }
        return builder.build();
    }

    public synchronized void addChangeListener(MemoryPoolId poolId, Consumer<MemoryPoolInfo> listener) {
        Verify.verify((boolean)this.memoryPoolExists(poolId), (String)"Memory pool does not exist: %s", (Object)poolId);
        this.changeListeners.computeIfAbsent(poolId, id -> new ArrayList()).add(listener);
    }

    public synchronized boolean memoryPoolExists(MemoryPoolId poolId) {
        return this.pools.containsKey(poolId);
    }

    /*
     * WARNING - void declaration
     */
    public synchronized void process(Iterable<QueryExecution> runningQueries, Supplier<List<BasicQueryInfo>> allQueryInfoSupplier) {
        void var10_12;
        if (!this.enabled) {
            return;
        }
        this.memoryLeakDetector.checkForMemoryLeaks(allQueryInfoSupplier, this.pools.get(LocalMemoryManager.GENERAL_POOL).getQueryMemoryReservations());
        boolean outOfMemory = this.isClusterOutOfMemory();
        if (!outOfMemory) {
            this.lastTimeNotOutOfMemory = System.nanoTime();
        }
        boolean queryKilled = false;
        long totalUserMemoryBytes = 0L;
        long totalMemoryBytes = 0L;
        for (QueryExecution queryExecution : runningQueries) {
            boolean resourceOvercommit = SystemSessionProperties.resourceOvercommit(queryExecution.getSession());
            long userMemoryReservation = queryExecution.getUserMemoryReservation().toBytes();
            long totalMemoryReservation = queryExecution.getTotalMemoryReservation().toBytes();
            if (resourceOvercommit && outOfMemory) {
                DataSize memory = DataSize.succinctBytes((long)this.getQueryMemoryReservation(queryExecution));
                queryExecution.fail((Throwable)new PrestoException((ErrorCodeSupplier)StandardErrorCode.CLUSTER_OUT_OF_MEMORY, String.format("The cluster is out of memory and %s=true, so this query was killed. It was using %s of memory", "resource_overcommit", memory)));
                queryKilled = true;
            }
            if (!resourceOvercommit) {
                long totalMemoryLimit;
                long userMemoryLimit = Math.min(this.maxQueryMemory.toBytes(), SystemSessionProperties.getQueryMaxMemory(queryExecution.getSession()).toBytes());
                if (userMemoryReservation > userMemoryLimit) {
                    queryExecution.fail((Throwable)((Object)ExceededMemoryLimitException.exceededGlobalUserLimit(DataSize.succinctBytes((long)userMemoryLimit))));
                    queryKilled = true;
                }
                if (totalMemoryReservation > (totalMemoryLimit = Math.min(this.maxQueryTotalMemory.toBytes(), SystemSessionProperties.getQueryMaxTotalMemory(queryExecution.getSession()).toBytes()))) {
                    queryExecution.fail((Throwable)((Object)ExceededMemoryLimitException.exceededGlobalTotalLimit(DataSize.succinctBytes((long)totalMemoryLimit))));
                    queryKilled = true;
                }
            }
            totalUserMemoryBytes += userMemoryReservation;
            totalMemoryBytes += totalMemoryReservation;
        }
        this.clusterUserMemoryReservation.set(totalUserMemoryBytes);
        this.clusterTotalMemoryReservation.set(totalMemoryBytes);
        if (!(this.lowMemoryKiller instanceof NoneLowMemoryKiller) && outOfMemory && !queryKilled && Duration.nanosSince((long)this.lastTimeNotOutOfMemory).compareTo(this.killOnOutOfMemoryDelay) > 0) {
            if (this.isLastKilledQueryGone()) {
                this.callOomKiller(runningQueries);
            } else {
                log.debug("Last killed query is still not gone: %s", new Object[]{this.lastKilledQuery});
            }
        }
        HashMap<MemoryPoolId, Integer> countByPool = new HashMap<MemoryPoolId, Integer>();
        for (QueryExecution query : runningQueries) {
            MemoryPoolId id = query.getMemoryPool().getId();
            countByPool.put(id, countByPool.getOrDefault(id, 0) + 1);
        }
        this.updatePools(countByPool);
        if (this.pools.containsKey(LocalMemoryManager.RESERVED_POOL)) {
            MemoryPoolAssignmentsRequest memoryPoolAssignmentsRequest = this.updateAssignments(runningQueries);
        } else {
            MemoryPoolAssignmentsRequest memoryPoolAssignmentsRequest = new MemoryPoolAssignmentsRequest(this.coordinatorId, Long.MIN_VALUE, (List<MemoryPoolAssignment>)ImmutableList.of());
        }
        this.updateNodes((MemoryPoolAssignmentsRequest)var10_12);
    }

    private synchronized void callOomKiller(Iterable<QueryExecution> runningQueries) {
        List nodeMemoryInfos;
        List queryMemoryInfoList = (List)Streams.stream(runningQueries).map(this::createQueryMemoryInfo).collect(ImmutableList.toImmutableList());
        Optional<QueryId> chosenQueryId = this.lowMemoryKiller.chooseQueryToKill(queryMemoryInfoList, nodeMemoryInfos = (List)this.nodes.values().stream().map(RemoteNodeMemory::getInfo).filter(Optional::isPresent).map(Optional::get).collect(ImmutableList.toImmutableList()));
        if (chosenQueryId.isPresent()) {
            log.debug("Low memory killer chose %s", new Object[]{chosenQueryId.get()});
            Optional chosenQuery = (Optional)Streams.stream(runningQueries).filter(query -> ((QueryId)chosenQueryId.get()).equals((Object)query.getQueryId())).collect(MoreCollectors.toOptional());
            if (chosenQuery.isPresent()) {
                ((QueryExecution)chosenQuery.get()).fail((Throwable)new PrestoException((ErrorCodeSupplier)StandardErrorCode.CLUSTER_OUT_OF_MEMORY, "Query killed because the cluster is out of memory. Please try again in a few minutes."));
                this.queriesKilledDueToOutOfMemory.incrementAndGet();
                this.lastKilledQuery = chosenQueryId.get();
                this.logQueryKill(chosenQueryId.get(), nodeMemoryInfos);
            }
        }
    }

    @GuardedBy(value="this")
    private boolean isLastKilledQueryGone() {
        if (this.lastKilledQuery == null) {
            return true;
        }
        if (this.memoryLeakDetector.wasQueryPossiblyLeaked(this.lastKilledQuery)) {
            this.lastKilledQuery = null;
            return true;
        }
        return !this.pools.get(LocalMemoryManager.GENERAL_POOL).getQueryMemoryReservations().containsKey(this.lastKilledQuery);
    }

    private void logQueryKill(QueryId killedQueryId, List<MemoryInfo> nodes) {
        if (!log.isInfoEnabled()) {
            return;
        }
        StringBuilder nodeDescription = new StringBuilder();
        nodeDescription.append("Query Kill Decision: Killed ").append(killedQueryId).append("\n");
        for (MemoryInfo node : nodes) {
            MemoryPoolInfo memoryPoolInfo = node.getPools().get(LocalMemoryManager.GENERAL_POOL);
            if (memoryPoolInfo == null) continue;
            nodeDescription.append("Query Kill Scenario: ");
            nodeDescription.append("MaxBytes ").append(memoryPoolInfo.getMaxBytes()).append(' ');
            nodeDescription.append("FreeBytes ").append(memoryPoolInfo.getFreeBytes() + memoryPoolInfo.getReservedRevocableBytes()).append(' ');
            nodeDescription.append("Queries ");
            Joiner.on((String)",").withKeyValueSeparator("=").appendTo(nodeDescription, memoryPoolInfo.getQueryMemoryReservations());
            nodeDescription.append('\n');
        }
        log.info(nodeDescription.toString());
    }

    @VisibleForTesting
    synchronized Map<MemoryPoolId, ClusterMemoryPool> getPools() {
        return ImmutableMap.copyOf(this.pools);
    }

    public synchronized Map<MemoryPoolId, MemoryPoolInfo> getMemoryPoolInfo() {
        ImmutableMap.Builder builder = new ImmutableMap.Builder();
        this.pools.forEach((poolId, memoryPool) -> builder.put(poolId, (Object)memoryPool.getInfo()));
        return builder.build();
    }

    private synchronized boolean isClusterOutOfMemory() {
        ClusterMemoryPool reservedPool = this.pools.get(LocalMemoryManager.RESERVED_POOL);
        ClusterMemoryPool generalPool = this.pools.get(LocalMemoryManager.GENERAL_POOL);
        if (reservedPool == null) {
            return generalPool.getBlockedNodes() > 0;
        }
        return reservedPool.getAssignedQueries() > 0 && generalPool.getBlockedNodes() > 0;
    }

    private synchronized MemoryPoolAssignmentsRequest updateAssignments(Iterable<QueryExecution> queries) {
        ClusterMemoryPool reservedPool = this.pools.get(LocalMemoryManager.RESERVED_POOL);
        ClusterMemoryPool generalPool = this.pools.get(LocalMemoryManager.GENERAL_POOL);
        Verify.verify((generalPool != null ? 1 : 0) != 0, (String)"generalPool is null", (Object[])new Object[0]);
        Verify.verify((reservedPool != null ? 1 : 0) != 0, (String)"reservedPool is null", (Object[])new Object[0]);
        long version = this.memoryPoolAssignmentsVersion.incrementAndGet();
        if (this.allAssignmentsHavePropagated(queries) && reservedPool.getAssignedQueries() == 0 && generalPool.getBlockedNodes() > 0) {
            QueryTracker.TrackedQuery biggestQuery = null;
            long maxMemory = -1L;
            for (QueryExecution queryExecution : queries) {
                long bytesUsed;
                if (SystemSessionProperties.resourceOvercommit(queryExecution.getSession()) || (bytesUsed = this.getQueryMemoryReservation(queryExecution)) <= maxMemory) continue;
                biggestQuery = queryExecution;
                maxMemory = bytesUsed;
            }
            if (biggestQuery != null) {
                log.info("Moving query %s to the reserved pool", new Object[]{biggestQuery.getQueryId()});
                biggestQuery.setMemoryPool(new VersionedMemoryPoolId(LocalMemoryManager.RESERVED_POOL, version));
            }
        }
        ImmutableList.Builder assignments = ImmutableList.builder();
        for (QueryExecution queryExecution : queries) {
            assignments.add((Object)new MemoryPoolAssignment(queryExecution.getQueryId(), queryExecution.getMemoryPool().getId()));
        }
        return new MemoryPoolAssignmentsRequest(this.coordinatorId, version, (List<MemoryPoolAssignment>)assignments.build());
    }

    private LowMemoryKiller.QueryMemoryInfo createQueryMemoryInfo(QueryExecution query) {
        return new LowMemoryKiller.QueryMemoryInfo(query.getQueryId(), query.getMemoryPool().getId(), query.getTotalMemoryReservation().toBytes());
    }

    private long getQueryMemoryReservation(QueryExecution query) {
        return query.getTotalMemoryReservation().toBytes();
    }

    private synchronized boolean allAssignmentsHavePropagated(Iterable<QueryExecution> queries) {
        long mostOutOfDateNode;
        if (this.nodes.isEmpty()) {
            return false;
        }
        long newestAssignment = ImmutableList.copyOf(queries).stream().map(QueryExecution::getMemoryPool).mapToLong(VersionedMemoryPoolId::getVersion).min().orElse(-1L);
        return newestAssignment <= (mostOutOfDateNode = this.nodes.values().stream().mapToLong(RemoteNodeMemory::getCurrentAssignmentVersion).min().orElse(Long.MAX_VALUE));
    }

    private synchronized void updateNodes(MemoryPoolAssignmentsRequest assignments) {
        ImmutableSet.Builder builder = ImmutableSet.builder();
        ImmutableSet aliveNodes = builder.addAll(this.nodeManager.getNodes(NodeState.ACTIVE)).addAll(this.nodeManager.getNodes(NodeState.SHUTTING_DOWN)).build();
        ImmutableSet aliveNodeIds = (ImmutableSet)aliveNodes.stream().map(InternalNode::getNodeIdentifier).collect(ImmutableSet.toImmutableSet());
        ImmutableSet deadNodes = ImmutableSet.copyOf((Collection)Sets.difference(this.nodes.keySet(), (Set)aliveNodeIds));
        this.nodes.keySet().removeAll((Collection<?>)deadNodes);
        for (InternalNode internalNode : aliveNodes) {
            if (this.nodes.containsKey(internalNode.getNodeIdentifier())) continue;
            this.nodes.put(internalNode.getNodeIdentifier(), new RemoteNodeMemory(internalNode, this.httpClient, this.memoryInfoCodec, this.assignmentsRequestJsonCodec, this.locationFactory.createMemoryInfoLocation(internalNode)));
        }
        if (!this.isWorkScheduledOnCoordinator) {
            this.nodes.remove(this.nodeManager.getCurrentNode().getNodeIdentifier());
        }
        for (RemoteNodeMemory remoteNodeMemory : this.nodes.values()) {
            remoteNodeMemory.asyncRefresh(assignments);
        }
    }

    private synchronized void updatePools(Map<MemoryPoolId, Integer> queryCounts) {
        List nodeMemoryInfos = (List)this.nodes.values().stream().map(RemoteNodeMemory::getInfo).filter(Optional::isPresent).map(Optional::get).collect(ImmutableList.toImmutableList());
        long totalProcessors = nodeMemoryInfos.stream().mapToLong(MemoryInfo::getAvailableProcessors).sum();
        this.totalAvailableProcessors.set(totalProcessors);
        long totalClusterMemory = nodeMemoryInfos.stream().map(MemoryInfo::getTotalNodeMemory).mapToLong(DataSize::toBytes).sum();
        this.clusterMemoryBytes.set(totalClusterMemory);
        for (ClusterMemoryPool pool : this.pools.values()) {
            pool.update(nodeMemoryInfos, queryCounts.getOrDefault(pool.getId(), 0));
            if (!this.changeListeners.containsKey(pool.getId())) continue;
            MemoryPoolInfo info = pool.getInfo();
            for (Consumer<MemoryPoolInfo> listener : this.changeListeners.get(pool.getId())) {
                this.listenerExecutor.execute(() -> listener.accept(info));
            }
        }
    }

    public synchronized Map<String, Optional<MemoryInfo>> getWorkerMemoryInfo() {
        HashMap<String, Optional<MemoryInfo>> memoryInfo = new HashMap<String, Optional<MemoryInfo>>();
        for (Map.Entry<String, RemoteNodeMemory> entry : this.nodes.entrySet()) {
            String workerId = entry.getKey() + " [" + entry.getValue().getNode().getHost() + "]";
            memoryInfo.put(workerId, entry.getValue().getInfo());
        }
        return memoryInfo;
    }

    @PreDestroy
    public synchronized void destroy() throws IOException {
        try (Closer closer = Closer.create();){
            for (ClusterMemoryPool pool : this.pools.values()) {
                closer.register(() -> this.exporter.unexport(ObjectNames.generatedNameOf(ClusterMemoryPool.class, (String)pool.getId().toString())));
            }
            closer.register(this.listenerExecutor::shutdownNow);
        }
    }

    @Managed
    public long getTotalAvailableProcessors() {
        return this.totalAvailableProcessors.get();
    }

    @Managed
    public int getNumberOfLeakedQueries() {
        return this.memoryLeakDetector.getNumberOfLeakedQueries();
    }

    @Managed
    public long getClusterUserMemoryReservation() {
        return this.clusterUserMemoryReservation.get();
    }

    @Managed
    public long getClusterTotalMemoryReservation() {
        return this.clusterTotalMemoryReservation.get();
    }

    @Managed
    public long getClusterMemoryBytes() {
        return this.clusterMemoryBytes.get();
    }

    @Managed
    public long getQueriesKilledDueToOutOfMemory() {
        return this.queriesKilledDueToOutOfMemory.get();
    }
}

