/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.transaction;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import io.airlift.concurrent.BoundedExecutor;
import io.airlift.concurrent.ExecutorServiceAdapter;
import io.airlift.concurrent.MoreFutures;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.prestosql.connector.CatalogName;
import io.prestosql.metadata.Catalog;
import io.prestosql.metadata.CatalogManager;
import io.prestosql.metadata.CatalogMetadata;
import io.prestosql.spi.ErrorCodeSupplier;
import io.prestosql.spi.PrestoException;
import io.prestosql.spi.StandardErrorCode;
import io.prestosql.spi.connector.Connector;
import io.prestosql.spi.connector.ConnectorMetadata;
import io.prestosql.spi.connector.ConnectorTransactionHandle;
import io.prestosql.spi.transaction.IsolationLevel;
import io.prestosql.transaction.InternalConnector;
import io.prestosql.transaction.TransactionId;
import io.prestosql.transaction.TransactionInfo;
import io.prestosql.transaction.TransactionManager;
import io.prestosql.transaction.TransactionManagerConfig;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.joda.time.DateTime;

@ThreadSafe
public class InMemoryTransactionManager
implements TransactionManager {
    private static final Logger log = Logger.get(InMemoryTransactionManager.class);
    private final Duration idleTimeout;
    private final int maxFinishingConcurrency;
    private final ConcurrentMap<TransactionId, TransactionMetadata> transactions = new ConcurrentHashMap<TransactionId, TransactionMetadata>();
    private final CatalogManager catalogManager;
    private final Executor finishingExecutor;

    private InMemoryTransactionManager(Duration idleTimeout, int maxFinishingConcurrency, CatalogManager catalogManager, Executor finishingExecutor) {
        this.catalogManager = catalogManager;
        Objects.requireNonNull(idleTimeout, "idleTimeout is null");
        Preconditions.checkArgument((maxFinishingConcurrency > 0 ? 1 : 0) != 0, (Object)"maxFinishingConcurrency must be at least 1");
        Objects.requireNonNull(finishingExecutor, "finishingExecutor is null");
        this.idleTimeout = idleTimeout;
        this.maxFinishingConcurrency = maxFinishingConcurrency;
        this.finishingExecutor = finishingExecutor;
    }

    public static TransactionManager create(TransactionManagerConfig config, ScheduledExecutorService idleCheckExecutor, CatalogManager catalogManager, ExecutorService finishingExecutor) {
        InMemoryTransactionManager transactionManager = new InMemoryTransactionManager(config.getIdleTimeout(), config.getMaxFinishingConcurrency(), catalogManager, finishingExecutor);
        transactionManager.scheduleIdleChecks(config.getIdleCheckInterval(), idleCheckExecutor);
        return transactionManager;
    }

    public static TransactionManager createTestTransactionManager() {
        return InMemoryTransactionManager.createTestTransactionManager(new CatalogManager());
    }

    public static TransactionManager createTestTransactionManager(CatalogManager catalogManager) {
        return new InMemoryTransactionManager(new Duration(1.0, TimeUnit.DAYS), 1, catalogManager, MoreExecutors.directExecutor());
    }

    private void scheduleIdleChecks(Duration idleCheckInterval, ScheduledExecutorService idleCheckExecutor) {
        idleCheckExecutor.scheduleWithFixedDelay(() -> {
            try {
                this.cleanUpExpiredTransactions();
            }
            catch (Throwable t) {
                log.error(t, "Unexpected exception while cleaning up expired transactions");
            }
        }, idleCheckInterval.toMillis(), idleCheckInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    private synchronized void cleanUpExpiredTransactions() {
        Iterator iterator = this.transactions.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry entry = iterator.next();
            if (!((TransactionMetadata)entry.getValue()).isExpired(this.idleTimeout)) continue;
            iterator.remove();
            log.info("Removing expired transaction: %s", new Object[]{entry.getKey()});
            ((TransactionMetadata)entry.getValue()).asyncAbort();
        }
    }

    @Override
    public boolean transactionExists(TransactionId transactionId) {
        return this.tryGetTransactionMetadata(transactionId).isPresent();
    }

    @Override
    public TransactionInfo getTransactionInfo(TransactionId transactionId) {
        return this.getTransactionMetadata(transactionId).getTransactionInfo();
    }

    @Override
    public List<TransactionInfo> getAllTransactionInfos() {
        return (List)this.transactions.values().stream().map(TransactionMetadata::getTransactionInfo).collect(ImmutableList.toImmutableList());
    }

    @Override
    public TransactionId beginTransaction(boolean autoCommitContext) {
        return this.beginTransaction(DEFAULT_ISOLATION, false, autoCommitContext);
    }

    @Override
    public TransactionId beginTransaction(IsolationLevel isolationLevel, boolean readOnly, boolean autoCommitContext) {
        BoundedExecutor executor;
        TransactionMetadata transactionMetadata;
        TransactionId transactionId = TransactionId.create();
        Preconditions.checkState((this.transactions.put(transactionId, transactionMetadata = new TransactionMetadata(transactionId, isolationLevel, readOnly, autoCommitContext, this.catalogManager, (Executor)(executor = new BoundedExecutor(this.finishingExecutor, this.maxFinishingConcurrency)))) == null ? 1 : 0) != 0, (String)"Duplicate transaction ID: %s", (Object)transactionId);
        return transactionId;
    }

    @Override
    public Map<String, CatalogName> getCatalogNames(TransactionId transactionId) {
        return this.getTransactionMetadata(transactionId).getCatalogNames();
    }

    @Override
    public Optional<CatalogMetadata> getOptionalCatalogMetadata(TransactionId transactionId, String catalogName) {
        TransactionMetadata transactionMetadata = this.getTransactionMetadata(transactionId);
        return transactionMetadata.getConnectorId(catalogName).map(x$0 -> transactionMetadata.getTransactionCatalogMetadata(x$0));
    }

    @Override
    public CatalogMetadata getCatalogMetadata(TransactionId transactionId, CatalogName catalogName) {
        return this.getTransactionMetadata(transactionId).getTransactionCatalogMetadata(catalogName);
    }

    @Override
    public CatalogMetadata getCatalogMetadataForWrite(TransactionId transactionId, CatalogName catalogName) {
        CatalogMetadata catalogMetadata = this.getCatalogMetadata(transactionId, catalogName);
        this.checkConnectorWrite(transactionId, catalogName);
        return catalogMetadata;
    }

    @Override
    public CatalogMetadata getCatalogMetadataForWrite(TransactionId transactionId, String catalogName) {
        TransactionMetadata transactionMetadata = this.getTransactionMetadata(transactionId);
        CatalogName catalog = (CatalogName)transactionMetadata.getConnectorId(catalogName).orElseThrow(() -> new PrestoException((ErrorCodeSupplier)StandardErrorCode.NOT_FOUND, "Catalog does not exist: " + catalogName));
        return this.getCatalogMetadataForWrite(transactionId, catalog);
    }

    @Override
    public ConnectorTransactionHandle getConnectorTransaction(TransactionId transactionId, CatalogName catalogName) {
        return this.getCatalogMetadata(transactionId, catalogName).getTransactionHandleFor(catalogName);
    }

    private void checkConnectorWrite(TransactionId transactionId, CatalogName catalogName) {
        this.getTransactionMetadata(transactionId).checkConnectorWrite(catalogName);
    }

    @Override
    public void checkAndSetActive(TransactionId transactionId) {
        TransactionMetadata metadata = this.getTransactionMetadata(transactionId);
        metadata.checkOpenTransaction();
        metadata.setActive();
    }

    @Override
    public void trySetActive(TransactionId transactionId) {
        this.tryGetTransactionMetadata(transactionId).ifPresent(TransactionMetadata::setActive);
    }

    @Override
    public void trySetInactive(TransactionId transactionId) {
        this.tryGetTransactionMetadata(transactionId).ifPresent(TransactionMetadata::setInactive);
    }

    private TransactionMetadata getTransactionMetadata(TransactionId transactionId) {
        TransactionMetadata transactionMetadata = (TransactionMetadata)this.transactions.get(transactionId);
        if (transactionMetadata == null) {
            throw InMemoryTransactionManager.unknownTransactionError(transactionId);
        }
        return transactionMetadata;
    }

    private Optional<TransactionMetadata> tryGetTransactionMetadata(TransactionId transactionId) {
        return Optional.ofNullable(this.transactions.get(transactionId));
    }

    private ListenableFuture<TransactionMetadata> removeTransactionMetadataAsFuture(TransactionId transactionId) {
        TransactionMetadata transactionMetadata = (TransactionMetadata)this.transactions.remove(transactionId);
        if (transactionMetadata == null) {
            return Futures.immediateFailedFuture((Throwable)InMemoryTransactionManager.unknownTransactionError(transactionId));
        }
        return Futures.immediateFuture((Object)transactionMetadata);
    }

    private static PrestoException unknownTransactionError(TransactionId transactionId) {
        return new PrestoException((ErrorCodeSupplier)StandardErrorCode.UNKNOWN_TRANSACTION, String.format("Unknown transaction ID: %s. Possibly expired? Commands ignored until end of transaction block", transactionId));
    }

    @Override
    public ListenableFuture<?> asyncCommit(TransactionId transactionId) {
        return Futures.nonCancellationPropagating((ListenableFuture)Futures.transformAsync(this.removeTransactionMetadataAsFuture(transactionId), TransactionMetadata::asyncCommit, (Executor)MoreExecutors.directExecutor()));
    }

    @Override
    public ListenableFuture<?> asyncAbort(TransactionId transactionId) {
        return Futures.nonCancellationPropagating((ListenableFuture)Futures.transformAsync(this.removeTransactionMetadataAsFuture(transactionId), TransactionMetadata::asyncAbort, (Executor)MoreExecutors.directExecutor()));
    }

    @Override
    public void fail(TransactionId transactionId) {
        this.tryGetTransactionMetadata(transactionId).ifPresent(TransactionMetadata::asyncAbort);
    }

    @ThreadSafe
    private static class TransactionMetadata {
        private final DateTime createTime = DateTime.now();
        private final CatalogManager catalogManager;
        private final TransactionId transactionId;
        private final IsolationLevel isolationLevel;
        private final boolean readOnly;
        private final boolean autoCommitContext;
        @GuardedBy(value="this")
        private final Map<CatalogName, ConnectorTransactionMetadata> connectorIdToMetadata = new ConcurrentHashMap<CatalogName, ConnectorTransactionMetadata>();
        @GuardedBy(value="this")
        private final AtomicReference<CatalogName> writtenConnectorId = new AtomicReference();
        private final ListeningExecutorService finishingExecutor;
        private final AtomicReference<Boolean> completedSuccessfully = new AtomicReference();
        private final AtomicReference<Long> idleStartTime = new AtomicReference();
        @GuardedBy(value="this")
        private final Map<String, Optional<Catalog>> catalogByName = new ConcurrentHashMap<String, Optional<Catalog>>();
        @GuardedBy(value="this")
        private final Map<CatalogName, Catalog> catalogsByName = new ConcurrentHashMap<CatalogName, Catalog>();
        @GuardedBy(value="this")
        private final Map<CatalogName, CatalogMetadata> catalogMetadata = new ConcurrentHashMap<CatalogName, CatalogMetadata>();

        public TransactionMetadata(TransactionId transactionId, IsolationLevel isolationLevel, boolean readOnly, boolean autoCommitContext, CatalogManager catalogManager, Executor finishingExecutor) {
            this.transactionId = Objects.requireNonNull(transactionId, "transactionId is null");
            this.isolationLevel = Objects.requireNonNull(isolationLevel, "isolationLevel is null");
            this.readOnly = readOnly;
            this.autoCommitContext = autoCommitContext;
            this.catalogManager = Objects.requireNonNull(catalogManager, "catalogManager is null");
            this.finishingExecutor = MoreExecutors.listeningDecorator((ExecutorService)ExecutorServiceAdapter.from((Executor)Objects.requireNonNull(finishingExecutor, "finishingExecutor is null")));
        }

        public void setActive() {
            this.idleStartTime.set(null);
        }

        public void setInactive() {
            this.idleStartTime.set(System.nanoTime());
        }

        public boolean isExpired(Duration idleTimeout) {
            Long idleStartTime = this.idleStartTime.get();
            return idleStartTime != null && Duration.nanosSince((long)idleStartTime).compareTo(idleTimeout) > 0;
        }

        public void checkOpenTransaction() {
            Boolean completedStatus = this.completedSuccessfully.get();
            if (completedStatus != null) {
                if (completedStatus.booleanValue()) {
                    throw new IllegalStateException("Current transaction already committed");
                }
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.TRANSACTION_ALREADY_ABORTED, "Current transaction is aborted, commands ignored until end of transaction block");
            }
        }

        private synchronized Map<String, CatalogName> getCatalogNames() {
            HashMap catalogNames = new HashMap();
            this.catalogByName.values().stream().filter(Optional::isPresent).map(Optional::get).forEach(catalog -> catalogNames.put(catalog.getCatalogName(), catalog.getConnectorCatalogName()));
            this.catalogManager.getCatalogs().stream().forEach(catalog -> catalogNames.putIfAbsent(catalog.getCatalogName(), catalog.getConnectorCatalogName()));
            return ImmutableMap.copyOf(catalogNames);
        }

        private synchronized Optional<CatalogName> getConnectorId(String catalogName) {
            Optional<Catalog> catalog = this.catalogByName.get(catalogName);
            if (catalog == null) {
                catalog = this.catalogManager.getCatalog(catalogName);
                this.catalogByName.put(catalogName, catalog);
                if (catalog.isPresent()) {
                    this.registerCatalog(catalog.get());
                }
            }
            return catalog.map(Catalog::getConnectorCatalogName);
        }

        private synchronized void registerCatalog(Catalog catalog) {
            this.catalogsByName.put(catalog.getConnectorCatalogName(), catalog);
            this.catalogsByName.put(catalog.getInformationSchemaId(), catalog);
            this.catalogsByName.put(catalog.getSystemTablesId(), catalog);
        }

        private synchronized CatalogMetadata getTransactionCatalogMetadata(CatalogName catalogName) {
            this.checkOpenTransaction();
            CatalogMetadata catalogMetadata = this.catalogMetadata.get(catalogName);
            if (catalogMetadata == null) {
                Catalog catalog = this.catalogsByName.get(catalogName);
                Verify.verify((catalog != null ? 1 : 0) != 0, (String)"Unknown catalog: %s", (Object)catalogName);
                Connector connector = catalog.getConnector(catalogName);
                ConnectorTransactionMetadata metadata = this.createConnectorTransactionMetadata(catalog.getConnectorCatalogName(), catalog);
                ConnectorTransactionMetadata informationSchema = this.createConnectorTransactionMetadata(catalog.getInformationSchemaId(), catalog);
                ConnectorTransactionMetadata systemTables = this.createConnectorTransactionMetadata(catalog.getSystemTablesId(), catalog);
                catalogMetadata = new CatalogMetadata(metadata.getCatalogName(), metadata.getConnectorMetadata(), metadata.getTransactionHandle(), informationSchema.getCatalogName(), informationSchema.getConnectorMetadata(), informationSchema.getTransactionHandle(), systemTables.getCatalogName(), systemTables.getConnectorMetadata(), systemTables.getTransactionHandle(), connector.getCapabilities());
                this.catalogMetadata.put(catalog.getConnectorCatalogName(), catalogMetadata);
                this.catalogMetadata.put(catalog.getInformationSchemaId(), catalogMetadata);
                this.catalogMetadata.put(catalog.getSystemTablesId(), catalogMetadata);
            }
            return catalogMetadata;
        }

        public synchronized ConnectorTransactionMetadata createConnectorTransactionMetadata(CatalogName catalogName, Catalog catalog) {
            Connector connector = catalog.getConnector(catalogName);
            ConnectorTransactionMetadata transactionMetadata = new ConnectorTransactionMetadata(catalogName, connector, this.beginTransaction(connector));
            Preconditions.checkState((this.connectorIdToMetadata.put(catalogName, transactionMetadata) == null ? 1 : 0) != 0);
            return transactionMetadata;
        }

        private ConnectorTransactionHandle beginTransaction(Connector connector) {
            if (connector instanceof InternalConnector) {
                return ((InternalConnector)connector).beginTransaction(this.transactionId, this.isolationLevel, this.readOnly);
            }
            return connector.beginTransaction(this.isolationLevel, this.readOnly);
        }

        public synchronized void checkConnectorWrite(CatalogName catalogName) {
            this.checkOpenTransaction();
            ConnectorTransactionMetadata transactionMetadata = this.connectorIdToMetadata.get(catalogName);
            Preconditions.checkArgument((transactionMetadata != null ? 1 : 0) != 0, (Object)"Cannot record write for connector not part of transaction");
            if (this.readOnly) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.READ_ONLY_VIOLATION, "Cannot execute write in a read-only transaction");
            }
            if (!this.writtenConnectorId.compareAndSet(null, catalogName) && !this.writtenConnectorId.get().equals(catalogName)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.MULTI_CATALOG_WRITE_CONFLICT, "Multi-catalog writes not supported in a single transaction. Already wrote to catalog " + this.writtenConnectorId.get());
            }
            if (transactionMetadata.isSingleStatementWritesOnly() && !this.autoCommitContext) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.AUTOCOMMIT_WRITE_CONFLICT, "Catalog " + catalogName + " only supports writes using autocommit");
            }
        }

        public synchronized ListenableFuture<?> asyncCommit() {
            if (!this.completedSuccessfully.compareAndSet(null, true)) {
                if (this.completedSuccessfully.get().booleanValue()) {
                    return Futures.immediateFuture(null);
                }
                return Futures.immediateFailedFuture((Throwable)new PrestoException((ErrorCodeSupplier)StandardErrorCode.TRANSACTION_ALREADY_ABORTED, "Current transaction has already been aborted"));
            }
            CatalogName writeCatalogName = this.writtenConnectorId.get();
            if (writeCatalogName == null) {
                ListenableFuture future = Futures.allAsList((Iterable)this.connectorIdToMetadata.values().stream().map(transactionMetadata -> this.finishingExecutor.submit(transactionMetadata::commit)).collect(Collectors.toList()));
                MoreFutures.addExceptionCallback((ListenableFuture)future, throwable -> {
                    this.abortInternal();
                    log.error(throwable, "Read-only connector should not throw exception on commit");
                });
                return Futures.nonCancellationPropagating((ListenableFuture)future);
            }
            Supplier<ListenableFuture> commitReadOnlyConnectors = () -> {
                ListenableFuture future = Futures.allAsList((Iterable)this.connectorIdToMetadata.entrySet().stream().filter(entry -> !((CatalogName)entry.getKey()).equals(writeCatalogName)).map(Map.Entry::getValue).map(transactionMetadata -> this.finishingExecutor.submit(transactionMetadata::commit)).collect(Collectors.toList()));
                MoreFutures.addExceptionCallback((ListenableFuture)future, throwable -> log.error(throwable, "Read-only connector should not throw exception on commit"));
                return future;
            };
            ConnectorTransactionMetadata writeConnector = this.connectorIdToMetadata.get(writeCatalogName);
            ListenableFuture commitFuture = this.finishingExecutor.submit(writeConnector::commit);
            ListenableFuture readOnlyCommitFuture = Futures.transformAsync((ListenableFuture)commitFuture, ignored -> (ListenableFuture)commitReadOnlyConnectors.get(), (Executor)MoreExecutors.directExecutor());
            MoreFutures.addExceptionCallback((ListenableFuture)readOnlyCommitFuture, this::abortInternal);
            return Futures.nonCancellationPropagating((ListenableFuture)readOnlyCommitFuture);
        }

        public synchronized ListenableFuture<?> asyncAbort() {
            if (!this.completedSuccessfully.compareAndSet(null, false)) {
                if (this.completedSuccessfully.get().booleanValue()) {
                    return Futures.immediateFailedFuture((Throwable)new IllegalStateException("Current transaction already committed"));
                }
                return Futures.immediateFuture(null);
            }
            return this.abortInternal();
        }

        private synchronized ListenableFuture<?> abortInternal() {
            return Futures.nonCancellationPropagating((ListenableFuture)Futures.allAsList((Iterable)this.connectorIdToMetadata.values().stream().map(connection -> this.finishingExecutor.submit(() -> TransactionMetadata.safeAbort(connection))).collect(Collectors.toList())));
        }

        private static void safeAbort(ConnectorTransactionMetadata connection) {
            try {
                connection.abort();
            }
            catch (Exception e) {
                log.error((Throwable)e, "Connector threw exception on abort");
            }
        }

        public TransactionInfo getTransactionInfo() {
            Duration idleTime = Optional.ofNullable(this.idleStartTime.get()).map(Duration::nanosSince).orElse(new Duration(0.0, TimeUnit.MILLISECONDS));
            Optional<CatalogName> writtenConnectorId = Optional.ofNullable(this.writtenConnectorId.get());
            ImmutableList catalogNames = ImmutableList.copyOf(this.connectorIdToMetadata.keySet());
            return new TransactionInfo(this.transactionId, this.isolationLevel, this.readOnly, this.autoCommitContext, this.createTime, idleTime, (List<CatalogName>)catalogNames, writtenConnectorId);
        }

        private static class ConnectorTransactionMetadata {
            private final CatalogName catalogName;
            private final Connector connector;
            private final ConnectorTransactionHandle transactionHandle;
            private final ConnectorMetadata connectorMetadata;
            private final AtomicBoolean finished = new AtomicBoolean();

            public ConnectorTransactionMetadata(CatalogName catalogName, Connector connector, ConnectorTransactionHandle transactionHandle) {
                this.catalogName = Objects.requireNonNull(catalogName, "catalogName is null");
                this.connector = Objects.requireNonNull(connector, "connector is null");
                this.transactionHandle = Objects.requireNonNull(transactionHandle, "transactionHandle is null");
                this.connectorMetadata = connector.getMetadata(transactionHandle);
            }

            public CatalogName getCatalogName() {
                return this.catalogName;
            }

            public boolean isSingleStatementWritesOnly() {
                return this.connector.isSingleStatementWritesOnly();
            }

            public synchronized ConnectorMetadata getConnectorMetadata() {
                Preconditions.checkState((!this.finished.get() ? 1 : 0) != 0, (Object)"Already finished");
                return this.connectorMetadata;
            }

            public ConnectorTransactionHandle getTransactionHandle() {
                Preconditions.checkState((!this.finished.get() ? 1 : 0) != 0, (Object)"Already finished");
                return this.transactionHandle;
            }

            public void commit() {
                if (this.finished.compareAndSet(false, true)) {
                    this.connector.commit(this.transactionHandle);
                }
            }

            public void abort() {
                if (this.finished.compareAndSet(false, true)) {
                    this.connector.rollback(this.transactionHandle);
                }
            }
        }
    }
}

