/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.type;

import com.google.common.base.Preconditions;
import io.prestosql.spi.block.Block;
import io.prestosql.spi.connector.ConnectorSession;
import io.prestosql.spi.type.DateTimeEncoding;
import io.prestosql.spi.type.LongTimestamp;
import io.prestosql.spi.type.LongTimestampWithTimeZone;
import io.prestosql.spi.type.TimeZoneKey;
import io.prestosql.spi.type.TimestampType;
import io.prestosql.spi.type.TimestampWithTimeZoneType;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoField;
import java.util.List;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class Timestamps {
    public static final Pattern DATETIME_PATTERN = Pattern.compile("(?<year>[-+]?\\d{4,})-(?<month>\\d{1,2})-(?<day>\\d{1,2})(?: (?<hour>\\d{1,2}):(?<minute>\\d{1,2})(?::(?<second>\\d{1,2})(?:\\.(?<fraction>\\d+))?)?)?\\s*(?<timezone>.+)?");
    private static final DateTimeFormatter TIMESTAMP_FORMATTER = DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss");
    private static final long[] POWERS_OF_TEN = new long[]{1L, 10L, 100L, 1000L, 10000L, 100000L, 1000000L, 10000000L, 100000000L, 1000000000L, 10000000000L, 100000000000L, 1000000000000L};
    public static final int MILLISECONDS_PER_SECOND = 1000;
    public static final int MICROSECONDS_PER_SECOND = 1000000;
    public static final int MICROSECONDS_PER_MILLISECOND = 1000;
    public static final long PICOSECONDS_PER_SECOND = 1000000000000L;
    public static final int NANOSECONDS_PER_MILLISECOND = 1000000;
    public static final int NANOSECONDS_PER_MICROSECOND = 1000;
    public static final int PICOSECONDS_PER_MILLISECOND = 1000000000;
    public static final int PICOSECONDS_PER_MICROSECOND = 1000000;
    public static final int PICOSECONDS_PER_NANOSECOND = 1000;

    private Timestamps() {
    }

    private static long roundDiv(long value, long factor) {
        Preconditions.checkArgument((factor > 0L ? 1 : 0) != 0, (Object)"factor must be positive");
        if (factor == 1L) {
            return value;
        }
        if (value >= 0L) {
            return (value + factor / 2L) / factor;
        }
        return (value + 1L - factor / 2L) / factor;
    }

    public static long scaleEpochMicrosToMillis(long value) {
        return Math.floorDiv(value, 1000);
    }

    private static long scaleEpochMillisToSeconds(long epochMillis) {
        return Math.floorDiv(epochMillis, 1000);
    }

    private static long scaleEpochMicrosToSeconds(long epochMicros) {
        return Math.floorDiv(epochMicros, 1000000);
    }

    public static long scaleEpochMillisToMicros(long epochMillis) {
        return Math.multiplyExact(epochMillis, 1000);
    }

    public static long epochSecondToMicrosWithRounding(long epochSecond, long picoOfSecond) {
        return epochSecond * 1000000L + Timestamps.roundDiv(picoOfSecond, 1000000L);
    }

    public static int getMicrosOfSecond(long epochMicros) {
        return Math.floorMod(epochMicros, 1000000);
    }

    public static int getMillisOfSecond(long epochMillis) {
        return Math.floorMod(epochMillis, 1000);
    }

    public static int getMicrosOfMilli(long epochMicros) {
        return Math.floorMod(epochMicros, 1000);
    }

    public static long toEpochMicros(long epochMillis, int fraction) {
        return Timestamps.scaleEpochMillisToMicros(epochMillis) + (long)(fraction / 1000000);
    }

    public static long round(long value, int magnitude) {
        return Timestamps.roundToNearest(value, POWERS_OF_TEN[magnitude]);
    }

    public static long roundToNearest(long value, long bound) {
        return Timestamps.roundDiv(value, bound) * bound;
    }

    private static long scaleFactor(int fromPrecision, int toPrecision) {
        if (fromPrecision > toPrecision) {
            throw new IllegalArgumentException("fromPrecision must be <= toPrecision");
        }
        return POWERS_OF_TEN[toPrecision - fromPrecision];
    }

    public static long rescale(long value, int fromPrecision, int toPrecision) {
        if (value < 0L) {
            throw new IllegalArgumentException("value must be >= 0");
        }
        value = fromPrecision <= toPrecision ? (value *= Timestamps.scaleFactor(fromPrecision, toPrecision)) : (value /= Timestamps.scaleFactor(toPrecision, fromPrecision));
        return value;
    }

    public static boolean timestampHasTimeZone(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Invalid timestamp '%s'", value));
        }
        return matcher.group("timezone") != null;
    }

    public static int extractTimestampPrecision(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Invalid timestamp '%s'", value));
        }
        String fraction = matcher.group("fraction");
        if (fraction == null) {
            return 0;
        }
        return fraction.length();
    }

    public static LocalDateTime toLocalDateTime(TimestampType type, ConnectorSession session, Block block, int position) {
        long epochMicros;
        int precision = type.getPrecision();
        int picosOfMicro = 0;
        if (precision <= 3) {
            epochMicros = Timestamps.scaleEpochMillisToMicros(type.getLong(block, position));
        } else if (precision <= 6) {
            epochMicros = type.getLong(block, position);
        } else {
            LongTimestamp timestamp = (LongTimestamp)type.getObject(block, position);
            epochMicros = timestamp.getEpochMicros();
            picosOfMicro = timestamp.getPicosOfMicro();
        }
        long epochSecond = Timestamps.scaleEpochMicrosToSeconds(epochMicros);
        int nanoFraction = Timestamps.getMicrosOfSecond(epochMicros) * 1000 + (int)(Timestamps.roundToNearest(picosOfMicro, 1000L) / 1000L);
        Instant instant = Instant.ofEpochSecond(epochSecond, nanoFraction);
        if (session.isLegacyTimestamp()) {
            return LocalDateTime.ofInstant(instant, session.getTimeZoneKey().getZoneId());
        }
        return LocalDateTime.ofInstant(instant, ZoneOffset.UTC);
    }

    public static ZonedDateTime toZonedDateTime(TimestampWithTimeZoneType type, Block block, int position) {
        ZoneId zoneId;
        long epochMillis;
        int precision = type.getPrecision();
        int picosOfMilli = 0;
        if (precision <= 3) {
            long packedEpochMillis = type.getLong(block, position);
            epochMillis = DateTimeEncoding.unpackMillisUtc((long)packedEpochMillis);
            zoneId = DateTimeEncoding.unpackZoneKey((long)packedEpochMillis).getZoneId();
        } else {
            LongTimestampWithTimeZone timestamp = (LongTimestampWithTimeZone)type.getObject(block, position);
            epochMillis = timestamp.getEpochMillis();
            picosOfMilli = timestamp.getPicosOfMilli();
            zoneId = TimeZoneKey.getTimeZoneKey((short)timestamp.getTimeZoneKey()).getZoneId();
        }
        long epochSecond = Timestamps.scaleEpochMillisToSeconds(epochMillis);
        int nanoFraction = Timestamps.getMillisOfSecond(epochMillis) * 1000000 + (int)(Timestamps.roundToNearest(picosOfMilli, 1000L) / 1000L);
        return Instant.ofEpochSecond(epochSecond, nanoFraction).atZone(zoneId);
    }

    public static String formatTimestamp(int precision, long epochMicros, int picosOfMicro, ZoneId zoneId) {
        return Timestamps.formatTimestamp(precision, epochMicros, picosOfMicro, zoneId, TIMESTAMP_FORMATTER);
    }

    public static String formatTimestamp(int precision, long epochMicros, int picosOfMicro, ZoneId zoneId, DateTimeFormatter yearToSecondFormatter) {
        Preconditions.checkArgument((picosOfMicro >= 0 && picosOfMicro < 1000000 ? 1 : 0) != 0, (Object)"picosOfMicro is out of range [0, 1_000_000]");
        Instant instant = Instant.ofEpochSecond(Timestamps.scaleEpochMicrosToSeconds(epochMicros));
        LocalDateTime dateTime = LocalDateTime.ofInstant(instant, zoneId);
        long picoFraction = (long)Timestamps.getMicrosOfSecond(epochMicros) * 1000000L + (long)picosOfMicro;
        return Timestamps.formatTimestamp(precision, dateTime, picoFraction, yearToSecondFormatter, (StringBuilder builder) -> {});
    }

    public static String formatTimestampWithTimeZone(int precision, long epochMillis, int picoSecondOfMilli, ZoneId zoneId) {
        Instant instant = Instant.ofEpochMilli(epochMillis);
        LocalDateTime dateTime = LocalDateTime.ofInstant(instant, zoneId);
        long picoFraction = (long)Timestamps.getMillisOfSecond(epochMillis) * 1000000000L + (long)picoSecondOfMilli;
        return Timestamps.formatTimestamp(precision, dateTime, picoFraction, TIMESTAMP_FORMATTER, (StringBuilder builder) -> builder.append(" ").append(zoneId));
    }

    public static String formatTimestamp(int precision, LocalDateTime dateTime, long picoFraction, DateTimeFormatter yearToSecondFormatter, Consumer<StringBuilder> zoneIdFormatter) {
        StringBuilder builder = new StringBuilder();
        builder.append(yearToSecondFormatter.format(dateTime));
        if (precision > 0) {
            builder.append(".");
            builder.append(String.format("%0" + precision + "d", Timestamps.rescale(picoFraction, 12, precision)));
        }
        zoneIdFormatter.accept(builder);
        return builder.toString();
    }

    public static Object parseTimestamp(int precision, String value) {
        if (precision <= 6) {
            return Timestamps.parseShortTimestamp(value, ZoneOffset.UTC);
        }
        return Timestamps.parseLongTimestamp(value, ZoneOffset.UTC);
    }

    public static Object parseLegacyTimestamp(int precision, TimeZoneKey timeZoneKey, String value) {
        if (precision <= 6) {
            return Timestamps.parseShortTimestamp(value, timeZoneKey.getZoneId());
        }
        return Timestamps.parseLongTimestamp(value, timeZoneKey.getZoneId());
    }

    public static Object parseTimestampWithTimeZone(int precision, String value) {
        if (precision <= 3) {
            return Timestamps.parseShortTimestampWithTimeZone(value);
        }
        return Timestamps.parseLongTimestampWithTimeZone(value);
    }

    private static long parseShortTimestamp(String value, ZoneId zoneId) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("timezone") != null) {
            throw new IllegalArgumentException("Invalid timestamp: " + value);
        }
        String year = matcher.group("year");
        String month = matcher.group("month");
        String day = matcher.group("day");
        String hour = matcher.group("hour");
        String minute = matcher.group("minute");
        String second = matcher.group("second");
        String fraction = matcher.group("fraction");
        long epochSecond = Timestamps.toEpochSecond(year, month, day, hour, minute, second, zoneId);
        int precision = 0;
        long fractionValue = 0L;
        if (fraction != null) {
            precision = fraction.length();
            fractionValue = Long.parseLong(fraction);
        }
        if (precision <= 3) {
            return epochSecond * 1000L + Timestamps.rescale(fractionValue, precision, 3);
        }
        if (precision <= 6) {
            return epochSecond * 1000000L + Timestamps.rescale(fractionValue, precision, 6);
        }
        throw new IllegalArgumentException(String.format("Cannot parse '%s' as short timestamp. Max allowed precision = %s", value, 6));
    }

    private static LongTimestamp parseLongTimestamp(String value, ZoneId zoneId) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("timezone") != null) {
            throw new IllegalArgumentException("Invalid timestamp: " + value);
        }
        String year = matcher.group("year");
        String month = matcher.group("month");
        String day = matcher.group("day");
        String hour = matcher.group("hour");
        String minute = matcher.group("minute");
        String second = matcher.group("second");
        String fraction = matcher.group("fraction");
        if (fraction == null || fraction.length() <= 6) {
            throw new IllegalArgumentException(String.format("Cannot parse '%s' as long timestamp. Precision must be in the range [%s, %s]", value, 7, 12));
        }
        int precision = fraction.length();
        long epochSecond = Timestamps.toEpochSecond(year, month, day, hour, minute, second, zoneId);
        long picoFraction = Timestamps.rescale(Long.parseLong(fraction), precision, 12);
        return Timestamps.longTimestamp(epochSecond, picoFraction);
    }

    private static long parseShortTimestampWithTimeZone(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("timezone") == null) {
            throw new IllegalArgumentException("Invalid timestamp with time zone: " + value);
        }
        String year = matcher.group("year");
        String month = matcher.group("month");
        String day = matcher.group("day");
        String hour = matcher.group("hour");
        String minute = matcher.group("minute");
        String second = matcher.group("second");
        String fraction = matcher.group("fraction");
        String timezone = matcher.group("timezone");
        ZoneId zoneId = ZoneId.of(timezone);
        long epochSecond = Timestamps.toEpochSecond(year, month, day, hour, minute, second, zoneId);
        int precision = 0;
        long fractionValue = 0L;
        if (fraction != null) {
            precision = fraction.length();
            fractionValue = Long.parseLong(fraction);
        }
        if (precision > 6) {
            throw new IllegalArgumentException(String.format("Cannot parse '%s' as short timestamp. Max allowed precision = %s", value, 6));
        }
        long epochMillis = epochSecond * 1000L + Timestamps.rescale(fractionValue, precision, 3);
        return DateTimeEncoding.packDateTimeWithZone((long)epochMillis, (String)timezone);
    }

    private static LongTimestampWithTimeZone parseLongTimestampWithTimeZone(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("timezone") == null) {
            throw new IllegalArgumentException("Invalid timestamp: " + value);
        }
        String year = matcher.group("year");
        String month = matcher.group("month");
        String day = matcher.group("day");
        String hour = matcher.group("hour");
        String minute = matcher.group("minute");
        String second = matcher.group("second");
        String fraction = matcher.group("fraction");
        String timezone = matcher.group("timezone");
        if (fraction == null || fraction.length() <= 3) {
            throw new IllegalArgumentException(String.format("Cannot parse '%s' as long timestamp. Precision must be in the range [%s, %s]", value, 4, 12));
        }
        ZoneId zoneId = ZoneId.of(timezone);
        long epochSecond = Timestamps.toEpochSecond(year, month, day, hour, minute, second, zoneId);
        return LongTimestampWithTimeZone.fromEpochSecondsAndFraction((long)epochSecond, (long)Timestamps.rescale(Long.parseLong(fraction), fraction.length(), 12), (TimeZoneKey)TimeZoneKey.getTimeZoneKey((String)timezone));
    }

    private static long toEpochSecond(String year, String month, String day, String hour, String minute, String second, ZoneId zoneId) {
        LocalDateTime timestamp = LocalDateTime.of(Integer.parseInt(year), Integer.parseInt(month), Integer.parseInt(day), hour == null ? 0 : Integer.parseInt(hour), minute == null ? 0 : Integer.parseInt(minute), second == null ? 0 : Integer.parseInt(second), 0);
        List<ZoneOffset> offsets = zoneId.getRules().getValidOffsets(timestamp);
        if (offsets.isEmpty()) {
            throw new IllegalArgumentException("Invalid timestamp due to daylight savings transition");
        }
        return timestamp.toEpochSecond(offsets.get(0));
    }

    public static LongTimestamp longTimestamp(long precision, Instant start) {
        Preconditions.checkArgument((precision > 6L && precision <= 12L ? 1 : 0) != 0, (Object)"Precision is out of range");
        return new LongTimestamp(start.getEpochSecond() * 1000000L + start.getLong(ChronoField.MICRO_OF_SECOND), (int)Timestamps.round(start.getNano() % 1000 * 1000, (int)(12L - precision)));
    }

    public static LongTimestamp longTimestamp(long epochSecond, long fractionInPicos) {
        return new LongTimestamp(Math.multiplyExact(epochSecond, 1000000) + fractionInPicos / 1000000L, (int)(fractionInPicos % 1000000L));
    }

    public static LongTimestampWithTimeZone longTimestampWithTimeZone(long precision, Instant start, TimeZoneKey timeZoneKey) {
        Preconditions.checkArgument((precision <= 12L ? 1 : 0) != 0, (Object)"Precision is out of range");
        return LongTimestampWithTimeZone.fromEpochMillisAndFraction((long)start.toEpochMilli(), (int)((int)Timestamps.round(start.getNano() % 1000000 * 1000, (int)(12L - precision))), (TimeZoneKey)timeZoneKey);
    }

    public static LongTimestampWithTimeZone longTimestampWithTimeZone(long epochSecond, long fractionInPicos, ZoneId zoneId) {
        return LongTimestampWithTimeZone.fromEpochMillisAndFraction((long)(Math.multiplyExact(epochSecond, 1000) + fractionInPicos / 1000000000L), (int)((int)(fractionInPicos % 1000000000L)), (TimeZoneKey)TimeZoneKey.getTimeZoneKey((String)zoneId.getId()));
    }

    public static long roundToEpochMillis(LongTimestampWithTimeZone timestamp) {
        long epochMillis = timestamp.getEpochMillis();
        if (Timestamps.roundToNearest(timestamp.getPicosOfMilli(), 1000000000L) == 1000000000L) {
            ++epochMillis;
        }
        return epochMillis;
    }
}

