/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.operator.scalar;

import com.google.common.math.LongMath;
import com.google.common.primitives.Doubles;
import com.google.common.primitives.Shorts;
import com.google.common.primitives.SignedBytes;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.prestosql.metadata.Signature;
import io.prestosql.metadata.SqlScalarFunction;
import io.prestosql.operator.aggregation.TypedSet;
import io.prestosql.spi.ErrorCodeSupplier;
import io.prestosql.spi.PrestoException;
import io.prestosql.spi.StandardErrorCode;
import io.prestosql.spi.block.Block;
import io.prestosql.spi.function.Description;
import io.prestosql.spi.function.LiteralParameter;
import io.prestosql.spi.function.LiteralParameters;
import io.prestosql.spi.function.ScalarFunction;
import io.prestosql.spi.function.SqlNullable;
import io.prestosql.spi.function.SqlType;
import io.prestosql.spi.type.Decimals;
import io.prestosql.spi.type.DoubleType;
import io.prestosql.spi.type.Type;
import io.prestosql.spi.type.UnscaledDecimal128Arithmetic;
import io.prestosql.spi.type.VarcharType;
import io.prestosql.type.Constraint;
import io.prestosql.type.Constraints;
import io.prestosql.type.DecimalOperators;
import io.prestosql.util.Failures;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.commons.math3.distribution.BetaDistribution;
import org.apache.commons.math3.special.Erf;

public final class MathFunctions {
    public static final SqlScalarFunction DECIMAL_MOD_FUNCTION = MathFunctions.decimalModFunction();
    private static final Slice[] DECIMAL_HALF_UNSCALED_FOR_SCALE = new Slice[38];
    private static final Slice[] DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE = new Slice[38];

    private MathFunctions() {
    }

    @Description(value="Absolute value")
    @ScalarFunction(value="abs")
    @SqlType(value="tinyint")
    public static long absTinyint(@SqlType(value="tinyint") long num) {
        Failures.checkCondition(num != -128L, (ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Value -128 is out of range for abs(tinyint)", new Object[0]);
        return Math.abs(num);
    }

    @Description(value="Absolute value")
    @ScalarFunction(value="abs")
    @SqlType(value="smallint")
    public static long absSmallint(@SqlType(value="smallint") long num) {
        Failures.checkCondition(num != -32768L, (ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Value -32768 is out of range for abs(smallint)", new Object[0]);
        return Math.abs(num);
    }

    @Description(value="Absolute value")
    @ScalarFunction(value="abs")
    @SqlType(value="integer")
    public static long absInteger(@SqlType(value="integer") long num) {
        Failures.checkCondition(num != Integer.MIN_VALUE, (ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Value -2147483648 is out of range for abs(integer)", new Object[0]);
        return Math.abs(num);
    }

    @Description(value="Absolute value")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long abs(@SqlType(value="bigint") long num) {
        Failures.checkCondition(num != Long.MIN_VALUE, (ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Value -9223372036854775808 is out of range for abs(bigint)", new Object[0]);
        return Math.abs(num);
    }

    @Description(value="Absolute value")
    @ScalarFunction
    @SqlType(value="double")
    public static double abs(@SqlType(value="double") double num) {
        return Math.abs(num);
    }

    @Description(value="Absolute value")
    @ScalarFunction(value="abs")
    @SqlType(value="real")
    public static long absFloat(@SqlType(value="real") long num) {
        return Float.floatToRawIntBits(Math.abs(Float.intBitsToFloat((int)num)));
    }

    @Description(value="Arc cosine")
    @ScalarFunction
    @SqlType(value="double")
    public static double acos(@SqlType(value="double") double num) {
        return Math.acos(num);
    }

    @Description(value="Arc sine")
    @ScalarFunction
    @SqlType(value="double")
    public static double asin(@SqlType(value="double") double num) {
        return Math.asin(num);
    }

    @Description(value="Arc tangent")
    @ScalarFunction
    @SqlType(value="double")
    public static double atan(@SqlType(value="double") double num) {
        return Math.atan(num);
    }

    @Description(value="Arc tangent of given fraction")
    @ScalarFunction
    @SqlType(value="double")
    public static double atan2(@SqlType(value="double") double num1, @SqlType(value="double") double num2) {
        return Math.atan2(num1, num2);
    }

    @Description(value="Cube root")
    @ScalarFunction
    @SqlType(value="double")
    public static double cbrt(@SqlType(value="double") double num) {
        return Math.cbrt(num);
    }

    @Description(value="Round up to nearest integer")
    @ScalarFunction(value="ceiling", alias={"ceil"})
    @SqlType(value="tinyint")
    public static long ceilingTinyint(@SqlType(value="tinyint") long num) {
        return num;
    }

    @Description(value="Round up to nearest integer")
    @ScalarFunction(value="ceiling", alias={"ceil"})
    @SqlType(value="smallint")
    public static long ceilingSmallint(@SqlType(value="smallint") long num) {
        return num;
    }

    @Description(value="Round up to nearest integer")
    @ScalarFunction(value="ceiling", alias={"ceil"})
    @SqlType(value="integer")
    public static long ceilingInteger(@SqlType(value="integer") long num) {
        return num;
    }

    @Description(value="Round up to nearest integer")
    @ScalarFunction(alias={"ceil"})
    @SqlType(value="bigint")
    public static long ceiling(@SqlType(value="bigint") long num) {
        return num;
    }

    @Description(value="Round up to nearest integer")
    @ScalarFunction(alias={"ceil"})
    @SqlType(value="double")
    public static double ceiling(@SqlType(value="double") double num) {
        return Math.ceil(num);
    }

    @Description(value="Round up to nearest integer")
    @ScalarFunction(value="ceiling", alias={"ceil"})
    @SqlType(value="real")
    public static long ceilingFloat(@SqlType(value="real") long num) {
        return Float.floatToRawIntBits((float)MathFunctions.ceiling(Float.intBitsToFloat((int)num)));
    }

    @Description(value="Round to integer by dropping digits after decimal point")
    @ScalarFunction
    @SqlType(value="double")
    public static double truncate(@SqlType(value="double") double num) {
        return Math.signum(num) * Math.floor(Math.abs(num));
    }

    @Description(value="Round to integer by dropping digits after decimal point")
    @ScalarFunction
    @SqlType(value="real")
    public static long truncate(@SqlType(value="real") long num) {
        float numInFloat = Float.intBitsToFloat((int)num);
        return Float.floatToRawIntBits((float)((double)Math.signum(numInFloat) * Math.floor(Math.abs(numInFloat))));
    }

    @Description(value="Cosine")
    @ScalarFunction
    @SqlType(value="double")
    public static double cos(@SqlType(value="double") double num) {
        return Math.cos(num);
    }

    @Description(value="Hyperbolic cosine")
    @ScalarFunction
    @SqlType(value="double")
    public static double cosh(@SqlType(value="double") double num) {
        return Math.cosh(num);
    }

    @Description(value="Converts an angle in radians to degrees")
    @ScalarFunction
    @SqlType(value="double")
    public static double degrees(@SqlType(value="double") double radians) {
        return Math.toDegrees(radians);
    }

    @Description(value="Euler's number")
    @ScalarFunction
    @SqlType(value="double")
    public static double e() {
        return Math.E;
    }

    @Description(value="Euler's number raised to the given power")
    @ScalarFunction
    @SqlType(value="double")
    public static double exp(@SqlType(value="double") double num) {
        return Math.exp(num);
    }

    @Description(value="Round down to nearest integer")
    @ScalarFunction(value="floor")
    @SqlType(value="tinyint")
    public static long floorTinyint(@SqlType(value="tinyint") long num) {
        return num;
    }

    @Description(value="Round down to nearest integer")
    @ScalarFunction(value="floor")
    @SqlType(value="smallint")
    public static long floorSmallint(@SqlType(value="smallint") long num) {
        return num;
    }

    @Description(value="Round down to nearest integer")
    @ScalarFunction(value="floor")
    @SqlType(value="integer")
    public static long floorInteger(@SqlType(value="integer") long num) {
        return num;
    }

    @Description(value="Round down to nearest integer")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long floor(@SqlType(value="bigint") long num) {
        return num;
    }

    @Description(value="Round down to nearest integer")
    @ScalarFunction
    @SqlType(value="double")
    public static double floor(@SqlType(value="double") double num) {
        return Math.floor(num);
    }

    @Description(value="Round down to nearest integer")
    @ScalarFunction(value="floor")
    @SqlType(value="real")
    public static long floorFloat(@SqlType(value="real") long num) {
        return Float.floatToRawIntBits((float)MathFunctions.floor(Float.intBitsToFloat((int)num)));
    }

    @Description(value="Natural logarithm")
    @ScalarFunction
    @SqlType(value="double")
    public static double ln(@SqlType(value="double") double num) {
        return Math.log(num);
    }

    @Description(value="Logarithm to given base")
    @ScalarFunction
    @SqlType(value="double")
    public static double log(@SqlType(value="double") double base, @SqlType(value="double") double number) {
        return Math.log(number) / Math.log(base);
    }

    @Description(value="Logarithm to base 2")
    @ScalarFunction
    @SqlType(value="double")
    public static double log2(@SqlType(value="double") double num) {
        return Math.log(num) / Math.log(2.0);
    }

    @Description(value="Logarithm to base 10")
    @ScalarFunction
    @SqlType(value="double")
    public static double log10(@SqlType(value="double") double num) {
        return Math.log10(num);
    }

    @Description(value="Remainder of given quotient")
    @ScalarFunction(value="mod")
    @SqlType(value="tinyint")
    public static long modTinyint(@SqlType(value="tinyint") long num1, @SqlType(value="tinyint") long num2) {
        return num1 % num2;
    }

    @Description(value="Remainder of given quotient")
    @ScalarFunction(value="mod")
    @SqlType(value="smallint")
    public static long modSmallint(@SqlType(value="smallint") long num1, @SqlType(value="smallint") long num2) {
        return num1 % num2;
    }

    @Description(value="Remainder of given quotient")
    @ScalarFunction(value="mod")
    @SqlType(value="integer")
    public static long modInteger(@SqlType(value="integer") long num1, @SqlType(value="integer") long num2) {
        return num1 % num2;
    }

    @Description(value="Remainder of given quotient")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long mod(@SqlType(value="bigint") long num1, @SqlType(value="bigint") long num2) {
        return num1 % num2;
    }

    @Description(value="Remainder of given quotient")
    @ScalarFunction
    @SqlType(value="double")
    public static double mod(@SqlType(value="double") double num1, @SqlType(value="double") double num2) {
        return num1 % num2;
    }

    private static SqlScalarFunction decimalModFunction() {
        Signature signature = DecimalOperators.modulusSignatureBuilder().name("mod").build();
        return DecimalOperators.modulusScalarFunction(signature);
    }

    @Description(value="Remainder of given quotient")
    @ScalarFunction(value="mod")
    @SqlType(value="real")
    public static long modFloat(@SqlType(value="real") long num1, @SqlType(value="real") long num2) {
        return Float.floatToRawIntBits(Float.intBitsToFloat((int)num1) % Float.intBitsToFloat((int)num2));
    }

    @Description(value="The constant Pi")
    @ScalarFunction
    @SqlType(value="double")
    public static double pi() {
        return Math.PI;
    }

    @Description(value="Value raised to the power of exponent")
    @ScalarFunction(alias={"pow"})
    @SqlType(value="double")
    public static double power(@SqlType(value="double") double num, @SqlType(value="double") double exponent) {
        return Math.pow(num, exponent);
    }

    @Description(value="Converts an angle in degrees to radians")
    @ScalarFunction
    @SqlType(value="double")
    public static double radians(@SqlType(value="double") double degrees) {
        return Math.toRadians(degrees);
    }

    @Description(value="A pseudo-random value")
    @ScalarFunction(alias={"rand"}, deterministic=false)
    @SqlType(value="double")
    public static double random() {
        return ThreadLocalRandom.current().nextDouble();
    }

    @Description(value="A pseudo-random number between 0 and value (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="tinyint")
    public static long randomTinyint(@SqlType(value="tinyint") long value) {
        Failures.checkCondition(value > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bound must be positive", new Object[0]);
        return ThreadLocalRandom.current().nextInt((int)value);
    }

    @Description(value="A pseudo-random number between 0 and value (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="smallint")
    public static long randomSmallint(@SqlType(value="smallint") long value) {
        Failures.checkCondition(value > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bound must be positive", new Object[0]);
        return ThreadLocalRandom.current().nextInt((int)value);
    }

    @Description(value="A pseudo-random number between 0 and value (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="integer")
    public static long randomInteger(@SqlType(value="integer") long value) {
        Failures.checkCondition(value > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bound must be positive", new Object[0]);
        return ThreadLocalRandom.current().nextInt((int)value);
    }

    @Description(value="A pseudo-random number between 0 and value (exclusive)")
    @ScalarFunction(alias={"rand"}, deterministic=false)
    @SqlType(value="bigint")
    public static long random(@SqlType(value="bigint") long value) {
        Failures.checkCondition(value > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bound must be positive", new Object[0]);
        return ThreadLocalRandom.current().nextLong(value);
    }

    @Description(value="A pseudo-random number between start and stop (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="tinyint")
    public static long randomTinyint(@SqlType(value="tinyint") long start, @SqlType(value="tinyint") long stop) {
        Failures.checkCondition(start < stop, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "start value must be less than stop value", new Object[0]);
        return ThreadLocalRandom.current().nextLong(start, stop);
    }

    @Description(value="A pseudo-random number between start and stop (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="smallint")
    public static long randomSmallint(@SqlType(value="smallint") long start, @SqlType(value="smallint") long stop) {
        Failures.checkCondition(start < stop, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "start value must be less than stop value", new Object[0]);
        return ThreadLocalRandom.current().nextInt((int)start, (int)stop);
    }

    @Description(value="A pseudo-random number between start and stop (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="integer")
    public static long randomInteger(@SqlType(value="integer") long start, @SqlType(value="integer") long stop) {
        Failures.checkCondition(start < stop, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "start value must be less than stop value", new Object[0]);
        return ThreadLocalRandom.current().nextInt((int)start, (int)stop);
    }

    @Description(value="A pseudo-random number between start and stop (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="bigint")
    public static long random(@SqlType(value="bigint") long start, @SqlType(value="bigint") long stop) {
        Failures.checkCondition(start < stop, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "start value must be less than stop value", new Object[0]);
        return ThreadLocalRandom.current().nextLong(start, stop);
    }

    @Description(value="Inverse of normal cdf given a mean, std, and probability")
    @ScalarFunction
    @SqlType(value="double")
    public static double inverseNormalCdf(@SqlType(value="double") double mean, @SqlType(value="double") double sd, @SqlType(value="double") double p) {
        Failures.checkCondition(p > 0.0 && p < 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "p must be 0 > p > 1", new Object[0]);
        Failures.checkCondition(sd > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "sd must be > 0", new Object[0]);
        return mean + sd * 1.4142135623730951 * Erf.erfInv((double)(2.0 * p - 1.0));
    }

    @Description(value="Normal cdf given a mean, standard deviation, and value")
    @ScalarFunction
    @SqlType(value="double")
    public static double normalCdf(@SqlType(value="double") double mean, @SqlType(value="double") double standardDeviation, @SqlType(value="double") double value) {
        Failures.checkCondition(standardDeviation > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "standardDeviation must be > 0", new Object[0]);
        return 0.5 * (1.0 + Erf.erf((double)((value - mean) / (standardDeviation * Math.sqrt(2.0)))));
    }

    @Description(value="Inverse of Beta cdf given a, b parameters and probability")
    @ScalarFunction
    @SqlType(value="double")
    public static double inverseBetaCdf(@SqlType(value="double") double a, @SqlType(value="double") double b, @SqlType(value="double") double p) {
        Failures.checkCondition(p >= 0.0 && p <= 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "p must be 0 >= p >= 1", new Object[0]);
        Failures.checkCondition(a > 0.0 && b > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "a, b must be > 0", new Object[0]);
        BetaDistribution distribution = new BetaDistribution(null, a, b, 1.0E-9);
        return distribution.inverseCumulativeProbability(p);
    }

    @Description(value="Beta cdf given the a, b parameters and value")
    @ScalarFunction
    @SqlType(value="double")
    public static double betaCdf(@SqlType(value="double") double a, @SqlType(value="double") double b, @SqlType(value="double") double value) {
        Failures.checkCondition(value >= 0.0 && value <= 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "value must be 0 >= v >= 1", new Object[0]);
        Failures.checkCondition(a > 0.0 && b > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "a, b must be > 0", new Object[0]);
        BetaDistribution distribution = new BetaDistribution(null, a, b, 1.0E-9);
        return distribution.cumulativeProbability(value);
    }

    @Description(value="Round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="tinyint")
    public static long roundTinyint(@SqlType(value="tinyint") long num) {
        return num;
    }

    @Description(value="Round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="smallint")
    public static long roundSmallint(@SqlType(value="smallint") long num) {
        return num;
    }

    @Description(value="Round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="integer")
    public static long roundInteger(@SqlType(value="integer") long num) {
        return num;
    }

    @Description(value="Round to nearest integer")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long round(@SqlType(value="bigint") long num) {
        return num;
    }

    @Description(value="Round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="tinyint")
    public static long roundTinyint(@SqlType(value="tinyint") long num, @SqlType(value="integer") long decimals) {
        long rounded = MathFunctions.roundLong(num, decimals);
        try {
            return SignedBytes.checkedCast((long)rounded);
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Out of range for tinyint: " + rounded, (Throwable)e);
        }
    }

    @Description(value="Round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="smallint")
    public static long roundSmallint(@SqlType(value="smallint") long num, @SqlType(value="integer") long decimals) {
        long rounded = MathFunctions.roundLong(num, decimals);
        try {
            return Shorts.checkedCast((long)rounded);
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Out of range for smallint: " + rounded, (Throwable)e);
        }
    }

    @Description(value="Round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="integer")
    public static long roundInteger(@SqlType(value="integer") long num, @SqlType(value="integer") long decimals) {
        long rounded = MathFunctions.roundLong(num, decimals);
        try {
            return Math.toIntExact(rounded);
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Out of range for integer: " + rounded, (Throwable)e);
        }
    }

    @Description(value="Round to nearest integer")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long round(@SqlType(value="bigint") long num, @SqlType(value="integer") long decimals) {
        return MathFunctions.roundLong(num, decimals);
    }

    private static long roundLong(long num, long decimals) {
        if (decimals >= 0L) {
            return num;
        }
        try {
            long factor = LongMath.checkedPow((long)10L, (int)Math.toIntExact(-decimals));
            return Math.multiplyExact(LongMath.divide((long)num, (long)factor, (RoundingMode)RoundingMode.HALF_UP), factor);
        }
        catch (ArithmeticException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "numerical overflow: " + num, (Throwable)e);
        }
    }

    @Description(value="Round to nearest integer")
    @ScalarFunction
    @SqlType(value="double")
    public static double round(@SqlType(value="double") double num) {
        return MathFunctions.round(num, 0L);
    }

    @Description(value="Round to given number of decimal places")
    @ScalarFunction(value="round")
    @SqlType(value="real")
    public static long roundFloat(@SqlType(value="real") long num) {
        return MathFunctions.roundFloat(num, 0L);
    }

    @Description(value="Round to given number of decimal places")
    @ScalarFunction
    @SqlType(value="double")
    public static double round(@SqlType(value="double") double num, @SqlType(value="integer") long decimals) {
        if (Double.isNaN(num) || Double.isInfinite(num)) {
            return num;
        }
        double factor = Math.pow(10.0, decimals);
        if (num < 0.0) {
            return -((double)Math.round(-num * factor) / factor);
        }
        return (double)Math.round(num * factor) / factor;
    }

    @Description(value="Round to given number of decimal places")
    @ScalarFunction(value="round")
    @SqlType(value="real")
    public static long roundFloat(@SqlType(value="real") long num, @SqlType(value="integer") long decimals) {
        float numInFloat = Float.intBitsToFloat((int)num);
        if (Float.isNaN(numInFloat) || Float.isInfinite(numInFloat)) {
            return num;
        }
        double factor = Math.pow(10.0, decimals);
        if (numInFloat < 0.0f) {
            return Float.floatToRawIntBits((float)(-((double)Math.round((double)(-numInFloat) * factor) / factor)));
        }
        return Float.floatToRawIntBits((float)((double)Math.round((double)numInFloat * factor) / factor));
    }

    @ScalarFunction
    @SqlType(value="bigint")
    public static long sign(@SqlType(value="bigint") long num) {
        return (long)Math.signum(num);
    }

    @Description(value="Signum")
    @ScalarFunction(value="sign")
    @SqlType(value="integer")
    public static long signInteger(@SqlType(value="integer") long num) {
        return (long)Math.signum(num);
    }

    @Description(value="Signum")
    @ScalarFunction(value="sign")
    @SqlType(value="smallint")
    public static long signSmallint(@SqlType(value="smallint") long num) {
        return (long)Math.signum(num);
    }

    @Description(value="Signum")
    @ScalarFunction(value="sign")
    @SqlType(value="tinyint")
    public static long signTinyint(@SqlType(value="tinyint") long num) {
        return (long)Math.signum(num);
    }

    @Description(value="Signum")
    @ScalarFunction
    @SqlType(value="double")
    public static double sign(@SqlType(value="double") double num) {
        return Math.signum(num);
    }

    @Description(value="Signum")
    @ScalarFunction(value="sign")
    @SqlType(value="real")
    public static long signFloat(@SqlType(value="real") long num) {
        return Float.floatToRawIntBits(Math.signum(Float.intBitsToFloat((int)num)));
    }

    @Description(value="Sine")
    @ScalarFunction
    @SqlType(value="double")
    public static double sin(@SqlType(value="double") double num) {
        return Math.sin(num);
    }

    @Description(value="Square root")
    @ScalarFunction
    @SqlType(value="double")
    public static double sqrt(@SqlType(value="double") double num) {
        return Math.sqrt(num);
    }

    @Description(value="Tangent")
    @ScalarFunction
    @SqlType(value="double")
    public static double tan(@SqlType(value="double") double num) {
        return Math.tan(num);
    }

    @Description(value="Hyperbolic tangent")
    @ScalarFunction
    @SqlType(value="double")
    public static double tanh(@SqlType(value="double") double num) {
        return Math.tanh(num);
    }

    @Description(value="Test if value is not-a-number")
    @ScalarFunction(value="is_nan")
    @SqlType(value="boolean")
    public static boolean isNaN(@SqlType(value="double") double num) {
        return Double.isNaN(num);
    }

    @Description(value="Test if value is finite")
    @ScalarFunction
    @SqlType(value="boolean")
    public static boolean isFinite(@SqlType(value="double") double num) {
        return Doubles.isFinite((double)num);
    }

    @Description(value="Test if value is infinite")
    @ScalarFunction
    @SqlType(value="boolean")
    public static boolean isInfinite(@SqlType(value="double") double num) {
        return Double.isInfinite(num);
    }

    @Description(value="Constant representing not-a-number")
    @ScalarFunction(value="nan")
    @SqlType(value="double")
    public static double NaN() {
        return Double.NaN;
    }

    @Description(value="Infinity")
    @ScalarFunction
    @SqlType(value="double")
    public static double infinity() {
        return Double.POSITIVE_INFINITY;
    }

    @Description(value="Convert a number to a string in the given base")
    @ScalarFunction
    @SqlType(value="varchar(64)")
    public static Slice toBase(@SqlType(value="bigint") long value, @SqlType(value="bigint") long radix) {
        MathFunctions.checkRadix(radix);
        return Slices.utf8Slice((String)Long.toString(value, (int)radix));
    }

    @Description(value="Convert a string in the given base to a number")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="bigint")
    public static long fromBase(@SqlType(value="varchar(x)") Slice value, @SqlType(value="bigint") long radix) {
        MathFunctions.checkRadix(radix);
        try {
            return Long.parseLong(value.toStringUtf8(), (int)radix);
        }
        catch (NumberFormatException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Not a valid base-%d number: %s", radix, value.toStringUtf8()), (Throwable)e);
        }
    }

    private static void checkRadix(long radix) {
        Failures.checkCondition(radix >= 2L && radix <= 36L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Radix must be between %d and %d", 2, 36);
    }

    @Description(value="The bucket number of a value given a lower and upper bound and the number of buckets")
    @ScalarFunction(value="width_bucket")
    @SqlType(value="bigint")
    public static long widthBucket(@SqlType(value="double") double operand, @SqlType(value="double") double bound1, @SqlType(value="double") double bound2, @SqlType(value="bigint") long bucketCount) {
        long result;
        Failures.checkCondition(bucketCount > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bucketCount must be greater than 0", new Object[0]);
        Failures.checkCondition(!MathFunctions.isNaN(operand), (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "operand must not be NaN", new Object[0]);
        Failures.checkCondition(MathFunctions.isFinite(bound1), (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "first bound must be finite", new Object[0]);
        Failures.checkCondition(MathFunctions.isFinite(bound2), (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "second bound must be finite", new Object[0]);
        Failures.checkCondition(bound1 != bound2, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bounds cannot equal each other", new Object[0]);
        double lower = Math.min(bound1, bound2);
        double upper = Math.max(bound1, bound2);
        if (operand < lower) {
            result = 0L;
        } else if (operand >= upper) {
            try {
                result = Math.addExact(bucketCount, 1L);
            }
            catch (ArithmeticException e) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, String.format("Bucket for value %s is out of range", operand));
            }
        } else {
            result = (long)((double)bucketCount * (operand - lower) / (upper - lower) + 1.0);
        }
        if (bound1 > bound2) {
            result = bucketCount - result + 1L;
        }
        return result;
    }

    @Description(value="The bucket number of a value given an array of bins")
    @ScalarFunction(value="width_bucket")
    @SqlType(value="bigint")
    public static long widthBucket(@SqlType(value="double") double operand, @SqlType(value="array(double)") Block bins) {
        int numberOfBins = bins.getPositionCount();
        Failures.checkCondition(numberOfBins > 0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Bins cannot be an empty array", new Object[0]);
        Failures.checkCondition(!MathFunctions.isNaN(operand), (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Operand cannot be NaN", new Object[0]);
        int lower = 0;
        int upper = numberOfBins;
        while (lower < upper) {
            if (DoubleType.DOUBLE.getDouble(bins, lower) > DoubleType.DOUBLE.getDouble(bins, upper - 1)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Bin values are not sorted in ascending order");
            }
            int index = (lower + upper) / 2;
            double bin = DoubleType.DOUBLE.getDouble(bins, index);
            if (!MathFunctions.isFinite(bin)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Bin value must be finite, got " + bin);
            }
            if (operand < bin) {
                upper = index;
                continue;
            }
            lower = index + 1;
        }
        return lower;
    }

    @Description(value="Cosine similarity between the given sparse vectors")
    @ScalarFunction
    @SqlNullable
    @SqlType(value="double")
    public static Double cosineSimilarity(@SqlType(value="map(varchar,double)") Block leftMap, @SqlType(value="map(varchar,double)") Block rightMap) {
        Double normLeftMap = MathFunctions.mapL2Norm(leftMap);
        Double normRightMap = MathFunctions.mapL2Norm(rightMap);
        if (normLeftMap == null || normRightMap == null) {
            return null;
        }
        double dotProduct = MathFunctions.mapDotProduct(leftMap, rightMap);
        return dotProduct / (normLeftMap * normRightMap);
    }

    private static double mapDotProduct(Block leftMap, Block rightMap) {
        TypedSet rightMapKeys = new TypedSet((Type)VarcharType.VARCHAR, rightMap.getPositionCount(), "cosine_similarity");
        for (int i = 0; i < rightMap.getPositionCount(); i += 2) {
            rightMapKeys.add(rightMap, i);
        }
        double result = 0.0;
        for (int i = 0; i < leftMap.getPositionCount(); i += 2) {
            int position = rightMapKeys.positionOf(leftMap, i);
            if (position == -1) continue;
            result += DoubleType.DOUBLE.getDouble(leftMap, i + 1) * DoubleType.DOUBLE.getDouble(rightMap, 2 * position + 1);
        }
        return result;
    }

    private static Double mapL2Norm(Block map) {
        double norm = 0.0;
        for (int i = 1; i < map.getPositionCount(); i += 2) {
            if (map.isNull(i)) {
                return null;
            }
            norm += DoubleType.DOUBLE.getDouble(map, i) * DoubleType.DOUBLE.getDouble(map, i);
        }
        return Math.sqrt(norm);
    }

    static {
        MathFunctions.DECIMAL_HALF_UNSCALED_FOR_SCALE[0] = UnscaledDecimal128Arithmetic.unscaledDecimal((long)0L);
        MathFunctions.DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE[0] = UnscaledDecimal128Arithmetic.unscaledDecimal((long)0L);
        for (int scale = 1; scale < 38; ++scale) {
            MathFunctions.DECIMAL_HALF_UNSCALED_FOR_SCALE[scale] = UnscaledDecimal128Arithmetic.unscaledDecimal((BigInteger)BigInteger.TEN.pow(scale).divide(BigInteger.valueOf(2L)));
            MathFunctions.DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE[scale] = UnscaledDecimal128Arithmetic.unscaledDecimal((BigInteger)BigInteger.TEN.pow(scale).divide(BigInteger.valueOf(2L)).subtract(BigInteger.ONE));
        }
    }

    @Description(value="Signum")
    @ScalarFunction(value="sign")
    public static final class Sign {
        private Sign() {
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(1,0)")
        public static long signDecimalShort(@SqlType(value="decimal(p, s)") long num) {
            return (long)Math.signum(num);
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(1,0)")
        public static long signDecimalLong(@SqlType(value="decimal(p, s)") Slice num) {
            if (UnscaledDecimal128Arithmetic.isZero((Slice)num)) {
                return 0L;
            }
            if (UnscaledDecimal128Arithmetic.isNegative((Slice)num)) {
                return -1L;
            }
            return 1L;
        }
    }

    @ScalarFunction(value="truncate")
    @Description(value="Round to integer by dropping given number of digits after decimal point")
    public static final class TruncateN {
        private TruncateN() {
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(p, s)")
        public static long truncateShort(@LiteralParameter(value="p") long numPrecision, @LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num, @SqlType(value="integer") long roundScale) {
            if (num == 0L || numPrecision - numScale + roundScale <= 0L) {
                return 0L;
            }
            if (roundScale >= numScale) {
                return num;
            }
            long rescaleFactor = Decimals.longTenToNth((int)((int)(numScale - roundScale)));
            long remainder = num % rescaleFactor;
            return num - remainder;
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(p, s)")
        public static Slice truncateLong(@LiteralParameter(value="p") long numPrecision, @LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num, @SqlType(value="integer") long roundScale) {
            if (numPrecision - numScale + roundScale <= 0L) {
                return UnscaledDecimal128Arithmetic.unscaledDecimal((long)0L);
            }
            if (roundScale >= numScale) {
                return num;
            }
            int rescaleFactor = (int)(numScale - roundScale);
            return UnscaledDecimal128Arithmetic.rescaleTruncate((Slice)UnscaledDecimal128Arithmetic.rescaleTruncate((Slice)num, (int)(-rescaleFactor)), (int)rescaleFactor);
        }
    }

    @ScalarFunction(value="truncate")
    @Description(value="Round to integer by dropping digits after decimal point")
    public static final class Truncate {
        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="max(1, p - s)")
        public static long truncateShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num) {
            if (num == 0L) {
                return 0L;
            }
            if (numScale == 0L) {
                return num;
            }
            long rescaleFactor = Decimals.longTenToNth((int)((int)numScale));
            return num / rescaleFactor;
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="max(1, p - s)")
        public static Slice truncateLong(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            if (numScale == 0L) {
                return num;
            }
            return UnscaledDecimal128Arithmetic.rescaleTruncate((Slice)num, (int)(-((int)numScale)));
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="max(1, p - s)")
        public static long truncateLongShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            return UnscaledDecimal128Arithmetic.unscaledDecimalToUnscaledLong((Slice)UnscaledDecimal128Arithmetic.rescaleTruncate((Slice)num, (int)(-((int)numScale))));
        }
    }

    @ScalarFunction(value="round")
    @Description(value="Round to given number of decimal places")
    public static final class RoundN {
        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp, s)")
        @Constraint(variable="rp", expression="min(38, p + 1)")
        public static long roundNShort(@LiteralParameter(value="p") long numPrecision, @LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num, @SqlType(value="integer") long decimals) {
            if (num == 0L || numPrecision - numScale + decimals <= 0L) {
                return 0L;
            }
            if (decimals >= numScale) {
                return num;
            }
            if (num < 0L) {
                return -RoundN.roundNShort(numPrecision, numScale, -num, decimals);
            }
            long rescaleFactor = Decimals.longTenToNth((int)((int)(numScale - decimals)));
            long remainder = num % rescaleFactor;
            int roundUp = remainder >= rescaleFactor / 2L ? 1 : 0;
            return (num / rescaleFactor + (long)roundUp) * rescaleFactor;
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp, s)")
        @Constraint(variable="rp", expression="min(38, p + 1)")
        public static Slice roundNLong(@LiteralParameter(value="s") long numScale, @LiteralParameter(value="rp") long resultPrecision, @SqlType(value="decimal(p, s)") Slice num, @SqlType(value="integer") long decimals) {
            if (decimals >= numScale) {
                return num;
            }
            int rescaleFactor = (int)numScale - (int)decimals;
            try {
                Slice result = UnscaledDecimal128Arithmetic.rescale((Slice)UnscaledDecimal128Arithmetic.rescale((Slice)num, (int)(-rescaleFactor)), (int)rescaleFactor);
                UnscaledDecimal128Arithmetic.throwIfOverflows((Slice)result, (int)((int)resultPrecision));
                return result;
            }
            catch (ArithmeticException e) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "decimal overflow: " + num, (Throwable)e);
            }
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp, s)")
        @Constraint(variable="rp", expression="min(38, p + 1)")
        public static Slice roundNShortLong(@LiteralParameter(value="s") long numScale, @LiteralParameter(value="rp") long resultPrecision, @SqlType(value="decimal(p, s)") long num, @SqlType(value="integer") long decimals) {
            return RoundN.roundNLong(numScale, resultPrecision, UnscaledDecimal128Arithmetic.unscaledDecimal((long)num), decimals);
        }
    }

    @ScalarFunction(value="round")
    @Description(value="Round to nearest integer")
    public static final class Round {
        private Round() {
        }

        @LiteralParameters(value={"p", "s", "rp", "rs"})
        @SqlType(value="decimal(rp, rs)")
        @Constraints(value={@Constraint(variable="rp", expression="min(38, p - s + min(1, s))"), @Constraint(variable="rs", expression="0")})
        public static long roundShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num) {
            long remainderBoundary;
            if (num == 0L) {
                return 0L;
            }
            if (numScale == 0L) {
                return num;
            }
            if (num < 0L) {
                return -Round.roundShort(numScale, -num);
            }
            long rescaleFactor = Decimals.longTenToNth((int)((int)numScale));
            long remainder = num % rescaleFactor;
            int roundUp = remainder >= (remainderBoundary = rescaleFactor / 2L) ? 1 : 0;
            return num / rescaleFactor + (long)roundUp;
        }

        @LiteralParameters(value={"p", "s", "rp", "rs"})
        @SqlType(value="decimal(rp, rs)")
        @Constraints(value={@Constraint(variable="rp", expression="min(38, p - s + min(1, s))"), @Constraint(variable="rs", expression="0")})
        public static Slice roundLongLong(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            if (numScale == 0L) {
                return num;
            }
            return UnscaledDecimal128Arithmetic.rescale((Slice)num, (int)(-((int)numScale)));
        }

        @LiteralParameters(value={"p", "s", "rp", "rs"})
        @SqlType(value="decimal(rp, rs)")
        @Constraints(value={@Constraint(variable="rp", expression="min(38, p - s + min(1, s))"), @Constraint(variable="rs", expression="0")})
        public static long roundLongShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            return UnscaledDecimal128Arithmetic.unscaledDecimalToUnscaledLong((Slice)UnscaledDecimal128Arithmetic.rescale((Slice)num, (int)(-((int)numScale))));
        }
    }

    @ScalarFunction(value="floor")
    @Description(value="Round down to nearest integer")
    public static final class Floor {
        private Floor() {
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static long floorShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num) {
            long rescaleFactor = Decimals.longTenToNth((int)((int)numScale));
            long increment = num % rescaleFactor < 0L ? -1L : 0L;
            return num / rescaleFactor + increment;
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static Slice floorLong(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            if (UnscaledDecimal128Arithmetic.isZero((Slice)num)) {
                return num;
            }
            Slice tmp = UnscaledDecimal128Arithmetic.isNegative((Slice)num) ? UnscaledDecimal128Arithmetic.subtract((Slice)num, (Slice)DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE[(int)numScale]) : UnscaledDecimal128Arithmetic.subtract((Slice)num, (Slice)DECIMAL_HALF_UNSCALED_FOR_SCALE[(int)numScale]);
            return UnscaledDecimal128Arithmetic.rescale((Slice)tmp, (int)(-((int)numScale)));
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static long floorLongShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            return UnscaledDecimal128Arithmetic.unscaledDecimalToUnscaledLong((Slice)Floor.floorLong(numScale, num));
        }
    }

    @ScalarFunction(value="ceiling", alias={"ceil"})
    @Description(value="Round up to nearest integer")
    public static final class Ceiling {
        private Ceiling() {
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static long ceilingShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num) {
            long rescaleFactor = Decimals.longTenToNth((int)((int)numScale));
            long increment = num % rescaleFactor > 0L ? 1L : 0L;
            return num / rescaleFactor + increment;
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static Slice ceilingLong(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            Slice tmp = UnscaledDecimal128Arithmetic.isNegative((Slice)num) ? UnscaledDecimal128Arithmetic.add((Slice)num, (Slice)DECIMAL_HALF_UNSCALED_FOR_SCALE[(int)numScale]) : UnscaledDecimal128Arithmetic.add((Slice)num, (Slice)DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE[(int)numScale]);
            return UnscaledDecimal128Arithmetic.rescale((Slice)tmp, (int)(-((int)numScale)));
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static long ceilingLongShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            return UnscaledDecimal128Arithmetic.unscaledDecimalToUnscaledLong((Slice)Ceiling.ceilingLong(numScale, num));
        }
    }

    @ScalarFunction(value="abs")
    @Description(value="Absolute value")
    public static final class Abs {
        private Abs() {
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(p, s)")
        public static long absShort(@SqlType(value="decimal(p, s)") long arg) {
            return arg > 0L ? arg : -arg;
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(p, s)")
        public static Slice absLong(@SqlType(value="decimal(p, s)") Slice arg) {
            if (UnscaledDecimal128Arithmetic.isNegative((Slice)arg)) {
                Slice result = UnscaledDecimal128Arithmetic.unscaledDecimal((Slice)arg);
                UnscaledDecimal128Arithmetic.negate((Slice)result);
                return result;
            }
            return arg;
        }
    }
}

