/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.plugin.raptor.legacy.storage;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.airlift.concurrent.MoreFutures;
import io.airlift.concurrent.Threads;
import io.airlift.json.JsonCodec;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.airlift.slice.XxHash64;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.prestosql.memory.context.AggregatedMemoryContext;
import io.prestosql.orc.FileOrcDataSource;
import io.prestosql.orc.OrcDataSource;
import io.prestosql.orc.OrcPredicate;
import io.prestosql.orc.OrcReader;
import io.prestosql.orc.OrcRecordReader;
import io.prestosql.orc.TupleDomainOrcPredicate;
import io.prestosql.orc.metadata.OrcType;
import io.prestosql.plugin.raptor.legacy.RaptorColumnHandle;
import io.prestosql.plugin.raptor.legacy.RaptorConnectorId;
import io.prestosql.plugin.raptor.legacy.RaptorErrorCode;
import io.prestosql.plugin.raptor.legacy.backup.BackupManager;
import io.prestosql.plugin.raptor.legacy.backup.BackupStore;
import io.prestosql.plugin.raptor.legacy.metadata.ColumnInfo;
import io.prestosql.plugin.raptor.legacy.metadata.ColumnStats;
import io.prestosql.plugin.raptor.legacy.metadata.ShardDelta;
import io.prestosql.plugin.raptor.legacy.metadata.ShardInfo;
import io.prestosql.plugin.raptor.legacy.metadata.ShardRecorder;
import io.prestosql.plugin.raptor.legacy.storage.OrcFileMetadata;
import io.prestosql.plugin.raptor.legacy.storage.OrcFileRewriter;
import io.prestosql.plugin.raptor.legacy.storage.OrcFileWriter;
import io.prestosql.plugin.raptor.legacy.storage.OrcPageSource;
import io.prestosql.plugin.raptor.legacy.storage.ReaderAttributes;
import io.prestosql.plugin.raptor.legacy.storage.Row;
import io.prestosql.plugin.raptor.legacy.storage.ShardRecoveryManager;
import io.prestosql.plugin.raptor.legacy.storage.ShardRewriter;
import io.prestosql.plugin.raptor.legacy.storage.ShardStats;
import io.prestosql.plugin.raptor.legacy.storage.StorageManager;
import io.prestosql.plugin.raptor.legacy.storage.StorageManagerConfig;
import io.prestosql.plugin.raptor.legacy.storage.StoragePageSink;
import io.prestosql.plugin.raptor.legacy.storage.StorageService;
import io.prestosql.spi.ErrorCodeSupplier;
import io.prestosql.spi.NodeManager;
import io.prestosql.spi.Page;
import io.prestosql.spi.PrestoException;
import io.prestosql.spi.connector.ConnectorPageSource;
import io.prestosql.spi.predicate.TupleDomain;
import io.prestosql.spi.type.ArrayType;
import io.prestosql.spi.type.BigintType;
import io.prestosql.spi.type.BooleanType;
import io.prestosql.spi.type.CharType;
import io.prestosql.spi.type.DecimalType;
import io.prestosql.spi.type.DoubleType;
import io.prestosql.spi.type.MapType;
import io.prestosql.spi.type.NamedTypeSignature;
import io.prestosql.spi.type.RowFieldName;
import io.prestosql.spi.type.RowType;
import io.prestosql.spi.type.TimestampType;
import io.prestosql.spi.type.Type;
import io.prestosql.spi.type.TypeManager;
import io.prestosql.spi.type.TypeSignature;
import io.prestosql.spi.type.TypeSignatureParameter;
import io.prestosql.spi.type.VarbinaryType;
import io.prestosql.spi.type.VarcharType;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.joda.time.DateTimeZone;

public class OrcStorageManager
implements StorageManager {
    private static final JsonCodec<ShardDelta> SHARD_DELTA_CODEC = JsonCodec.jsonCodec(ShardDelta.class);
    private static final long MAX_ROWS = 1000000000L;
    private static final DataSize HUGE_MAX_READ_BLOCK_SIZE = new DataSize(1.0, DataSize.Unit.PETABYTE);
    private static final JsonCodec<OrcFileMetadata> METADATA_CODEC = JsonCodec.jsonCodec(OrcFileMetadata.class);
    private final String nodeId;
    private final StorageService storageService;
    private final Optional<BackupStore> backupStore;
    private final ReaderAttributes defaultReaderAttributes;
    private final BackupManager backupManager;
    private final ShardRecoveryManager recoveryManager;
    private final ShardRecorder shardRecorder;
    private final Duration recoveryTimeout;
    private final long maxShardRows;
    private final DataSize maxShardSize;
    private final DataSize minAvailableSpace;
    private final TypeManager typeManager;
    private final ExecutorService deletionExecutor;
    private final ExecutorService commitExecutor;

    @Inject
    public OrcStorageManager(NodeManager nodeManager, StorageService storageService, Optional<BackupStore> backupStore, ReaderAttributes readerAttributes, StorageManagerConfig config, RaptorConnectorId connectorId, BackupManager backgroundBackupManager, ShardRecoveryManager recoveryManager, ShardRecorder shardRecorder, TypeManager typeManager) {
        this(nodeManager.getCurrentNode().getNodeIdentifier(), storageService, backupStore, readerAttributes, backgroundBackupManager, recoveryManager, shardRecorder, typeManager, connectorId.toString(), config.getDeletionThreads(), config.getShardRecoveryTimeout(), config.getMaxShardRows(), config.getMaxShardSize(), config.getMinAvailableSpace());
    }

    public OrcStorageManager(String nodeId, StorageService storageService, Optional<BackupStore> backupStore, ReaderAttributes readerAttributes, BackupManager backgroundBackupManager, ShardRecoveryManager recoveryManager, ShardRecorder shardRecorder, TypeManager typeManager, String connectorId, int deletionThreads, Duration shardRecoveryTimeout, long maxShardRows, DataSize maxShardSize, DataSize minAvailableSpace) {
        this.nodeId = Objects.requireNonNull(nodeId, "nodeId is null");
        this.storageService = Objects.requireNonNull(storageService, "storageService is null");
        this.backupStore = Objects.requireNonNull(backupStore, "backupStore is null");
        this.defaultReaderAttributes = Objects.requireNonNull(readerAttributes, "readerAttributes is null");
        this.backupManager = Objects.requireNonNull(backgroundBackupManager, "backgroundBackupManager is null");
        this.recoveryManager = Objects.requireNonNull(recoveryManager, "recoveryManager is null");
        this.recoveryTimeout = Objects.requireNonNull(shardRecoveryTimeout, "shardRecoveryTimeout is null");
        Preconditions.checkArgument((maxShardRows > 0L ? 1 : 0) != 0, (Object)"maxShardRows must be > 0");
        this.maxShardRows = Math.min(maxShardRows, 1000000000L);
        this.maxShardSize = Objects.requireNonNull(maxShardSize, "maxShardSize is null");
        this.minAvailableSpace = Objects.requireNonNull(minAvailableSpace, "minAvailableSpace is null");
        this.shardRecorder = Objects.requireNonNull(shardRecorder, "shardRecorder is null");
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.deletionExecutor = Executors.newFixedThreadPool(deletionThreads, Threads.daemonThreadsNamed((String)("raptor-delete-" + connectorId + "-%s")));
        this.commitExecutor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)("raptor-commit-" + connectorId + "-%s")));
    }

    @PreDestroy
    public void shutdown() {
        this.deletionExecutor.shutdownNow();
        this.commitExecutor.shutdown();
    }

    @Override
    public ConnectorPageSource getPageSource(UUID shardUuid, OptionalInt bucketNumber, List<Long> columnIds, List<Type> columnTypes, TupleDomain<RaptorColumnHandle> effectivePredicate, ReaderAttributes readerAttributes, OptionalLong transactionId) {
        OrcDataSource dataSource = this.openShard(shardUuid, readerAttributes);
        AggregatedMemoryContext systemMemoryUsage = AggregatedMemoryContext.newSimpleAggregatedMemoryContext();
        try {
            OrcReader reader = new OrcReader(dataSource, readerAttributes.getMaxMergeDistance(), readerAttributes.getTinyStripeThreshold(), HUGE_MAX_READ_BLOCK_SIZE);
            Map<Long, Integer> indexMap = OrcStorageManager.columnIdIndex(reader.getColumnNames());
            ImmutableMap.Builder includedColumns = ImmutableMap.builder();
            ImmutableList.Builder columnIndexes = ImmutableList.builder();
            for (int i = 0; i < columnIds.size(); ++i) {
                long columnId = columnIds.get(i);
                if (RaptorColumnHandle.isHiddenColumn(columnId)) {
                    columnIndexes.add((Object)OrcStorageManager.toSpecialIndex(columnId));
                    continue;
                }
                Integer index = indexMap.get(columnId);
                if (index == null) {
                    columnIndexes.add((Object)-1);
                    continue;
                }
                columnIndexes.add((Object)index);
                includedColumns.put((Object)index, (Object)OrcStorageManager.toOrcFileType(columnTypes.get(i), this.typeManager));
            }
            OrcPredicate predicate = OrcStorageManager.getPredicate(effectivePredicate, indexMap);
            OrcRecordReader recordReader = reader.createRecordReader((Map)includedColumns.build(), predicate, DateTimeZone.UTC, systemMemoryUsage, 1);
            Optional<ShardRewriter> shardRewriter = Optional.empty();
            if (transactionId.isPresent()) {
                shardRewriter = Optional.of(this.createShardRewriter(transactionId.getAsLong(), bucketNumber, shardUuid));
            }
            return new OrcPageSource(shardRewriter, recordReader, dataSource, columnIds, columnTypes, (List<Integer>)columnIndexes.build(), shardUuid, bucketNumber, systemMemoryUsage);
        }
        catch (IOException | RuntimeException e) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to create page source for shard " + shardUuid, (Throwable)e);
        }
        finally {
            OrcStorageManager.closeQuietly((Closeable)dataSource);
        }
    }

    private static int toSpecialIndex(long columnId) {
        if (RaptorColumnHandle.isShardRowIdColumn(columnId)) {
            return -2;
        }
        if (RaptorColumnHandle.isShardUuidColumn(columnId)) {
            return -3;
        }
        if (RaptorColumnHandle.isBucketNumberColumn(columnId)) {
            return -4;
        }
        throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Invalid column ID: " + columnId);
    }

    @Override
    public StoragePageSink createStoragePageSink(long transactionId, OptionalInt bucketNumber, List<Long> columnIds, List<Type> columnTypes, boolean checkSpace) {
        if (checkSpace && this.storageService.getAvailableBytes() < this.minAvailableSpace.toBytes()) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_LOCAL_DISK_FULL, "Local disk is full on node " + this.nodeId);
        }
        return new OrcStoragePageSink(transactionId, columnIds, columnTypes, bucketNumber);
    }

    private ShardRewriter createShardRewriter(long transactionId, OptionalInt bucketNumber, UUID shardUuid) {
        return rowsToDelete -> {
            if (rowsToDelete.isEmpty()) {
                return CompletableFuture.completedFuture(ImmutableList.of());
            }
            return CompletableFuture.supplyAsync(() -> this.rewriteShard(transactionId, bucketNumber, shardUuid, rowsToDelete), this.deletionExecutor);
        };
    }

    private void writeShard(UUID shardUuid) {
        if (this.backupStore.isPresent() && !this.backupStore.get().shardExists(shardUuid)) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Backup does not exist after write");
        }
        File stagingFile = this.storageService.getStagingFile(shardUuid);
        File storageFile = this.storageService.getStorageFile(shardUuid);
        this.storageService.createParents(storageFile);
        try {
            Files.move(stagingFile.toPath(), storageFile.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to move shard file", (Throwable)e);
        }
    }

    @VisibleForTesting
    OrcDataSource openShard(UUID shardUuid, ReaderAttributes readerAttributes) {
        File file = this.storageService.getStorageFile(shardUuid).getAbsoluteFile();
        if (!file.exists() && this.backupStore.isPresent()) {
            try {
                Future<?> future = this.recoveryManager.recoverShard(shardUuid);
                future.get(this.recoveryTimeout.toMillis(), TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
            catch (ExecutionException e) {
                if (e.getCause() != null) {
                    Throwables.throwIfInstanceOf((Throwable)e.getCause(), PrestoException.class);
                }
                throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_RECOVERY_ERROR, "Error recovering shard " + shardUuid, e.getCause());
            }
            catch (TimeoutException e) {
                throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_RECOVERY_TIMEOUT, "Shard is being recovered from backup. Please retry in a few minutes: " + shardUuid);
            }
        }
        try {
            return OrcStorageManager.fileOrcDataSource(readerAttributes, file);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to open shard file: " + file, (Throwable)e);
        }
    }

    private static FileOrcDataSource fileOrcDataSource(ReaderAttributes readerAttributes, File file) throws FileNotFoundException {
        return new FileOrcDataSource(file, readerAttributes.getMaxMergeDistance(), readerAttributes.getMaxReadSize(), readerAttributes.getStreamBufferSize(), readerAttributes.isLazyReadSmallRanges());
    }

    private ShardInfo createShardInfo(UUID shardUuid, OptionalInt bucketNumber, File file, Set<String> nodes, long rowCount, long uncompressedSize) {
        return new ShardInfo(shardUuid, bucketNumber, nodes, this.computeShardStats(file), rowCount, file.length(), uncompressedSize, OrcStorageManager.xxhash64(file));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private List<ColumnStats> computeShardStats(File file) {
        try (FileOrcDataSource dataSource = OrcStorageManager.fileOrcDataSource(this.defaultReaderAttributes, file);){
            OrcReader reader = new OrcReader((OrcDataSource)dataSource, this.defaultReaderAttributes.getMaxMergeDistance(), this.defaultReaderAttributes.getTinyStripeThreshold(), HUGE_MAX_READ_BLOCK_SIZE);
            ImmutableList.Builder list = ImmutableList.builder();
            for (ColumnInfo info : this.getColumnInfo(reader)) {
                ShardStats.computeColumnStats(reader, info.getColumnId(), info.getType(), this.typeManager).ifPresent(arg_0 -> ((ImmutableList.Builder)list).add(arg_0));
            }
            ImmutableList immutableList = list.build();
            return immutableList;
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to read file: " + file, (Throwable)e);
        }
    }

    @VisibleForTesting
    Collection<Slice> rewriteShard(long transactionId, OptionalInt bucketNumber, UUID shardUuid, BitSet rowsToDelete) {
        File output;
        if (rowsToDelete.isEmpty()) {
            return ImmutableList.of();
        }
        UUID newShardUuid = UUID.randomUUID();
        File input = this.storageService.getStorageFile(shardUuid);
        OrcFileRewriter.OrcFileInfo info = OrcStorageManager.rewriteFile(input, output = this.storageService.getStagingFile(newShardUuid), rowsToDelete);
        long rowCount = info.getRowCount();
        if (rowCount == 0L) {
            return OrcStorageManager.shardDelta(shardUuid, Optional.empty());
        }
        this.shardRecorder.recordCreatedShard(transactionId, newShardUuid);
        MoreFutures.getFutureValue(this.backupManager.submit(newShardUuid, output));
        ImmutableSet nodes = ImmutableSet.of((Object)this.nodeId);
        long uncompressedSize = info.getUncompressedSize();
        ShardInfo shard = this.createShardInfo(newShardUuid, bucketNumber, output, (Set<String>)nodes, rowCount, uncompressedSize);
        this.writeShard(newShardUuid);
        return OrcStorageManager.shardDelta(shardUuid, Optional.of(shard));
    }

    private static Collection<Slice> shardDelta(UUID oldShardUuid, Optional<ShardInfo> shardInfo) {
        List newShards = (List)shardInfo.map(ImmutableList::of).orElse(ImmutableList.of());
        ShardDelta delta = new ShardDelta((List<UUID>)ImmutableList.of((Object)oldShardUuid), newShards);
        return ImmutableList.of((Object)Slices.wrappedBuffer((byte[])SHARD_DELTA_CODEC.toJsonBytes((Object)delta)));
    }

    private static OrcFileRewriter.OrcFileInfo rewriteFile(File input, File output, BitSet rowsToDelete) {
        try {
            return OrcFileRewriter.rewrite(input, output, rowsToDelete);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to rewrite shard file: " + input, (Throwable)e);
        }
    }

    private List<ColumnInfo> getColumnInfo(OrcReader reader) {
        Optional<OrcFileMetadata> metadata = OrcStorageManager.getOrcFileMetadata(reader);
        if (metadata.isPresent()) {
            return this.getColumnInfoFromOrcUserMetadata(metadata.get());
        }
        return this.getColumnInfoFromOrcColumnTypes(reader.getColumnNames(), reader.getFooter().getTypes());
    }

    private List<ColumnInfo> getColumnInfoFromOrcColumnTypes(List<String> orcColumnNames, List<OrcType> orcColumnTypes) {
        Type rowType = this.getType(orcColumnTypes, 0);
        if (orcColumnNames.size() != rowType.getTypeParameters().size()) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Column names and types do not match");
        }
        ImmutableList.Builder list = ImmutableList.builder();
        for (int i = 0; i < orcColumnNames.size(); ++i) {
            list.add((Object)new ColumnInfo(Long.parseLong(orcColumnNames.get(i)), (Type)rowType.getTypeParameters().get(i)));
        }
        return list.build();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    static long xxhash64(File file) {
        try (FileInputStream in = new FileInputStream(file);){
            long l = XxHash64.hash((InputStream)in);
            return l;
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed to read file: " + file, (Throwable)e);
        }
    }

    private static Optional<OrcFileMetadata> getOrcFileMetadata(OrcReader reader) {
        return Optional.ofNullable(reader.getFooter().getUserMetadata().get("metadata")).map(slice -> (OrcFileMetadata)METADATA_CODEC.fromJson(slice.getBytes()));
    }

    private List<ColumnInfo> getColumnInfoFromOrcUserMetadata(OrcFileMetadata orcFileMetadata) {
        return orcFileMetadata.getColumnTypes().entrySet().stream().sorted(Map.Entry.comparingByKey()).map(entry -> new ColumnInfo((Long)entry.getKey(), this.typeManager.getType((TypeSignature)entry.getValue()))).collect(Collectors.toList());
    }

    private Type getType(List<OrcType> types, int index) {
        OrcType type = types.get(index);
        switch (type.getOrcTypeKind()) {
            case BOOLEAN: {
                return BooleanType.BOOLEAN;
            }
            case LONG: {
                return BigintType.BIGINT;
            }
            case DOUBLE: {
                return DoubleType.DOUBLE;
            }
            case STRING: {
                return VarcharType.createUnboundedVarcharType();
            }
            case VARCHAR: {
                return VarcharType.createVarcharType((int)((Integer)type.getLength().get()));
            }
            case CHAR: {
                return CharType.createCharType((long)((Integer)type.getLength().get()).intValue());
            }
            case BINARY: {
                return VarbinaryType.VARBINARY;
            }
            case DECIMAL: {
                return DecimalType.createDecimalType((int)((Integer)type.getPrecision().get()), (int)((Integer)type.getScale().get()));
            }
            case LIST: {
                TypeSignature elementType = this.getType(types, type.getFieldTypeIndex(0)).getTypeSignature();
                return this.typeManager.getParameterizedType("array", (List)ImmutableList.of((Object)TypeSignatureParameter.of((TypeSignature)elementType)));
            }
            case MAP: {
                TypeSignature keyType = this.getType(types, type.getFieldTypeIndex(0)).getTypeSignature();
                TypeSignature valueType = this.getType(types, type.getFieldTypeIndex(1)).getTypeSignature();
                return this.typeManager.getParameterizedType("map", (List)ImmutableList.of((Object)TypeSignatureParameter.of((TypeSignature)keyType), (Object)TypeSignatureParameter.of((TypeSignature)valueType)));
            }
            case STRUCT: {
                List fieldNames = type.getFieldNames();
                ImmutableList.Builder fieldTypes = ImmutableList.builder();
                for (int i = 0; i < type.getFieldCount(); ++i) {
                    fieldTypes.add((Object)TypeSignatureParameter.of((NamedTypeSignature)new NamedTypeSignature(Optional.of(new RowFieldName((String)fieldNames.get(i), false)), this.getType(types, type.getFieldTypeIndex(i)).getTypeSignature())));
                }
                return this.typeManager.getParameterizedType("row", (List)fieldTypes.build());
            }
        }
        throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Unhandled ORC type: " + type);
    }

    static Type toOrcFileType(Type raptorType, TypeManager typeManager) {
        if (raptorType == TimestampType.TIMESTAMP) {
            return BigintType.BIGINT;
        }
        if (raptorType instanceof ArrayType) {
            Type elementType = OrcStorageManager.toOrcFileType(((ArrayType)raptorType).getElementType(), typeManager);
            return new ArrayType(elementType);
        }
        if (raptorType instanceof MapType) {
            TypeSignature keyType = OrcStorageManager.toOrcFileType(((MapType)raptorType).getKeyType(), typeManager).getTypeSignature();
            TypeSignature valueType = OrcStorageManager.toOrcFileType(((MapType)raptorType).getValueType(), typeManager).getTypeSignature();
            return typeManager.getParameterizedType("map", (List)ImmutableList.of((Object)TypeSignatureParameter.of((TypeSignature)keyType), (Object)TypeSignatureParameter.of((TypeSignature)valueType)));
        }
        if (raptorType instanceof RowType) {
            List fields = (List)((RowType)raptorType).getFields().stream().map(field -> new RowType.Field(field.getName(), OrcStorageManager.toOrcFileType(field.getType(), typeManager))).collect(ImmutableList.toImmutableList());
            return RowType.from((List)fields);
        }
        return raptorType;
    }

    private static OrcPredicate getPredicate(TupleDomain<RaptorColumnHandle> effectivePredicate, Map<Long, Integer> indexMap) {
        ImmutableList.Builder columns = ImmutableList.builder();
        for (RaptorColumnHandle column : ((Map)effectivePredicate.getDomains().get()).keySet()) {
            Integer index = indexMap.get(column.getColumnId());
            if (index == null) continue;
            columns.add((Object)new TupleDomainOrcPredicate.ColumnReference((Object)column, index.intValue(), column.getColumnType()));
        }
        return new TupleDomainOrcPredicate(effectivePredicate, (List)columns.build(), false);
    }

    private static Map<Long, Integer> columnIdIndex(List<String> columnNames) {
        ImmutableMap.Builder map = ImmutableMap.builder();
        for (int i = 0; i < columnNames.size(); ++i) {
            map.put((Object)Long.valueOf(columnNames.get(i)), (Object)i);
        }
        return map.build();
    }

    private static void closeQuietly(Closeable closeable) {
        try {
            closeable.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private class OrcStoragePageSink
    implements StoragePageSink {
        private final long transactionId;
        private final List<Long> columnIds;
        private final List<Type> columnTypes;
        private final OptionalInt bucketNumber;
        private final List<File> stagingFiles = new ArrayList<File>();
        private final List<ShardInfo> shards = new ArrayList<ShardInfo>();
        private final List<CompletableFuture<?>> futures = new ArrayList();
        private boolean committed;
        private OrcFileWriter writer;
        private UUID shardUuid;

        public OrcStoragePageSink(long transactionId, List<Long> columnIds, List<Type> columnTypes, OptionalInt bucketNumber) {
            this.transactionId = transactionId;
            this.columnIds = ImmutableList.copyOf((Collection)Objects.requireNonNull(columnIds, "columnIds is null"));
            this.columnTypes = ImmutableList.copyOf((Collection)Objects.requireNonNull(columnTypes, "columnTypes is null"));
            this.bucketNumber = Objects.requireNonNull(bucketNumber, "bucketNumber is null");
        }

        @Override
        public void appendPages(List<Page> pages) {
            this.createWriterIfNecessary();
            this.writer.appendPages(pages);
        }

        @Override
        public void appendPages(List<Page> inputPages, int[] pageIndexes, int[] positionIndexes) {
            this.createWriterIfNecessary();
            this.writer.appendPages(inputPages, pageIndexes, positionIndexes);
        }

        @Override
        public void appendRow(Row row) {
            this.createWriterIfNecessary();
            this.writer.appendRow(row);
        }

        @Override
        public boolean isFull() {
            if (this.writer == null) {
                return false;
            }
            return this.writer.getRowCount() >= OrcStorageManager.this.maxShardRows || this.writer.getUncompressedSize() >= OrcStorageManager.this.maxShardSize.toBytes();
        }

        @Override
        public void flush() {
            if (this.writer != null) {
                this.writer.close();
                OrcStorageManager.this.shardRecorder.recordCreatedShard(this.transactionId, this.shardUuid);
                File stagingFile = OrcStorageManager.this.storageService.getStagingFile(this.shardUuid);
                this.futures.add(OrcStorageManager.this.backupManager.submit(this.shardUuid, stagingFile));
                ImmutableSet nodes = ImmutableSet.of((Object)OrcStorageManager.this.nodeId);
                long rowCount = this.writer.getRowCount();
                long uncompressedSize = this.writer.getUncompressedSize();
                this.shards.add(OrcStorageManager.this.createShardInfo(this.shardUuid, this.bucketNumber, stagingFile, (Set)nodes, rowCount, uncompressedSize));
                this.writer = null;
                this.shardUuid = null;
            }
        }

        @Override
        public CompletableFuture<List<ShardInfo>> commit() {
            Preconditions.checkState((!this.committed ? 1 : 0) != 0, (Object)"already committed");
            this.committed = true;
            this.flush();
            return MoreFutures.allAsList(this.futures).thenApplyAsync(ignored -> {
                for (ShardInfo shard : this.shards) {
                    OrcStorageManager.this.writeShard(shard.getShardUuid());
                }
                return ImmutableList.copyOf(this.shards);
            }, (Executor)OrcStorageManager.this.commitExecutor);
        }

        @Override
        public void rollback() {
            try {
                if (this.writer != null) {
                    this.writer.close();
                    this.writer = null;
                }
            }
            finally {
                for (File file : this.stagingFiles) {
                    file.delete();
                }
                this.futures.forEach(future -> future.cancel(true));
                OrcStorageManager.this.backupStore.ifPresent(backupStore -> {
                    for (ShardInfo shard : this.shards) {
                        backupStore.deleteShard(shard.getShardUuid());
                    }
                });
            }
        }

        private void createWriterIfNecessary() {
            if (this.writer == null) {
                this.shardUuid = UUID.randomUUID();
                File stagingFile = OrcStorageManager.this.storageService.getStagingFile(this.shardUuid);
                OrcStorageManager.this.storageService.createParents(stagingFile);
                this.stagingFiles.add(stagingFile);
                this.writer = new OrcFileWriter(this.columnIds, this.columnTypes, stagingFile);
            }
        }
    }
}

