/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.plugin.raptor.legacy.storage;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import io.airlift.concurrent.MoreFutures;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.prestosql.plugin.raptor.legacy.RaptorErrorCode;
import io.prestosql.plugin.raptor.legacy.backup.BackupStore;
import io.prestosql.plugin.raptor.legacy.metadata.ShardManager;
import io.prestosql.plugin.raptor.legacy.metadata.ShardMetadata;
import io.prestosql.plugin.raptor.legacy.storage.OrcStorageManager;
import io.prestosql.plugin.raptor.legacy.storage.ShardRecoveryStats;
import io.prestosql.plugin.raptor.legacy.storage.StorageManagerConfig;
import io.prestosql.plugin.raptor.legacy.storage.StorageService;
import io.prestosql.plugin.raptor.legacy.util.PrioritizedFifoExecutor;
import io.prestosql.spi.ErrorCodeSupplier;
import io.prestosql.spi.NodeManager;
import io.prestosql.spi.PrestoException;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Comparator;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.weakref.jmx.Flatten;
import org.weakref.jmx.Managed;

public class ShardRecoveryManager {
    private static final Logger log = Logger.get(ShardRecoveryManager.class);
    private final StorageService storageService;
    private final Optional<BackupStore> backupStore;
    private final String nodeIdentifier;
    private final ShardManager shardManager;
    private final Duration missingShardDiscoveryInterval;
    private final AtomicBoolean started = new AtomicBoolean();
    private final MissingShardsQueue shardQueue;
    private final ScheduledExecutorService missingShardExecutor = Executors.newScheduledThreadPool(1, Threads.daemonThreadsNamed((String)"missing-shard-discovery"));
    private final ExecutorService executorService = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)"shard-recovery-%s"));
    private final ShardRecoveryStats stats;

    @Inject
    public ShardRecoveryManager(StorageService storageService, Optional<BackupStore> backupStore, NodeManager nodeManager, ShardManager shardManager, StorageManagerConfig config) {
        this(storageService, backupStore, nodeManager, shardManager, config.getMissingShardDiscoveryInterval(), config.getRecoveryThreads());
    }

    public ShardRecoveryManager(StorageService storageService, Optional<BackupStore> backupStore, NodeManager nodeManager, ShardManager shardManager, Duration missingShardDiscoveryInterval, int recoveryThreads) {
        this.storageService = Objects.requireNonNull(storageService, "storageService is null");
        this.backupStore = Objects.requireNonNull(backupStore, "backupStore is null");
        this.nodeIdentifier = Objects.requireNonNull(nodeManager, "nodeManager is null").getCurrentNode().getNodeIdentifier();
        this.shardManager = Objects.requireNonNull(shardManager, "shardManager is null");
        this.missingShardDiscoveryInterval = Objects.requireNonNull(missingShardDiscoveryInterval, "missingShardDiscoveryInterval is null");
        this.shardQueue = new MissingShardsQueue(new PrioritizedFifoExecutor<MissingShardRunnable>(this.executorService, recoveryThreads, new MissingShardComparator()));
        this.stats = new ShardRecoveryStats();
    }

    @PostConstruct
    public void start() {
        if (this.backupStore.isEmpty()) {
            return;
        }
        if (this.started.compareAndSet(false, true)) {
            this.scheduleRecoverMissingShards();
        }
    }

    @PreDestroy
    public void shutdown() {
        this.executorService.shutdownNow();
        this.missingShardExecutor.shutdownNow();
    }

    private void scheduleRecoverMissingShards() {
        this.missingShardExecutor.scheduleWithFixedDelay(() -> {
            try {
                long interval = this.missingShardDiscoveryInterval.roundTo(TimeUnit.SECONDS);
                TimeUnit.SECONDS.sleep(ThreadLocalRandom.current().nextLong(1L, interval));
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            this.enqueueMissingShards();
        }, 0L, this.missingShardDiscoveryInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    @Managed
    public void recoverMissingShards() {
        this.missingShardExecutor.submit(this::enqueueMissingShards);
    }

    private synchronized void enqueueMissingShards() {
        try {
            for (ShardMetadata shard : this.getMissingShards()) {
                this.stats.incrementBackgroundShardRecovery();
                ListenableFuture<?> future = this.shardQueue.submit(new MissingShard(shard.getShardUuid(), shard.getCompressedSize(), shard.getXxhash64(), false));
                MoreFutures.addExceptionCallback(future, t -> log.warn(t, "Error recovering shard: %s", new Object[]{shard.getShardUuid()}));
            }
        }
        catch (Throwable t2) {
            log.error(t2, "Error creating shard recovery tasks");
        }
    }

    private Set<ShardMetadata> getMissingShards() {
        return this.shardManager.getNodeShards(this.nodeIdentifier).stream().filter(shard -> this.shardNeedsRecovery(shard.getShardUuid(), shard.getCompressedSize())).collect(Collectors.toSet());
    }

    private boolean shardNeedsRecovery(UUID shardUuid, long shardSize) {
        File storageFile = this.storageService.getStorageFile(shardUuid);
        return !storageFile.exists() || storageFile.length() != shardSize;
    }

    public Future<?> recoverShard(UUID shardUuid) throws ExecutionException {
        ShardMetadata shard = this.shardManager.getShard(shardUuid);
        if (shard == null) {
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Shard does not exist in database: " + shardUuid);
        }
        this.stats.incrementActiveShardRecovery();
        return this.shardQueue.submit(new MissingShard(shardUuid, shard.getCompressedSize(), shard.getXxhash64(), true));
    }

    @VisibleForTesting
    void restoreFromBackup(UUID shardUuid, long shardSize, OptionalLong shardXxhash64) {
        File storageFile = this.storageService.getStorageFile(shardUuid);
        if (!this.backupStore.get().shardExists(shardUuid)) {
            this.stats.incrementShardRecoveryBackupNotFound();
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_RECOVERY_ERROR, "No backup file found for shard: " + shardUuid);
        }
        if (storageFile.exists()) {
            if (!ShardRecoveryManager.isFileCorrupt(storageFile, shardSize, shardXxhash64)) {
                return;
            }
            this.stats.incrementCorruptLocalFile();
            this.quarantineFile(shardUuid, storageFile, "Local file is corrupt.");
        }
        File stagingFile = ShardRecoveryManager.temporarySuffix(this.storageService.getStagingFile(shardUuid));
        this.storageService.createParents(stagingFile);
        log.info("Copying shard %s from backup...", new Object[]{shardUuid});
        long start = System.nanoTime();
        try {
            this.backupStore.get().restoreShard(shardUuid, stagingFile);
        }
        catch (PrestoException e) {
            this.stats.incrementShardRecoveryFailure();
            stagingFile.delete();
            throw e;
        }
        Duration duration = Duration.nanosSince((long)start);
        DataSize size = DataSize.succinctBytes((long)stagingFile.length());
        DataSize rate = ShardRecoveryManager.dataRate(size, duration).succinct();
        this.stats.addShardRecoveryDataRate(rate, size, duration);
        log.info("Copied shard %s from backup in %s (%s at %s/s)", new Object[]{shardUuid, duration, size, rate});
        this.storageService.createParents(storageFile);
        try {
            Files.move(stagingFile.toPath(), storageFile.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        catch (FileAlreadyExistsException fileAlreadyExistsException) {
        }
        catch (IOException e) {
            this.stats.incrementShardRecoveryFailure();
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_RECOVERY_ERROR, "Failed to move shard: " + shardUuid, (Throwable)e);
        }
        finally {
            stagingFile.delete();
        }
        if (!storageFile.exists()) {
            this.stats.incrementShardRecoveryFailure();
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_RECOVERY_ERROR, "File does not exist after recovery: " + shardUuid);
        }
        if (ShardRecoveryManager.isFileCorrupt(storageFile, shardSize, shardXxhash64)) {
            this.stats.incrementShardRecoveryFailure();
            this.stats.incrementCorruptRecoveredFile();
            this.quarantineFile(shardUuid, storageFile, "Local file is corrupt after recovery.");
            throw new PrestoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_BACKUP_CORRUPTION, "Backup is corrupt after read: " + shardUuid);
        }
        this.stats.incrementShardRecoverySuccess();
    }

    private void quarantineFile(UUID shardUuid, File file, String message) {
        File quarantine = new File(this.storageService.getQuarantineFile(shardUuid).getPath() + ".corrupt");
        if (quarantine.exists()) {
            log.warn("%s Quarantine already exists: %s", new Object[]{message, quarantine});
            return;
        }
        log.error("%s Quarantining corrupt file: %s", new Object[]{message, quarantine});
        try {
            Files.move(file.toPath(), quarantine.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            log.warn((Throwable)e, "Quarantine of corrupt file failed: " + quarantine);
            file.delete();
        }
    }

    private static boolean isFileCorrupt(File file, long size, OptionalLong xxhash64) {
        return file.length() != size || xxhash64.isPresent() && OrcStorageManager.xxhash64(file) != xxhash64.getAsLong();
    }

    static DataSize dataRate(DataSize size, Duration duration) {
        double rate = (double)size.toBytes() / duration.getValue(TimeUnit.SECONDS);
        if (Double.isNaN(rate) || Double.isInfinite(rate)) {
            rate = 0.0;
        }
        return DataSize.succinctBytes((long)Math.round(rate));
    }

    private static File temporarySuffix(File file) {
        return new File(file.getPath() + ".tmp-" + UUID.randomUUID());
    }

    @Managed
    @Flatten
    public ShardRecoveryStats getStats() {
        return this.stats;
    }

    private static <T> FutureCallback<T> failureCallback(final Consumer<Throwable> callback) {
        return new FutureCallback<T>(){

            public void onSuccess(T result) {
            }

            public void onFailure(Throwable throwable) {
                callback.accept(throwable);
            }
        };
    }

    private class MissingShardsQueue {
        private final LoadingCache<MissingShard, ListenableFuture<?>> queuedMissingShards;

        public MissingShardsQueue(final PrioritizedFifoExecutor<MissingShardRunnable> shardRecoveryExecutor) {
            Objects.requireNonNull(shardRecoveryExecutor, "shardRecoveryExecutor is null");
            this.queuedMissingShards = CacheBuilder.newBuilder().build(new CacheLoader<MissingShard, ListenableFuture<?>>(){

                public ListenableFuture<?> load(MissingShard missingShard) {
                    MissingShardRecovery task = new MissingShardRecovery(missingShard.getShardUuid(), missingShard.getShardSize(), missingShard.getShardXxhash64(), missingShard.isActive());
                    ListenableFuture<?> future = shardRecoveryExecutor.submit(task);
                    future.addListener(() -> MissingShardsQueue.this.queuedMissingShards.invalidate((Object)missingShard), MoreExecutors.directExecutor());
                    return future;
                }
            });
        }

        public ListenableFuture<?> submit(MissingShard shard) throws ExecutionException {
            return (ListenableFuture)this.queuedMissingShards.get((Object)shard);
        }
    }

    private static final class MissingShard {
        private final UUID shardUuid;
        private final long shardSize;
        private final OptionalLong shardXxhash64;
        private final boolean active;

        public MissingShard(UUID shardUuid, long shardSize, OptionalLong shardXxhash64, boolean active) {
            this.shardUuid = Objects.requireNonNull(shardUuid, "shardUuid is null");
            this.shardSize = shardSize;
            this.shardXxhash64 = Objects.requireNonNull(shardXxhash64, "shardXxhash64 is null");
            this.active = active;
        }

        public UUID getShardUuid() {
            return this.shardUuid;
        }

        public long getShardSize() {
            return this.shardSize;
        }

        public OptionalLong getShardXxhash64() {
            return this.shardXxhash64;
        }

        public boolean isActive() {
            return this.active;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            MissingShard other = (MissingShard)o;
            return Objects.equals(this.active, other.active) && Objects.equals(this.shardUuid, other.shardUuid);
        }

        public int hashCode() {
            return Objects.hash(this.shardUuid, this.active);
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("shardUuid", (Object)this.shardUuid).add("active", this.active).toString();
        }
    }

    private class MissingShardRecovery
    implements MissingShardRunnable {
        private final UUID shardUuid;
        private final long shardSize;
        private final OptionalLong shardXxhash64;
        private final boolean active;

        public MissingShardRecovery(UUID shardUuid, long shardSize, OptionalLong shardXxhash64, boolean active) {
            this.shardUuid = Objects.requireNonNull(shardUuid, "shardUuid is null");
            this.shardSize = shardSize;
            this.shardXxhash64 = Objects.requireNonNull(shardXxhash64, "shardXxhash64 is null");
            this.active = active;
        }

        @Override
        public void run() {
            ShardRecoveryManager.this.restoreFromBackup(this.shardUuid, this.shardSize, this.shardXxhash64);
        }

        @Override
        public boolean isActive() {
            return this.active;
        }
    }

    static interface MissingShardRunnable
    extends Runnable {
        public boolean isActive();
    }

    @VisibleForTesting
    static class MissingShardComparator
    implements Comparator<MissingShardRunnable> {
        MissingShardComparator() {
        }

        @Override
        public int compare(MissingShardRunnable shard1, MissingShardRunnable shard2) {
            if (shard1.isActive() == shard2.isActive()) {
                return 0;
            }
            return shard1.isActive() ? -1 : 1;
        }
    }
}

