package io.privy.analytics

import io.ktor.resources.Resource
import io.privy.logging.PrivyLogger
import io.privy.network.ApiResult
import io.privy.network.AuthType
import io.privy.network.KtorWrapper
import me.tatarka.inject.annotations.Inject

public interface AnalyticsRepository {
  public suspend fun logEvent(event: AnalyticsEvent)
}

@Inject
public class RealAnalyticsRepository(
    private val ktorWrapper: KtorWrapper,
    private val privyLogger: PrivyLogger,
    private val clientAnalyticsIdRepository: ClientAnalyticsIdRepository,
) : AnalyticsRepository {
  override suspend fun logEvent(event: AnalyticsEvent) {
    val analyticsResponse: ApiResult<Unit> =
        ktorWrapper.postResult(
            resource = AnalyticsResource,
            body =
                event.requestBody(
                    clientAnalyticsId = clientAnalyticsIdRepository.loadClientAnalyticsId()),
            authType = AuthType.None,
        )

    privyLogger.internal("Logged ${event.name()} analytics event. Result: $analyticsResponse")
  }
}

@Resource("analytics_events") private data object AnalyticsResource

private fun AnalyticsEvent.name(): String {
  return when (this) {
    AnalyticsEvent.SdkInitialize -> "sdk_initialize"
  }
}

private fun AnalyticsEvent.requestBody(clientAnalyticsId: String): Map<String, String> {
  return mapOf("event_name" to this.name(), "client_id" to clientAnalyticsId)
}
