package io.privy.auth.jwt

import kotlinx.datetime.Clock
import kotlinx.datetime.Instant
import kotlinx.datetime.format
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.longOrNull
import kotlin.time.Duration.Companion.seconds

public data class Jwt(
  val header:  Map<String, JsonPrimitive>,
  val claims:  Map<String, JsonPrimitive>,
  val signature: String,
) {
  // By default, a session will be considered unauthenticated 60 seconds prior
  // to its token's expiration time. This is so we can eagerly re-authenticate
  // before the server would reject requests with a 401.
  private val expirationThreshold: Long = 60 * 1000

  public fun isExpired(): Boolean {
    return claims["exp"]?.longOrNull?.seconds?.let { expirationInSeconds ->
      // Add a 60 second padding to expiration
      val jwtExpirationInMilliseconds = expirationInSeconds.inWholeMilliseconds
      val expirationInMilliseconds = jwtExpirationInMilliseconds - expirationThreshold
      val currentTimeMilliseconds = Clock.System.now().toEpochMilliseconds()

      return expirationInMilliseconds < currentTimeMilliseconds
    } ?: kotlin.run {
      // If expiration couldn't be parsed from JWT, consider it invalid and expired
      true
    }
  }
}

